/*
 *  Abstract routines for MIXER control
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "mixer.h"
#include "control.h"

#define SND_MIXERS		(SND_CARDS * SND_MINOR_MIXERS)

static snd_kmixer_t *snd_mixers[ SND_MIXERS ] = { [ 0 ... SND_MIXERS ] = NULL };

MUTEX_DEFINE_STATIC( register );

snd_kmixer_channel_t *snd_mixer_oss_channel( snd_kmixer_t *mixer, int device )
{
  int idx;
  snd_kmixer_channel_t *channel;
  
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    channel = mixer -> channels[ idx ];
    if ( channel -> hw.ossdev == device )
      return channel;
  }
  return NULL;
}

static int snd_mixer_open( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_kmixer_file_t *mfile, *mfile1;
  snd_kmixer_t *mixer;

  if ( !(mixer = snd_mixers[ (cardnum * SND_MINOR_MIXERS) + device ]) ) return -ENODEV;
  mfile = (snd_kmixer_file_t *)snd_malloc( sizeof( snd_kmixer_file_t ) );
  if ( !mfile ) return -ENOMEM;
  MEMSET( mfile, 0, sizeof( snd_kmixer_file_t ) );
  SLEEP_PREPARE( mfile, change );
  mfile -> mixer = mixer;
  if ( !mfile -> mixer ) {
    snd_free( mfile, sizeof( snd_kmixer_file_t ) );
    return -ENOMEM;
  }
  file -> private_data = mfile;
  MOD_INC_USE_COUNT;
  mixer -> card -> use_inc();
  MUTEX_DOWN( mixer, ffile );
  if ( mixer -> ffile ) {
    for ( mfile1 = mixer -> ffile; mfile1 -> next; mfile1 = mfile1 -> next );
    mfile1 -> next = mfile;
  } else {
    mixer -> ffile = mfile;
  }
  MUTEX_UP( mixer, ffile );
  return 0;
}

static int snd_mixer_release( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_kmixer_file_t *mfile, *mfile1;
  snd_kmixer_t *mixer;

  if ( file -> private_data ) {
    mfile = (snd_kmixer_file_t *)file -> private_data;
    mixer = mfile -> mixer;
    MUTEX_DOWN( mixer, ffile );
    if ( mixer -> ffile == mfile ) {
      mixer -> ffile = mfile -> next;
    } else {
      for ( mfile1 = mixer -> ffile; mfile1 -> next != mfile; mfile1 = mfile1 -> next );
      mfile1 -> next = mfile -> next;
    }
    MUTEX_UP( mixer, ffile );
    snd_free( file -> private_data, sizeof( snd_kmixer_file_t ) );
    file -> private_data = NULL;
    mixer -> card -> use_dec();
  }
  MOD_DEC_USE_COUNT;
  return 0;
}

void snd_mixer_set_kernel_mute( snd_kmixer_t *mixer, unsigned int priority, unsigned short mute ) {
  int idx, mask, left, right;
  snd_kmixer_channel_t *channel;

  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    channel = mixer -> channels[ idx ];
    if ( channel -> hw.priority == priority ) {
      if ( !channel -> hw.stereo ) mute = SND_MIX_MUTE;
      if ( channel -> kmute != mute ) {
        mask = channel -> kmute = mute;
        mask |= channel -> umute;
        if ( mask != channel -> mute ) { 
          if ( channel -> hw.mute && channel -> hw.set_mute ) {
            channel -> hw.set_mute( mixer, channel, mask );
          } else {
            left = mask & SND_MIX_MUTE_LEFT ? 0 : channel -> uleft;
            right = mask & SND_MIX_MUTE_RIGHT ? 0 : channel -> uright;
            if ( channel -> left != left || channel -> right != right ) {
              if ( channel -> hw.set_volume_level )
                channel -> hw.set_volume_level( mixer, channel, left, right );
              channel -> left = left;
              channel -> right = right;
            }        
          }
        }
      }
    }
  }
}

static void snd_mixer_notify_change( snd_kmixer_file_t *mfile, unsigned int channel )
{
  unsigned long flags;
  snd_kmixer_file_t *mfile1;
  snd_kmixer_t *mixer;
  
  mixer = mfile -> mixer;
  channel = 1 << channel;
  for ( mfile1 = mixer -> ffile; mfile1; mfile1 = mfile1 -> next ) {
    if ( mfile1 != mfile ) {
      CLI( &flags );
      if ( mfile1 -> changes & channel ) {
        STI( &flags );
        continue;
      }
      mfile1 -> changes |= channel;
      STI( &flags );
      CLI( &flags );
      if ( GETLOCK( mfile1, change ) & SND_WK_SLEEP ) {
        GETLOCK( mfile1, change ) &= ~SND_WK_SLEEP;
        WAKEUP( mfile1, change );
      }
      STI( &flags );
    }
  }
}

static int snd_mixer_device_read( snd_kmixer_file_t *mfile, snd_mixer_channel_t *device, int space )
{
  unsigned long flags;
  snd_kmixer_t *mixer;
  snd_mixer_channel_t sdevice, *old_device = NULL;
  snd_kmixer_channel_t *channel;
  int tmp;

  mixer = mfile -> mixer;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, device, sizeof( *device ) ) ) return -EINVAL;
    MEMCPY_FROMFS( &sdevice, device, sizeof( sdevice ) );
    old_device = device;
    device = &sdevice;
  }
  tmp = device -> channel;
  MEMSET( device, 0, sizeof( snd_mixer_channel_t ) );
  device -> channel = tmp;
  if ( tmp >= mixer -> channels_count ) return -ENODEV;
  channel = mixer -> channels[ tmp ];
  CLI( &flags );
  device -> flags = 0;
  if ( channel -> record )
    device -> flags |= SND_MIXER_FLG_RECORD;
  if ( channel -> umute & SND_MIX_MUTE_LEFT )
    device -> flags |= SND_MIXER_FLG_MUTE_LEFT;
  if ( channel -> umute & SND_MIX_MUTE_RIGHT )
    device -> flags |= SND_MIXER_FLG_MUTE_RIGHT;
  if ( mfile -> exact ) {
    device -> left = channel -> uleft;
    device -> right = channel -> uright;
  } else {
    device -> left = channel -> aleft;
    device -> right = channel -> aright;
  }
  if ( channel -> hw.compute_dB ) {
    device -> left_dB = channel -> hw.compute_dB( mixer, channel, channel -> uleft );
    device -> right_dB = channel -> hw.compute_dB( mixer, channel, channel -> uright );
  } else {
    tmp = channel -> hw.max - channel -> hw.min;
    device -> left_dB = ((((channel -> hw.max_dB - channel -> hw.min_dB) * (channel -> uleft - channel -> hw.min)) + (tmp - 1)) / tmp) + channel -> hw.min_dB;
    device -> left_dB -= device -> left_dB % channel -> hw.step_dB;
    device -> right_dB = ((((channel -> hw.max_dB - channel -> hw.min_dB) * (channel -> uright - channel -> hw.min)) + (tmp - 1)) / tmp) + channel -> hw.min_dB;
    device -> right_dB -= device -> right_dB % channel -> hw.step_dB;
  }
  STI( &flags );
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_WRITE, old_device, sizeof( *old_device ) ) ) return -EINVAL;
    MEMCPY_TOFS( old_device, &sdevice, sizeof( sdevice ) );
  }  
  return 0;
}

static int snd_mixer_device_write( snd_kmixer_file_t *mfile, snd_mixer_channel_t *device, int space )
{
  unsigned int mask, left, right;
  snd_mixer_channel_t sdevice, *old_device;
  snd_kmixer_t *mixer;
  snd_kmixer_channel_t *channel;
  int force = 0, tmp, aleft, aright;

  mixer = mfile -> mixer;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, device, sizeof( *device ) ) ) return -EINVAL;
    MEMCPY_FROMFS( &sdevice, device, sizeof( sdevice ) );
    old_device = device;
    device = &sdevice;
  }
  if ( device -> flags & SND_MIXER_FLG_DECIBEL ) return -EIO;	/* not implemented yet */
  if ( device -> channel >= mixer -> channels_count ) return -ENODEV;
  channel = mixer -> channels[ device -> channel ];
  force = (device -> flags & SND_MIXER_FLG_FORCE) != 0;
  mask = 0;
  if ( device -> flags & SND_MIXER_FLG_RECORD ) mask = 1;
  if ( force || channel -> record != mask ) {
    if ( channel -> hw.set_record_source )
      channel -> hw.set_record_source( mixer, channel, mask );
    channel -> record = mask;
    snd_mixer_notify_change( mfile, device -> channel );
  }
  mask = 0;
  if ( channel -> hw.stereo ) {
    if ( device -> flags & SND_MIXER_FLG_MUTE_LEFT ) mask |= SND_MIX_MUTE_LEFT;
    if ( device -> flags & SND_MIXER_FLG_MUTE_RIGHT ) mask |= SND_MIX_MUTE_RIGHT;
  } else {
    if ( device -> flags & SND_MIXER_FLG_MUTE ) mask = SND_MIX_MUTE;
  }
  left = channel -> uleft;
  right = channel -> uright;
  if ( force || channel -> umute != mask ) {
    channel -> umute = mask;
    if ( channel -> hw.mute ) {
      mask |= channel -> kmute;
      channel -> mute = mask;
      if ( channel -> hw.set_mute ) {
        channel -> hw.set_mute( mixer, channel, mask );
      } else {
        if ( mask & SND_MIX_MUTE_LEFT ) left = 0;
        if ( mask & SND_MIX_MUTE_RIGHT ) right = 0;
        if ( force || channel -> left != left || channel -> right != right ) {
          if ( channel -> hw.set_volume_level )
            channel -> hw.set_volume_level( mixer, channel, left, right );
          left = channel -> left;
          right = channel -> right;
        }
      }
      snd_mixer_notify_change( mfile, device -> channel );
    }
  }
  if ( !(device -> flags & SND_MIXER_FLG_DECIBEL) ) {
    left = device -> left;
    right = device -> right;
  } else {
    if ( device -> left_dB < channel -> hw.min_dB ) device -> left_dB = channel -> hw.min_dB;
    if ( device -> right_dB < channel -> hw.min_dB ) device -> right_dB = channel -> hw.min_dB;
    if ( device -> left_dB > channel -> hw.max_dB ) device -> left_dB = channel -> hw.max_dB;
    if ( device -> right_dB > channel -> hw.max_dB ) device -> right_dB = channel -> hw.max_dB;
    if ( channel -> hw.compute_linear ) {
      left = channel -> hw.compute_linear( mixer, channel, device -> left_dB );
      right = channel -> hw.compute_linear( mixer, channel, device -> right_dB );
    } else {
      tmp = channel -> hw.max_dB - channel -> hw.min_dB;
      device -> left_dB %= channel -> hw.step_dB;
      device -> right_dB %= channel -> hw.step_dB;
      left = ((((channel -> hw.max - channel -> hw.min) * (device -> left_dB - channel -> hw.min_dB)) + (tmp - 1)) / tmp) + channel -> hw.min;
      right = ((((channel -> hw.max - channel -> hw.min) * (device -> right_dB - channel -> hw.min_dB)) + (tmp - 1)) / tmp) + channel -> hw.min;
    }
  }  
  if ( mfile -> exact || (device -> flags & SND_MIXER_FLG_DECIBEL) ) {
    if ( left < channel -> hw.min ) left = channel -> hw.min;
    if ( right < channel -> hw.min ) right = channel -> hw.min;
    if ( left > channel -> hw.max ) left = channel -> hw.max;
    if ( right > channel -> hw.max ) right = channel -> hw.max;
    tmp = channel -> hw.max - channel -> hw.min;
    aleft = ( ( 100 * ( left - channel -> hw.min ) ) + ( tmp - 1 ) ) / tmp;
    aright = ( ( 100 * ( right - channel -> hw.min ) ) + ( tmp - 1 ) ) / tmp;
  } else {
    if ( left < 0 ) left = 0;
    if ( right < 0 ) right = 0;
    if ( left > 100 ) left = 100;
    if ( right > 100 ) right = 100;
    aleft = left;
    aright = right;
  }
  if ( force || aleft != channel -> aleft || aright != channel -> aright ) {
    channel -> aleft = aleft;
    channel -> aright = aright;
    if ( !mfile -> exact ) {
      left = ( ( channel -> hw.max * left ) + 99 ) / 100;
      right = ( ( channel -> hw.max * right ) + 99 ) / 100;
    }
    channel -> uleft = left;
    channel -> uright = right;
    if ( channel -> hw.mute && channel -> hw.set_mute == NULL ) {
      if ( channel -> mute & SND_MIX_MUTE_LEFT ) left = 0;
      if ( channel -> mute & SND_MIX_MUTE_RIGHT ) right = 0;
    }
    if ( force || channel -> left != left || channel -> right != right ) {
      channel -> left = left;
      channel -> right = right;
      if ( channel -> hw.set_volume_level )
        channel -> hw.set_volume_level( mixer, channel, left, right );
    }
    snd_mixer_notify_change( mfile, device -> channel );
  }
  return 0;
}

static int snd_mixer_special_read( snd_kmixer_t *mixer, struct snd_mixer_special *special, int space )
{
  int err = 0;
  struct snd_mixer_special sspecial, *old_special = NULL;
  
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, special, sizeof( *special ) ) ) return -EINVAL;
    MEMCPY_FROMFS( &sspecial, old_special = special, sizeof( sspecial ) );
    special = &sspecial;
  }
  if ( special -> what == SND_MIXER_S_NONE )
    return 0;		/* can be used for feature testing */
  if ( mixer -> hw.get_special )
    err = mixer -> hw.get_special( mixer, special );
   else
    err = -ENXIO;
  if ( !err && space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_WRITE, old_special, sizeof( *special ) ) ) return -EINVAL;
    MEMCPY_TOFS( old_special, special, sizeof( sspecial ) );
  }
  return err;
}

static int snd_mixer_special_write( snd_kmixer_t *mixer, struct snd_mixer_special *special, int space )
{
  int err = 0;
  struct snd_mixer_special sspecial;
  
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, special, sizeof( *special ) ) ) return -EINVAL;
    MEMCPY_FROMFS( &sspecial, special, sizeof( sspecial ) );
    special = &sspecial;
  }
  if ( special -> what == SND_MIXER_S_NONE )
    return 0;		/* can be used for feature testing */
  if ( mixer -> hw.get_special )
    err = mixer -> hw.get_special( mixer, special );
   else
    err = -ENXIO;
  return err;
}

static int snd_mixer_info( snd_kmixer_t *mixer, snd_mixer_info_t *info, int space )
{
  snd_mixer_info_t sinfo;
  
  MEMSET( &sinfo, 0, sizeof( snd_mixer_info_t ) );
  sinfo.type = mixer -> card -> type;
  sinfo.channels = mixer -> channels_count;
  sinfo.caps = mixer -> hw.caps;
  strncpy( sinfo.id, mixer -> id, sizeof( sinfo.id ) );
  strncpy( sinfo.name, mixer -> info, sizeof( sinfo.name ) );
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_WRITE, info, sizeof( *info ) ) ) return -EINVAL;
    MEMCPY_TOFS( info, &sinfo, sizeof( sinfo ) );
  }  
  return 0;
}

static int snd_mixer_device_info( snd_kmixer_file_t *mfile, snd_mixer_channel_info_t *info, int space )
{
  snd_kmixer_t *mixer;
  snd_mixer_channel_info_t sinfo, *old_info = NULL;
  snd_kmixer_channel_t *channel;
  int channel_idx, i;

  mixer = mfile -> mixer;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_READ, info, sizeof( *info ) ) ) return -EINVAL;
    MEMCPY_FROMFS( &sinfo, info, sizeof( sinfo ) );
    old_info = info;
    info = &sinfo;
  }
  channel_idx = info -> channel;
  MEMSET( info, 0, sizeof( snd_mixer_channel_info_t ) );
  info -> channel = channel_idx;
  if ( channel_idx >= mixer -> channels_count ) return -ENODEV;
  channel = mixer -> channels[ channel_idx ];  
  strncpy( info -> name, channel -> hw.name, sizeof( sinfo.name ) );
  info -> name[ sizeof( sinfo.name ) - 1 ] = 0;
  if ( channel -> hw.parent_priority != SND_MIXER_PRI_PARENT ) {
    for ( i = 0; i < mixer -> channels_count; i++ ) {
      if ( mixer -> channels[ i ] -> hw.priority == channel -> hw.parent_priority ) {
        info -> parent = i;
        break;
      }
    }
    if ( i >= mixer -> channels_count ) {
      PRINTD( "Oops... Parent 0x%x not found!!!\n", channel -> hw.parent_priority );
      info -> parent = SND_MIXER_PRI_PARENT;
    }
  } else {
    info -> parent = SND_MIXER_PRI_PARENT;
  }
  info -> caps = 0;
  if ( channel -> hw.record )
    info -> caps |= SND_MIXER_CINFO_CAP_RECORD;
  if ( channel -> hw.stereo )
    info -> caps |= SND_MIXER_CINFO_CAP_STEREO;
  info -> caps |= SND_MIXER_CINFO_CAP_MUTE;	/* mute is always emulated */
  if ( channel -> hw.mute )
    info -> caps |= SND_MIXER_CINFO_CAP_HWMUTE;
  if ( channel -> hw.digital )
    info -> caps |= SND_MIXER_CINFO_CAP_DIGITAL;
  if ( channel -> hw.input )
    info -> caps |= SND_MIXER_CINFO_CAP_INPUT;
  if ( mfile -> exact ) {
    info -> min = channel -> hw.min;
    info -> max = channel -> hw.max;
  } else {
    info -> min = 0;
    info -> max = 100;
  }
  info -> min_dB = channel -> hw.min_dB;
  info -> max_dB = channel -> hw.max_dB;
  info -> step_dB = channel -> hw.step_dB;
  if ( space == SND_SP_USER ) {
    if ( VERIFY_AREA( VERIFY_WRITE, old_info, sizeof( *old_info ) ) ) return -EINVAL;
    MEMCPY_TOFS( old_info, &sinfo, sizeof( sinfo ) );
  }  
  return 0;
}

static int snd_mixer_set_recsrc( snd_kmixer_file_t *mfile, int recsrc )
{
  int idx, value;
  snd_kmixer_t *mixer;
  snd_kmixer_channel_t *channel;

  mixer = mfile -> mixer;
  for ( idx = 0; idx < SND_MIXER_OSSDEVS; idx++ ) {
    channel = snd_mixer_oss_channel( mixer, idx );
    if ( channel == NULL ) continue;
    value = (recsrc & (1 << idx)) ? 1 : 0;
    if ( channel -> record != value ) {
      channel -> record = value;
      if ( channel -> hw.record && channel -> hw.set_record_source )
        channel -> hw.set_record_source( mixer, channel, value );
      snd_mixer_notify_change( mfile, channel -> channel );
    }
  }
  return 0;
}

static int snd_mixer_devmask( snd_kmixer_t *mixer )
{
  int result, idx, ossdev;
  
  result = 0;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    ossdev = mixer -> channels[ idx ] -> hw.ossdev;
    if ( ossdev == SND_MIXER_UNKNOWN ) continue;
    result |= (1 << ossdev);
  }
  return result;
}

static int snd_mixer_stereodevs( snd_kmixer_t *mixer )
{
  int result, idx;
  snd_kmixer_channel_t *channel;
  
  result = 0;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    channel = mixer -> channels[ idx ];
    if ( channel -> hw.ossdev == SND_MIXER_UNKNOWN ) continue;
    if ( channel -> hw.stereo )
      result |= (1 << channel -> hw.ossdev);
  }
  return result;
}

static int snd_mixer_recmask( snd_kmixer_t *mixer )
{
  int result, idx;
  snd_kmixer_channel_t *channel;
  
  result = 0;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    channel = mixer -> channels[ idx ];
    if ( channel -> hw.ossdev == SND_MIXER_UNKNOWN ) continue;
    if ( channel -> hw.record )
      result |= (1 << channel -> hw.ossdev);
  }
  return result;
}

static int snd_mixer_get_recsrc( snd_kmixer_t *mixer )
{
  int result, idx;
  snd_kmixer_channel_t *channel;
  
  result = 0;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    channel = mixer -> channels[ idx ];
    if ( channel -> hw.ossdev == SND_MIXER_UNKNOWN ) continue;
    if ( channel -> record )
      result |= (1 << channel -> hw.ossdev);
  }
  return result;
}

static int snd_mixer_osscaps( snd_kmixer_t *mixer )
{
  int result;

  result = 0;
  if ( mixer -> hw.caps & SND_MIXER_INFO_CAP_EXCL_RECORD )
    result |= SND_MIXER_CAP_EXCL_INPUT;
  return result;
}

static int snd_mixer_set_volume( snd_kmixer_file_t *mfile, int oss_device, int volume )
{
  int err;
  snd_kmixer_t *mixer;
  snd_kmixer_channel_t *channel;
  snd_mixer_channel_t device;

  mixer = mfile -> mixer;
  channel = snd_mixer_oss_channel( mixer, oss_device );
  if ( !channel ) return -ENXIO;
  device.channel = channel -> channel;
  if ( (err = snd_mixer_device_read( mfile, &device, SND_SP_KERNEL )) < 0 )
    return err;
  device.left = volume & 0xff;
  device.right = (volume >> 8) & 0xff;
  if ( (err = snd_mixer_device_write( mfile, &device, SND_SP_KERNEL )) < 0 )
    return err;
  return (unsigned short)device.left |
         ((unsigned short)device.right << 8);
}

static int snd_mixer_get_volume( snd_kmixer_file_t *mfile, int oss_device )
{
  int err;
  snd_kmixer_t *mixer;
  snd_kmixer_channel_t *channel;
  snd_mixer_channel_t device;

  mixer = mfile -> mixer;
  channel = snd_mixer_oss_channel( mixer, oss_device );
  if ( !channel ) return -ENXIO;
  device.channel = channel -> channel;
  if ( (err = snd_mixer_device_read( mfile, &device, SND_SP_KERNEL )) < 0 )
    return err;
  return (unsigned short)device.left |
         ((unsigned short)device.right << 8);
}

int snd_mixer_ioctl_card( snd_card_t *card, struct file *file, unsigned int cmd, unsigned long arg )
{
  snd_kmixer_file_t *mfile, sfile;
  snd_kmixer_t *mixer;
  int tmp;

  if ( card ) {
    /* note: this finds only first mixer for card */
    for ( tmp = 0; tmp < SND_MIXERS; tmp++ ) {
      mixer = snd_mixers[ tmp ];
      if ( mixer && mixer -> card == card ) break;
    }
    if ( !mixer ) return -ENXIO;
    MEMSET( &sfile, 0, sizeof( sfile ) );
    sfile.mixer = mixer;
    mfile = &sfile;
  } else {
    mfile = (snd_kmixer_file_t *)file -> private_data;
    if ( !mfile ) return -EIO;
    mixer = mfile -> mixer;
  }
  if ( ( ( cmd >> 8 ) & 0xff ) == 'R' ) {
    switch ( cmd ) {
      case SND_MIXER_IOCTL_PVERSION:
        return SND_IOCTL_OUT( arg, SND_MIXER_VERSION );
      case SND_MIXER_IOCTL_CHANNELS:
        return SND_IOCTL_OUT( arg, mixer -> channels_count );
      case SND_MIXER_IOCTL_INFO:
        return snd_mixer_info( mixer, (snd_mixer_info_t *)arg, SND_SP_USER );
      case SND_MIXER_IOCTL_EXACT:
        tmp = SND_IOCTL_IN( arg );
        mfile -> exact = tmp != 0;
        SND_IOCTL_OUT( arg, mfile -> exact );
        return 0;
      case SND_MIXER_IOCTL_CHANNEL_INFO:
        return snd_mixer_device_info( mfile, (snd_mixer_channel_info_t *)arg, SND_SP_USER );
      case SND_MIXER_IOCTL_CHANNEL_READ:
        return snd_mixer_device_read( mfile, (snd_mixer_channel_t *)arg, SND_SP_USER );
      case SND_MIXER_IOCTL_CHANNEL_WRITE:
        return snd_mixer_device_write( mfile, (snd_mixer_channel_t *)arg, SND_SP_USER );
      case SND_MIXER_IOCTL_SPECIAL_READ:
        return snd_mixer_special_read( mixer, (struct snd_mixer_special *)arg, SND_SP_USER );
      case SND_MIXER_IOCTL_SPECIAL_WRITE:
        return snd_mixer_special_write( mixer, (struct snd_mixer_special *)arg, SND_SP_USER );
    }
    return -ENXIO;
  }
  if ( ( ( cmd >> 8 ) & 0xff ) != 'M' ) return -ENXIO;
#if 0
  PRINTK( ">> mixer ioctl - cmd = 0x%x\n", cmd );
#endif
  if ( cmd == SND_MIXER_OSS_INFO )
    {
      struct snd_oss_mixer_info info;

      strncpy( info.id, mixer -> id, sizeof( info.id ) );
      strncpy( info.name, mixer -> info, sizeof( info.name ) );
      if ( VERIFY_AREA( VERIFY_WRITE, (void *)arg, sizeof( snd_mixer_info ) ) ) return -EIO;
      MEMCPY_TOFS( (void *)arg, &snd_mixer_info, sizeof( snd_mixer_info ) );
      return 0;
    }
  if ( cmd & IOC_IN )
    switch ( cmd ) {
      case SND_MIXER_SET_RECSRC:
        return SND_IOCTL_OUT( arg, snd_mixer_set_recsrc( mfile, SND_IOCTL_IN( arg ) ) );
      default:
        return SND_IOCTL_OUT( arg, snd_mixer_set_volume( mfile, cmd & 0xff, SND_IOCTL_IN( arg ) ) );
    }
   else
  if ( cmd & IOC_OUT )
    switch ( cmd ) {
      case SND_MIXER_DEVMASK:
        return SND_IOCTL_OUT( arg, snd_mixer_devmask( mixer ) );
      case SND_MIXER_STEREODEVS:
        return SND_IOCTL_OUT( arg, snd_mixer_stereodevs( mixer ) );
      case SND_MIXER_RECMASK:
        return SND_IOCTL_OUT( arg, snd_mixer_recmask( mixer ) );
      case SND_MIXER_CAPS:
        return SND_IOCTL_OUT( arg, snd_mixer_osscaps( mixer ) );
      case SND_MIXER_RECSRC:
        return SND_IOCTL_OUT( arg, snd_mixer_get_recsrc( mixer ) );
      default:
        return SND_IOCTL_OUT( arg, snd_mixer_get_volume( mfile, cmd & 0xff ) );
    }
#ifdef SNDCFG_DEBUG
  PRINTK( "MIXER ERR\n" );
  PRINTK( "  MIXER REQUEST: 0x%x, 0x%x\n", cmd, (int)SND_IOCTL_IN( arg ) );
#endif
  return -ENXIO;
}

static int snd_mixer_control_ioctl( snd_card_t *card, snd_control_t *control, unsigned int cmd, unsigned long arg )
{
  switch ( cmd ) {
    case SND_CTL_IOCTL_HW_INFO:
      {
        struct snd_ctl_hw_info *ptr = (struct snd_ctl_hw_info *)arg;
        if ( snd_mixers[ (card -> number << 1) + 1 ] ) ptr -> mixerdevs = 2; else
        if ( snd_mixers[ (card -> number << 1) + 0 ] ) ptr -> mixerdevs = 1; else
        ptr -> mixerdevs = 0;
        return 0;
      }
    case SND_CTL_IOCTL_MIXER_DEVICE:
      {
        int val = SND_IOCTL_IN( arg );
        if ( val < 0 || val > 1 ) return -EINVAL;
        if ( !snd_mixers[ (card -> number << 1) + 1 ] && val > 1 ) return -EINVAL;
        if ( !snd_mixers[ (card -> number << 1) + 0 ] ) return -EINVAL;
        control -> mixer_device = val;
        return 0;
      }
    case SND_CTL_IOCTL_MIXER_INFO:
      return snd_mixer_info( snd_mixers[ (card -> number << 1) + control -> mixer_device ], (snd_mixer_info_t *)arg, SND_SP_USER );
  }
  return -EAGAIN;
}

static int snd_mixer_ioctl( struct file *file, unsigned int cmd, unsigned long arg )
{
  return snd_mixer_ioctl_card( NULL, file, cmd, arg );
}

static long snd_mixer_read( struct file *file, char *buffer, long count )
{
  unsigned long flags;
  unsigned int device;
  unsigned char buf[ 8 ] = { 0, 0, 0, 0, 0, 0, 0, 0 };
  snd_kmixer_file_t *mfile;
  long size = 0;

  mfile = (snd_kmixer_file_t *)file -> private_data;
  if ( !mfile ) return -EIO;
  if ( count < 8 ) return 0;
  for ( device = 0; device < mfile -> mixer -> channels_count; device++ ) {
    CLI( &flags );
    if ( mfile -> changes & (1 << device) ) {
      mfile -> changes &= ~(1 << device);
      STI( &flags );
      *(unsigned int *)&buf[ 4 ] = device;
      MEMCPY_TOFS( buffer, buf, 8 );
      buffer += 8;
      size += 8;
      count -= 8;
      if ( count < 8 ) return size;
    } else {
      STI( &flags );
    }
  }
  return size;
}

#ifdef SND_POLL
static unsigned int snd_mixer_poll( struct file *file, poll_table *wait )
{
  unsigned long flags;
  unsigned int mask;
  snd_kmixer_file_t *mfile;
  snd_kmixer_t *mixer;

  mfile = (snd_kmixer_file_t *)file -> private_data;
  if ( !mfile ) return -EIO;
  mixer = mfile -> mixer;
 
  CLI( &flags );
  GETLOCK( mfile, change ) |= SND_WK_SLEEP;
  SLEEP_POLL( file, mfile, change, wait );
  STI( &flags );
  
  mask = 0;
  if ( mfile -> changes )
    mask |= POLLIN | POLLRDNORM;

  return mask;
}
#else
static int snd_mixer_select( struct file *file, int sel_type, select_table *wait )
{
  unsigned long flags;
  snd_kmixer_file_t *mfile;
  snd_kmixer_t *mixer;

  mfile = (snd_kmixer_file_t *)file -> private_data;
  if ( !mfile ) return -EIO;  
  mixer = mfile -> mixer;

  switch ( sel_type ) {
    case SEL_IN:
      CLI( &flags );
      if ( !mfile -> changes )
        {
          GETLOCK( mfile, change ) |= SND_WK_SLEEP;
          SLEEPS( mfile, change, wait );
          STI( &flags );
          return 0;
        }
      GETLOCK( mfile, change ) &= ~SND_WK_SLEEP;
      STI( &flags );
      return 1;
    case SEL_OUT:
      break;
    case SEL_EX:
      break;
  }
  return 0;
}
#endif

/*
 *  REGISTRATION PART
 */

static snd_minor_t snd_mixer_reg = {
  "mixer",

  NULL,					/* unregister */

  NULL,					/* lseek */
  snd_mixer_read,			/* read */
  NULL,					/* write */
  snd_mixer_open,			/* open */
  snd_mixer_release,			/* release */
#ifdef SND_POLL
  snd_mixer_poll,			/* poll */
#else
  snd_mixer_select,			/* select */
#endif
  snd_mixer_ioctl,			/* ioctl */
  NULL					/* mmap */
};

snd_kmixer_t *snd_mixer_new( snd_card_t *card, char *id, char *info, unsigned short port )
{
  snd_kmixer_t *mixer;
  
  mixer = (snd_kmixer_t *)snd_malloc( sizeof( snd_kmixer_t ) );
  if ( !mixer ) return NULL;
  MEMSET( mixer, 0, sizeof( snd_kmixer_t ) );
  mixer -> card = card;
  if ( id ) {
    strncpy( mixer -> id, id, sizeof( mixer -> id ) - 1 );
    mixer -> id[ sizeof( mixer -> id ) - 1 ] = 0;
  }
  if ( info ) {
    strncpy( mixer -> info, info, sizeof( mixer -> info ) - 1 );
    mixer -> info[ sizeof( mixer -> info ) - 1 ] = 0;
  }
  mixer -> port = port;
  MUTEX_PREPARE( mixer, ffile );
  return mixer;
}

int snd_mixer_free( snd_kmixer_t *mixer )
{
  int idx;

  if ( !mixer ) return -EINVAL;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    snd_free( mixer -> channels[ idx ], sizeof( snd_kmixer_channel_t ) );
  }
  if ( mixer -> private_data && mixer -> private_free )
    mixer -> private_free( mixer -> private_data );
  snd_free( mixer, sizeof( snd_kmixer_t ) );
  return 0;
}

snd_kmixer_channel_t *snd_mixer_new_channel( snd_kmixer_t *mixer, struct snd_stru_mixer_channel_hw *hw )
{
  int idx, idx1, priority;
  snd_kmixer_channel_t *channel;

  MUTEX_DOWN( mixer, ffile );
  if ( !mixer || mixer -> channels_count >= SND_MIXER_CHANNELS ) {
    MUTEX_UP( mixer, ffile );
    return NULL;
  }
  channel = (snd_kmixer_channel_t *)snd_malloc( sizeof( snd_kmixer_channel_t ) );
  if ( !channel ) {
    MUTEX_UP( mixer, ffile );
    return NULL;
  }
  MEMSET( channel, 0, sizeof( snd_kmixer_channel_t ) );
  MEMCPY( &channel -> hw, hw, sizeof( struct snd_stru_mixer_channel_hw ) );
  priority = channel -> hw.priority;
  for ( idx = 0; idx < mixer -> channels_count; idx++ )
    if ( mixer -> channels[ idx ] -> hw.priority > priority ) break;
  for ( idx1 = mixer -> channels_count; idx1 > idx; idx1-- ) {
    mixer -> channels[ idx1 ] = mixer -> channels[ idx1 - 1 ];
  }
  mixer -> channels_count++;
  mixer -> channels[ idx ] = channel;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    mixer -> channels[ idx ] -> channel = idx;
  }
  MUTEX_UP( mixer, ffile );
  return channel;
}

void snd_mixer_reorder_channel( snd_kmixer_t *mixer, snd_kmixer_channel_t *channel )
{
  int idx, idx1;

  if ( !mixer || !channel ) return;
  MUTEX_DOWN( mixer, ffile );
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    if ( mixer -> channels[ idx ] == channel ) {
      for ( idx1 = idx; idx1 + 1 < mixer -> channels_count; idx1++ )
        mixer -> channels[ idx1 ] = mixer -> channels[ idx1 + 1 ];
      mixer -> channels_count--;
      break;
    }
  }
  for ( idx = 0; idx < mixer -> channels_count; idx++ )
    if ( mixer -> channels[ idx ] -> hw.priority > channel -> hw.priority ) break;
  for ( idx1 = mixer -> channels_count; idx1 > idx; idx1-- ) {
    mixer -> channels[ idx1 ] = mixer -> channels[ idx1 - 1 ];
  }
  mixer -> channels[ idx ] = channel;
  mixer -> channels_count++;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    mixer -> channels[ idx ] -> channel = idx;
  }
  MUTEX_UP( mixer, ffile );
}

int snd_mixer_register( snd_kmixer_t *mixer, int device )
{
  int idx, err, cardnum;

  if ( device < 0 || device > 1 ) return -EINVAL;
  if ( !mixer -> card ) return -EINVAL;
  idx = ((cardnum = mixer -> card -> number) * SND_MINOR_MIXERS) + device;
  MUTEX_DOWN_STATIC( register ); 
  if ( snd_mixers[ idx ] ) {
    MUTEX_UP_STATIC( register );
    return -EBUSY;
  }
  mixer -> device = device;
  snd_mixers[ idx ] = mixer;
  if ( (err = snd_register_minor( SND_MINOR_MIXER + (mixer -> card -> number * SND_MINOR_MIXERS) + device, &snd_mixer_reg )) < 0 ) {
    MUTEX_UP_STATIC( register );
    snd_mixers[ idx ] = NULL;
    return err;
  }
  if ( (idx & (SND_MINOR_MIXERS-1)) == 0 )
    snd_register_minor( (cardnum << 4) + SND_MINOR_OSS_MIXER, &snd_mixer_reg );
  if ( (idx & (SND_MINOR_MIXERS-1)) == 1 )
    snd_register_minor( (cardnum << 4) + SND_MINOR_OSS_MIXER1, &snd_mixer_reg );
  MUTEX_UP_STATIC( register );
  return 0;
}

int snd_mixer_unregister( snd_kmixer_t *mixer )
{
  int idx, cardnum;

  MUTEX_DOWN_STATIC( register );
  idx = (mixer -> card -> number * SND_MINOR_MIXERS) + mixer -> device;
  if ( snd_mixers[ idx ] != mixer ) {
    MUTEX_UP_STATIC( register );
    return -EINVAL;
  }
  snd_unregister_minor( SND_MINOR_MIXER + idx );
  if ( (idx & (SND_MINOR_MIXERS-1)) == 0 ) {
    cardnum = idx >> 1;
    snd_unregister_minor( (cardnum << 4) + SND_MINOR_OSS_MIXER );
  }
  if ( (idx & (SND_MINOR_MIXERS-1)) == 1 ) {
    cardnum = idx >> 1;
    snd_unregister_minor( (cardnum << 4) + SND_MINOR_OSS_MIXER1 );
  }
  snd_mixers[ idx ] = NULL;
  MUTEX_UP_STATIC( register );
  return snd_mixer_free( mixer );
}

/*
 *  INIT PART
 */

snd_kmixer_channel_t *snd_mixer_find_channel( snd_kmixer_t *mixer, unsigned int priority )
{
  int idx;
  snd_kmixer_channel_t *channel;

  if ( !mixer ) return NULL;
  for ( idx = 0; idx < mixer -> channels_count; idx++ ) {
    channel = mixer -> channels[ idx ];
    if ( channel -> hw.priority == priority )
      return channel;
  }
  return NULL;
}

int snd_mixer_channel_init( snd_kmixer_t *mixer, unsigned int priority, unsigned char left, unsigned char right, unsigned int flags )
{
  int err;
  snd_kmixer_file_t sfile, *mfile;
  snd_kmixer_channel_t *channel;
  snd_mixer_channel_t device;

  MEMSET( &sfile, 0, sizeof( sfile ) );
  sfile.mixer = mixer;
  mfile = &sfile;
  channel = snd_mixer_find_channel( mixer, priority );
  if ( !channel ) {
    PRINTD( "snd_mixer_channel_init: Oops, device with priority 0x%x not found!!!\n", priority );
    return -ENODEV;
  }
  device.channel = channel -> channel;
  device.flags = flags | SND_MIXER_FLG_FORCE;
  device.left = (unsigned char)left;
  device.right = (unsigned char)right;
  if ( (err = snd_mixer_device_write( mfile, &device, SND_SP_KERNEL )) < 0 ) {
    PRINTD( "snd_mixer_channel_init: Oops, can't setup device with priority 0x%x (%i)!!!\n", priority, err );
    return err;
  }
  return 0;
}

int init_module( void )
{
  snd_control_register_ioctl( snd_mixer_control_ioctl );
  return 0;
}

void cleanup_module( void )
{
  snd_control_unregister_ioctl( snd_mixer_control_ioctl );
}
