/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Memory allocation routines.
 */

#include "driver.h"

/*
 *  page allocation
 */

MUTEX_DEFINE_STATIC( memory );
MUTEX_DEFINE_STATIC( dma );
static long snd_pages;

#ifdef SNDCFG_DEBUG
static void snd_memory_debug1( void );
#endif

char *snd_malloc_pages( unsigned long size, int *res_pg, int dma )
{
  int pg;
  char *buf;
  
  for ( pg = 0; PAGE_SIZE * ( 1 << pg ) < size; pg++ );
#if LinuxVersionCode( 2, 1, 85 ) <= LINUX_VERSION_CODE
  buf = (char *)__get_free_pages( GFP_KERNEL | (dma ? GFP_DMA : 0), pg );
#else
  buf = (char *)__get_free_pages( GFP_KERNEL, pg, dma ? MAX_DMA_ADDRESS : 0 );
#endif
#if 0 
  PRINTK( "pg = %i, buf = 0x%x\n", pg, (int)buf );
#endif
  if ( buf )
    {
      int page, last_page;
      
      page = MAP_NR( buf );
      last_page = page + ( 1 << pg );
      while ( page < last_page )
        set_bit( PG_reserved, &mem_map[ page++ ].flags );
      snd_pages += 1 << pg;
    }
  *res_pg = pg;
  return buf;
}

void snd_free_pages( char *ptr, unsigned long size )
{
  int pg, page, last_page;

  for ( pg = 0; PAGE_SIZE * ( 1 << pg ) < size; pg++ );
  page = MAP_NR( ptr );
  last_page = page + ( 1 << pg );
  while ( page < last_page )
    clear_bit( PG_reserved, &mem_map[ page++ ].flags );
  free_pages( (unsigned long)ptr, pg );
  snd_pages -= 1 << pg; 
}

/*
 *  ok.. new allocation routines..
 */

#define BLOCKS_IN_PAGE		(PAGE_SIZE / 32)
#define HEADER_SIZE		sizeof( struct alloc_header )
#define FREE_SPACE		(PAGE_SIZE - HEADER_SIZE)
#define ALIGN_VALUE( x )	( ( (x) + ( sizeof( void * ) - 1 ) ) & ~( sizeof( void * ) - 1 ) )

struct alloc_header {
  struct alloc_header *next;			/* next page */
  unsigned short sizes[ BLOCKS_IN_PAGE ];	/* size of block */
};

static struct alloc_header *snd_alloc_first_page;

#ifdef SNDCFG_DEBUG_MEMORY
static void *sndmem_alloc_ptr[ 1024 ];
static unsigned long sndmem_alloc_size[ 1024 ];
#endif

void snd_malloc_init( void )
{
  snd_pages = 0;
  snd_alloc_first_page = NULL;
#ifdef SNDCFG_DEBUG_MEMORY
  memset( sndmem_alloc_ptr, 0, sizeof( sndmem_alloc_ptr ) );
#endif
}

void snd_malloc_done( void )
{
  struct alloc_header *hdr, *next;

  if ( ( hdr = snd_alloc_first_page ) != NULL )
    {
#ifdef SNDCFG_DEBUG
      PRINTK( "snd: malloc done - trouble\n" );
      snd_memory_debug1();
#endif
      while ( hdr )
        {
          next = hdr -> next;
          snd_free_pages( (char *)hdr, PAGE_SIZE );
          hdr = next;
        }
    }    
#ifdef SNDCFG_DEBUG
  if ( snd_pages > 0 )
    PRINTK( "snd: malloc done - snd_pages = %li\n", snd_pages );
#endif
}

void *snd_malloc( unsigned long size )
{
  int pg;

  if ( !size ) return NULL;
#if 0
  PRINTK( "mallocccccccccccccccc %li\n", size );
#endif
  if ( size > FREE_SPACE )
    {
      void *res;
      
      res = snd_malloc_pages( size, &pg, 0 );
#ifdef SNDCFG_DEBUG_MEMORY
      {
        int idx;

        for ( idx = 0; idx < 1024; idx++ )
          if ( !sndmem_alloc_ptr[ idx ] )
            {
              sndmem_alloc_ptr[ idx ] = res;
              sndmem_alloc_size[ idx ] = size;
              break;
            }
      }
#endif
      return res;
    }
   else
    {
      struct alloc_header *hdr;
      struct alloc_header *phdr = NULL;

      MUTEX_DOWN_STATIC( memory );      
      hdr = snd_alloc_first_page;
      size = ALIGN_VALUE( size );
      while ( hdr )
        {
          int idx;
          unsigned short used;
          
          for ( idx = 0, used = 0; idx < BLOCKS_IN_PAGE; idx++ )
            {
              unsigned short bused;
              unsigned short bsize;
              
              bsize = hdr -> sizes[ idx ];
              bused = bsize & 0x8000;
              bsize &= 0x7fff;
              if ( ( !bsize && ( used + HEADER_SIZE + size <= PAGE_SIZE ) ) || 
                   ( !bused && bsize >= size ) )  /* free & passed */
                {
                  if ( bsize > size )
                    {
                      int i;
                      if ( hdr -> sizes[ BLOCKS_IN_PAGE - 1 ] & 0x8000 ) continue;	/* next block */
                      for ( i = BLOCKS_IN_PAGE - 1; i > idx + 1; i-- )
                        hdr -> sizes[ i ] = hdr -> sizes[ i - 1 ];
                      hdr -> sizes[ idx + 1 ] = bsize - size;
                    }
                  hdr -> sizes[ idx ] = 0x8000 | size;
                  MUTEX_UP_STATIC( memory );
                  return (void *)((char *)hdr + HEADER_SIZE + used);
                }
              used += bsize;
            }
          hdr = ( phdr = hdr ) -> next;
        }
      hdr = (struct alloc_header *)snd_malloc_pages( PAGE_SIZE, &pg, 0 );
      if ( !hdr ) {
        MUTEX_UP_STATIC( memory );
        return NULL;
      }
      memset( hdr, 0, HEADER_SIZE );
      if ( !phdr )
        snd_alloc_first_page = hdr; 
       else
        phdr -> next = hdr;
      hdr -> sizes[ 0 ] = 0x8000 | size;
      MUTEX_UP_STATIC( memory );
      return (void *)((char *)hdr + HEADER_SIZE);
    }
}

void snd_free( void *obj, unsigned long size )
{
  if ( !size ) return;
  if ( !obj )
    {
#ifdef SNDCFG_DEBUG_MEMORY
      PRINTK( "snd: snd_free - NULL?\n" ); 
#endif
      return;
    }
#if 0
  PRINTK( "freeeeeeeeeeee 0x%x, %li\n", obj, size );
#endif
  if ( size > FREE_SPACE )
    {
      snd_free_pages( obj, size );
#ifdef SNDCFG_DEBUG_MEMORY
      {
        int idx;
     
        for ( idx = 0; idx < 1024; idx++ )
          if ( sndmem_alloc_ptr[ idx ] == obj )
            {
              sndmem_alloc_ptr[ idx ] = NULL;
              return;
            }
        PRINTK( "snd: snd_free (1) - free of unallocated block at 0x%x size %li\n", (int)obj, size );
      }
#endif
    }
   else
    {
      struct alloc_header *hdr;
      struct alloc_header *phdr = NULL;	/* previous header */

      MUTEX_DOWN_STATIC( memory );
      hdr = snd_alloc_first_page;
      while ( hdr )
        {
          int idx;
          unsigned short used;
          
          for ( idx = 0, used = 0; hdr -> sizes[ idx ] != 0 && idx < BLOCKS_IN_PAGE; idx++ )
            {
              if ( (char *)hdr + HEADER_SIZE + used == obj )
                {
                  hdr -> sizes[ idx ] &= ~0x8000;
                  if ( idx + 1 < BLOCKS_IN_PAGE && hdr -> sizes[ idx + 1 ] == 0 )
                    hdr -> sizes[ idx ] = 0;
                  __again:
                  for ( idx = 0, used = 0; hdr -> sizes[ idx ] != 0 && idx < BLOCKS_IN_PAGE; idx++ )
                    {
                      if ( idx + 1 < BLOCKS_IN_PAGE &&
                           !( hdr -> sizes[ idx ] & 0x8000 ) &&		/* join blocks */
                           !( hdr -> sizes[ idx + 1 ] & 0x8000 ) &&
                           hdr -> sizes[ idx + 1 ] != 0 )
                        {
                          int i;
                        
                          hdr -> sizes[ idx ] += hdr -> sizes[ idx + 1 ];
                          for ( i = idx + 1; hdr -> sizes[ i + 1 ] != 0 && i + 1 < BLOCKS_IN_PAGE; i++ )
                            hdr -> sizes[ i ] = hdr -> sizes[ i + 1 ];
			  hdr -> sizes[ i ] = 0;
			  goto __again;
                        }
                      if ( hdr -> sizes[ idx ] & 0x8000 ) used++;
                    }
                  if ( idx > 0 && !(hdr -> sizes[ idx - 1 ] & 0x8000) )
                    hdr -> sizes[ idx - 1 ] = 0;
                  if ( !used )
                    {
                      if ( phdr )
                        phdr -> next = hdr -> next;
                       else
                        snd_alloc_first_page = hdr -> next;
                      snd_free_pages( (char *)hdr, PAGE_SIZE );
                    }
                  MUTEX_UP_STATIC( memory );
                  return;		/* ok.. successfull */
                }
              used += hdr -> sizes[ idx ] & 0x7fff;
            }
          phdr = hdr;
          hdr = hdr -> next;
        }
#ifdef SNDCFG_DEBUG_MEMORY
      PRINTK( "snd: gus free (2) - free of unallocated block at 0x%x size %li\n", (int)obj, size );
#endif
      MUTEX_UP_STATIC( memory ); 
    }
}

char *snd_malloc_strdup( char *string, int space )
{
  int len;
  char *res;

  if ( !string ) return NULL;
  if ( space == SND_SP_USER )
    {
      for ( res = string, len = 0; get_fs_byte( res ) != 0; res++, len++ );
      if ( ( res = snd_malloc( ++len ) ) == NULL ) return NULL;
      MEMCPY_FROMFS( res, string, len );
    }
   else
    {
      len = strlen( string ) + 1;
      if ( ( res = snd_malloc( len ) ) == NULL ) return NULL;
      MEMCPY( res, string, len );
    }
  return res;
}

void snd_free_str( char *string )
{
  if ( string )
    snd_free( string, strlen( string ) + 1 );
}

void snd_memory_info( snd_info_buffer_t *buffer )
{
  snd_iprintf( buffer, "Driver using %i page%s (%li bytes) of kernel memory for data.\n",
          			snd_pages,
          			snd_pages > 1 ? "s" : "",
		          	snd_pages * PAGE_SIZE );
}

#ifdef SNDCFG_DEBUG_MEMORY
void snd_memory_debug( snd_info_buffer_t *buffer )
{
  int i, used, free, total_blocks, total_used;
  struct alloc_header *hdr;

  MUTEX_DOWN_STATIC( memory );
  snd_iprintf( buffer, "\n-----\n\n"
  		         "Memory allocation table:\n" );
  for ( i = used = 0; i < 1024; i++ )
    if ( sndmem_alloc_ptr[ i ] )
      {
        used++;
        snd_iprintf( buffer, "  %04i: 0x%08lx [%08i] = %8li\n",
      					i,
        				sndmem_alloc_ptr[ i ],
        				MAP_NR( sndmem_alloc_ptr[ i ] ),
        				sndmem_alloc_size[ i ] );
      }
  for ( hdr = snd_alloc_first_page; hdr; hdr = hdr -> next, i++ )
    snd_iprintf( buffer, "  %04i: 0x%08lx [%08i] = %8li (internal)\n", i, (long)hdr, MAP_NR( hdr ), PAGE_SIZE );
  if ( !used && !snd_alloc_first_page )
    snd_iprintf( buffer, "  -none-\n" );
  snd_iprintf( buffer, "\nMemory allocation table (internal):\n" );
  hdr = snd_alloc_first_page;
  if ( !hdr )
    snd_iprintf( buffer, "  -none-\n" );
   else
    {
      total_blocks = total_used = 0;
      while ( hdr )
        {
          snd_iprintf( buffer, "  page at 0x%x:\n", (int)hdr );
          for ( i = used = free = 0; i < BLOCKS_IN_PAGE; i++ )
            {
              unsigned short bsize, bused;
              
              bsize = hdr -> sizes[ i ];
              if ( !bsize ) continue;
              bused = bsize & 0x8000;
              bsize &= 0x7fff;
              total_blocks++;
              snd_iprintf( buffer, "    %04i: 0x%08x = %4li (%s)\n", i, (int)((char *)hdr + HEADER_SIZE + used), bsize, bused ? "allocated" : "free" );
              if ( !bused ) free += bsize; else total_used += bsize;
              used += bsize;
            }
          snd_iprintf( buffer, "    sum : size = %li, used = %li, free = %li, unused = %li\n", PAGE_SIZE - HEADER_SIZE, used - free, free, PAGE_SIZE - HEADER_SIZE - used );
          hdr = hdr -> next;
        }
      snd_iprintf( buffer, "    total: blocks = %i, used = %i, recomended block allocation = %i\n",
      			total_blocks, total_used,
      			PAGE_SIZE / ( total_used / total_blocks ) );
    }
  MUTEX_UP_STATIC( memory );
}
#endif

#ifdef SNDCFG_DEBUG
static void snd_memory_debug1( void )
{
  int i, used, free, total_blocks, total_used;
  struct alloc_header *hdr;

  MUTEX_DOWN_STATIC( memory );
  printk( "Memory allocation table:\n" );
  for ( i = used = 0; i < 1024; i++ )
    if ( sndmem_alloc_ptr[ i ] )
      {
        used++;
        printk( "  %04i: 0x%08lx [%08li] = %8li\n",
      					i,
        				(long)sndmem_alloc_ptr[ i ],
        				(long)MAP_NR( sndmem_alloc_ptr[ i ] ),
        				sndmem_alloc_size[ i ] );
      }
  for ( hdr = snd_alloc_first_page; hdr; hdr = hdr -> next, i++ )
    printk( "  %04i: 0x%08lx [%08li] = %8li (internal)\n", i, (long)hdr, (long)MAP_NR( hdr ), (long)PAGE_SIZE );
  if ( !used && !snd_alloc_first_page )
    printk( "  -none-\n" );
  printk( "Memory allocation table (internal):\n" );
  hdr = snd_alloc_first_page;
  if ( !hdr )
    printk( "  -none-\n" );
   else
    {
      total_blocks = total_used = 0;
      while ( hdr )
        {
          printk( "  page at 0x%x:\n", (int)hdr );
          for ( i = used = free = 0; i < BLOCKS_IN_PAGE; i++ )
            {
              unsigned short bsize, bused;
              
              bsize = hdr -> sizes[ i ];
              if ( !bsize ) continue;
              bused = bsize & 0x8000;
              bsize &= 0x7fff;
              total_blocks++;
              printk( "    %04i: 0x%08x = %4i (%s)\n", i, (int)((char *)hdr + HEADER_SIZE + used), bsize, bused ? "allocated" : "free" );
              if ( !bused ) free += bsize; else total_used += bsize;
              used += bsize;
            }
          printk( "    sum : size = %li, used = %i, free = %li, unused = %li\n", PAGE_SIZE - HEADER_SIZE, used - free, (long)free, PAGE_SIZE - HEADER_SIZE - used );
          hdr = hdr -> next;
        }
      printk( "    total: blocks = %i, used = %i, recomended block allocation = %li\n",
      			total_blocks, total_used,
      			(long)(PAGE_SIZE / ( total_used / total_blocks )) );
    }
  MUTEX_UP_STATIC( memory );
}
#endif

/*
 *  DMA allocation routines
 */

int snd_dma_malloc( snd_card_t *card, int idx, char *owner, int lock )
{
  struct snd_stru_dma *dma;
  unsigned char *buf;
  long size;
  int pg = 0;

  MUTEX_DOWN_STATIC( dma );
  dma = card -> dmas[ idx ];
  if ( dma -> used > 0 )
    {
      if ( dma -> lock && lock )
        {
#if 0
          PRINTD( "snd: snd_dma_malloc - trying allocate locked DMA channel\n" );
#endif
          MUTEX_UP_STATIC( dma );
          return -EBUSY;
        }
      if ( lock )
        {
          dma -> old_owner = dma -> owner;
          dma -> owner = owner;
          dma -> lock = 1;
        }
       else
        dma -> old_owner = owner;
      dma -> used++;
      MUTEX_UP_STATIC( dma );
      return 0;		/* ok... dma is allocated */
    }
  if ( dma -> buf )
    {
      if ( !dma -> static_alloc ) {
        if ( !dma -> mmaped ) {	/* oops, oops, oops... */
          PRINTD( "snd: snd_dma_malloc - already?\n" );
          MUTEX_UP_STATIC( dma );
          return -EBUSY;
        }
      } else {
        if ( !dma -> used ) {
          dma -> usize = dma -> ursize;
          if ( dma -> usize > dma -> size )
            dma -> usize = dma -> size;
        }
      }
      dma -> used++;
      dma -> lock = lock ? 1 : 0;
      dma -> owner = owner;
      MUTEX_UP_STATIC( dma );
      return 0;
    }
  if ( dma -> rsize < PAGE_SIZE ) dma -> rsize = PAGE_SIZE;
  buf = NULL;
  size = 64 * 1024L;
  if ( dma -> dma > 3 ) size <<= 1;
  if ( size > dma -> rsize ) size = dma -> rsize;
  while ( !buf && size >= PAGE_SIZE )
    {
      buf = snd_malloc_pages( size, &pg, 1 );
      if ( !buf ) size >>= 1;
    }
  if ( buf )
    {
      dma -> size = ( 1 << pg ) * PAGE_SIZE;
      dma -> usize = dma -> ursize;
      if ( dma -> usize > dma -> size )
        dma -> usize = dma -> size;
      dma -> buf = buf;
      if ( lock < 0 )
        {
          dma -> used = 0;
          dma -> static_alloc = 1;
          dma -> owner = NULL;
          dma -> lock = 0;
        }
       else
        {
          dma -> used++;
          dma -> static_alloc = 0;
          dma -> owner = owner;
          dma -> lock = lock ? 1 : 0;
        }
      dma -> mmaped = 0;
    }
  MUTEX_UP_STATIC( dma );
  return buf == NULL ? -ENOMEM : 0;
}

void snd_dma_free( snd_card_t *card, int idx, int lock )
{
  struct snd_stru_dma *dma;

  MUTEX_DOWN_STATIC( dma );
  dma = card -> dmas[ idx ];
  if ( lock >= 0 )
    {
      if ( dma -> used ) dma -> used--;
      if ( dma -> used )
        {
          if ( lock )
            {
              dma -> owner = dma -> old_owner;
              dma -> old_owner = NULL;
              dma -> lock = 0;
            }
           else
            dma -> old_owner = NULL;
          MUTEX_UP_STATIC( dma );
          return;
        }
    }
#ifdef SNDCFG_DEBUG
   else
    if ( dma -> used > 0 )
      PRINTK( "snd: snd_dma_free - static error?\n" );
#endif
  if ( !dma -> static_alloc )
    {
      if ( dma -> mmaped ) {
        MUTEX_UP_STATIC( dma ); 
        return;
      }
      if ( dma -> buf )
        {
          snd_free_pages( dma -> buf, dma -> size );
          dma -> buf = NULL;
        }
      dma -> size = dma -> usize = 0;
    }
   else
    {
      if ( lock < 0 )
        {
          snd_free_pages( dma -> buf, dma -> size );
          dma -> buf = NULL;
          dma -> static_alloc = 0;
          dma -> size = dma -> usize = 0;
        }
    }  
  dma -> lock = 0;
  dma -> owner = dma -> old_owner = NULL;
  MUTEX_UP_STATIC( dma ); 
}

void snd_dma_notify_vma_close( struct vm_area_struct *area )
{
  int idx, idx1;
  snd_card_t *card;
  struct snd_stru_dma *dma;

  for ( idx = 0; idx < snd_cards_limit; idx++ ) {
    if ( !(snd_cards_bitmap & (1 << idx)) ) continue;
    card = snd_cards[ idx ];
    if ( !card ) continue;
    for ( idx1 = 0; idx1 < SND_DMAS; idx1++ ) {
      dma = card -> dmas[ idx1 ];
      if ( !dma ) continue;
      if ( dma -> mmaped && dma -> vma == area ) {	/* yep! */
        dma -> vma = NULL;
        dma -> mmaped = 0;
        if ( !dma -> used && !dma -> static_alloc )
          snd_dma_free( card, idx1, -1 );
      }
    }
  }
}
