/*
 *  Initialization & resource registration routines
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 */

#include "driver.h"
#include "control.h"

int snd_cards_count;
unsigned int snd_cards_bitmap;
snd_card_t *snd_cards[ SND_CARDS ];

void snd_driver_init( void )
{
  snd_cards_count = 0;
  snd_cards_bitmap = 0;
  MEMSET( &snd_cards, 0, sizeof( snd_cards ) );
}

snd_card_t *snd_card_new( int idx, char *xid, char *name, void (*use_inc)( void ), void (*use_dec)( void ) )
{
  snd_card_t *card;
  
  if ( !use_inc || !use_dec ) return NULL;
  card = (snd_card_t *)snd_malloc( sizeof( snd_card_t ) );
  if ( !card ) return NULL;
  MEMSET( card, 0, sizeof( snd_card_t ) );
  if ( xid )
    strncpy( card -> id, xid, 8 );
  if ( name )
    strncpy( card -> name, name, sizeof( card -> name ) - 1 );
  if ( idx < 0 ) {
    for ( idx = 0; idx < snd_cards_limit; idx++ )
      if ( !(snd_cards_bitmap & (1 << idx)) ) break;
  } else {
    idx--;
  }
  if ( idx < 0 || idx >= snd_cards_limit ) {
    snd_free( card, sizeof( snd_card_t ) );
    return NULL;
  }
  card -> number = idx;
  if ( !card -> id[ 0 ] )
    sprintf( card -> id, "card%i", card -> number + 1 );
  card -> use_inc = use_inc;
  card -> use_dec = use_dec;
  MUTEX_PREPARE( card, control );
  return card;
}

int snd_card_free( snd_card_t *card )
{
  snd_unregister_interrupts( card );
  snd_unregister_dma_channels( card );
  snd_unregister_ioports( card );
  if ( card -> private_free )
    card -> private_free( card -> private );
  snd_free( card, sizeof( snd_card_t ) );
  return 0;
}

int snd_card_register( snd_card_t *card )
{
  unsigned long flags;
  
  CLI( &flags );
  snd_cards_bitmap |= 1 << card -> number;
  snd_cards[ card -> number ] = card;
  snd_cards_count++;
  STI( &flags );
  if ( snd_control_register( card -> number ) < 0 ) {
    PRINTD( "snd: unable to register control minors\n" );
    /* Not fatal error */
  }
  return 0;
}

int snd_card_unregister( snd_card_t *card )
{
  unsigned long flags;
  
  if ( !card ) return -EINVAL;
  if ( snd_control_unregister( card -> number ) < 0 ) {
    PRINTD( "snd: unable to unregister control minors\n" );
    /* Not fatal error */
  }
  CLI( &flags );
  snd_cards_bitmap &= ~(1 << card -> number);
  snd_cards[ card -> number ] = NULL;
  snd_cards_count--;
  STI( &flags );
  snd_card_free( card );
  return 0;
}

char *snd_get_id( snd_card_t *card, char *id )
{
  strncpy( id, card -> id, 8 );
  id[ 8 ] = 0;
  return id;
}

void snd_info_card( snd_info_buffer_t *buffer )
{
  unsigned long flags;
  int idx, count;
  snd_card_t *card;
  char id[ 9 ];
  
  snd_iprintf( buffer, "\nRegistered soundcards:\n" );
  for ( idx = count = 0; idx < SND_CARDS; idx++ ) {
    CLI( &flags );
    if ( snd_cards_bitmap & (1 << idx) ) {
      count++;
      card = snd_cards[ idx ];
      snd_iprintf( buffer, "  %i [%-8s]: %s\n", idx + 1, snd_get_id( card, id ), card -> name );
    }
    STI( &flags );
  }
  if ( !count ) {
    snd_iprintf( buffer, "  --- no soundcards registered ---\n" );
  }
}

/*
 *  Misc...
 */

void snd_delay( int loops )
{
  int i;

  while( loops-- > 0 )
      for ( i = 0; i < 16; i++ ) INB( 0x80 );
}      

/*
 *  I/O port registration
 */

int snd_register_ioport( snd_card_t *card, int port, int size, char *name )
{
  int idx;
  char *xname;
  unsigned long flags;
  struct snd_stru_port *pport;
  
  if ( !card ) return -EINVAL;
  for ( idx = 0; idx < SND_PORTS; idx++ ) {
    if ( card -> ports_map & (1 << idx) ) continue;
    xname = snd_malloc_strdup( name, SND_SP_KERNEL );
    CLI( &flags );
    if ( check_region( port, size ) ) {
      STI( &flags );
      snd_free_str( xname );
      return -EBUSY;
    }
    pport = (struct snd_stru_port *)snd_malloc( sizeof( struct snd_stru_port ) );
    if ( !pport ) {
      snd_free_str( xname );
      return -ENOMEM;
    }
    MEMSET( pport, 0, sizeof( *pport ) );
    request_region( port, size, xname );
    card -> ports_map |= 1 << idx;
    pport -> port = port;
    pport -> size = size;
    pport -> name = xname;
    card -> ports[ idx ] = pport;
    STI( &flags );
    return 0;
  }
  return -ENOMEM;
}

int snd_unregister_ioports( snd_card_t *card )
{
  int idx;
  struct snd_stru_port *pport;
  
  if ( !card ) return -EINVAL;
  for ( idx = 0; idx < SND_PORTS; idx++ ) {
    if ( card -> ports_map & (1 << idx) ) {
      pport = card -> ports[ idx ];
      if ( !pport ) { PRINTD( "unregister_ioports: ugh!!!\n" ); continue; }
      card -> ports[ idx ] = NULL;
      release_region( pport -> port, pport -> size );
      snd_free_str( pport -> name );
      snd_free( pport, sizeof( struct snd_stru_port ) );
    }
  }
  card -> ports_map = 0;
  return 0;
}

/*
 *  DMA channel registration
 */

int snd_register_dma_channel( snd_card_t *card, char *name, int number, int rsize, int *possible_numbers )
{
  int idx;
  struct snd_stru_dma *dmaptr;

  if ( !card ) return -EINVAL;
  if ( rsize == SND_AUTO_DMA_SIZE ) rsize = 128;
  if ( rsize < 4 || rsize > 128 ) return -EINVAL;
  if ( !possible_numbers ) return -EINVAL;
  if ( number == SND_AUTO_DMA ) {
    for ( ; ( number = *possible_numbers ) >= 0 ; possible_numbers++ ) {
      if ( request_dma( number, "snd test" ) ) continue;
      free_dma( number );
      break;
    }
    if ( number < 0 ) return -ENOMEM;
  } else {
    if ( *possible_numbers >= 0 ) {
      for ( ; *possible_numbers >= 0; possible_numbers++ )
        if ( *possible_numbers == number ) break;
      if ( *possible_numbers < 0 ) return -EINVAL;
    }
  }
  if ( number < 4 && rsize > 64 ) rsize = 64;
  if ( number < 0 || number > 7 ) return -EINVAL;
  for ( idx = 0; idx < SND_DMAS; idx++ ) {
    if ( card -> dmas_map & (1 << idx) ) continue;
    break;
  }
  if ( idx >= SND_DMAS ) return -ENOMEM;
  dmaptr = (struct snd_stru_dma *)snd_malloc( sizeof( struct snd_stru_dma ) );
  if ( !dmaptr ) return -ENOMEM;
  MEMSET( dmaptr, 0, sizeof( *dmaptr ) );
  dmaptr -> dma = number;
  dmaptr -> rsize = dmaptr -> ursize = rsize * 1024;
  dmaptr -> name = snd_malloc_strdup( name, SND_SP_KERNEL );
  if ( request_dma( number, dmaptr -> name ) ) {
    PRINTK( "snd: unable to grab DMA %i for %s\n", number, dmaptr -> name );
    snd_free_str( dmaptr -> name );
    dmaptr -> name = NULL;
    return -EBUSY;
  }
  card -> dmas[ idx ] = dmaptr;
  card -> dmas_map |= 1 << idx;
  return idx;
}

int snd_unregister_dma_channels( snd_card_t *card )
{
  int idx;
  struct snd_stru_dma *dmaptr;

  if ( !card ) return -EINVAL;
  for ( idx = 0; idx < SND_DMAS; idx++ ) {
    if ( !(card -> dmas_map & (1 << idx)) ) continue;
    dmaptr = card -> dmas[ idx ];
    card -> dmas[ idx ] = NULL;
    disable_dma( dmaptr -> dma );
    free_dma( dmaptr -> dma );
    snd_free_str( dmaptr -> name );
    snd_free( dmaptr, sizeof( struct snd_stru_dma ) );
  }
  card -> dmas_map = 0;
  return 0;
}

/*
 *  IRQ channel registration
 */

int snd_register_interrupt( snd_card_t *card, char *name, int number, snd_irq_handler_t *handler, void *dev_id, int *possible_numbers )
{
  int idx;
  char *xname;
  struct snd_stru_irq *irqptr;

  if ( !card ) return -EINVAL;
  if ( !possible_numbers ) return -EINVAL;
  if ( number == SND_AUTO_IRQ ) {
    for ( ; ( number = *possible_numbers ) >= 0; possible_numbers++ ) {
      if ( snd_request_irq( number, handler, SND_SA_FLAGS, "snd test", dev_id ) ) continue;
      snd_free_irq( number, dev_id );
      break;
    }
    if ( number < 0 ) return -ENOMEM;
  } else {
    if ( *possible_numbers >= 0 ) {
      for ( ; *possible_numbers >= 0; possible_numbers++ )
        if ( *possible_numbers == number ) break;
      if ( *possible_numbers < 0 ) return -EINVAL;
    }
  }
  if ( number < 0 || number > 15 ) return -EINVAL;
  for ( idx = 0; idx < SND_IRQS; idx++ ) {
    if ( card -> irqs_map & (1 << idx) ) continue;
    break;
  }
  if ( idx >= SND_IRQS ) return -ENOMEM;
  xname = snd_malloc_strdup( name, SND_SP_KERNEL );
  if ( !xname ) return -ENOMEM;
  irqptr = (struct snd_stru_irq *)snd_malloc( sizeof( struct snd_stru_irq ) );
  if ( !irqptr ) {
    snd_free_str( xname );
    return -ENOMEM;
  }
  MEMSET( irqptr, 0, sizeof( struct snd_stru_irq ) );
  if ( snd_request_irq( number, handler, SND_SA_FLAGS, xname, dev_id ) ) {
    PRINTK( "snd: unable to grab IRQ %i for %s\n", number, xname );
    snd_free_str( xname );
    return -EBUSY;
  }
  irqptr -> irq = number;
  irqptr -> name = xname;
  irqptr -> dev_id = dev_id;
  card -> irqs[ idx ] = irqptr;
  card -> irqs_map |= (1 << idx);
  return idx;
}

int snd_unregister_interrupts( snd_card_t *card )
{
  int idx;
  struct snd_stru_irq *irqptr;
  
  if ( !card ) return -EINVAL;
  for ( idx = 0; idx < SND_IRQS; idx++ ) {
    if ( card -> irqs_map & (1 << idx) ) {
      irqptr = card -> irqs[ idx ];
      disable_irq( irqptr -> irq );
      snd_free_irq( irqptr -> irq, irqptr -> dev_id );
      snd_free_str( irqptr -> name );
      snd_free( irqptr, sizeof( struct snd_stru_irq ) );
    }
  }
  card -> irqs_map = 0;
  return 0;
}
