/*
 *  Get information about ALSA driver
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 */

#include "driver.h"
#include "info.h"
#include <stdarg.h>

#ifdef SNDCFG_DEBUG_MEMORY
#define INFO_BUFFER_SIZE	( 700 * 1024 )
#else
#define INFO_BUFFER_SIZE	( 32 * 1024 )
#endif

struct snd_info_call {
  void *private_data;
  snd_info_call_t call;
  struct snd_info_call *next;
};

MUTEX_DEFINE_STATIC( info );

static struct snd_info_call *snd_info_first = NULL;

int snd_iprintf( snd_info_buffer_t *buffer, char *fmt, ... )
{
  va_list args;
  int res;
  char sbuffer[ 512 ];
  
  if ( buffer -> stop ) return 0;
  va_start( args, fmt );
  res = vsprintf( sbuffer, fmt, args );
  va_end( args );
  if ( buffer -> size + res >= INFO_BUFFER_SIZE )
    {
      buffer -> stop = 1;
      return 0;
    }
  strcpy( buffer -> curr, sbuffer );
  buffer -> curr += res;
  buffer -> size += res;
  return res;
}

static void snd_info_init( snd_info_buffer_t *buffer )
{
#ifdef MODULE_PARM
  static char *kernel_version = UTS_RELEASE;
#else
  extern char kernel_version[];
#endif
  struct snd_info_call *call;

  buffer -> curr = buffer -> buffer;
  *buffer -> buffer = 0;
  MUTEX_DOWN_STATIC( info );
  snd_iprintf( buffer,
                       "Advanced Linux Sound Architecture Driver Version " SND_VERSION ".\n"
  		       COPYRIGHT "\n"
                       "Compiled in " __DATE__ " for kernel %s"
#ifdef __SMP__
		       " (SMP)"
#endif
#ifdef MODVERSIONS
                       " with versioned symbols"
#endif
                       ".\n", kernel_version );

  snd_memory_info( buffer );
  snd_minor_info( buffer );
  snd_info_card( buffer );
  for ( call = snd_info_first; call; call = call -> next ) {
    call -> call( buffer, call -> private_data );
  }
#ifdef SNDCFG_DEBUG_MEMORY
  snd_memory_debug( buffer );
#endif
  MUTEX_UP_STATIC( info );
}

static long snd_info_read( struct file *file, char *buffer, long count )
{
  snd_info_buffer_t *buf;
  int size, size1;
  
  buf = (snd_info_buffer_t *)file -> private_data;
  if ( !buf ) return -EIO;
  if ( !buf -> size )
    snd_info_init( buf );
  if ( file -> f_pos >= buf -> size ) return 0;
  size = buf -> size < count ? buf -> size : count;
  size1 = buf -> size - file -> f_pos;
  if ( size1 < size ) size = size1;
  MEMCPY_TOFS( buffer, buf -> buffer + file -> f_pos, size );
  file -> f_pos += size;
  return size;
}

static int snd_info_open( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_info_buffer_t *buf;
  
  buf = (snd_info_buffer_t *)snd_malloc( sizeof( snd_info_buffer_t ) );
  if ( !buf ) return -ENOMEM;
  MEMSET( buf, 0, sizeof( snd_info_buffer_t ) ); 
  buf -> buffer = (char *)VMALLOC( INFO_BUFFER_SIZE );
  if ( !buf -> buffer ) {
    snd_free( buf, sizeof( snd_info_buffer_t ) );
    return -ENOMEM;
  }
  file -> private_data = buf;
  MOD_INC_USE_COUNT;
  return 0;
}

static int snd_info_release( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_info_buffer_t *buf;
  
  buf = (snd_info_buffer_t *)file -> private_data;
  file -> private_data = NULL;
  if ( buf ) {
    if ( buf -> buffer ) VFREE( buf -> buffer );
    snd_free( buf, sizeof( snd_info_buffer_t ) );
  }  
  MOD_DEC_USE_COUNT;
  return 0;
}

int snd_register_info( snd_info_call_t call, void *private_data )
{
  struct snd_info_call *ncall, *pcall;
  
  ncall = (struct snd_info_call *)snd_malloc( sizeof( struct snd_info_call ) );
  if ( !ncall ) return -ENOMEM;
  MEMSET( ncall, 0, sizeof( struct snd_info_call ) );
  ncall -> call = call;
  ncall -> private_data = private_data;
  MUTEX_DOWN_STATIC( info );
  if ( !snd_info_first ) {
    snd_info_first = ncall;
  } else {
    for ( pcall = snd_info_first; pcall -> next; pcall = pcall -> next ) ;
    pcall -> next = ncall;
  }
  MUTEX_UP_STATIC( info );
  return 0;
}

int snd_unregister_info( snd_info_call_t call, void *private_data )
{
  struct snd_info_call *p, *pn, *po;

  MUTEX_DOWN_STATIC( info );
  if ( !snd_info_first ) {
    MUTEX_UP_STATIC( info );
    return -EINVAL;
  }
  p = snd_info_first;
  if ( p -> call == call && p -> private_data == private_data ) {
    snd_info_first = p -> next;
    MUTEX_UP_STATIC( info );
    snd_free( p, sizeof( struct snd_info_call ) );
    return 0;
  }
  for ( ; p; p = pn ) {
    pn = p -> next;
    if ( pn && pn -> call == call && pn -> private_data == private_data ) {
      po = pn;
      p -> next = pn = pn -> next;
      MUTEX_UP_STATIC( info );
      snd_free( po, sizeof( struct snd_info_call ) );
      return 0;
    }
  }
  MUTEX_UP_STATIC( info );
  return -EINVAL;
}

static long snd_sndstat_read( struct file *file, char *buffer, long count )
{
  char *s = SNDSTAT_STRING;
  int size;

  size = strlen( s ) + 1;
  if ( file -> f_pos >= size || count < size - file -> f_pos )
    return 0;
  size -= file -> f_pos;
  MEMCPY_TOFS( buffer, s + file -> f_pos, size );
  file -> f_pos += size;
  return size;
}

static int snd_sndstat_open( unsigned short minor, int cardnum, int device, struct file *file )
{
  MOD_INC_USE_COUNT;
  return 0;
}

static int snd_sndstat_release( unsigned short minor, int cardnum, int device, struct file *file )
{
  MOD_DEC_USE_COUNT;
  return 0;
}

/*
 *
 */

static snd_minor_t snd_info_reg = {
  "sndinfo",

  NULL,					/* unregister */

  NULL,					/* lseek */
  snd_info_read,			/* read */
  NULL,					/* write */
  snd_info_open,			/* open */
  snd_info_release,			/* release */
  NULL,					/* poll or select */
  NULL,					/* ioctl */
  NULL					/* mmap */
};

static snd_minor_t snd_sndstat_reg = {
  "sndstat",

  NULL,					/* unregister */

  NULL,					/* lseek */
  snd_sndstat_read,			/* read */
  NULL,					/* write */
  snd_sndstat_open,			/* open */
  snd_sndstat_release,			/* release */
  NULL,					/* poll or select */
  NULL,					/* ioctl */
  NULL					/* mmap */
};

int snd_info_register( void )
{
  int err;

  if ( ( err = snd_register_minor( SND_MINOR_INFO, &snd_info_reg ) ) < 0 )
    return err;
  if ( ( err = snd_register_minor( SND_MINOR_OSS_SNDSTAT, &snd_sndstat_reg ) ) < 0 ) {
    snd_unregister_minor( SND_MINOR_INFO );
    return err;
  }
  return 0;
}
