/*
 *  Main detection routines
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include <stdarg.h>

#define SND_DETECT_VERSION	"1.0"
#define SND_DETECT_BUFFER_SIZE	(32 * 1024)

typedef int (*snd_isa_probe)( snd_card_t *card, char *name, unsigned int *type, unsigned short *port );

extern int snd_detect_interwave( snd_card_t *card, char *name, unsigned int *type, unsigned short *port );
extern int snd_detect_gus( snd_card_t *card, char *name, unsigned int *type, unsigned short *port );
extern int snd_detect_es1688( snd_card_t *card, char *name, unsigned int *type, unsigned short *port );
extern int snd_detect_sb( snd_card_t *card, char *name, unsigned int *type, unsigned short *port );

static snd_isa_probe snd_isa_probes[] = {
  snd_detect_es1688,
  snd_detect_interwave,
  snd_detect_gus,
  snd_detect_sb,
};

#define SND_DETECT_ISA_COUNT (sizeof(snd_isa_probes)/sizeof(void *))

static void snd_detect_use_inc( void )
{
  MOD_INC_USE_COUNT;
}

static void snd_detect_use_dec( void )
{
  MOD_DEC_USE_COUNT;
}

static int snd_detect_init( snd_info_buffer_t *buffer )
{
  int idx, cardidx;
  snd_card_t *cards[ SND_CARDS ];	/* detected cards */
  char name[64];
  unsigned int type;
  unsigned short port;

  buffer -> curr = buffer -> buffer;
  *buffer -> buffer = 0;
  snd_iprintf( buffer, "Version: " SND_DETECT_VERSION "\n" );
  if ( snd_cards_count > 0 ) {
    return -EBUSY;
  }
  MEMSET( &cards, 0, sizeof( cards ) );
  cardidx = 0;
  for ( idx = 0; idx < SND_CARDS; idx++ ) {
    cards[ idx ] = snd_card_new( idx + 1, NULL, NULL, snd_detect_use_inc, snd_detect_use_dec );
    if ( !cards[ idx ] ) {
      for ( idx = 0; idx < SND_CARDS; idx++ ) {
        if ( cards[ idx ] ) snd_card_free( cards[ idx ] );
      }
      return -ENOMEM;
    }
  }
  for ( idx = 0; cardidx < SND_CARDS && idx < SND_DETECT_ISA_COUNT; idx++ ) {
    while ( !snd_isa_probes[ idx ]( cards[ cardidx ], name, &type, &port ) ) {
      snd_iprintf( buffer, "ISA#0x%x#%s#0x%x\n", type, name, port );
      cardidx++;
    }
  }
  for ( idx = 0; idx < SND_CARDS; idx++ ) {
    if ( cards[ idx ] ) snd_card_free( cards[ idx ] );
  }
  return 0;
}

static long snd_detect_read( struct file *file, char *buffer, long count )
{
  snd_info_buffer_t *buf;
  int size, size1, err;
  
  buf = (snd_info_buffer_t *)file -> private_data;
  if ( !buf ) return -EIO;
  if ( !buf -> size ) {
    err = snd_detect_init( buf );
    if ( err < 0 ) return err;
  }
  if ( file -> f_pos >= buf -> size ) return 0;
  size = buf -> size < count ? buf -> size : count;
  size1 = buf -> size - file -> f_pos;
  if ( size1 < size ) size = size1;
  MEMCPY_TOFS( buffer, buf -> buffer + file -> f_pos, size );
  file -> f_pos += size;
  return size;
}

static int snd_detect_open( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_info_buffer_t *buf;
  
  buf = (snd_info_buffer_t *)snd_malloc( sizeof( snd_info_buffer_t ) );
  if ( !buf ) return -ENOMEM;
  MEMSET( buf, 0, sizeof( snd_info_buffer_t ) ); 
  buf -> buffer = (char *)VMALLOC( SND_DETECT_BUFFER_SIZE );
  if ( !buf -> buffer ) {
    snd_free( buf, sizeof( snd_info_buffer_t ) );
    return -ENOMEM;
  }
  file -> private_data = buf;
  MOD_INC_USE_COUNT;
  return 0;
}

static int snd_detect_release( unsigned short minor, int cardnum, int device, struct file *file )
{
  snd_info_buffer_t *buf;
  
  buf = (snd_info_buffer_t *)file -> private_data;
  file -> private_data = NULL;
  if ( buf ) {
    if ( buf -> buffer ) VFREE( buf -> buffer );
    snd_free( buf, sizeof( snd_info_buffer_t ) );
  }  
  MOD_DEC_USE_COUNT;
  return 0;
}

/*
 *
 */

static snd_minor_t snd_detect_reg = {
  "snddetect",

  NULL,					/* unregister */

  NULL,					/* lseek */
  snd_detect_read,			/* read */
  NULL,					/* write */
  snd_detect_open,			/* open */
  snd_detect_release,			/* release */
  NULL,					/* poll or select */
  NULL,					/* ioctl */
  NULL					/* mmap */
};

int init_module( void )
{
  int err;

  if ( ( err = snd_register_minor( SND_MINOR_DETECT, &snd_detect_reg ) ) < 0 )
    return err;
  return 0;
}

void cleanup_module( void )
{
  snd_unregister_minor( SND_MINOR_DETECT );
}
