/*
 *  Driver for SoundBlaster 16/AWE32/AWE64 soundcards
 *  Copyright (c) by Jaroslav Kysela <perex@jcu.cz>
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "pcm.h"
#include "mixer.h"
#include "sb.h"

int snd_index[ SND_CARDS ] = SND_DEFAULT_IDX;	/* Index 1-MAX */
char *snd_id[ SND_CARDS ] = SND_DEFAULT_STR;	/* ID for this card */
int snd_port[ SND_CARDS ] = SND_DEFAULT_PORT;	/* 0x220,0x240,0x260 */
int snd_irq[ SND_CARDS ] = SND_DEFAULT_IRQ;	/* 5,7,9,10 */
int snd_dma8[ SND_CARDS ] = SND_DEFAULT_DMA;	/* 0,1,3 */
int snd_dma8_size[ SND_CARDS ] = SND_DEFAULT_DMA_SIZE; /* 8,16,32,64,128 */
int snd_dma16[ SND_CARDS ] = SND_DEFAULT_DMA;	/* 0,1,3 */
int snd_dma16_size[ SND_CARDS ] = SND_DEFAULT_DMA_SIZE; /* 8,16,32,64,128 */
#ifdef MODULE_PARM
MODULE_PARM( snd_index, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_index, "Index value for SoundBlaster 16 soundcard." );
MODULE_PARM( snd_id, "1-" __MODULE_STRING(SND_CARDS) "s" );
MODULE_PARM_DESC( snd_id, "ID string for SoundBlaster 16 soundcard." );
MODULE_PARM( snd_port, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_port, "Port # for SB16 driver." );
MODULE_PARM( snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_irq, "IRQ # for SB16 driver." );
MODULE_PARM( snd_dma8, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma8, "8-bit DMA # for SB16 driver." );
MODULE_PARM( snd_dma16, "1-" __MODULE_STRING(SND_CARDS) "i" );
MODULE_PARM_DESC( snd_dma16, "16-bit DMA # for SB16 driver." );
#endif

static struct snd_sb16 {
  int irqnum;
  int dma8num;
  int dma16num;
  snd_card_t *card;
  snd_pcm_t *pcm;
  snd_kmixer_t *mixer;
} *snd_sb16_cards[ SND_CARDS ] = SND_DEFAULT_PTR;

static void snd_sb16_use_inc( void )
{
  MOD_INC_USE_COUNT;
}

static void snd_sb16_use_dec( void )
{
  MOD_DEC_USE_COUNT;
}

static snd_pcm_t *snd_sb16_detect( snd_card_t *card,
                                   unsigned short port,
                                   unsigned short irqnum,
                                   unsigned short dma8num,
                                   unsigned short dma16num )
{
  snd_pcm_t *pcm;
  sbdsp_t *codec;

  if ( snd_register_ioport( card, port, 16, "Sound Blaster 16" ) < 0 )
    return NULL;
  pcm = snd_sbdsp_new_device( card, port, irqnum, dma8num, dma16num, SB_HW_16 );
  if ( !pcm ) {
    snd_unregister_ioports( card );
    return NULL;
  }
  codec = (sbdsp_t *)pcm -> playback.hw.private_data;
  if ( codec -> hardware != SB_HW_16 ) {
    PRINTDD( "sound: SB 16 soundcard weren't detected at 0x%x\n", port );
    __return1:
    snd_pcm_free( pcm );
    snd_unregister_ioports( card );
    return NULL;
  }
  if ( snd_sbdsp_sb16_configure( pcm ) < 0 ) {
    goto __return1;
  }
  return pcm;
}

static void snd_sb16_interrupt( int irq, void *dev_id, struct pt_regs *regs )
{
  struct snd_sb16 *acard;
  snd_pcm_t *pcm;
  register unsigned short status;
  
  acard = (struct snd_sb16 *)dev_id;
  if ( acard == NULL ) return;
  pcm = acard -> pcm;
  if ( pcm == NULL ) return;
  status = snd_sbdsp_mixer_read( (sbdsp_t *)pcm -> playback.hw.private_data, 0x82 );
  if ( status & 0x03 )
    snd_sbdsp_sb16_interrupt( pcm, status );
}

static int snd_sb16_resources( int dev, struct snd_sb16 *acard, snd_card_t *card )
{
  static int possible_irqs[] = { 5, 9, 10, 7, -1 };
  static int possible_dmas8[] = { 1, 3, 0, -1 };
  static int possible_dmas16[] = { 5, 6, 7, -1 };
  int ok = 0;

  if ( (acard -> irqnum = snd_register_interrupt( card, "Sound Blaster 16", snd_irq[ dev ], snd_sb16_interrupt, acard, possible_irqs )) < 0 )
    return acard -> irqnum;
  if ( snd_dma8[ dev ] >= 0 ) {
    if ( (acard -> dma8num = snd_register_dma_channel( card, "Sound Blaster 16", snd_dma8[ dev ], snd_dma8_size[ dev ], possible_dmas8 )) < 0 ) {
      return acard -> dma8num;
    }
    ok++;
  } else {
    acard -> dma8num = SND_DMA_DISABLE;
  }
  if ( snd_dma16[ dev ] >= 0 ) {
    if ( (acard -> dma16num = snd_register_dma_channel( card, "Sound Blaster 16", snd_dma16[ dev ], snd_dma16_size[ dev ], possible_dmas16 )) < 0 ) {
      return acard -> dma16num;
    }
    ok++;
  } else {
    acard -> dma16num = SND_DMA_DISABLE;
  }
  if ( !ok ) {
    PRINTK( "sound: SB16 - bad DMA #\n" );
    return -ENOMEM;
  }
  return 0;
}

static int snd_sb16_probe( int dev, struct snd_sb16 *acard )
{
  static int possible_ports[] = { 0x220,0x240,0x260,-1 };
  int *ports = possible_ports;
  sbdsp_t *codec;
  snd_card_t *card;
  snd_pcm_t *pcm = NULL;
  snd_kmixer_t *mixer = NULL;

  card = snd_card_new( snd_index[ dev ], snd_id[ dev ],
  		       "Sound Blaster 16",
  		       snd_sb16_use_inc, snd_sb16_use_dec );
  if ( !card ) return -ENOMEM;
  card -> type = SND_CARD_TYPE_SB_16;		/* overriden in probe function */
  if ( snd_sb16_resources( dev, acard, card ) < 0 ) {
    snd_card_free( card );
    return -EBUSY;
  }
  pcm = NULL;
  if ( snd_port[ dev ] == SND_AUTO_PORT ) {
    for ( ports = possible_ports; *ports >= 0; ports++ ) {
      pcm = snd_sb16_detect( card, *ports, acard -> irqnum, acard -> dma8num, acard -> dma16num );
      if ( pcm ) break;
    }
    if ( !pcm ) {
      snd_card_free( card );
      return -ENODEV;
    }
  } else {
    pcm = snd_sb16_detect( card, snd_port[ dev ], acard -> irqnum, acard -> dma8num, acard -> dma16num );
    if ( !pcm ) {
      snd_card_free( card );
      return -ENODEV;
    }
  }
  codec = (sbdsp_t *)pcm -> playback.hw.private_data;
  mixer = snd_sbdsp_new_mixer( card, codec -> port, "Sound Blaster 16", codec -> hardware );
  if ( !mixer ) goto __nodev;

  if ( snd_mixer_register( mixer, 0 ) < 0 ) goto __nodev;
  if ( snd_pcm_register( pcm, 0 ) ) {
    snd_mixer_unregister( mixer ); mixer = NULL;
    goto __nodev;
  }
  if ( snd_card_register( card ) ) {
    snd_pcm_unregister( pcm ); pcm = NULL;
    snd_mixer_unregister( mixer ); mixer = NULL;
    goto __nodev;
  }

  snd_mixer_channel_init( mixer, SND_MIXER_PRI_MASTER, 70, 70, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_BASS, 50, 50, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_TREBLE, 25, 25, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_SYNTHESIZER, 75, 75, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_PCM, 85, 85, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_SPEAKER, 0, 0, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_LINE, 0, 0, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_MIC, 0, 0, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_CD, 0, 0, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_IGAIN, 0, 0, 0 );
  snd_mixer_channel_init( mixer, SND_MIXER_PRI_OGAIN, 0, 0, 0 );

  snd_enable_irq( card, acard -> irqnum );
  sprintf( card -> name, "%s at 0x%x, irq %i, dma ",
  	pcm -> name,
  	codec -> port,
  	card -> irqs[ acard -> irqnum ] -> irq );
  if ( acard -> dma8num != SND_DMA_DISABLE )
    sprintf( card -> name + strlen( card -> name ), "%i",
    		card -> dmas[ acard -> dma8num ] -> dma );
  if ( acard -> dma16num != SND_DMA_DISABLE )
    sprintf( card -> name + strlen( card -> name ), "%s%i",
    		acard -> dma8num != SND_DMA_DISABLE ? "&" : "",
    		card -> dmas[ acard -> dma16num ] -> dma );
  acard -> card = card;
  acard -> pcm = pcm;
  acard -> mixer = mixer;
  return 0;

  __nodev:
  if ( mixer ) snd_mixer_free( mixer );
  if ( pcm ) snd_pcm_free( pcm );
  if ( card ) snd_card_free( card );
  return -ENXIO;
}

int init_module( void )
{
  int dev, cards;
  struct snd_sb16 *acard;

  for ( dev = cards = 0; dev < SND_CARDS && snd_port[ dev ] > 0; dev++ ) {
    acard = (struct snd_sb16 *)snd_malloc( sizeof( struct snd_sb16 ) );
    if ( !acard ) continue;
    MEMSET( acard, 0, sizeof( struct snd_sb16 ) );
    if ( snd_sb16_probe( dev, acard ) < 0 ) {
      snd_free( acard, sizeof( struct snd_sb16 ) );
      if ( snd_port[ dev ] == SND_AUTO_PORT ) break;
      PRINTK( "sound: Sound Blaster 16 soundcard #%i not found at 0x%x or device busy\n", dev + 1, snd_port[ dev ] );
      continue;
    }
    snd_sb16_cards[ dev ] = acard;
    cards++;
  }
  if ( !cards ) { 
    PRINTK( "sound: Sound Blaster 16 soundcard #%i not found or device busy\n", dev + 1 );
    return -ENODEV;
  }
  return 0;
}

void cleanup_module( void )
{
  int idx;
  struct snd_sb16 *acard;

  for ( idx = 0; idx < SND_CARDS; idx++ ) {
    acard = snd_sb16_cards[ idx ];
    if ( acard ) {
      if ( acard -> mixer )
        snd_mixer_unregister( acard -> mixer );
      if ( acard -> pcm ) {
        snd_pcm_unregister( acard -> pcm );
      }
      snd_card_unregister( acard -> card );
      snd_free( acard, sizeof( struct snd_sb16 ) );
    }
  }
}
