/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of CS4231(A)/CS4232/InterWave & compatible chips
 *
 *  Bugs:
 *     - sometimes record brokes playback with WSS portion of 
 *       Yamaha OPL3-SA3 chip
 *     - CS4231 (GUS MAX) - still trouble with occasional noises
 *                        - broken initialization?
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "cs4231.h"
#include "ulaw.h"

#if 0
#define SND_DEBUG_MCE
#endif

/*
 *  Some variables
 */

static struct snd_stru_cs4231_freq snd_cs4231_freq[14] =
{
	{5, 5510, 0x00 | CS4231_XTAL2},
	{6, 6620, 0x0E | CS4231_XTAL2},
	{8, 8000, 0x00 | CS4231_XTAL1},
	{9, 9600, 0x0E | CS4231_XTAL1},
	{11, 11025, 0x02 | CS4231_XTAL2},
	{16, 16000, 0x02 | CS4231_XTAL1},
	{18, 18900, 0x04 | CS4231_XTAL2},
	{22, 22050, 0x06 | CS4231_XTAL2},
	{27, 27042, 0x04 | CS4231_XTAL1},
	{32, 32000, 0x06 | CS4231_XTAL1},
	{33, 33075, 0x0C | CS4231_XTAL2},
	{37, 37800, 0x08 | CS4231_XTAL2},
	{44, 44100, 0x0A | CS4231_XTAL2},
	{48, 48000, 0x0C | CS4231_XTAL1}
};

static unsigned char snd_cs4231_original_image[32] =
{
	0x00,			/* 00/00 - lic */
	0x00,			/* 01/01 - ric */
	0x80,			/* 02/02 - la1ic */
	0x80,			/* 03/03 - ra1ic */
	0x80,			/* 04/04 - la2ic */
	0x80,			/* 05/05 - ra2ic */
	0x80,			/* 06/06 - loc */
	0x80,			/* 07/07 - roc */
	0x20,			/* 08/08 - pdfr */
	CS4231_AUTOCALIB,	/* 09/09 - ic */
	0x00,			/* 0a/10 - pc */
	0x00,			/* 0b/11 - ti */
	CS4231_MODE2,		/* 0c/12 - mi */
	0x00,			/* 0d/13 - lbc */
	0x00,			/* 0e/14 - pbru */
	0x00,			/* 0f/15 - pbrl */
	0x80,			/* 10/16 - afei */
	0x01,			/* 11/17 - afeii */
	0x00,			/* 12/18 - llic */
	0x00,			/* 13/19 - rlic */
	0x00,			/* 14/20 - tlb */
	0x00,			/* 15/21 - thb */
	0x00,			/* 16/22 - la3mic/reserved */
	0x00,			/* 17/23 - ra3mic/reserved */
	0x00,			/* 18/24 - afs */
	0x00,			/* 19/25 - lamoc/version */
	0xc0,			/* 1a/26 - mioc */
	0x00,			/* 1b/27 - ramoc/reserved */
	0x20,			/* 1c/28 - cdfr */
	0x00,			/* 1d/29 - res4 */
	0x00,			/* 1e/30 - cbru */
	0x00,			/* 1f/31 - cbrl */
};

/*
 *  Basic I/O functions
 */

void snd_cs4231_outm(cs4231_t * codec, unsigned char reg,
		     unsigned char mask, unsigned char value)
{
	int timeout;
	unsigned char tmp;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_outm: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	if (codec->calibrate_mute) {
		codec->image[reg] &= mask;
		codec->image[reg] |= value;
	} else {
		outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
		mb();
		tmp = (codec->image[reg] & mask) | value;
		outb(tmp, CS4231P(codec, REG));
		codec->image[reg] = tmp;
		mb();
	}
}

static void snd_cs4231_dout(cs4231_t * codec, unsigned char reg, unsigned char value)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	outb(value, CS4231P(codec, REG));
	mb();
}

void snd_cs4231_out(cs4231_t * codec, unsigned char reg, unsigned char value)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_out: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	outb(value, CS4231P(codec, REG));
	codec->image[reg] = value;
	mb();
#if 0
	printk("codec out - reg 0x%x = 0x%x\n", codec->mce_bit | reg, value);
#endif
}

unsigned char snd_cs4231_in(cs4231_t * codec, unsigned char reg)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_in: auto calibration time out - reg = 0x%x\n", reg);
#endif
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	mb();
	return inb(CS4231P(codec, REG));
}

void snd_cs4236_ext_out(cs4231_t * codec, unsigned char reg, unsigned char val)
{
	outb(codec->mce_bit | 0x17, CS4231P(codec, REGSEL));
	outb(reg | (codec->image[CS4236_EXT_REG] & 0x01), CS4231P(codec, REG));
	outb(val, CS4231P(codec, REG));
	codec->eimage[CS4236_REG(reg)] = val;
#if 0
	printk("ext out : reg = 0x%x, val = 0x%x\n", reg, val);
#endif
}

unsigned char snd_cs4236_ext_in(cs4231_t * codec, unsigned char reg)
{
	outb(codec->mce_bit | 0x17, CS4231P(codec, REGSEL));
	outb(reg | (codec->image[CS4236_EXT_REG] & 0x01), CS4231P(codec, REG));
#if 1
	return inb(CS4231P(codec, REG));
#else
	{
		unsigned char res;
		res = inb(CS4231P(codec, REG));
		printk("ext in : reg = 0x%x, val = 0x%x\n", reg, res);
		return res;
	}
#endif
}

#ifdef CONFIG_SND_DEBUG

void snd_cs4231_debug(cs4231_t * codec)
{
	printk("CS4231 REGS:      INDEX = 0x%02x  ", inb(CS4231P(codec, REGSEL)));
	printk("                 STATUS = 0x%02x\n", inb(CS4231P(codec, STATUS)));
	printk("  0x00: left input      = 0x%02x  ", snd_cs4231_in(codec, 0x00));
	printk("  0x10: alt 1 (CFIG 2)  = 0x%02x\n", snd_cs4231_in(codec, 0x10));
	printk("  0x01: right input     = 0x%02x  ", snd_cs4231_in(codec, 0x01));
	printk("  0x11: alt 2 (CFIG 3)  = 0x%02x\n", snd_cs4231_in(codec, 0x11));
	printk("  0x02: GF1 left input  = 0x%02x  ", snd_cs4231_in(codec, 0x02));
	printk("  0x12: left line in    = 0x%02x\n", snd_cs4231_in(codec, 0x12));
	printk("  0x03: GF1 right input = 0x%02x  ", snd_cs4231_in(codec, 0x03));
	printk("  0x13: right line in   = 0x%02x\n", snd_cs4231_in(codec, 0x13));
	printk("  0x04: CD left input   = 0x%02x  ", snd_cs4231_in(codec, 0x04));
	printk("  0x14: timer low       = 0x%02x\n", snd_cs4231_in(codec, 0x14));
	printk("  0x05: CD right input  = 0x%02x  ", snd_cs4231_in(codec, 0x05));
	printk("  0x15: timer high      = 0x%02x\n", snd_cs4231_in(codec, 0x15));
	printk("  0x06: left output     = 0x%02x  ", snd_cs4231_in(codec, 0x06));
	printk("  0x16: left MIC (PnP)  = 0x%02x\n", snd_cs4231_in(codec, 0x16));
	printk("  0x07: right output    = 0x%02x  ", snd_cs4231_in(codec, 0x07));
	printk("  0x17: right MIC (PnP) = 0x%02x\n", snd_cs4231_in(codec, 0x17));
	printk("  0x08: playback format = 0x%02x  ", snd_cs4231_in(codec, 0x08));
	printk("  0x18: IRQ status      = 0x%02x\n", snd_cs4231_in(codec, 0x18));
	printk("  0x09: iface (CFIG 1)  = 0x%02x  ", snd_cs4231_in(codec, 0x09));
	printk("  0x19: left line out   = 0x%02x\n", snd_cs4231_in(codec, 0x19));
	printk("  0x0a: pin control     = 0x%02x  ", snd_cs4231_in(codec, 0x0a));
	printk("  0x1a: mono control    = 0x%02x\n", snd_cs4231_in(codec, 0x1a));
	printk("  0x0b: init & status   = 0x%02x  ", snd_cs4231_in(codec, 0x0b));
	printk("  0x1b: right line out  = 0x%02x\n", snd_cs4231_in(codec, 0x1b));
	printk("  0x0c: revision & mode = 0x%02x  ", snd_cs4231_in(codec, 0x0c));
	printk("  0x1c: record format   = 0x%02x\n", snd_cs4231_in(codec, 0x1c));
	printk("  0x0d: loopback        = 0x%02x  ", snd_cs4231_in(codec, 0x0d));
	printk("  0x1d: var freq (PnP)  = 0x%02x\n", snd_cs4231_in(codec, 0x1d));
	printk("  0x0e: ply upr count   = 0x%02x  ", snd_cs4231_in(codec, 0x0e));
	printk("  0x1e: ply lwr count   = 0x%02x\n", snd_cs4231_in(codec, 0x1e));
	printk("  0x0f: rec upr count   = 0x%02x  ", snd_cs4231_in(codec, 0x0f));
	printk("  0x1f: rec lwr count   = 0x%02x\n", snd_cs4231_in(codec, 0x1f));
}

#endif

/*
 *  CS4231 detection / MCE routines
 */

static void snd_cs4231_busy_wait(cs4231_t * codec)
{
	int timeout;

	/* huh.. looks like this sequence is proper for CS4231A chip (GUS MAX) */
	for (timeout = 5; timeout > 0; timeout--)
		inb(CS4231P(codec, REGSEL));
	/* end of cleanup sequence */
	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
		snd_delay(1);
}

void snd_cs4231_mce_up(cs4231_t * codec)
{
	unsigned long flags;
	int timeout;

	snd_spin_lock(codec, reg, &flags);
	for (timeout = 250; timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT); timeout--)
		snd_delay(1);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("cs4231_mce_up - auto calibration time out (0)\n");
#endif
	codec->mce_bit |= CS4231_MCE;
	timeout = inb(CS4231P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_cs4231_up [0x%x]: serious init problem - codec still busy\n", codec->port);
	if (!(timeout & CS4231_MCE))
		outb(codec->mce_bit | (timeout & 0x1f), CS4231P(codec, REGSEL));
	snd_spin_unlock(codec, reg, &flags);
}

void snd_cs4231_mce_down(cs4231_t * codec)
{
	unsigned long flags;
	int timeout;
	signed long time;

	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_busy_wait(codec);
#if 0
	printk("(1) timeout = %i\n", timeout);
#endif
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("cs4231_mce_down [0x%x] - auto calibration time out (0)\n", CS4231P(codec, REGSEL));
#endif
	codec->mce_bit &= ~CS4231_MCE;
	timeout = inb(CS4231P(codec, REGSEL));
	outb(codec->mce_bit | (timeout & 0x1f), CS4231P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_cs4231_down [0x%x]: serious init problem - codec still busy\n", codec->port);
	if ((timeout & CS4231_MCE) == 0 ||
	    !(codec->hardware & (CS4231_HW_CS4231_MASK | CS4231_HW_CS4232_MASK))) {
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
	snd_cs4231_busy_wait(codec);

	/* calibration process */

	for (timeout = 500; timeout > 0 && (snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) == 0; timeout--);
	if ((snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) == 0) {
		snd_printd("cs4231_mce_down - auto calibration time out (1)\n");
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
#if 0
	printk("(2) timeout = %i, jiffies = %li\n", timeout, jiffies);
#endif
	time = HZ / 4;
	while (snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) {
		snd_spin_unlock(codec, reg, &flags);
		if (time <= 0) {
			snd_printk("cs4231_mce_down - auto calibration time out (2)\n");
			return;
		}
		snd_schedule(codec, mce, time);
		time = snd_timevalue(codec, mce);
		snd_spin_lock(codec, reg, &flags);
	}
#if 0
	printk("(3) jiffies = %li\n", jiffies);
#endif
	time = HZ / 10;
	while (inb(CS4231P(codec, REGSEL)) & CS4231_INIT) {
		snd_spin_unlock(codec, reg, &flags);
		if (time <= 0) {
			snd_printk("cs4231_mce_down - auto calibration time out (3)\n");
			return;
		}
		snd_schedule(codec, mce, time);
		time = snd_timevalue(codec, mce);
		snd_spin_lock(codec, reg, &flags);
	}
	snd_spin_unlock(codec, reg, &flags);
#if 0
	printk("(4) jiffies = %li\n", jiffies);
	snd_printk("cs4231_mce_down - exit = 0x%x\n", inb(CS4231P(codec, REGSEL)));
#endif
}

static unsigned int snd_cs4231_get_count(unsigned char format, unsigned int size)
{
	switch (format & 0xe0) {
	case CS4231_LINEAR_16:
	case CS4231_LINEAR_16_BIG:
		size >>= 1;
		break;
	case CS4231_ADPCM_16:
		return size >> 2;
	}
	if (format & CS4231_STEREO)
		size >>= 1;
	return size;
}

static void snd_cs4231_trigger(cs4231_t * codec, unsigned char what, int enable)
{
	unsigned long flags;

#if 0
	printk("codec trigger!!! - what = %i, enable = %i, status = 0x%x\n", what, enable, inb(CS4231P(card, STATUS)));
#endif
	snd_spin_lock(codec, reg, &flags);
	if (enable) {
		if (codec->image[CS4231_IFACE_CTRL] & what) {
			snd_spin_unlock(codec, reg, &flags);
			return;
		}
		snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL] |= what);
	} else {
		if (!(codec->image[CS4231_IFACE_CTRL] & what)) {
			snd_spin_unlock(codec, reg, &flags);
			return;
		}
		snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL] &= ~what);
	}
	snd_spin_unlock(codec, reg, &flags);
#if 0
	snd_cs4231_debug(codec);
#endif
}

/*
 *  CODEC I/O
 */

static unsigned char snd_cs4231_get_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_cs4231_freq[i].hertz)
			return snd_cs4231_freq[i].bits;
	return snd_cs4231_freq[13].bits;
}

static unsigned int snd_cs4231_get_real_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_cs4231_freq[i].hertz)
			return snd_cs4231_freq[i].rate;
	return snd_cs4231_freq[13].rate;
}

static unsigned int snd_cs4231_xrate(snd_pcm1_t * pcm1, cs4231_t * codec,
                                     unsigned int rate)
{
	return snd_cs4231_get_real_freq(rate);
}

static int snd_cs4231_playback_ioctl(snd_pcm1_t * pcm1, unsigned int cmd,
                                     unsigned long *arg)
{
	cs4231_t *codec = (cs4231_t *) pcm1->private_data;

	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->playback.real_rate = codec->set_playback_rate(pcm1, codec, pcm1->playback.rate);
		return 0;
	}
	return -ENXIO;
}

static int snd_cs4231_record_ioctl(snd_pcm1_t * pcm1, unsigned int cmd,
                                   unsigned long *arg)
{
	cs4231_t *codec = (cs4231_t *) pcm1->private_data;

	switch (cmd) {
	case SND_PCM1_IOCTL_RATE:
		pcm1->record.real_rate = codec->set_record_rate(pcm1, codec, pcm1->record.rate);
		return 0;
	}
	return -ENXIO;
}

static unsigned char snd_cs4231_get_format(cs4231_t * codec,
				           unsigned int mode,
                                           int voices)
{
	unsigned char format;

	format = 0;
	if (mode & SND_PCM1_MODE_16) {
		if (codec->hardware == CS4231_HW_CS4235 ||
		    codec->hardware == CS4231_HW_CS4239) {
			format |= CS4231_LINEAR_16;
		} else {
			if (mode & SND_PCM1_MODE_ADPCM)
				format |= CS4231_ADPCM_16;
			else if (mode & SND_PCM1_MODE_BIG)
				format |= CS4231_LINEAR_16_BIG;
			else
				format |= CS4231_LINEAR_16;
		}
	} else {
		if (codec->hardware == CS4231_HW_CS4235 ||
		    codec->hardware == CS4231_HW_CS4239) {
			/* nothing */
		} else {
			if (mode & SND_PCM1_MODE_ALAW)
				format |= CS4231_ALAW_8;
			else if (mode & SND_PCM1_MODE_ULAW)
				format |= CS4231_ULAW_8;
#if 0				/* CS4231_LINEAR_8 == 0 */
			else
				format |= CS4231_LINEAR_8;
#endif
		}
	}
	if (voices > 1)
		format |= CS4231_STEREO;
#if 0
	snd_printk("snd_cs4231_get_format: 0x%x (mode=0x%x)\n", format, mode);
#endif
	return format;
}

static void snd_cs4231_calibrate_mute(cs4231_t * codec, int mute)
{
	unsigned long flags;

	mute = mute ? 1 : 0;
	snd_spin_lock(codec, reg, &flags);
	if (codec->calibrate_mute == mute) {
		snd_spin_unlock(codec, reg, &flags);
		return;
	}
	if (!mute) {
		snd_cs4231_dout(codec, CS4231_LEFT_INPUT, codec->image[CS4231_LEFT_INPUT]);
		snd_cs4231_dout(codec, CS4231_RIGHT_INPUT, codec->image[CS4231_RIGHT_INPUT]);
		snd_cs4231_dout(codec, CS4231_LOOPBACK, codec->image[CS4231_LOOPBACK]);
	}
	snd_cs4231_dout(codec, CS4231_AUX1_LEFT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX1_LEFT_INPUT]);
	snd_cs4231_dout(codec, CS4231_AUX1_RIGHT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX1_RIGHT_INPUT]);
	snd_cs4231_dout(codec, CS4231_AUX2_LEFT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX2_LEFT_INPUT]);
	snd_cs4231_dout(codec, CS4231_AUX2_RIGHT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX2_RIGHT_INPUT]);
	snd_cs4231_dout(codec, CS4231_LEFT_OUTPUT, mute ? 0x80 : codec->image[CS4231_LEFT_OUTPUT]);
	snd_cs4231_dout(codec, CS4231_RIGHT_OUTPUT, mute ? 0x80 : codec->image[CS4231_RIGHT_OUTPUT]);
	snd_cs4231_dout(codec, CS4231_LEFT_LINE_IN, mute ? 0x80 : codec->image[CS4231_LEFT_LINE_IN]);
	snd_cs4231_dout(codec, CS4231_RIGHT_LINE_IN, mute ? 0x80 : codec->image[CS4231_RIGHT_LINE_IN]);
	snd_cs4231_dout(codec, CS4231_MONO_CTRL, mute ? 0xc0 : codec->image[CS4231_MONO_CTRL]);
	if (codec->hardware == CS4231_HW_INTERWAVE) {
		snd_cs4231_dout(codec, CS4231_LEFT_MIC_INPUT, mute ? 0x80 : codec->image[CS4231_LEFT_MIC_INPUT]);
		snd_cs4231_dout(codec, CS4231_RIGHT_MIC_INPUT, mute ? 0x80 : codec->image[CS4231_RIGHT_MIC_INPUT]);		
		snd_cs4231_dout(codec, CS4231_LINE_LEFT_OUTPUT, mute ? 0x80 : codec->image[CS4231_LINE_LEFT_OUTPUT]);
		snd_cs4231_dout(codec, CS4231_LINE_RIGHT_OUTPUT, mute ? 0x80 : codec->image[CS4231_LINE_RIGHT_OUTPUT]);
	}
	codec->calibrate_mute = mute;
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_playback_format(snd_pcm1_t * pcm1,
				       cs4231_t * codec,
				       unsigned char pdfr)
{
	unsigned long flags;
	int full_calib = 1;

	snd_mutex_down(codec, mce);
	snd_cs4231_calibrate_mute(codec, 1);
	if (codec->hardware == CS4231_HW_CS4231A ||
	    (codec->hardware & CS4231_HW_CS4232_MASK)) {
		snd_spin_lock(codec, reg, &flags);
		if ((codec->image[CS4231_PLAYBK_FORMAT] & 0x0f) == (pdfr & 0x0f)) {	/* rate is same? */
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] | 0x10);
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image[CS4231_PLAYBK_FORMAT] = pdfr);
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] &= ~0x10);
			full_calib = 0;
		}
		snd_spin_unlock(codec, reg, &flags);
	}
	if (full_calib) {
		snd_cs4231_mce_up(codec);
		snd_spin_lock(codec, reg, &flags);
		if (codec->hardware != CS4231_HW_INTERWAVE && !codec->single_dma) {
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT,
					(codec->image[CS4231_IFACE_CTRL] & CS4231_RECORD_ENABLE) ?
					(pdfr & 0xf0) | (codec->image[CS4231_REC_FORMAT] & 0x0f) :
				        pdfr);
		} else {
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image[CS4231_PLAYBK_FORMAT] = pdfr);
		}
		snd_spin_unlock(codec, reg, &flags);
		snd_cs4231_mce_down(codec);
	}
	snd_cs4231_calibrate_mute(codec, 0);
	snd_mutex_up(codec, mce);
}

static void snd_cs4231_record_format(snd_pcm1_t * pcm1,
                                     cs4231_t * codec,
                                     unsigned char cdfr)
{
	unsigned long flags;
	int full_calib = 1;

	snd_mutex_down(codec, mce);
	snd_cs4231_calibrate_mute(codec, 1);
	if (codec->hardware == CS4231_HW_CS4231A ||
	    (codec->hardware & CS4231_HW_CS4232_MASK)) {
		snd_spin_lock(codec, reg, &flags);
		if ((codec->image[CS4231_PLAYBK_FORMAT] & 0x0f) == (cdfr & 0x0f) ||	/* rate is same? */
		    (codec->image[CS4231_IFACE_CTRL] & CS4231_PLAYBACK_ENABLE)) {
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] | 0x20);
			snd_cs4231_out(codec, CS4231_REC_FORMAT, (codec->image[CS4231_REC_FORMAT] = cdfr) & 0xf0);
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] &= ~0x20);
			full_calib = 0;
		}
		snd_spin_unlock(codec, reg, &flags);
	}
	if (full_calib) {
		snd_cs4231_mce_up(codec);
		snd_spin_lock(codec, reg, &flags);
		if (codec->hardware != CS4231_HW_INTERWAVE) {
			if (!(codec->image[CS4231_IFACE_CTRL] & CS4231_PLAYBACK_ENABLE)) {
				snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT,
					       ((codec->single_dma ? cdfr : codec->image[CS4231_PLAYBK_FORMAT]) & 0xf0) |
					       (cdfr & 0x0f));
				snd_spin_unlock(codec, reg, &flags);
				snd_cs4231_mce_down(codec);
				snd_cs4231_mce_up(codec);
				snd_spin_lock(codec, reg, &flags);
			}
			snd_cs4231_out(codec, CS4231_REC_FORMAT, cdfr & 0xf0);
		} else {
			snd_cs4231_out(codec, CS4231_REC_FORMAT, cdfr);
		}
		snd_spin_unlock(codec, reg, &flags);
		snd_cs4231_mce_down(codec);
	}
	snd_cs4231_calibrate_mute(codec, 0);
	snd_mutex_up(codec, mce);
}

/*
 *  Timer interface
 */

static unsigned long snd_cs4231_timer_resolution(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	if (codec->hardware & CS4231_HW_CS4236_MASK)
		return 14467;
	else
		return codec->image[CS4231_PLAYBK_FORMAT] & 1 ? 9969 : 9920;
}

static void snd_cs4231_timer_start(snd_timer_t * timer)
{
	unsigned long flags;
	unsigned int ticks;
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_spin_lock(codec, reg, &flags);
	ticks = timer->sticks;
	if ((codec->image[CS4231_ALT_FEATURE_1] & CS4231_TIMER_ENABLE) == 0 ||
	    (unsigned char)(ticks >> 8) != codec->image[CS4231_TIMER_HIGH] ||
	    (unsigned char)ticks != codec->image[CS4231_TIMER_LOW]) {
		snd_cs4231_out(codec, CS4231_TIMER_HIGH, codec->image[CS4231_TIMER_HIGH] = (unsigned char) (ticks >> 8));
		snd_cs4231_out(codec, CS4231_TIMER_LOW, codec->image[CS4231_TIMER_LOW] = (unsigned char) ticks);
		snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] | CS4231_TIMER_ENABLE);
	}
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_timer_stop(snd_timer_t * timer)
{
	unsigned long flags;
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] &= ~CS4231_TIMER_ENABLE);
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_init(cs4231_t * codec)
{
	unsigned long flags;

	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (1)\n");
#endif
	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO |
			     CS4231_RECORD_ENABLE | CS4231_RECORD_PIO |
			     CS4231_CALIB_MODE);
	codec->image[CS4231_IFACE_CTRL] |= CS4231_AUTOCALIB;
	snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL]);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (2)\n");
#endif

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1]);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (3) - afei = 0x%x\n", codec->image[CS4231_ALT_FEATURE_1]);
#endif

	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_2, codec->image[CS4231_ALT_FEATURE_2]);
	snd_spin_unlock(codec, reg, &flags);

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image[CS4231_PLAYBK_FORMAT]);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (4)\n");
#endif

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_REC_FORMAT,
		       codec->hardware == CS4231_HW_INTERWAVE ?
		       codec->image[CS4231_REC_FORMAT] :
		       (codec->image[CS4231_REC_FORMAT] & 0xf0));
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (5)\n");
#endif
}

static void snd_cs4231_open(cs4231_t * codec, unsigned int mode)
{
	unsigned long flags;

	snd_mutex_down(codec, open);
	if (codec->mode & CS4231_MODE_OPEN) {
		codec->mode |= mode;
		snd_mutex_up(codec, open);
		return;
	}
	/* ok. now enable and ack CODEC IRQ */
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_PLAYBACK_IRQ |
		       CS4231_RECORD_IRQ |
		       CS4231_TIMER_IRQ);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	codec->image[CS4231_PIN_CTRL] |= CS4231_IRQ_ENABLE;
	snd_cs4231_out(codec, CS4231_PIN_CTRL, codec->image[CS4231_PIN_CTRL]);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_PLAYBACK_IRQ |
		       CS4231_RECORD_IRQ |
		       CS4231_TIMER_IRQ);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	snd_spin_unlock(codec, reg, &flags);

	codec->mode = mode;
	snd_mutex_up(codec, open);
}

static void snd_cs4231_close(cs4231_t * codec, unsigned int mode)
{
	unsigned long flags;

	snd_mutex_down(codec, open);
	codec->mode &= ~mode;
	if (codec->mode & CS4231_MODE_OPEN) {
		snd_mutex_up(codec, open);
		return;
	}
	snd_cs4231_calibrate_mute(codec, 1);

	/* disable IRQ */
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	codec->image[CS4231_PIN_CTRL] &= ~CS4231_IRQ_ENABLE;
	snd_cs4231_out(codec, CS4231_PIN_CTRL, codec->image[CS4231_PIN_CTRL]);
	snd_spin_unlock(codec, reg, &flags);

	/* now disable record & playback */

	snd_cs4231_mce_up(codec);
	snd_spin_lock(codec, reg, &flags);
	codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO |
			     CS4231_RECORD_ENABLE | CS4231_RECORD_PIO);
	snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL]);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_down(codec);

	/* clear IRQ again */
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	snd_spin_unlock(codec, reg, &flags);

	snd_cs4231_calibrate_mute(codec, 0);

	codec->mode = 0;
	snd_mutex_up(codec, open);
}

/*
 *  timer open/close
 */

static int snd_cs4231_timer_open(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_cs4231_open(codec, CS4231_MODE_TIMER);
	return 0;
}

static int snd_cs4231_timer_close(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = (cs4231_t *) timer->private_data;
	snd_cs4231_close(codec, CS4231_MODE_TIMER);
	return 0;
}

static struct snd_stru_timer_hardware snd_cs4231_timer =
{
	SND_TIMER_HW_AUTO,	/* flags */
	9945,			/* resolution in us * 1000 */
	65535,			/* high timer mark */
	snd_cs4231_timer_open,	/* open */
	snd_cs4231_timer_close,	/* close */
	snd_cs4231_timer_resolution, /* resolution */
	snd_cs4231_timer_start,	/* start */
	snd_cs4231_timer_stop,	/* stop */
};

/*
 *  ok.. exported functions..
 */

static void snd_cs4231_playback_trigger(snd_pcm1_t * pcm1, int up)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;

	snd_cs4231_trigger(codec, CS4231_PLAYBACK_ENABLE, up);

#if 0
	if (up)
		snd_cs4231_debug(codec);
#endif
}

static void snd_cs4231_record_trigger(snd_pcm1_t * pcm1, int up)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	snd_cs4231_trigger(codec, CS4231_RECORD_ENABLE, up);
}

static void snd_cs4231_playback_prepare(snd_pcm1_t * pcm1,
					unsigned char *buffer,
					unsigned int size,
					unsigned int offset,
					unsigned int count)
{
	cs4231_t *codec;
	unsigned long flags;
	unsigned int mode;
	unsigned char new_pdfr;

	codec = (cs4231_t *) pcm1->private_data;
	/* Note: offset is always 0 if AUTO DMA */
#if 0
	snd_printk("cs4231_start_playback - dma = %i, buffer = 0x%lx, size = 0x%x, offset = 0x%x, count = 0x%x\n", codec->dma1, (long) buffer, size, offset, count);
#endif
	mode = pcm1->playback.mode;
	if ((mode & SND_PCM1_MODE_ULAW) &&
	    codec->hardware == CS4231_HW_INTERWAVE &&
	    codec->dma1 > 3) {
		mode = SND_PCM1_MODE_U;
	}
	new_pdfr = snd_cs4231_get_format(codec, mode, pcm1->playback.voices) |
		   snd_cs4231_get_freq(pcm1->playback.real_rate);
	codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO);
	snd_dma_program(codec->dma1, buffer, size, DMA_MODE_WRITE | DMA_MODE_AUTOINIT);
	codec->set_playback_format(pcm1, codec, new_pdfr);
	count = snd_cs4231_get_count(codec->image[CS4231_PLAYBK_FORMAT], count) - 1;
	snd_spin_lock(codec, reg, &flags);
	snd_cs4231_out(codec, CS4231_PLY_LWR_CNT, (unsigned char) count);
	snd_cs4231_out(codec, CS4231_PLY_UPR_CNT, (unsigned char) (count >> 8));
	snd_spin_unlock(codec, reg, &flags);
#if 0
	snd_cs4231_debug(codec);
#endif
}

static void snd_cs4231_record_prepare(snd_pcm1_t * pcm1,
				      unsigned char *buffer,
				      unsigned int size,
				      unsigned int offset,
				      unsigned int count)
{
	cs4231_t *codec;
	unsigned long flags;
	unsigned char new_cdfr;

	codec = (cs4231_t *) pcm1->private_data;
#if 0
	snd_printk("cs4231_start_record: start - buffer = 0x%lx, offset = 0x%x, size = 0x%x, count = 0x%x\n", (long) buffer, offset, size, count);
#endif
	new_cdfr = snd_cs4231_get_format(codec, pcm1->record.mode, pcm1->record.voices) |
		   snd_cs4231_get_freq(pcm1->record.real_rate);
	codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_RECORD_ENABLE | CS4231_RECORD_PIO);
	snd_dma_program(codec->dma2, buffer, size, DMA_MODE_READ | DMA_MODE_AUTOINIT);
#if 0
	printk("cdfr = 0x%x, rport = 0x%x, pport = 0x%x, status = 0x%x\n", codec->image[CS4231_REC_FORMAT], snd_cs4231_in(codec, CS4231_REC_FORMAT), snd_cs4231_in(codec, CS4231_PLAYBK_FORMAT), inb(CS4231P(codec, REGSEL)));
#endif
	codec->set_record_format(pcm1, codec, new_cdfr);
	count = snd_cs4231_get_count(codec->image[CS4231_REC_FORMAT], count) - 1;
	snd_spin_lock(codec, reg, &flags);
	if (codec->single_dma && codec->hardware != CS4231_HW_INTERWAVE) {
		snd_cs4231_out(codec, CS4231_PLY_LWR_CNT, (unsigned char) count);
		snd_cs4231_out(codec, CS4231_PLY_UPR_CNT, (unsigned char) (count >> 8));
	} else {
		snd_cs4231_out(codec, CS4231_REC_LWR_CNT, (unsigned char) count);
		snd_cs4231_out(codec, CS4231_REC_UPR_CNT, (unsigned char) (count >> 8));
	}
	snd_spin_unlock(codec, reg, &flags);
}

static void snd_cs4231_overrange(snd_pcm1_t * pcm1)
{
	cs4231_t *codec;
	unsigned long flags;
	unsigned char res;

	codec = (cs4231_t *) pcm1->private_data;
	snd_spin_lock(codec, reg, &flags);
	res = snd_cs4231_in(codec, CS4231_TEST_INIT);
	snd_spin_unlock(codec, reg, &flags);
	if (res & (0x08 | 0x02))	/* detect overrange only above 0dB; may be user selectable? */
		pcm1->record.overrange++;
}

void snd_cs4231_interrupt(snd_pcm_t * pcm, unsigned char status)
{
	unsigned long flags;
	snd_pcm1_t *pcm1;
	cs4231_t *codec;

#if 0
	snd_printk("snd_cs4231_interrupt: status=0x%x\n", status);
#endif
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (cs4231_t *) pcm1->private_data;
	if (!codec)
		return;
	status = snd_cs4231_in(codec, CS4231_IRQ_STATUS);

	if ((status & CS4231_TIMER_IRQ) && codec->timer)
		snd_timer_interrupt(codec->timer, codec->timer->sticks);
	if (codec->single_dma && codec->hardware != CS4231_HW_INTERWAVE) {
		if (status & CS4231_PLAYBACK_IRQ) {
			if ((codec->mode & CS4231_MODE_PLAY) && pcm1->playback.ack)
				pcm1->playback.ack(pcm1);
			if ((codec->mode & CS4231_MODE_RECORD) && pcm1->record.ack) {
				snd_cs4231_overrange(pcm1);
				pcm1->record.ack(pcm1);
			}
		}
	} else {
		if ((status & CS4231_PLAYBACK_IRQ) && pcm1->playback.ack)
			pcm1->playback.ack(pcm1);
		if ((status & CS4231_RECORD_IRQ) && pcm1->record.ack) {
			snd_cs4231_overrange(pcm1);
			pcm1->record.ack(pcm1);
		}
	}

	snd_spin_lock(codec, reg, &flags);
	outb(0, CS4231P(codec, STATUS));	/* clear global interrupt bit */
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_ALL_IRQS & ~status);
	snd_spin_unlock(codec, reg, &flags);
}

/*

 */

static int snd_cs4231_playback_open(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;
	int err;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_PLAYBACK, codec->dmaptr1, "CS4231 (playback)")) < 0)
		return err;
	snd_cs4231_open(codec, CS4231_MODE_PLAY);
	return 0;
}

static int snd_cs4231_record_open(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;
	int err;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	if ((err = snd_pcm1_dma_alloc(pcm1, SND_PCM1_RECORD, codec->dmaptr2, "CS4231 (record)")) < 0)
		return err;
	snd_cs4231_open(codec, CS4231_MODE_RECORD);
	return 0;
}

static void snd_cs4231_playback_close(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	snd_cs4231_close(codec, CS4231_MODE_PLAY);
	snd_pcm1_dma_free(pcm1, SND_PCM1_PLAYBACK, codec->dmaptr1);
}

static void snd_cs4231_record_close(snd_pcm1_t * pcm1)
{
	snd_card_t *card;
	cs4231_t *codec;

	card = pcm1->card;
	codec = (cs4231_t *) pcm1->private_data;
	snd_cs4231_close(codec, CS4231_MODE_RECORD);
	snd_pcm1_dma_free(pcm1, SND_PCM1_RECORD, codec->dmaptr2);
}

static void snd_cs4231_playback_dma(snd_pcm1_t * pcm1,
			      unsigned char *buffer, unsigned int offset,
				 unsigned char *user, unsigned int count)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	if ((pcm1->playback.mode & SND_PCM1_MODE_ULAW) &&
	    codec->hardware == CS4231_HW_INTERWAVE && codec->dma1 > 3) {
		snd_translate_from_user(snd_ulaw_dsp_loud, &buffer[offset], user, count);
	} else {
		copy_from_user(&buffer[offset], user, count);
	}
}

static unsigned int snd_cs4231_playback_pointer(snd_pcm1_t * pcm1, unsigned int used_size)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	if (!(codec->image[CS4231_IFACE_CTRL] & CS4231_PLAYBACK_ENABLE))
		return 0;
	return used_size - snd_dma_residue(codec->dma1);
}

static unsigned int snd_cs4231_record_pointer(snd_pcm1_t * pcm1, unsigned int used_size)
{
	cs4231_t *codec;

	codec = (cs4231_t *) pcm1->private_data;
	if (!(codec->image[CS4231_IFACE_CTRL] & CS4231_RECORD_ENABLE))
		return 0;
	return used_size - snd_dma_residue(codec->dma2);
}

/*

 */

#if 0
void snd_cs4231_info(snd_card_t * card, snd_info_buffer_t * buffer)
{
	if (!card->use_codec)
		return;
	snd_iprintf(buffer, "CS4231:\n");
	snd_iprintf(buffer, "  mode               : %s\n", card->codec.hardware != CS4231_HW_INTERWAVE ? "2" : "3");
	snd_iprintf(buffer, "  record overflow    : %i\n", card->codec.record_overflow);
	if (card->pnp_flag) {
		snd_iprintf(buffer, "  playback fifo      : %i\n", card->codec.playback_fifo_size);
		snd_iprintf(buffer, "  record fifo        : %i\n", card->codec.record_fifo_size);
	}
}
#endif

/*

 */

static int snd_cs4231_probe(snd_pcm1_t * pcm1)
{
	unsigned long flags;
	snd_pcm_t *pcm;
	cs4231_t *codec;
	int i, id, rev;
	unsigned char *ptr;
	unsigned int hw;

	codec = (cs4231_t *) pcm1->private_data;
	if (!codec)
		return -EINVAL;
	pcm = pcm1->pcm;
	if (!pcm)
		return -EINVAL;

#if 0
	snd_cs4231_debug(codec);
#endif
	id = 0;
	for (i = 0; i < 1000; i++) {
		mb();
		if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
			snd_delay(50 * 8);
		else {
			snd_spin_lock(codec, reg, &flags);
			snd_cs4231_out(codec, CS4231_MISC_INFO, CS4231_MODE2);
			id = snd_cs4231_in(codec, CS4231_MISC_INFO) & 0x0f;
			snd_spin_unlock(codec, reg, &flags);
			if (id == 0x0a)
				break;	/* this is valid value */
		}
	}
	if (id != 0x0a)
		return -ENODEV;	/* no valid device found */

	if (((hw = codec->hardware) & CS4231_HW_TYPE_MASK) == CS4231_HW_DETECT) {
		rev = snd_cs4231_in(codec, CS4231_VERSION) & 0xe7;
		snd_printdd("CS4231: VERSION (I25) = 0x%x\n", rev);
		if (rev == 0x80) {
			codec->hardware = CS4231_HW_CS4231;
		} else if (rev == 0xa0) {
			codec->hardware = CS4231_HW_CS4231A;
		} else if (rev == 0xa2) {
			codec->hardware = CS4231_HW_CS4232;
		} else if (rev == 0xb2) {
			codec->hardware = CS4231_HW_CS4232A;
		} else if (rev == 0x83) {	/* CS4236 -> down to CS4232 */
			codec->hardware = CS4231_HW_CS4232A;
			snd_printk("Original CS4236 chip is not fully supported!!!\n");
		} else if (rev == 0x03) {
			codec->hardware = CS4231_HW_CS4236;
		} else {
			snd_printk("cs4231: unknown CS chip with version 0x%x\n", rev);
			return -ENODEV;		/* unknown CS4231 chip? */
		}
	}
	snd_spin_lock(codec, reg, &flags);
	inb(CS4231P(codec, STATUS));	/* clear any pendings IRQ */
	outb(0, CS4231P(codec, STATUS));
	mb();
	snd_spin_unlock(codec, reg, &flags);

	codec->image[CS4231_MISC_INFO] = CS4231_MODE2;
	switch (codec->hardware) {
	case CS4231_HW_INTERWAVE:
		codec->image[CS4231_MISC_INFO] = CS4231_IW_MODE3;
		break;
	case CS4231_HW_CS4235:
	case CS4231_HW_CS4236:
	case CS4231_HW_CS4236B:
	case CS4231_HW_CS4237B:
	case CS4231_HW_CS4238B:
	case CS4231_HW_CS4239:
		if (hw == CS4231_HW_DETECT3)
			codec->image[CS4231_MISC_INFO] = CS4231_4236_MODE3;
		else
			codec->hardware = CS4231_HW_CS4232A;
		break;
	}

	codec->image[CS4231_IFACE_CTRL] =
	    (codec->image[CS4231_IFACE_CTRL] & ~CS4231_SINGLE_DMA) |
	    (codec->single_dma ? CS4231_SINGLE_DMA : 0);
	codec->image[CS4231_ALT_FEATURE_1] = 0x80;
	codec->image[CS4231_ALT_FEATURE_2] = codec->hardware == CS4231_HW_INTERWAVE ? 0xc2 : 0x01;
	ptr = (unsigned char *) &codec->image;
	snd_cs4231_mce_down(codec);
	snd_spin_lock(codec, reg, &flags);
	for (i = 0; i < 32; i++)	/* ok.. fill all CS4231 registers */
		snd_cs4231_out(codec, i, *ptr++);
	snd_spin_unlock(codec, reg, &flags);
	snd_cs4231_mce_up(codec);
	snd_cs4231_mce_down(codec);

	/* ok.. try check hardware version for CS4236+ chips */
	if ((hw & CS4231_HW_TYPE_MASK) == CS4231_HW_DETECT) {
		if (codec->hardware == CS4231_HW_CS4236) {
			rev = snd_cs4236_ext_in(codec, CS4236_VERSION);
			snd_cs4236_ext_out(codec, CS4236_VERSION, 0xff);
			id = snd_cs4236_ext_in(codec, CS4236_VERSION);
			snd_cs4236_ext_out(codec, CS4236_VERSION, rev);
			snd_printdd("CS4231: ext version; rev = 0x%x, id = 0x%x\n", rev, id);
			if ((id & 0x1f) == 0x1d) {	/* CS4235 */
				codec->hardware = CS4231_HW_CS4235;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
					break;
				default:
					snd_printk("cs4231: unknown CS4235 chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x0b) {	/* CS4236/B */
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
				case 7:
					codec->hardware = CS4231_HW_CS4236B;
					break;
				default:
					snd_printk("cs4231: unknown CS4236 chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x08) {	/* CS4237B */
				codec->hardware = CS4231_HW_CS4237B;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
				case 7:
					break;
				default:
					snd_printk("cs4231: unknown CS4237B chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x09) {	/* CS4238B */
				codec->hardware = CS4231_HW_CS4238B;
				switch (id >> 5) {
				case 5:
				case 6:
				case 7:
					break;
				default:
					snd_printk("cs4231: unknown CS4238B chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x1e) {	/* CS4239 */
				codec->hardware = CS4231_HW_CS4239;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
					break;
				default:
					snd_printk("cs4231: unknown CS4239 chip (enhanced version = 0x%x)\n", id);
				}
			} else {
				snd_printk("cs4231: unknown CS4236/CS423xB chip (enhanced version = 0x%x)\n", id);
			}
		}
	}
	return 0;		/* all things are ok.. */
}

/*

 */

static struct snd_stru_pcm1_hardware snd_cs4231_playback =
{
	NULL,			/* private data */
	NULL,			/* private_free */
	SND_PCM1_HW_AUTODMA,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* formats */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* hardware formats */
	0,			/* align value */
	6,			/* minimal fragment */
	5510,			/* min. rate */
	48000,			/* max. rate */
	2,			/* max. voices */
	snd_cs4231_playback_open,
	snd_cs4231_playback_close,
	snd_cs4231_playback_ioctl,
	snd_cs4231_playback_prepare,
	snd_cs4231_playback_trigger,
	snd_cs4231_playback_pointer,
	snd_cs4231_playback_dma,
	snd_pcm1_dma_move,
	snd_pcm1_playback_dma_neutral
};

static struct snd_stru_pcm1_hardware snd_cs4231_record =
{
	NULL,			/* private data */
	NULL,			/* private free */
	SND_PCM1_HW_AUTODMA,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* formats */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* hardware formats */
	0,			/* align value */
	6,			/* minimal fragment */
	5510,			/* min. rate */
	48000,			/* max. rate */
	2,			/* max. voices */
	snd_cs4231_record_open,
	snd_cs4231_record_close,
	snd_cs4231_record_ioctl,
	snd_cs4231_record_prepare,
	snd_cs4231_record_trigger,
	snd_cs4231_record_pointer,
	snd_pcm1_record_dma,
	snd_pcm1_dma_move,
	NULL
};

static void snd_cs4231_free(void *private_data)
{
	cs4231_t *codec;

	codec = (cs4231_t *) private_data;
	if (codec->timer)
		snd_timer_unregister(codec->timer);
	snd_free(codec, sizeof(cs4231_t));
}

snd_pcm_t *snd_cs4231_new_device(snd_card_t * card,
				 unsigned short port,
				 snd_irq_t * irqptr,
				 snd_dma_t * dmaptr1,
				 snd_dma_t * dmaptr2,
				 unsigned short hardware,
				 int timer_dev)
{
	snd_pcm_t *pcm;
	snd_pcm1_t *pcm1;
	cs4231_t *codec;
	snd_timer_t *timer;
	char *str;

	pcm = snd_pcm1_new_device(card, "CS4231");
	if (!pcm)
		return NULL;
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (cs4231_t *) snd_calloc(sizeof(cs4231_t));
	if (!codec)
		return NULL;
	snd_spin_prepare(codec, reg);
	snd_mutex_prepare(codec, mce);
	snd_mutex_prepare(codec, open);
	snd_sleep_prepare(codec, mce);
	snd_sleep_prepare(codec, iec958);
	codec->pcm = pcm;
	codec->card = pcm->card;
	codec->port = port;
	codec->irq = irqptr->irq;
	codec->irqptr = irqptr;
	codec->dmaptr1 = dmaptr1;
	codec->dma1 = dmaptr1->dma;
	codec->dmaptr2 = dmaptr2;
	codec->dma2 = dmaptr2->dma;
	if (codec->dma1 == codec->dma2)
		codec->single_dma = 1;
	codec->hardware = hardware;
	codec->set_playback_rate =
	    codec->set_record_rate = snd_cs4231_xrate;
	codec->set_playback_format = snd_cs4231_playback_format;
	codec->set_record_format = snd_cs4231_record_format;
	memcpy(&codec->image, &snd_cs4231_original_image, sizeof(snd_cs4231_original_image));
	memcpy(&pcm1->playback.hw, &snd_cs4231_playback, sizeof(snd_cs4231_playback));
	memcpy(&pcm1->record.hw, &snd_cs4231_record, sizeof(snd_cs4231_record));
	if (codec->hardware == CS4231_HW_INTERWAVE && codec->dma1 > 3)
		pcm1->playback.hw.hw_formats &= ~SND_PCM_FMT_MU_LAW;	/* bug reached */
	pcm1->private_data = codec;
	pcm1->private_free = snd_cs4231_free;
	pcm->info_flags = SND_PCM_INFO_CODEC | SND_PCM_INFO_MMAP |
	    SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_RECORD;
	if (!codec->single_dma)
		pcm->info_flags |= SND_PCM_INFO_DUPLEX;
	if (codec->hardware != CS4231_HW_INTERWAVE)
		pcm->info_flags |= SND_PCM_INFO_DUPLEX_LIMIT;
	if (snd_cs4231_probe(pcm1) < 0) {
		snd_pcm_free(pcm);
		return NULL;
	}
	switch (codec->hardware) {
	case CS4231_HW_CS4231:
		str = "CS4231";
		break;
	case CS4231_HW_CS4231A:
		str = "CS4231A";
		break;
	case CS4231_HW_CS4232:
		str = "CS4232";
		break;
	case CS4231_HW_CS4232A:
		str = "CS4232A";
		break;
	case CS4231_HW_CS4235:
		str = "CS4235";
		break;
	case CS4231_HW_CS4236B:
		str = "CS4236B";
		break;
	case CS4231_HW_CS4237B:
		str = "CS4237B";
		break;
	case CS4231_HW_CS4238B:
		str = "CS4238B";
		break;
	case CS4231_HW_CS4239:
		str = "CS4239";
		break;
	case CS4231_HW_INTERWAVE:
		str = "AMD InterWave";
		break;
	case CS4231_HW_OPL3SA:
		str = card->shortname;
		break;
	default:
		str = "???";
	}
	strcpy(pcm->name, str);
	snd_cs4231_init(codec);
	timer = snd_timer_new_device(card, "CS4231");
	if (timer) {
		strcpy(timer->name, pcm->name);
		timer->private_data = codec;
		memcpy(&timer->hw, &snd_cs4231_timer, sizeof(snd_cs4231_timer));
		if (snd_timer_register(timer, timer_dev) < 0) {
			snd_timer_free(timer);
			timer = NULL;
		}
	}
	codec->timer = timer;
	return pcm;
}

/*
 *  MIXER part
 */

static int snd_cs4231_mixer_get_recsrc(cs4231_t *codec, snd_kmixer_element_t *element)
{
	if (element == codec->me_mux_line)
		return CS4231_MIXS_LINE;
	if (element == codec->me_mux_aux1)
		return CS4231_MIXS_AUX1;
	if (element == codec->me_mux_mic)
		return CS4231_MIXS_MIC;
	if (element == codec->me_mux_mix)
		return CS4231_MIXS_ALL;
	return -EINVAL;
}

static snd_kmixer_element_t *snd_cs4231_mixer_get_recsrc_element(cs4231_t *codec, int value)
{
	value &= CS4231_MIXS_ALL;
	switch (value) {
	case CS4231_MIXS_LINE:
		return codec->me_mux_line;
	case CS4231_MIXS_AUX1:
		return codec->me_mux_aux1;
	case CS4231_MIXS_MIC:
		return codec->me_mux_mic;
	case CS4231_MIXS_ALL:
		return codec->me_mux_mix;
	}
	return NULL;
}

static int snd_cs4231_mixer_mux(int w_flag, snd_kmixer_element_t **elements, cs4231_t *codec)
{
	unsigned long flags;
	int change = 0;
	int left, right, nleft, nright;

	snd_spin_lock(codec, reg, &flags);
	left = codec->image[CS4231_LEFT_INPUT] & CS4231_MIXS_ALL;
	right = codec->image[CS4231_RIGHT_INPUT] & CS4231_MIXS_ALL;
	if (w_flag) {
		nleft = snd_cs4231_mixer_get_recsrc(codec, elements[0]);
		nright = snd_cs4231_mixer_get_recsrc(codec, elements[1]);
		if (nleft >= 0 && nright >= 0) {
			change = nleft != left || nright != right;
			snd_cs4231_outm(codec, CS4231_LEFT_INPUT, (unsigned char)~CS4231_MIXS_ALL, (unsigned char)nleft);
			snd_cs4231_outm(codec, CS4231_RIGHT_INPUT, (unsigned char)~CS4231_MIXS_ALL, (unsigned char)nright);
		} else {
			change = -EINVAL;
		}
	} else {
		elements[0] = snd_cs4231_mixer_get_recsrc_element(codec, left);
		elements[1] = snd_cs4231_mixer_get_recsrc_element(codec, right);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;
}

int snd_cs4231_mixer_stereo_volume(int w_flag, int *voices, cs4231_t *codec,
					  int max, int invert, int shift,
					  unsigned char left_reg,
					  unsigned char right_reg)
{
	int change = 0, left, right;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	left = (codec->image[left_reg] >> shift) & max;
	right = (codec->image[right_reg] >> shift) & max;
	if (!w_flag) {
		if (invert) {
			voices[0] = max - left;
			voices[1] = max - right;
		} else {
			voices[0] = left;
			voices[1] = right;
		}
	} else {
		if (invert) {
			change = max - left != voices[0] || max - right != voices[1];
		} else {
			change = left != voices[0] || right != voices[1];
		}
		left = voices[0]; right = voices[1];
		if (invert) {
			left = max - left;
			right = max - right;
		}
		snd_cs4231_outm(codec, left_reg, ~(max << shift), left << shift);
		snd_cs4231_outm(codec, right_reg, ~(max << shift), right << shift);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

int snd_cs4231_mixer_mono_volume(int w_flag, int *voices, cs4231_t *codec,
					  int max, int invert, int shift,
					  unsigned char reg)
{
	int change = 0, val;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	val = (codec->image[reg] >> shift) & max;
	if (!w_flag) {
		voices[0] = invert ? max - val : val;
	} else {
		change = (invert ? max - val : val) != voices[0];
		val = voices[0];
		if (invert)
			val = max - val;
		snd_cs4231_outm(codec, reg, ~(max << shift), val << shift);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

int snd_cs4231_mixer_stereo_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec,
					  int bit, int invert,
					  unsigned char left_reg,
					  unsigned char right_reg)
{
	int change = 0, left, right;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	left = (codec->image[left_reg] >> bit) & 1;
	right = (codec->image[right_reg] >> bit) & 1;
	if (!w_flag) {
		if (invert) {
			snd_mixer_set_bit(bitmap, 0, left ^ 1);
			snd_mixer_set_bit(bitmap, 1, right ^ 1);
		} else {
			snd_mixer_set_bit(bitmap, 0, left);
			snd_mixer_set_bit(bitmap, 1, right);
		}
	} else {
		if (invert) {
			change = (left ^ 1) != snd_mixer_get_bit(bitmap, 0) ||
			         (right ^ 1) != snd_mixer_get_bit(bitmap, 1);
		} else {
			change = left != snd_mixer_get_bit(bitmap, 0) ||
			         right != snd_mixer_get_bit(bitmap, 1);
		}
		left = snd_mixer_get_bit(bitmap, 0);
		right = snd_mixer_get_bit(bitmap, 1);
		if (invert) {
			left ^= 1;
			right ^= 1;
		}
		snd_cs4231_outm(codec, left_reg, ~(1 << bit), left << bit);
		snd_cs4231_outm(codec, right_reg, ~(1 << bit), right << bit);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

int snd_cs4231_mixer_mono_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec,
					  int bit, int invert,
					  unsigned char reg)
{
	int change = 0, val;
	unsigned long flags;

	snd_spin_lock(codec, reg, &flags);
	val = (codec->image[reg] >> bit) & 1;
	if (!w_flag) {
		snd_mixer_set_bit(bitmap, 0, invert ? val ^ 1 : val);
	} else {
		change = (invert ? val ^ 1 : val) != snd_mixer_get_bit(bitmap, 0);
		val = snd_mixer_get_bit(bitmap, 0);
		if (invert)
			val ^= 1;
		snd_cs4231_outm(codec, reg, ~(1 << bit), val << bit);
	}
	snd_spin_unlock(codec, reg, &flags);
	return change;	
}

static int snd_cs4231_mixer_mic_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_volume(w_flag, voices, codec,
					      1, 0, 5,
					      CS4231_LEFT_INPUT,
					      CS4231_RIGHT_INPUT);
}

int snd_cs4231_mixer_line_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_volume(w_flag, voices, codec,
					      31, 1, 0,
					      CS4231_LEFT_LINE_IN,
					      CS4231_RIGHT_LINE_IN);
}

int snd_cs4231_mixer_line_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_switch(w_flag, bitmap, codec,
					      7, 1,
					      CS4231_LEFT_LINE_IN,
					      CS4231_RIGHT_LINE_IN);
}

int snd_cs4231_mixer_aux1_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_volume(w_flag, voices, codec,
					      31, 1, 0,
					      CS4231_AUX1_LEFT_INPUT,
					      CS4231_AUX1_RIGHT_INPUT);
}

int snd_cs4231_mixer_aux1_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_switch(w_flag, bitmap, codec,
					      7, 1,
					      CS4231_AUX1_LEFT_INPUT,
					      CS4231_AUX1_RIGHT_INPUT);
}

int snd_cs4231_mixer_aux2_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_volume(w_flag, voices, codec,
					      31, 1, 0,
					      CS4231_AUX2_LEFT_INPUT,
					      CS4231_AUX2_RIGHT_INPUT);
}

int snd_cs4231_mixer_aux2_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_switch(w_flag, bitmap, codec,
					      7, 1,
					      CS4231_AUX2_LEFT_INPUT,
					      CS4231_AUX2_RIGHT_INPUT);
}

int snd_cs4231_mixer_monoin_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_mono_volume(w_flag, voices, codec,
					      15, 1, 0,
					      CS4231_MONO_CTRL);
}

int snd_cs4231_mixer_monoin_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec)
{
	return snd_cs4231_mixer_mono_switch(w_flag, bitmap, codec,
					      7, 1,
					      CS4231_MONO_CTRL);
}

int snd_cs4231_mixer_mono_bypass_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec)
{
	return snd_cs4231_mixer_mono_switch(w_flag, bitmap, codec,
					      5, 0,
					      CS4231_MONO_CTRL);
}

int snd_cs4231_mixer_igain_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_volume(w_flag, voices, codec,
					      15, 0, 0,
					      CS4231_LEFT_INPUT,
					      CS4231_RIGHT_INPUT);
}

static int snd_cs4231_mixer_loopback_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_mono_volume(w_flag, voices, codec,
					      63, 1, 2,
					      CS4231_LOOPBACK);
}

static int snd_cs4231_mixer_loopback_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec)
{
	return snd_cs4231_mixer_mono_switch(w_flag, bitmap, codec,
					      0, 0,
					      CS4231_LOOPBACK);
}

int snd_cs4231_mixer_dac_volume(int w_flag, int *voices, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_volume(w_flag, voices, codec,
					      63, 1, 0,
					      CS4231_LEFT_OUTPUT,
					      CS4231_RIGHT_OUTPUT);
}

int snd_cs4231_mixer_dac_switch(int w_flag, unsigned int *bitmap, cs4231_t *codec)
{
	return snd_cs4231_mixer_stereo_switch(w_flag, bitmap, codec,
					      7, 1,
					      CS4231_LEFT_OUTPUT,
					      CS4231_RIGHT_OUTPUT);
}

snd_kmixer_t *snd_cs4231_new_mixer(snd_pcm_t * pcm, int pcm_dev)
{
	snd_pcm1_t *pcm1;
	cs4231_t *codec;
	snd_kmixer_t *mixer;
	snd_kmixer_element_t *mux;		/* input MUX */
	snd_kmixer_element_t *accu;		/* output accumulator */
	snd_kmixer_element_t *mono_accu;	/* mono output accumulator */
	snd_kmixer_element_t *dig_accu;		/* digital accumulator */
	snd_kmixer_element_t *element1, *element2, *element3, *element4;
	snd_kmixer_group_t *group;
	static struct snd_mixer_element_volume1_range mic_range[2] = {
		{0, 1, 0, 2000},
		{0, 1, 0, 2000}
	};
	static struct snd_mixer_element_volume1_range table4_range[2] = {
		{0, 15, 0, 2250},
		{0, 15, 0, 2250}
	};
	static struct snd_mixer_element_volume1_range table5_range[2] = {
		{0, 31, -3450, 1200},
		{0, 31, -3450, 1200}
	};
	static struct snd_mixer_element_volume1_range table6_range[2] = {
		{0, 63, -9450, 0},
		{0, 63, -9450, 0}
	};
	static struct snd_mixer_element_volume1_range table7_range[1] = {
		{0, 15, -4500, 0},
	};

	if (!pcm || !pcm->card)
		return NULL;
	pcm1 = (snd_pcm1_t *) pcm->private_data;
	codec = (cs4231_t *) pcm1->private_data;
	if (!codec)
		return NULL;
	mixer = snd_mixer_new(pcm->card, pcm->id);
	if (!mixer)
		return NULL;
	strcpy(mixer->name, pcm->name);

	/* build input MUX and accumulators */
	if ((mux = snd_mixer_lib_mux1(mixer, SND_MIXER_ELEMENT_INPUT_MUX, 0, 0, 2, (snd_mixer_mux1_control_t *)snd_cs4231_mixer_mux, codec)) == NULL)
		goto __error;
	if ((accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, accu, mux) < 0)
		goto __error;
	codec->me_mux_mix = accu;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_ETYPE_OUTPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, accu, element1) < 0)
		goto __error;
	if ((mono_accu = snd_mixer_lib_accu2(mixer, SND_MIXER_ELEMENT_MONO_OUT_ACCU, 0, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, accu, mono_accu) < 0)
		goto __error;
	if ((element1 = snd_mixer_lib_io_mono(mixer, SND_MIXER_OUT_MASTER_MONO, 0, SND_MIXER_ETYPE_OUTPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, mono_accu, element1) < 0)
		goto __error;
	if ((dig_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_DIGITAL_ACCU, 0, 0)) == NULL)
		goto __error;
	/* build MIC input */ 
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_MIC, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "MIC Volume", 0, 2, mic_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_mic_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, mux) < 0)
		goto __error;
	codec->me_mux_mic = element2;
	/* build Line-In */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_LINE, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, mux) < 0)
		goto __error;
	codec->me_mux_line = element1;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Line Input Volume", 0, 2, table5_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_line_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "Line Input Switch", 0, 2, (snd_mixer_sw1_control_t *)snd_cs4231_mixer_line_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;
	/* build AUX1 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_AUX, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, mux) < 0)
		goto __error;
	codec->me_mux_aux1 = element1;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 0, 2, table5_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_aux1_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 0, 2, (snd_mixer_sw1_control_t *)snd_cs4231_mixer_aux1_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;
	/* build AUX2 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_AUX, 1)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 1, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 1, 2, table5_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_aux2_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 1, 2, (snd_mixer_sw1_control_t *)snd_cs4231_mixer_aux2_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;
	/* build MIN */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_MONO, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_io_mono(mixer, SND_MIXER_IN_MONO, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "Mono Input Volume", 0, 1, table7_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_monoin_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "Mono Input Switch", 0, 1, (snd_mixer_sw1_control_t *)snd_cs4231_mixer_monoin_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;
	/* build input GAIN */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_GRP_IGAIN, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_volume1(mixer, "Input Gain Volume", 0, 2, table4_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_igain_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, mux, element1) < 0)
		goto __error;
	/* build ADC */
	if ((element2 = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_ADC, 0, SND_MIXER_ETYPE_ADC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	/* build capture endpoint */
	if ((element3 = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE, 1, &pcm_dev)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	/* build loopback */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_GRP_DIGITAL_LOOPBACK, 0)) == NULL)
		goto __error;
	if ((element3 = snd_mixer_lib_volume1(mixer, "Loopback Volume", 0, 1, table4_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_loopback_volume, codec)) == NULL)
		goto __error;	
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if ((element4 = snd_mixer_lib_sw1(mixer, "Loopback Switch", 0, 1, (snd_mixer_sw1_control_t *)snd_cs4231_mixer_loopback_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element4) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, element4) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element4, dig_accu) < 0)
		goto __error;
	/* build playback endpoint */
	if ((element1 = snd_mixer_lib_pcm(mixer, SND_MIXER_ELEMENT_PLAYBACK, 0, SND_MIXER_ETYPE_PLAYBACK, 1, &pcm_dev)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, dig_accu) < 0)
		goto __error;
	/* build DAC */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_PCM, 0)) == NULL)
		goto __error;
	if ((element1 = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_DAC, 0, SND_MIXER_ETYPE_DAC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, dig_accu, element1) < 0)
		goto __error;
	if ((element2 = snd_mixer_lib_volume1(mixer, "PCM Volume", 0, 2, table6_range, (snd_mixer_volume1_control_t *)snd_cs4231_mixer_dac_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element1, element2) < 0)
		goto __error;
	if ((element3 = snd_mixer_lib_sw1(mixer, "PCM Switch", 0, 2, (snd_mixer_sw1_control_t *)snd_cs4231_mixer_dac_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element2, element3) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element3, accu) < 0)
		goto __error;

	mixer->private_data = codec;
	codec->mixer = mixer;
	return mixer;
      __error:
      	snd_mixer_free(mixer);
      	return NULL;
}

/*
 *  INIT part
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_cs4231_export;
#endif

int init_module(void)
{
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_cs4231_export) < 0)
		return -ENOMEM;
#endif
	return 0;
}

void cleanup_module(void)
{
}
