/*
 *  Advanced Linux Sound Architecture
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "driver.h"
#include "minors.h"
#include "info.h"
#include "sndpersist.h"
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif
#ifdef CONFIG_KERNELD
#include <linux/kerneld.h>
#endif

#if 0
#define DEBUG_ACCESS( d, m ) printk( ">>" d "[%i] - begin\n", m )
#define DEBUG_RETURN( d, m, f ) \
	do { long result = f; \
	     printk( ">>" d "[%i] - end (result = %li)\n", m, result ); \
	     return result; } while( 0 )
#else
#define DEBUG_ACCESS( d, m )	/* nothing */
#define DEBUG_RETURN( d, m, f ) return(f)
#endif

int snd_major = CONFIG_SND_MAJOR;
#if !defined(LINUX_2_2) && defined(CONFIG_SND_OSSEMUL)
int snd_oss_major = 14;		/* not configurable item at this time */
#endif
int snd_cards_limit = SND_CARDS, snd_ecards_limit;
int snd_device_mode = S_IFCHR | S_IRUGO | S_IWUGO;
int snd_device_gid = 0;
int snd_device_uid = 0;
#ifdef MODULE_PARM		/* hey - we have new 2.1.18+ kernel... */
MODULE_AUTHOR("Jaroslav Kysela <perex@suse.cz>");
MODULE_DESCRIPTION("Advanced Linux Sound Architecture driver for soundcards.");
MODULE_SUPPORTED_DEVICE("sound");
MODULE_PARM(snd_major, "i");
MODULE_PARM_DESC(snd_major, "Major # for sound driver. (116 by default)");
MODULE_PARM(snd_cards_limit, "1-" __MODULE_STRING(SND_CARDS) "i");
#if !defined(LINUX_2_2) && defined(CONFIG_SND_OSSEMUL)
MODULE_PARM_DESC(snd_major, "Major # for sound driver. (14 by default)");
MODULE_PARM(snd_cards_limit, "1-" __MODULE_STRING(SND_CARDS) "i");
#endif
MODULE_PARM_DESC(snd_cards_limit, "Count of soundcards installed in the system.");
MODULE_PARM(snd_device_mode, "i");
MODULE_PARM_DESC(snd_device_mode, "Device file permission mask for sound dynamic device filesystem.");
MODULE_PARM(snd_device_gid, "i");
MODULE_PARM_DESC(snd_device_gid, "Device file GID for sound dynamic device filesystem.");
MODULE_PARM(snd_device_uid, "i");
MODULE_PARM_DESC(snd_device_uid, "Device file UID for sound dynamic device filesystem.");
#endif

#define SND_OS_MINORS		256

static int snd_minors_device[SND_OS_MINORS] = {[0 ... (32 - 1)] = -1};
static snd_minor_t *snd_minors[SND_OS_MINORS] = {[0 ... (SND_OS_MINORS - 1)] = NULL};

snd_mutex_define_static(sound);

static inline snd_minor_t *snd_verify_card(unsigned short minor)
{
	int dev;

	dev = minor & SND_MINOR_MASK;
	if (dev != SND_MINOR_SEQUENCER) {
		if (!(snd_cards_bitmap & (1 << (minor >> 5))))
			return NULL;
	}
	return snd_minors[minor];
}

#if defined(CONFIG_KERNELD) || defined(CONFIG_KMOD)

static void snd_request_card(int card)
{
	char str[32];

	if (card < 0 || card >= snd_ecards_limit)
		return;
	sprintf(str, "snd-card-%i", card);
	request_module(str);
}

#endif				/* request_module support */

static inline snd_minor_t *snd_verify_card_open(unsigned short minor)
{
	int dev = minor & SND_MINOR_MASK;

	dev = minor & SND_MINOR_MASK;
	if (dev != SND_MINOR_SEQUENCER) {
		if (!(snd_cards_bitmap & (1 << (minor >> 5)))) {
#if defined(CONFIG_KERNELD) || defined(CONFIG_KMOD)
			snd_request_card(minor >> 5);
			if (!(snd_cards_bitmap & (1 << (minor >> 5))))
#endif
				return NULL;
		}
	}
	return snd_minors[minor];
}

#ifdef LINUX_2_1
static loff_t snd_lseek(struct file *file, loff_t offset, int orig)
#else
static int snd_lseek(struct inode *inode, struct file *file,
		     off_t offset, int orig)
#endif				/* LINUX_2_1 */
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_lseek_t *ptr;

	DEBUG_ACCESS("lseek", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->lseek) == NULL)
		return -ESPIPE;
	DEBUG_RETURN("lseek", minor, ptr(file, offset, orig));
}

#ifdef LINUX_2_1
static ssize_t snd_read(struct file *file,
                        char *buf, size_t count,
                        loff_t * offset)
#else
static int snd_read(struct inode *inode, struct file *file,
                    char *buf, int count)
#endif
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_read_t *ptr;

	DEBUG_ACCESS("read", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->read) == NULL)
		return -ENXIO;
	DEBUG_RETURN("read", minor, ptr(file, buf, count));
}

#ifdef LINUX_2_1
static ssize_t snd_write(struct file *file,
			 const char *buf, size_t count,
			 loff_t * offset)
#else
static int snd_write(struct inode *inode, struct file *file,
		     const char *buf, int count)
#endif
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_write_t *ptr;

	DEBUG_ACCESS("write", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->write) == NULL)
		return -ENXIO;
	DEBUG_RETURN("write", minor, ptr(file, buf, count));
}

static int snd_open(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_open_t *ptr;

	DEBUG_ACCESS("open", minor);
	if ((mptr = snd_verify_card_open(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->open) == NULL)
		return -ENXIO;
	DEBUG_RETURN("open", minor, ptr(minor + 256, minor >> 5, snd_minors_device[minor], file));
}

static int snd_release(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_release_t *ptr;

	DEBUG_ACCESS("release", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->release) == NULL)
		return -ENXIO;
	DEBUG_RETURN("release", minor, ptr(minor + 256, minor >> 5, snd_minors_device[minor], file));
}

#ifdef SND_POLL
static unsigned int snd_poll(struct file *file, poll_table * wait)
{
#ifdef LINUX_2_1
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
#else
	unsigned short minor = MINOR(file->f_inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_poll_t *ptr;

	DEBUG_ACCESS("poll", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->poll) == NULL)
		return -ENODEV;
	DEBUG_RETURN("poll", minor, ptr(file, wait));
}
#else
static int snd_select(struct inode *inode, struct file *file,
		      int sel_type, select_table * wait)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_select_t *ptr;

	DEBUG_ACCESS("select", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->select) == NULL)
		return -ENXIO;
	DEBUG_RETURN("select", minor, ptr(file, sel_type, wait));
}
#endif

static int snd_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_ioctl_t *ptr;

	DEBUG_ACCESS("ioctl", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->ioctl) == NULL)
		return -ENXIO;
	DEBUG_RETURN("ioctl", minor, ptr(file, cmd, arg));
}

#ifdef LINUX_2_1
static int snd_mmap(struct file *file, struct vm_area_struct *vma)
#else
static int snd_mmap(struct inode *inode, struct file *file,
                    struct vm_area_struct *vma)
#endif
{
#ifdef LINUX_2_1
	struct inode *inode = file->f_dentry->d_inode;
	unsigned short minor = MINOR(inode->i_rdev);
#else
	unsigned short minor = MINOR(file->f_inode->i_rdev);
#endif
	snd_minor_t *mptr;
	snd_mmap_t *ptr;

	DEBUG_ACCESS("mmap", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->mmap) == NULL)
		return -ENXIO;
	DEBUG_RETURN("mmap", minor, ptr(inode, file, vma));
}

struct file_operations snd_fops =
{
	snd_lseek,		/* (l)lseek */
	snd_read,		/* read */
	snd_write,		/* write */
	NULL,			/* readdir */
#ifdef SND_POLL
	snd_poll,		/* poll */
#else
	snd_select,		/* select */
#endif
	snd_ioctl,		/* ioctl */
	snd_mmap,		/* mmap */
	snd_open,		/* open */
#ifdef SND_FOPS_FLUSH
	NULL,			/* flush */
#endif
#ifdef LINUX_2_1
	snd_release,		/* release */
#else
	(void (*)(struct inode * inode, struct file * file)) snd_release,
#endif
	NULL,			/* fsync */
#ifdef LINUX_2_1
	NULL,			/* fasync */
	NULL,			/* check_media_change */
	NULL,			/* revalidate */
	NULL,			/* lock */
#endif
};

#ifndef LINUX_2_1

int snd_ioctl_in(long *addr)
{
	if (verify_area(VERIFY_READ, addr, sizeof(int)) != 0) {
		snd_printd("snd_ioctl_in: bad address 0x%lx\n", (long) addr);
		return 0;
	}
	return get_fs_long(addr);	/* not really long - 32-bit int */
}

int snd_ioctl_out(long *addr, int value)
{
	int err;

	if ((err = verify_area(VERIFY_WRITE, addr, sizeof(int))) != 0)
		 return err;
	put_fs_long(value, addr);	/* not really long - 32-bit int */
	return 0;
}

#else

int snd_ioctl_in(long *addr)
{
	int value;

	if (verify_area(VERIFY_READ, addr, sizeof(int)) != 0) {
		snd_printd("snd_ioctl_in: bad address 0x%lx\n", (long) addr);
		return 0;
	}
	get_user(value, addr);
	return value;
}

int snd_ioctl_out(long *addr, int value)
{
	int err;

	if ((err = verify_area(VERIFY_WRITE, addr, sizeof(int))) != 0)
		 return err;
	put_user(value, addr);
	return 0;
}

#endif				/* !LINUX_2_1 */

static void snd_init_minors(void)
{
	int idx;

	for (idx = 0; idx <= SND_MINOR_MASK; idx++)
		snd_minors_device[idx] = -1;
	for (idx = 0; idx < SND_MINOR_MIXERS; idx++)
		snd_minors_device[idx + SND_MINOR_MIXER] = idx;
	for (idx = 0; idx < SND_MINOR_RAWMIDIS; idx++)
		snd_minors_device[idx + SND_MINOR_RAWMIDI] = idx;
	for (idx = 0; idx < SND_MINOR_PCMS; idx++)
		snd_minors_device[idx + SND_MINOR_PCM] = idx;
}

static int snd_kernel_minor(int type, snd_card_t * card, int dev)
{
	int minor;

	switch (type) {
	case SND_DEVICE_TYPE_SEQUENCER:
	case SND_DEVICE_TYPE_TIMER:
		minor = type;
		break;
	case SND_DEVICE_TYPE_CONTROL:
		if (!card)
			return -EINVAL;
		minor = (card->number * (SND_MINOR_MASK + 1)) + type;
		break;
	case SND_DEVICE_TYPE_MIXER:
	case SND_DEVICE_TYPE_RAWMIDI:
	case SND_DEVICE_TYPE_PCM:
		if (!card)
			return -EINVAL;
		minor = (card->number * (SND_MINOR_MASK + 1)) + type + dev;
		break;
	default:
		return -EINVAL;
	}
	if (minor < 0 || minor > SND_OS_MINORS)
		return -EINVAL;
	return minor;
}

int snd_register_device(int type, snd_card_t * card, int dev, snd_minor_t * reg, const char *name)
{
	int minor = snd_kernel_minor(type, card, dev);
	snd_minor_t *preg;

	if (minor < 0)
		return minor;
	preg = (snd_minor_t *)snd_malloc(sizeof(snd_minor_t));
	if (preg == NULL)
		return -ENOMEM;
	memcpy(preg, reg, sizeof(snd_minor_t));
	preg->dev = NULL;
	snd_mutex_down_static(sound);
	if (snd_minors[minor]) {
		snd_mutex_up_static(sound);
		snd_free(preg, sizeof(snd_minor_t));
		return -EBUSY;
	}
	preg->dev = snd_info_create_device(name, minor, 0);
	snd_minors[minor] = preg;
	snd_mutex_up_static(sound);
	return 0;
}

int snd_unregister_device(int type, snd_card_t * card, int dev)
{
	int minor = snd_kernel_minor(type, card, dev);
	snd_minor_t *mptr;

	if (minor < 0)
		return minor;
	snd_mutex_down_static(sound);
	if ((mptr = snd_minors[minor]) == NULL) {
		snd_mutex_up_static(sound);
		return -EINVAL;
	}
	if (mptr->dev)
		snd_info_free_device(mptr->dev);
	snd_minors[minor] = NULL;
	snd_mutex_up_static(sound);
	snd_free(mptr, sizeof(snd_minor_t));
	return 0;
}

/*
 *  INFO PART
 */

static snd_info_entry_t *snd_minor_info_entry = NULL;

static void snd_minor_info_read(snd_info_buffer_t * buffer, void *private_data)
{
	int idx, device;
	snd_minor_t *mptr;

	snd_mutex_down_static(sound);
	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		if ((mptr = snd_minors[idx]) == NULL)
			continue;
		if (idx != SND_MINOR_SEQUENCER && idx != SND_MINOR_TIMER) {
			if ((device = snd_minors_device[idx & SND_MINOR_MASK]) >= 0)
				snd_iprintf(buffer, "%3i: [%i-%2i]: %s\n", idx, idx >> 5, device, mptr->comment);
			else
				snd_iprintf(buffer, "%3i: [%i]   : %s\n", idx, idx >> 5, mptr->comment);
		} else {
			snd_iprintf(buffer, "%3i:       : %s\n", idx, mptr->comment);
		}
	}
	snd_mutex_up_static(sound);
}

int snd_minor_info_init(void)
{
	snd_info_entry_t *entry;

	entry = snd_info_create_entry(NULL, "devices");
	if (entry) {
		entry->t.text.read_size = PAGE_SIZE;
		entry->t.text.read = snd_minor_info_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	snd_minor_info_entry = entry;
	return 0;
}

int snd_minor_info_done(void)
{
	if (snd_minor_info_entry)
		snd_info_unregister(snd_minor_info_entry);
	return 0;
}

/*
 *  persistant part
 */

#if defined(LINUX_2_1) || LinuxVersionCode(2, 0, 34) <= LINUX_VERSION_CODE

void (*snd_L_persist_lock) (int lock);
int (*snd_L_persist_store) (char *key, const char *data, int data_len) = NULL;
int (*snd_L_persist_restore) (char *key, char *data, int data_len) = NULL;
int (*snd_L_persist_length) (char *key) = NULL;
int (*snd_L_persist_present) (char *key) = NULL;
int (*snd_L_persist_remove) (char *key) = NULL;

static void snd_persist_init(void)
{
	unsigned long flags;

	snd_cli(&flags);
	snd_L_persist_lock = (void *) get_module_symbol(NULL, "persist_lock");
	if (snd_L_persist_lock) {
		snd_L_persist_lock(1);
	} else {
		snd_sti(&flags);
		return;
	}
	snd_sti(&flags);
	snd_L_persist_store = (void *) get_module_symbol(NULL, "persist_store");
	snd_L_persist_restore = (void *) get_module_symbol(NULL, "persist_restore");
	snd_L_persist_length = (void *) get_module_symbol(NULL, "persist_length");
	snd_L_persist_present = (void *) get_module_symbol(NULL, "persist_present");
	snd_L_persist_remove = (void *) get_module_symbol(NULL, "persist_remove");
#if 0
	printk("persist ok - remove = 0x%lx\n", (long) snd_L_persist_remove);
#endif
}

static void snd_persist_done(void)
{
	if (snd_L_persist_lock)
		snd_L_persist_lock(0);
}

int snd_persist_store(char *key, const char *data, int data_len)
{
	if (snd_L_persist_store)
		return snd_L_persist_store(key, data, data_len);
	return -ENXIO;
}

int snd_persist_restore(char *key, char *data, int data_len)
{
	if (snd_L_persist_store)
		return snd_L_persist_restore(key, data, data_len);
	return -ENXIO;
}

int snd_persist_length(char *key)
{
	if (snd_L_persist_length)
		return snd_L_persist_length(key);
	return -ENXIO;
}

int snd_persist_present(char *key)
{
	if (snd_L_persist_present)
		return snd_L_persist_present(key);
	return -ENXIO;
}

int snd_persist_remove(char *key)
{
	if (snd_L_persist_remove)
		return snd_L_persist_remove(key);
	return -ENXIO;
}

#endif

/*
 *  INIT PART
 */

#ifndef LINUX_2_1
extern struct symbol_table snd_symbol_table_snd_export;
#endif

int init_module(void)
{
#ifdef CONFIG_SND_OSSEMUL
	int err;
#endif

	snd_ecards_limit = snd_cards_limit;
#ifndef LINUX_2_1
	if (register_symtab(&snd_symbol_table_snd_export) < 0)
		return -ENOMEM;
#endif
	snd_init_minors();
#ifdef CONFIG_SND_OSSEMUL
	if ((err = snd_oss_init_module()) < 0)
		return err;
#endif
	if (register_chrdev(snd_major, "alsa", &snd_fops)) {
		snd_printk("unable to register native major device number %d\n", snd_major);
#ifdef CONFIG_SND_OSSEMUL
		snd_oss_cleanup_module();
#endif
		return -EIO;
	}
	snd_malloc_init();
	snd_driver_init();
	if (snd_info_init() < 0) {
		snd_malloc_done();
#ifdef CONFIG_SND_OSSEMUL
		snd_oss_cleanup_module();
#endif
		return -ENOMEM;
	}
#ifdef CONFIG_SND_OSSEMUL
	snd_info_minor_register();
#endif
#if defined(LINUX_2_1) || LinuxVersionCode(2, 0, 34) <= LINUX_VERSION_CODE
	snd_persist_init();
#endif
	return 0;
}

void cleanup_module(void)
{
	int idx;
	snd_minor_t *minor;

	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		minor = snd_minors[idx];
		if (minor && minor->unregister)
			minor->unregister(idx);
		minor = snd_minors[idx];
		if (minor) {
			snd_minors[idx] = NULL;
			if (minor->dev)
				snd_info_free_device(minor->dev);
			snd_free(minor, sizeof(snd_minor_t));
		}
	}
#ifdef CONFIG_SND_OSSEMUL
	snd_info_minor_unregister();
	snd_oss_cleanup_module();
#endif
	snd_info_done();
	snd_malloc_done();
#if defined(LINUX_2_1) || LinuxVersionCode(2, 0, 34) <= LINUX_VERSION_CODE
	snd_persist_done();
#endif
	if (unregister_chrdev(snd_major, "alsa") != 0)
		snd_printk("unable to unregister major device number %d\n", snd_major);
}
