/*
 * OSS compatible sequencer driver
 *
 * seq_oss_readq.c - MIDI input queue
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "seq_oss_readq.h"
#include "seq_oss_event.h"
#include "seq_oss_legacy.h"

/*
 * constants
 */
//#define SND_SEQ_OSS_MAX_TIMEOUT	(unsigned long)(-1)
#define SND_SEQ_OSS_MAX_TIMEOUT	(HZ * 3600)


/*
 * prototypes
 */


/*
 * create a read queue
 */
seq_oss_readq_t *
snd_seq_oss_readq_new(seq_oss_devinfo_t *dp, int maxlen)
{
	seq_oss_readq_t *q;

	if ((q = snd_malloc(sizeof(*q))) == NULL) {
		snd_printk("sequencer: can't malloc read queue\n");
		return NULL;
	}

	if ((q->q = snd_malloc(sizeof(evrec_t) * maxlen)) == NULL) {
		snd_printk("sequencer: can't malloc read queue buffer\n");
		snd_free(q, sizeof(*q));
		return NULL;
	}

	q->maxlen = maxlen;
	q->qlen = 0;
	q->head = q->tail = 0;
	snd_sleep_prepare(q, midi);
	snd_getlock(q, midi) = 0;
	snd_spin_prepare(q, lock);
	q->pre_event_timeout = SND_SEQ_OSS_MAX_TIMEOUT;
	q->input_time = (unsigned long)-1;

	return q;
}

/*
 * delete the read queue
 */
void
snd_seq_oss_readq_delete(seq_oss_readq_t *q)
{
	if (q) {
		snd_seq_oss_readq_clear(q);	/* to be sure */
		if (q->q)
			snd_free(q->q, sizeof(evrec_t) * q->maxlen);
		snd_free(q, sizeof(*q));
	}
}

/*
 * reset the read queue
 */
void
snd_seq_oss_readq_clear(seq_oss_readq_t *q)
{
	if (q->qlen) {
		q->qlen = 0;
		q->head = q->tail = 0;
	}
	/* if someone sleeping, wake'em up */
	snd_seq_oss_readq_wakeup(q);
	q->input_time = (unsigned long)-1;
}

/*
 * put a midi byte
 */
int
snd_seq_oss_readq_putc(seq_oss_readq_t *q, int dev, int data)
{
	evrec_t rec;
	rec.c[0] = SND_OSS_SEQ_MIDIPUTC;
	rec.c[1] = data;
	rec.c[2] = dev;
	rec.c[3] = 0;
	return snd_seq_oss_readq_put_event(q, &rec);
}

/*
 * copy an event to input queue:
 * return zero if enqueued
 */
int
snd_seq_oss_readq_put_event(seq_oss_readq_t *q, evrec_t *ev)
{
	unsigned long flags;

	if (q->qlen >= q->maxlen - 1)
		return -ENOSPC;

	snd_spin_lock(q, lock, &flags);
	memcpy(&q->q[q->tail], ev, ev_length(ev));
	q->tail = (q->tail + 1) % q->maxlen;
	q->qlen++;
	snd_spin_unlock(q, lock, &flags);

	snd_seq_oss_readq_wakeup(q);
	return 0;
}

/*
 * drain queue
 */
void
snd_seq_oss_readq_free(seq_oss_readq_t *q, evrec_t *ev)
{
	if (q->qlen > 0) {
		unsigned long flags;
		snd_spin_lock(q, lock, &flags);
		q->head = (q->head + 1) % q->maxlen;
		q->qlen--;
		snd_spin_unlock(q, lock, &flags);
	}
}

/*
 * sleep until some data is sent
 */
void
snd_seq_oss_readq_sleep(seq_oss_readq_t *q)
{
	unsigned long flags;
	snd_spin_lock(q, lock, &flags);
	snd_getlock(q, midi) |= SND_WK_SLEEP;
	snd_sleep(q, midi, q->pre_event_timeout);
	snd_getlock(q, midi) &= ~SND_WK_SLEEP;
	snd_spin_unlock(q, lock, &flags);
}

/*
 * if someone is suspended for reading, wake them up
 */
void
snd_seq_oss_readq_wakeup(seq_oss_readq_t *q)
{
	unsigned long flags;
	snd_spin_lock(q, lock, &flags);
	if (q->qlen > 0 && snd_getlock(q, midi) & SND_WK_SLEEP) {
		snd_getlock(q, midi) &= ~SND_WK_SLEEP;
		snd_wakeup(q, midi);
	}
	snd_spin_unlock(q, lock, &flags);
}

/*
 * polling/select:
 * return non-zero if readq is not empty.
 */
#ifdef SND_POLL
unsigned int
snd_seq_oss_readq_poll(seq_oss_readq_t *q, struct file *file, poll_table *wait)
{
	unsigned long flags;
	snd_spin_lock(q, lock, &flags);
	if (! q->qlen) {
		snd_getlock(q, midi) |= SND_WK_SLEEP;
		snd_poll_wait(file, q, midi, wait);
	}
	snd_spin_unlock(q, lock, &flags);
	return q->qlen;
}
#else
int
snd_seq_oss_readq_select(seq_oss_readq_t *q, select_table *wait)
{
	unsigned long flags;
	snd_spin_lock(q, lock, &flags);
	if (! q->qlen) {
		snd_getlock(q, midi) |= SND_WK_SLEEP;
		snd_select_wait(q, midi, wait);
		snd_spin_unlock(q, lock, &flags);
		return 0;
	}
	snd_getlock(q, midi) &= ~SND_WK_SLEEP;
	snd_spin_unlock(q, lock, &flags);
	return 1;
}
#endif

/*
 * put a timestamp
 */
int
snd_seq_oss_readq_put_timestamp(seq_oss_readq_t *q, unsigned long curt, int seq_mode)
{
	if (curt != q->input_time) {
		evrec_t rec;
		switch (seq_mode) {
		case SND_SEQ_OSS_MODE_SYNTH:
			rec.echo = (curt << 8) | SND_OSS_SEQ_WAIT;
			snd_seq_oss_readq_put_event(q, &rec);
			break;
		case SND_SEQ_OSS_MODE_MUSIC:
			rec.t.code = SND_OSS_EV_TIMING;
			rec.t.cmd = SND_OSS_TMR_WAIT_ABS;
			rec.t.time = curt;
			snd_seq_oss_readq_put_event(q, &rec);
			break;
		}
		q->input_time = curt;
	}
	return 0;
}


/*
 * proc interface
 */
void
snd_seq_oss_readq_info_read(seq_oss_readq_t *q, snd_info_buffer_t *buf)
{
	snd_iprintf(buf, "  read queue length %d\n", q->qlen);
	snd_iprintf(buf, "  read queue tick %d\n", q->input_time);
	if (snd_getlock(q, midi) & SND_WK_SLEEP)
		snd_iprintf(buf, "  readq sleeping\n");
}
