/*
 *  Routines for control of the TEA6330T circuit via i2c bus
 *  Sound fader control circuit for car radios by Philips Semiconductors
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SNDRV_MAIN_OBJECT_FILE
#include <sound/driver.h>
#include <sound/tea6330t.h>

#define chip_t tea6330t_t

#define TEA6330T_ADDR			0x80	/* fixed address */

#define TEA6330T_SADDR_VOLUME_LEFT	0x00	/* volume left */
#define TEA6330T_SADDR_VOLUME_RIGHT	0x01	/* volume right */
#define TEA6330T_SADDR_BASS		0x02	/* bass control */
#define TEA6330T_SADDR_TREBLE		0x03	/* treble control */
#define TEA6330T_SADDR_FADER		0x04	/* fader control */
#define   TEA6330T_MFN			0x20	/* mute control for selected channels */
#define   TEA6330T_FCH			0x10	/* select fader channels - front or rear */
#define TEA6330T_SADDR_AUDIO_SWITCH	0x05	/* audio switch */
#define   TEA6330T_GMU			0x80	/* mute control, general mute */
#define   TEA6330T_EQN			0x40	/* equalizer switchover (0=equalizer-on) */

int snd_tea6330t_detect(struct snd_i2c_bus *bus, int equalizer)
{
	equalizer = equalizer ? 0 : TEA6330T_EQN;
	/* mute & turn on/off equalizer */
	if (snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_AUDIO_SWITCH, TEA6330T_GMU | equalizer, 1) < 0)
		return -ENODEV;
	/* fader off */
	if (snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_FADER, 0x3f, 1) < 0)
		return -ENODEV;
	return 0;
}

#if 0
static void snd_tea6330t_set(tea6330t_t *tea,
			     unsigned char addr, unsigned char value)
{
#if 0
	printk("set - 0x%x/0x%x\n", addr, value);
#endif
	snd_i2c_write(tea->bus, TEA6330T_ADDR, addr, value, 1);
}
#endif

#define TEA6330T_MASTER_VOLUME(xname, xindex) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_tea6330t_info_master_volume, \
  get: snd_tea6330t_get_master_volume, put: snd_tea6330t_put_master_volume }

static int snd_tea6330t_info_master_volume(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 43;
	return 0;
}

static int snd_tea6330t_get_master_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	
	spin_lock_irqsave(&tea->reg_lock, flags);
	ucontrol->value.integer.value[0] = tea->mleft - 0x14;
	ucontrol->value.integer.value[1] = tea->mright - 0x14;
	spin_unlock_irqrestore(&tea->reg_lock, flags);
	return 0;
}

static int snd_tea6330t_put_master_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int change;
	unsigned char val1, val2;
	
	val1 = (ucontrol->value.integer.value[0] % 44) + 0x14;
	val2 = (ucontrol->value.integer.value[1] % 44) + 0x14;
	spin_lock_irqsave(&tea->reg_lock, flags);
	change = val1 != tea->mleft || val2 != tea->mright;
	tea->mleft = val1;
	tea->mright = val2;
	if (tea->regs[TEA6330T_SADDR_VOLUME_LEFT] != 0)
		snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_LEFT, tea->regs[TEA6330T_SADDR_VOLUME_LEFT] = tea->mleft, 1);
	if (tea->regs[TEA6330T_SADDR_VOLUME_RIGHT] != 0)
		snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_RIGHT, tea->regs[TEA6330T_SADDR_VOLUME_RIGHT] = tea->mright, 1);
	spin_unlock_irqrestore(&tea->reg_lock, flags);
	return change;
}

#define TEA6330T_MASTER_SWITCH(xname, xindex) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_tea6330t_info_master_switch, \
  get: snd_tea6330t_get_master_switch, put: snd_tea6330t_put_master_switch }

static int snd_tea6330t_info_master_switch(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;
	return 0;
}

static int snd_tea6330t_get_master_switch(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	
	spin_lock_irqsave(&tea->reg_lock, flags);
	ucontrol->value.integer.value[0] = tea->regs[TEA6330T_SADDR_VOLUME_LEFT] == 0 ? 0 : 1;
	ucontrol->value.integer.value[1] = tea->regs[TEA6330T_SADDR_VOLUME_RIGHT] == 0 ? 0 : 1;
	spin_unlock_irqrestore(&tea->reg_lock, flags);
	return 0;
}

static int snd_tea6330t_put_master_switch(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int change;
	unsigned char oval1, oval2, val1, val2;
	
	val1 = ucontrol->value.integer.value[0] & 1;
	val2 = ucontrol->value.integer.value[1] & 1;
	spin_lock_irqsave(&tea->reg_lock, flags);
	oval1 = tea->regs[TEA6330T_SADDR_VOLUME_LEFT] == 0 ? 0 : 1;
	oval2 = tea->regs[TEA6330T_SADDR_VOLUME_RIGHT] == 0 ? 0 : 1;
	change = val1 != oval1 || val2 != oval2;
	tea->regs[TEA6330T_SADDR_VOLUME_LEFT] = val1 ? tea->mleft : 0;
	tea->regs[TEA6330T_SADDR_VOLUME_RIGHT] = val2 ? tea->mright : 0;
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_LEFT, tea->regs[TEA6330T_SADDR_VOLUME_LEFT], 1);
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_RIGHT, tea->regs[TEA6330T_SADDR_VOLUME_RIGHT], 1);
	spin_unlock_irqrestore(&tea->reg_lock, flags);
	return change;
}

#define TEA6330T_BASS(xname, xindex) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_tea6330t_info_bass, \
  get: snd_tea6330t_get_bass, put: snd_tea6330t_put_bass }

static int snd_tea6330t_info_bass(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);

	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = tea->max_bass;
	return 0;
}

static int snd_tea6330t_get_bass(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	
	ucontrol->value.integer.value[0] = tea->bass;
	return 0;
}

static int snd_tea6330t_put_bass(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int change;
	unsigned char val1;
	
	val1 = ucontrol->value.integer.value[0] % (tea->max_bass + 1);
	spin_lock_irqsave(&tea->reg_lock, flags);
	tea->bass = val1;
	val1 += tea->equalizer ? 7 : 3;
	change = tea->regs[TEA6330T_SADDR_BASS] != val1;
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_BASS, tea->regs[TEA6330T_SADDR_BASS] = val1, 1);
	spin_unlock_irqrestore(&tea->reg_lock, flags);
	return change;
}

#define TEA6330T_TREBLE(xname, xindex) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_tea6330t_info_treble, \
  get: snd_tea6330t_get_treble, put: snd_tea6330t_put_treble }

static int snd_tea6330t_info_treble(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);

	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = tea->max_treble;
	return 0;
}

static int snd_tea6330t_get_treble(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	
	ucontrol->value.integer.value[0] = tea->treble;
	return 0;
}

static int snd_tea6330t_put_treble(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	tea6330t_t *tea = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int change;
	unsigned char val1;
	
	val1 = ucontrol->value.integer.value[0] % (tea->max_treble + 1);
	spin_lock_irqsave(&tea->reg_lock, flags);
	tea->treble = val1;
	val1 += 3;
	change = tea->regs[TEA6330T_SADDR_TREBLE] != val1;
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_TREBLE, tea->regs[TEA6330T_SADDR_TREBLE] = val1, 1);
	spin_unlock_irqrestore(&tea->reg_lock, flags);
	return change;
}

#define TEA6330T_CONTROLS (sizeof(snd_tea6330t_controls)/sizeof(snd_kcontrol_new_t))

static snd_kcontrol_new_t snd_tea6330t_controls[] = {
TEA6330T_MASTER_SWITCH("Master Playback Switch", 0),
TEA6330T_MASTER_VOLUME("Master Playback Volume", 0),
TEA6330T_BASS("Tone Control - Bass", 0),
TEA6330T_TREBLE("Tone Control - Treble", 0)
};

void snd_tea6330_free(struct snd_i2c_bus *bus)
{
	tea6330t_t *tea = snd_magic_cast(tea6330t_t, bus->private_data, return);
	snd_magic_kfree(tea);
}
                                        
int snd_tea6330t_update_mixer(snd_card_t * card,
			      struct snd_i2c_bus *bus,
			      int equalizer, int fader)
{
	tea6330t_t *tea;
	snd_kcontrol_new_t *knew;
	int idx, err = -ENOMEM;
	u8 default_treble, default_bass;

	tea = snd_magic_kcalloc(tea6330t_t, 0, GFP_KERNEL);
	if (tea == NULL)
		return -ENOMEM;
	spin_lock_init(&tea->reg_lock);
	tea->bus = bus;
	tea->equalizer = equalizer;
	tea->fader = fader;

	/* turn fader off */
	tea->regs[TEA6330T_SADDR_FADER] = 0x3f;
	if (snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_FADER, 0x3f, 1) < 0) {
		snd_magic_kfree(tea);
		return -ENODEV;
	}
	equalizer = tea->regs[TEA6330T_SADDR_AUDIO_SWITCH] = equalizer ? 0 : TEA6330T_EQN;
	/* mute off & turn on/off equalizer */
	if (snd_i2c_write(bus, TEA6330T_ADDR, TEA6330T_SADDR_AUDIO_SWITCH, equalizer, 1) < 0) {
		snd_magic_kfree(tea);
		return -ENODEV;
	}

	bus->private_data = tea;
	bus->private_free = snd_tea6330_free;
	strcat(card->mixername, ",TEA6330T");
	if ((err = snd_component_add(card, "TEA6330T")) < 0)
		goto __error;

	if (!tea->equalizer) {
		tea->max_bass = 9;
		tea->max_treble = 8;
		default_bass = 3 + 4;
		tea->bass = 4;
		default_treble = 3 + 4;
		tea->treble = 4;
	} else {
		tea->max_bass = 5;
		tea->max_treble = 0;
		default_bass = 7 + 4;
		tea->bass = 4;
		default_treble = 3;
		tea->treble = 0;
	}

	for (idx = 0; idx < TEA6330T_CONTROLS; idx++) {
		knew = &snd_tea6330t_controls[idx];
		if (tea->treble == 0 && !strcmp(knew->name, "Tone Control - Treble"))
			continue;
		if ((err = snd_ctl_add(card, snd_ctl_new1(knew, tea))) < 0)
			goto __error;
	}

	/* initialize mixer */
	tea->mleft = tea->mright = 0x14;
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_LEFT, 0, 1);
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_VOLUME_RIGHT, 0, 1);
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_BASS, tea->regs[TEA6330T_SADDR_BASS] = default_bass, 1);
	snd_i2c_write(tea->bus, TEA6330T_ADDR, TEA6330T_SADDR_TREBLE, tea->regs[TEA6330T_SADDR_TREBLE] = default_treble, 1);
	return 0;
	
      __error:
      	bus->private_free = NULL;
      	bus->private_data = NULL;
      	snd_magic_kfree(tea);
      	return err;
}

EXPORT_SYMBOL(snd_tea6330t_detect);
EXPORT_SYMBOL(snd_tea6330t_update_mixer);

/*
 *  INIT part
 */

static int __init alsa_tea6330t_init(void)
{
	return 0;
}

static void __exit alsa_tea6330t_exit(void)
{
}

module_init(alsa_tea6330t_init)
module_exit(alsa_tea6330t_exit)
