///////////////////////////////////////////////////////////////////////////
//      MOTU Midi Timepiece ALSA Main routines
//      Copyright by Michael T. Mayers (c) Jan 09, 2000
//      mail: tweakoz@pacbell.net
//      Thanks to John Galbraith
///////////////////////////////////////////////////////////////////////////
//      This program is free software; you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation; either version 2 of the License, or
//      (at your option) any later version.
//
//      This program is distributed in the hope that it will be useful,
//      but WITHOUT ANY WARRANTY; without even the implied warranty of
//      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//      GNU General Public License for more details.
//
//      You should have received a copy of the GNU General Public License
//      along with this program; if not, write to the Free Software
//      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
///////////////////////////////////////////////////////////////////////////
//      This driver is for the 'Mark Of The Unicorn' (MOTU) MidiTimePiece AV multiport MIDI interface 
//
//      IOPORTS
//      ///////////
//      8 MIDI Ins and 8 MIDI outs
//      Video Sync In (BNC), Word Sync Out (BNC), 
//      ADAT Sync Out (DB9)
//      SMPTE in/out (1/4")
//      2 programmable pedal/footswitch inputs and 4 programmable MIDI controller knobs.
//      Macintosh RS422 serial port
//      RS422 "network" port for ganging multiple MTP's
//      PC Parallel Port ( which this driver currently uses )
//
//      MISC FEATURES
//      ///////////////
//      Hardware MIDI routing, merging, and filtering   
//      MIDI Synchronization to Video, ADAT, SMPTE and other Clock sources
//      128 'scene' memories, recallable from MIDI program change
///////////////////////////////////////////////////////////////////////////
//      remember to set SNDRV_RAWMIDI_PORTS in rawmidi.h to >= 8 for 1 mtp or 16 for 2 mtps
///////////////////////////////////////////////////////////////////////////

#define SNDRV_MAIN_OBJECT_FILE

/////////////////////////////////////////////////////////////////
//      includes

#include <sound/driver.h>
#define SNDRV_GET_ID
#include <sound/initval.h>
#include <sound/rawmidi.h>

/////////////////////////////////////////////////////////////////
//      globals

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("MOTU MidiTimePiece AV multiport MIDI");
MODULE_CLASSES("{sound}");
MODULE_DEVICES("{{MOTU,MidiTimePiece AV multiport MIDI}}");

static int snd_index = SNDRV_DEFAULT_IDX1;
static char *snd_id = SNDRV_DEFAULT_STR1;
static long snd_port = 0x378;		/* 0x378, 0x278 */
static int snd_irq = 7;		/* 7, 5 */
static int snd_hwports = 8;		/* use hardware ports 1-8 */

MODULE_PARM(snd_index, "i");
MODULE_PARM_DESC(snd_index, "Index value for MotuMTPAV MIDI.");
MODULE_PARM_SYNTAX(snd_index, SNDRV_INDEX_DESC);
MODULE_PARM(snd_id, "s");
MODULE_PARM_DESC(snd_id, "ID string for MotuMTPAV MIDI.");
MODULE_PARM_SYNTAX(snd_id, SNDRV_ID_DESC);
MODULE_PARM(snd_port, "l");
MODULE_PARM_DESC(snd_port, "Parallel port # for MotuMTPAV MIDI.");
MODULE_PARM_SYNTAX(snd_port, SNDRV_PORT_DESC);
MODULE_PARM(snd_irq, "i");
MODULE_PARM_DESC(snd_irq, "Parallel IRQ # for MotuMTPAV MIDI.");
MODULE_PARM_SYNTAX(snd_irq, SNDRV_IRQ_DESC);
MODULE_PARM(snd_hwports, "i");
MODULE_PARM_DESC(snd_hwports, "Hardware ports # for MotuMTPAV MIDI.");
MODULE_PARM_SYNTAX(snd_hwports, SNDRV_ENABLED ",allows:{{1,127}},dialog:list");

///////////////////////////////////////////////////////////////////
//      defines

//#define USE_FAKE_MTP //       dont actually read/write to MTP device (for debugging without an actual unit) (does not work yet)

// io resources (make these module options?)
#define MTPAV_IOBASE 0x378
#define MTPAV_IRQ    7

// parallel port usage masks
#define SIGS_BYTE 0x08
#define SIGS_RFD 0x80
#define SIGS_IRQ 0x40
#define SIGS_IN0 0x10
#define SIGS_IN1 0x20

#define SIGC_WRITE 0x04
#define SIGC_READ 0x08
#define SIGC_INTEN 0x10

#define DREG 0
#define SREG 1
#define CREG 2

//
#define MTPAV_MODE_INPUT_OPENED		0x01
#define MTPAV_MODE_OUTPUT_OPENED	0x02
#define MTPAV_MODE_INPUT_TRIGGERED	0x04
#define MTPAV_MODE_OUTPUT_TRIGGERED	0x08

// possible hardware ports (selected by 0xf5 port message)
//      0x00		all ports
//      0x01 .. 0x08    this MTP's ports 1..8
//      0x09 .. 0x10    networked MTP's ports (9..16)
//      0x11            networked MTP's computer port
//      0x63            to ADAT
#define NUMPORTS (0x12+1)

static inline int translate_subdevice_to_hwport(int subdev)
{
	switch (subdev) {
	case 0x00-0x10: return subdev+1; /* single MTP's ports */
	case 0x11: return 0x63;		/* ADAT */
	default: return 0;		/* all ports */
	}
}

static inline int translate_hwport_to_subdevice(int hwport)
{
	switch (hwport) {
	case 0x01-0x11: return hwport-1; /* single MTP's ports */
	case 0x63: return 0x11;		/* ADAT */
	default: return 0x12;		/* all ports */
	}
}

///////////////////////////////////////////////////////////////////

typedef struct SmtpPort {
	u8 number;
	u8 hwport;
	u8 mode;
	snd_rawmidi_substream_t *input;
	snd_rawmidi_substream_t *output;
} TSmtpPort;

typedef struct Smtp {
	snd_card_t *card;
	unsigned long port;
	struct resource *res_port;
	int irq;
	spinlock_t spinlock;
	int share_irq;
	int istimer;
	struct timer_list timer;
	snd_rawmidi_t *rmidi;
	TSmtpPort ports[NUMPORTS];

	u32 inmidiport;
	u32 inmidistate;

	u32 outmidihwport;
} TSmtp;

//////////////////////////////////////////////////////////////////
//      protos

static void snd_mtpav_output_write(snd_rawmidi_substream_t * substream);
static int snd_mtpav_input_open(snd_rawmidi_substream_t * substream);
static int snd_mtpav_input_close(snd_rawmidi_substream_t * substream);
static void snd_mtpav_input_trigger(snd_rawmidi_substream_t * substream, int up);
static int snd_mtpav_output_open(snd_rawmidi_substream_t * substream);
static int snd_mtpav_output_close(snd_rawmidi_substream_t * substream);
static void snd_mtpav_output_trigger(snd_rawmidi_substream_t * substream, int up);
static void snd_mtpav_mputreg(u16 reg, u8 val);

static void snd_mtpav_inmidi_h(TSmtp * mcrd, u8 inbyte);

/////////////////////////////////////////////////////////////////////

static TSmtp *mtp_card;

static snd_rawmidi_ops_t snd_mtpav_output = {
	open:		snd_mtpav_output_open,
	close:		snd_mtpav_output_close,
	trigger:	snd_mtpav_output_trigger,
};

static snd_rawmidi_ops_t snd_mtpav_input = {
	open:		snd_mtpav_input_open,
	close:		snd_mtpav_input_close,
	trigger:	snd_mtpav_input_trigger,
};

///////////////////////////////////////////////////////////////////////////////

static u8 snd_mtpav_getreg(u16 reg)
{
	u8 rval = 0;

	if (reg == SREG) {
		rval = inb(MTPAV_IOBASE + SREG);
		rval = (rval & 0xf8);
	} else if (reg == CREG) {
		rval = inb(MTPAV_IOBASE + CREG);
		rval = (rval & 0x1c);
	}

	return rval;
}

///////////////////////////////////////////////////////////////////////////////

static void snd_mtpav_mputreg(u16 reg, u8 val)
{
	if (reg == DREG) {
		outb(val, (MTPAV_IOBASE + DREG));
	} else if (reg == CREG) {
		outb(val, (MTPAV_IOBASE + CREG));
	}
}

//////////////////////////////////////////////////////////////////////////

static void snd_mtpav_wait_rfdhi(void)
{
	u8 sbyte;

	sbyte = snd_mtpav_getreg(SREG);
	while (!(sbyte & SIGS_RFD)) {
		sbyte = snd_mtpav_getreg(SREG);
	}

}

static void snd_mtpav_send_byte(u8 byte)
{
	u8 tcbyt;
	u8 clrwrite;
	u8 setwrite;

	snd_mtpav_wait_rfdhi();

	/////////////////

	tcbyt = snd_mtpav_getreg(CREG);
	clrwrite = tcbyt & (SIGC_WRITE ^ 0xff);
	setwrite = tcbyt | SIGC_WRITE;

	snd_mtpav_mputreg(DREG, byte);
	snd_mtpav_mputreg(CREG, clrwrite);	// clear write bit

	snd_mtpav_mputreg(CREG, setwrite);	// set write bit

}

//

static void snd_mtpav_output_write(snd_rawmidi_substream_t * substream)
{
	u8 outbyte;
	u8 ifsent;
	TSmtpPort *port = (TSmtpPort *)substream->runtime->private_data;
	unsigned long flags;


	/////////////////
	// send port change command if necessary

	spin_lock_irqsave(&mtp_card->spinlock, flags);

	if (port->hwport != mtp_card->outmidihwport) {
		mtp_card->outmidihwport = port->hwport;

		snd_mtpav_send_byte(0xf5);
		snd_mtpav_send_byte(port->hwport);
		//snd_printk("new outport: 0x%x\n", (unsigned int) port->hwport);

	}
	///////////////////
	//      send data

	ifsent = 1;
	while (ifsent == 1) {
		ifsent = snd_rawmidi_transmit(substream, &outbyte, 1);

		if (ifsent == 1)
			snd_mtpav_send_byte(outbyte);
	}

	spin_unlock_irqrestore(&mtp_card->spinlock, flags);
}

/////////////////////////////////////////////////////
//      mtpav control

static void snd_mtpav_portscan(void)	// put mtp into smart routing mode
{
	u8 port;

	for (port = 0; port < 8; port++) {
		snd_mtpav_send_byte(0xf5);
		snd_mtpav_send_byte(port);
		snd_mtpav_send_byte(0xfe);
	}
}

//////////////////////////////////////////////////////////////////////////

static int snd_mtpav_input_open(snd_rawmidi_substream_t * substream)
{
	unsigned long flags;
	TSmtpPort *port = &mtp_card->ports[substream->number];

	//printk("mtpav port: %d opened\n", (int) substream->number);
	spin_lock_irqsave(&mtp_card->spinlock, flags);
	port->mode |= MTPAV_MODE_INPUT_OPENED;
	port->input = substream;
	if (mtp_card->share_irq++ == 0)
		snd_mtpav_mputreg(CREG, (SIGC_INTEN | SIGC_WRITE));	// enable pport interrupts
	spin_unlock_irqrestore(&mtp_card->spinlock, flags);
	return 0;
}

//////////////////////////////////////////////////////////////////

static int snd_mtpav_input_close(snd_rawmidi_substream_t *substream)
{
	unsigned long flags;
	TSmtpPort *port = &mtp_card->ports[substream->number];

	//printk("mtpav port: %d closed\n", (int) port);

	spin_lock_irqsave(&mtp_card->spinlock, flags);

	port->mode &= (~MTPAV_MODE_INPUT_OPENED);
	port->input = NULL;
	if (--mtp_card->share_irq == 0)
		snd_mtpav_mputreg(CREG, 0);	// disable pport interrupts

	spin_unlock_irqrestore(&mtp_card->spinlock, flags);
	return 0;
}

//////////////////////////////////////////////////////////////////

static void snd_mtpav_input_trigger(snd_rawmidi_substream_t * substream, int up)
{
	unsigned long flags;
	TSmtpPort *port = &mtp_card->ports[substream->number];

	spin_lock_irqsave(&mtp_card->spinlock, flags);
	if (up)
		port->mode |= MTPAV_MODE_INPUT_TRIGGERED;
	else
		port->mode &= ~MTPAV_MODE_INPUT_TRIGGERED;
	spin_unlock_irqrestore(&mtp_card->spinlock, flags);

}

///////////////////////////////////////////////////////////////////////////////

static int snd_mtpav_output_open(snd_rawmidi_substream_t * substream)
{
	unsigned long flags;
	TSmtpPort *port = &mtp_card->ports[substream->number];

	spin_lock_irqsave(&mtp_card->spinlock, flags);
	port->mode |= MTPAV_MODE_OUTPUT_OPENED;
	port->output = substream;
	spin_unlock_irqrestore(&mtp_card->spinlock, flags);
	return 0;
};

///////////////////////////////////////////////////////////////////////////////

static int snd_mtpav_output_close(snd_rawmidi_substream_t * substream)
{
	unsigned long flags;
	TSmtpPort *port = &mtp_card->ports[substream->number];

	spin_lock_irqsave(&mtp_card->spinlock, flags);
	port->mode &= (~MTPAV_MODE_OUTPUT_OPENED);
	port->output = NULL;
	spin_unlock_irqrestore(&mtp_card->spinlock, flags);
	return 0;
};

//////////////////////////////////////////////////////////////////

static void snd_mtpav_output_trigger(snd_rawmidi_substream_t * substream, int up)
{
	unsigned long flags;
	TSmtpPort *port = &mtp_card->ports[substream->number];

	spin_lock_irqsave(&mtp_card->spinlock, flags);
	if (up) {
#if 0	/* FIXME */
		if ((mtp_card->mode[port] & MTPAV_MODE_OUTPUT_TRIGGERED) == 0) {
			if (!mtp_card->istimer) {
				mtp_card->timer.function = snd_mtpav_output_timer;
				mtp_card->timer.data = (unsigned long) mtp_card;
				mtp_card->timer.expires = 1 + jiffies;
				add_timer(&mtp_card->timer);
			}
			mtp_card->istimer++;
		}
#endif
		port->mode |= MTPAV_MODE_OUTPUT_TRIGGERED;
	} else {
		port->mode &= ~MTPAV_MODE_OUTPUT_TRIGGERED;
	}
	spin_unlock_irqrestore(&mtp_card->spinlock, flags);

	if (up)
		snd_mtpav_output_write(substream);
}

//////////////////////////////////////////////////////////////////

static void snd_mtpav_inmidi_h(TSmtp * mcrd, u8 inbyte)
{
	TSmtpPort *port;

	if (mcrd == NULL)
		return;

	if (inbyte == 0xfe)		// really? we shouldn't ignore these messages
		return;

	if (mcrd->inmidistate == 0) {	// awaiting command
		if (inbyte == 0xf5) {	// MTP port #
			mcrd->inmidistate = 1;
		} else {
			if (mcrd->inmidiport >= NUMPORTS)
				return;

			port = &mcrd->ports[mcrd->inmidiport];

			if (port->mode & MTPAV_MODE_INPUT_TRIGGERED)
				snd_rawmidi_receive(port->input, &inbyte, 1);
		}
	} else if (mcrd->inmidistate == 1) {
		mcrd->inmidiport = translate_hwport_to_subdevice(inbyte);
		mcrd->inmidistate = 0;
		//snd_printk("cable %d/%d\n", (int) mcrd->inmidiport, (int)inbyte);
	}
}

static void snd_mtpav_read_bytes(TSmtp * mcrd)
{
	u8 clrread, setread;
	u8 mtp_read_byte;
	u8 read_sr0, read_sr1, read_sr2, read_sr3;
	u8 cbyt;

	u8 sbyt = snd_mtpav_getreg(SREG);

	//printk("snd_mtpav_read_bytes() sbyt: 0x%x\n", sbyt);

	if (!(sbyt & SIGS_BYTE))
		return;

	cbyt = snd_mtpav_getreg(CREG);
	clrread = cbyt & (SIGC_READ ^ 0xff);
	setread = cbyt | SIGC_READ;

	do {

		snd_mtpav_mputreg(CREG, setread);
		read_sr0 = snd_mtpav_getreg(SREG);
		snd_mtpav_mputreg(CREG, clrread);

		snd_mtpav_mputreg(CREG, setread);
		read_sr1 = snd_mtpav_getreg(SREG);
		snd_mtpav_mputreg(CREG, clrread);

		snd_mtpav_mputreg(CREG, setread);
		read_sr2 = snd_mtpav_getreg(SREG);
		snd_mtpav_mputreg(CREG, clrread);

		snd_mtpav_mputreg(CREG, setread);
		read_sr3 = snd_mtpav_getreg(SREG);
		snd_mtpav_mputreg(CREG, clrread);

		mtp_read_byte = ((read_sr0 & (SIGS_IN0 | SIGS_IN1)) / 16);
		mtp_read_byte |=
		    (((read_sr1 & (SIGS_IN0 | SIGS_IN1)) / 16) << 2);
		mtp_read_byte |=
		    (((read_sr2 & (SIGS_IN0 | SIGS_IN1)) / 16) << 4);
		mtp_read_byte |=
		    (((read_sr3 & (SIGS_IN0 | SIGS_IN1)) / 16) << 6);

		snd_mtpav_inmidi_h(mcrd, mtp_read_byte);

		sbyt = snd_mtpav_getreg(SREG);

	} while (sbyt & SIGS_BYTE);
}

static void snd_mtpav_irqh(int irq, void *dev_id, struct pt_regs *regs)
{
	unsigned long flags;
	TSmtp *mcard = (TSmtp *) dev_id;

	//printk("irqh()\n");
	spin_lock_irqsave(&mcard->spinlock, flags);
	snd_mtpav_read_bytes(mcard);
	spin_unlock_irqrestore(&mcard->spinlock, flags);
}

/////////////////////////////////////////////////////////////////
// get ISA resources

static int snd_mtpav_get_ISA(TSmtp * mcard)
{
	if ((mcard->res_port = request_region(snd_port, 3, "MotuMTPAV MIDI")) == NULL) {
		snd_printk("MTVAP port 0x%lx is busy\n", snd_port);
		return -EBUSY;
	}
	if (request_irq(snd_irq, snd_mtpav_irqh, SA_INTERRUPT, "MOTU MTPAV", (void *)mcard)) {
		snd_printk("MTVAP IRQ %d busy\n", snd_irq);
		return -EBUSY;
	}
	mcard->irq = snd_irq;
	return 0;
}

/////////////////////////////////////////////////////////////////
// get RAWMIDI resources

static void snd_mtpav_set_name(snd_rawmidi_substream_t *substream)
{
	if (substream->number >= 0 && substream->number < 8) {
		sprintf(substream->name, "MTP direct %d", (substream->number & 7) + 1);
		return;
	}
	if (substream->number >= 8 && substream->number < 15) {
		sprintf(substream->name, "MTP remote %d", (substream->number & 7) + 1);
		return;
	}
	switch (substream->number) {
	case 0x10: strcpy(substream->name, "MTP computer"); break;
	case 0x11: strcpy(substream->name, "MTP ADAT"); break;
	case 0x12: strcpy(substream->name, "MTP broadcast"); break;
	}
}

static int snd_mtpav_get_RAWMIDI(TSmtp * mcard)
{
	int rval = 0;
	snd_rawmidi_t *rawmidi;
	snd_rawmidi_substream_t *substream;
	struct list_head *list;
	char rmidinames1[80];

	//printk("entering snd_mtpav_get_RAWMIDI\n");

	if (snd_hwports < 1)
		snd_hwports = 1;
	if (snd_hwports > NUMPORTS)
		snd_hwports = NUMPORTS;

	if ((rval = snd_rawmidi_new(mcard->card, rmidinames1,
				    0, snd_hwports, snd_hwports,
				    &mcard->rmidi)) < 0)
		return rval;
	rawmidi = mcard->rmidi;

	list_for_each(list, &rawmidi->streams[SNDRV_RAWMIDI_STREAM_INPUT].substreams) {
		substream = list_entry(list, snd_rawmidi_substream_t, list);
		snd_mtpav_set_name(substream);
		substream->ops = &snd_mtpav_input;
	}
	list_for_each(list, &rawmidi->streams[SNDRV_RAWMIDI_STREAM_OUTPUT].substreams) {
		substream = list_entry(list, snd_rawmidi_substream_t, list);
		snd_mtpav_set_name(substream);
		substream->ops = &snd_mtpav_output;
	}
	rawmidi->info_flags |= SNDRV_RAWMIDI_INFO_OUTPUT | SNDRV_RAWMIDI_INFO_INPUT |
			       SNDRV_RAWMIDI_INFO_DUPLEX;
	sprintf(rawmidi->name, "MotuMIDI");
	//printk("exiting snd_mtpav_get_RAWMIDI() \n");
	return 0;
}

/////////////////////////////////////////////////////////////////

static TSmtp *new_TSmtp(void)
{
	TSmtp *ncrd = (TSmtp *) snd_kcalloc(sizeof(TSmtp), GFP_KERNEL);
	if (ncrd != NULL) {
		spin_lock_init(&ncrd->spinlock);

		ncrd->card = NULL;
		ncrd->irq = -1;
		ncrd->share_irq = 0;

		ncrd->inmidiport = 0xffffffff;
		ncrd->inmidistate = 0;
		ncrd->outmidihwport = 0xffffffff;
	}
	return ncrd;
}

/////////////////////////////////////////////////////////////////

static void free_TSmtp(TSmtp * crd)
{
	if (crd->res_port)
		release_resource(crd->res_port);
	if (crd->irq >= 0)
		free_irq(crd->irq, (void *)crd);
	if (crd != NULL)
		kfree(crd);
}

/////////////////////////////////////////////////////////////////

static int __init alsa_card_mtpav_init(void)
{
	int err = 0;
	char longname_buffer[80];

	mtp_card = new_TSmtp();
	if (mtp_card == NULL)
		return -ENOMEM;

	mtp_card->card = snd_card_new(snd_index, snd_id, THIS_MODULE, 0);
	if (mtp_card->card == NULL) {
		free_TSmtp(mtp_card);
		return -ENOMEM;
	}

	mtp_card->card->type = SNDRV_CARD_TYPE_MTPAV;
	strcpy(mtp_card->card->abbreviation, "mtpav");
	strcpy(mtp_card->card->shortname, "mtpav on parallel port");
	memset(longname_buffer, 0, sizeof(longname_buffer));
	sprintf(longname_buffer, "mtpav on parallel port at");

	err = snd_mtpav_get_ISA(mtp_card);
	//printk("snd_mtpav_get_ISA returned: %d\n", err);
	if (err < 0)
		goto __error;

	err = snd_mtpav_get_RAWMIDI(mtp_card);
	//snd_printk("snd_mtapv_get_RAWMIDI returned: %d\n", err);
	if (err < 0)
		goto __error;

	err = snd_card_register(mtp_card->card);	// dont snd_card_register until AFTER all cards reources done!

	//printk("snd_card_register returned %d\n", err);
	if (err < 0)
		goto __error;


	snd_mtpav_portscan();

	snd_printk("Motu MidiTimePiece on parallel port irq: %d ioport: 0x%lx\n", snd_irq, snd_port);

	return 0;

      __error:
	snd_card_free(mtp_card->card);
	free_TSmtp(mtp_card);
	return err;
}

/////////////////////////////////////////////////////////////////

static void __exit alsa_card_mtpav_exit(void)
{
	if (mtp_card == NULL)
		return;
	if (mtp_card->card)
		snd_card_free(mtp_card->card);
	free_TSmtp(mtp_card);
}

/////////////////////////////////////////////////////////////////

module_init(alsa_card_mtpav_init)
module_exit(alsa_card_mtpav_exit)

#ifndef MODULE

/* format is: snd-card-mtpav=snd_enable,snd_index,snd_id,
			     snd_port,snd_irq,snd_hwports */

static int __init alsa_card_mtpav_setup(char *str)
{
        int __attribute__ ((__unused__)) enable = 1;

	(void)(get_option(&str,&enable) == 2 &&
	       get_option(&str,&snd_index) == 2 &&
	       get_id(&str,&snd_id) == 2 &&
	       get_option(&str,(int *)&snd_port) == 2 &&
	       get_option(&str,&snd_irq) == 2 &&
	       get_option(&str,&snd_hwports) == 2);
	return 1;
}

__setup("snd-card-mtpav=", alsa_card_mtpav_setup);

#endif /* ifndef MODULE */
