/*
 * Driver for C-Media CMI8338 and 8738 PCI soundcards.
 * Copyright (c) 2000 by Takashi Iwai <tiwai@suse.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#define SNDRV_MAIN_OBJECT_FILE

#include "../include/driver.h"
#include "../include/info.h"
#include "../include/control.h"
#include "../include/pcm.h"
#include "../include/rawmidi.h"
#include "../include/mpu401.h"
#include "../include/sb.h"	/* for SB16 mixer */
#define SNDRV_GET_ID
#include "../include/initval.h"

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("C-Media CMI8x38 PCI");
MODULE_CLASSES("{sound}");
MODULE_DEVICES("{{C-Media,CMI8738},"
		"{C-Media,CMI8738B},"
		"{C-Media,CMI8338A},"
		"{C-Media,CMI8338B}}");

static int snd_index[SNDRV_CARDS] = SNDRV_DEFAULT_IDX;	/* Index 0-MAX */
static char *snd_id[SNDRV_CARDS] = SNDRV_DEFAULT_STR;	/* ID for this card */
static int snd_enable[SNDRV_CARDS] = SNDRV_DEFAULT_ENABLE;	/* Enable switches */
static int snd_enable_midi[SNDRV_CARDS] = {[0 ... (SNDRV_CARDS-1)] = 0};
static int snd_enable_fm[SNDRV_CARDS] = {[0 ... (SNDRV_CARDS-1)] = 0};
static long snd_mpu_port[SNDRV_CARDS] = {0x330, [1 ... (SNDRV_CARDS-1)]=-1};
static long snd_fm_port[SNDRV_CARDS] = {0x388, [1 ... (SNDRV_CARDS-1)]=-1};

MODULE_PARM(snd_index, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for C-Media PCI soundcard.");
MODULE_PARM_SYNTAX(snd_index, SNDRV_INDEX_DESC);
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SNDRV_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for C-Media PCI soundcard.");
MODULE_PARM_SYNTAX(snd_id, SNDRV_ID_DESC);
MODULE_PARM(snd_enable, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(snd_enable, "Enable C-Media PCI soundcard.");
MODULE_PARM_SYNTAX(snd_enable, SNDRV_ENABLE_DESC);
MODULE_PARM(snd_enable_midi, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(snd_enable_midi, "Enable MPU-401 port.");
MODULE_PARM_SYNTAX(snd_enable_midi, SNDRV_ENABLED "," SNDRV_BOOLEAN_FALSE_DESC);
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SNDRV_CARDS) "l");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port.");
MODULE_PARM_SYNTAX(snd_mpu_port, "enable:(snd_enable_midi),allows:{{0x330},{0x320},{0x310},{0x300}},dialog:list");
MODULE_PARM(snd_enable_fm, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(snd_enable_fm, "Enable FM OPL-3 synth.");
MODULE_PARM_SYNTAX(snd_enable_fm, SNDRV_ENABLED "," SNDRV_BOOLEAN_FALSE_DESC);
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SNDRV_CARDS) "l");
MODULE_PARM_DESC(snd_fm_port, "FM port.");
MODULE_PARM_SYNTAX(snd_fm_port, "enable:(snd_enable_fm),allows:{{0x388},{0x3c8},{0x3e0},{0x3e8}},dialog:list");

/*
 * CM8338 registers definition
 */

#define CM_REG_FUNCTRL0		0x00
#define CM_RST_CH1		0x00080000
#define CM_RST_CH0		0x00040000
#define CM_CHEN1		0x00020000
#define CM_CHEN0		0x00010000
#define CM_PAUSE1		0x00000008
#define CM_PAUSE0		0x00000004
#define CM_CHADC1		0x00000002	/* ch1, 0:playback, 1:record */
#define CM_CHADC0		0x00000001	/* ch0, 0:playback, 1:record */

#define CM_REG_FUNCTRL1		0x04
#define CM_ASFC_MASK		0x0000E000	/* ADC sampling frequency */
#define CM_ASFC_SHIFT		13
#define CM_DSFC_MASK		0x00001C00	/* DAC sampling frequency */
#define CM_DSFC_SHIFT		10
#define CM_SPDF_1		0x00000200	/* SPDIF IN/OUT at channel B */
#define CM_SPDF_0		0x00000100	/* SPDIF OUT only channel A */
#define CM_SPDFLOOP		0x00000080	/* ext. SPDIIF/OUT -> IN loopback */
#define CM_SPDO2DAC		0x00000040	/* SPDIF/OUT can be heard from internal DAC */
#define CM_INTRM		0x00000020	/* master control block (MCB) interrupt enabled */
#define CM_BREQ			0x00000010	/* bus master enabled */
#define CM_VOICE_EN		0x00000008	/* legacy voice (SB16,FM) */
#define CM_UART_EN		0x00000004	/* UART */
#define CM_JYSTK_EN		0x00000002	/* joy stick */

#define CM_REG_CHFORMAT		0x08

#define CM_ADCBITLEN_MASK	0x0000C000	
#define CM_ADCBITLEN_16		0x00000000
#define CM_ADCBITLEN_15		0x00004000
#define CM_ADCBITLEN_14		0x00008000
#define CM_ADCBITLEN_13		0x0000C000

#define CM_ADCDACLEN_MASK	0x00003000
#define CM_ADCDACLEN_060	0x00000000
#define CM_ADCDACLEN_066	0x00001000
#define CM_ADCDACLEN_130	0x00002000
#define CM_ADCDACLEN_280	0x00003000

#define CM_CH1FMT_MASK		0x0000000C
#define CM_CH1FMT_SHIFT		2
#define CM_CH0FMT_MASK		0x00000003
#define CM_CH0FMT_SHIFT		0

#define CM_REG_INT_HLDCLR	0x0C
#define CM_TDMA_INT_EN		0x00040000
#define CM_CH1_INT_EN		0x00020000
#define CM_CH0_INT_EN		0x00010000
#define CM_INT_HOLD		0x00000002
#define CM_INT_CLEAR		0x00000001

#define CM_REG_INT_STATUS	0x10
#define CM_INTR			0x80000000
#define CM_UARTINT		0x00010000
#define CM_LTDMAINT		0x00008000
#define CM_HTDMAINT		0x00004000
#define CM_CH1BUSY		0x00000008
#define CM_CH0BUSY		0x00000004
#define CM_CHINT1		0x00000002
#define CM_CHINT0		0x00000001

#define CM_REG_LEGACY_CTRL	0x14
#define CM_VMPU_MASK		0x60000000	/* MPU401 i/o port address */
#define CM_VMPU_330		0x00000000
#define CM_VMPU_320		0x20000000
#define CM_VMPU_310		0x40000000
#define CM_VMPU_300		0x60000000
#define CM_VSBSEL_MASK		0x0C000000	/* SB16 base address */
#define CM_VSBSEL_220		0x00000000
#define CM_VSBSEL_240		0x04000000
#define CM_VSBSEL_260		0x08000000
#define CM_VSBSEL_280		0x0C000000
#define CM_FMSEL_MASK		0x03000000	/* FM OPL3 base address */
#define CM_FMSEL_388		0x00000000
#define CM_FMSEL_3C8		0x01000000
#define CM_FMSEL_3E0		0x02000000
#define CM_FMSEL_3E8		0x03000000
#define CM_ENSPDOUT		0x00800000	/* enable XPDIF/OUT to I/O interface */
#define CM_SPDCOPYRHT		0x00400000
#define CM_DAC2SPDO		0x00200000	/* enable wave+fm_midi -> SPDIF/OUT */
#define CM_SETRETRY		0x00010000	/* 0: legacy i/o wait (default), 1: legacy i/o bus retry */

#define CM_REG_MISC_CTRL	0x18
#define CM_PWD			0x80000000
#define CM_RESET		0x40000000
#define CM_SFIL_MASK		0x30000000
#define CM_TXVX			0x08000000
#define CM_N4SPK3D		0x04000000
#define CM_SPDO5V		0x02000000
#define CM_SPDIF48K		0x01000000	/* write */
#define CM_SPATUS48K		0x01000000	/* read */
#define CM_ENDBDAC		0x00800000
#define CM_XCHGDAC		0x00400000	/* 0: front=ch0, 1: front=ch1 */
#define CM_SPD32SEL		0x00200000	/* 0: 16bit SPDIF, 1: 32bit */
#define CM_SPDFLOOPI		0x00100000	/* int. SPDIF-IN -> int. OUT */
#define CM_FM_EN		0x00080000	/* enalbe FM */
#define CM_VIDWPDSB		0x00010000 
#define CM_SPDF_AC97		0x00008000	/* 0: SPDIF/OUT 44.1K, 1: 48K */
#define CM_MASK_EN		0x00004000
#define CM_VIDWPPRT		0x00002000
#define CM_SFILENB		0x00001000
#define CM_MMODE_MASK		0x00000E00
#define CM_FLINKON		0x00000080
#define CM_FLINKOFF		0x00000040
#define CM_MIDSMP		0x00000010
#define CM_UPDDMA_MASK		0x0000000C
#define CM_TWAIT_MASK		0x00000003

	/* byte */
#define CM_REG_MIXER0	0x20

#define CM_REG_SB16_DATA	0x22
#define CM_REG_SB16_ADDR	0x23

#define CM_REG_MIXER1		0x24
#define CM_FMMUTE		0x80	/* mute FM */
#define CM_FMMUTE_SHIFT		7
#define CM_WSMUTE		0x40	/* mute PCM */
#define CM_WSMUTE_SHIFT		6
#define CM_SPK4			0x20	/* lin-in -> line out */
#define CM_SPK4_SHIFT		5
#define CM_REAR2FRONT		0x10	/* exchange rear/front */
#define CM_REAR2FRONT_SHIFT	4
#define CM_WAVEINL		0x08	/* digital wave rec. left chan */
#define CM_WAVEINL_SHIFT	3
#define CM_WAVEINR		0x04
#define CM_WAVEINR_SHIFT	2
#define CM_X3DEN		0x02	/* 3D surround enable */
#define CM_X3DEN_SHIFT		1
#define CM_CDPLAY		0x01	/* enable SPDIF/IN PCM -> DAC */
#define CM_CDPLAY_SHIFT		0

#define CM_REG_MIXER2		0x25
#define CM_RAUXREN		0x80
#define CM_RAUXLEN		0x40
#define CM_VAUXRM		0x20
#define CM_VAUXLM		0x10
#define CM_VADMIC_MASK		0x0e
#define CM_MICGAINZ		0x01

#define CM_REG_AUX_VOL		0x26
#define CM_VAUXL_MASK		0xf0
#define CM_VAUXR_MASK		0x0f

#define CM_REG_MISC		0x27
#define CM_REG_AC97		0x28

#define CM_REG_CH0_FRAME1	0x80
#define CM_REG_CH0_FRAME2	0x84
#define CM_REG_CH1_FRAME1	0x88
#define CM_REG_CH1_FRAME2	0x8C

#define CM_EXTENT_CODEC	  0x100
#define CM_EXTENT_MIDI	  0x2
#define CM_EXTENT_SYNTH	  0x4

/*
 */

typedef struct snd_card_cmipci snd_card_cmipci_t;
typedef struct _snd_cmipci cmipci_t;
#define chip_t cmipci_t

/*
 */

typedef struct _snd_cmipci_pcm cmipci_pcm_t;
struct _snd_cmipci_pcm {
	snd_pcm_substream_t *substream;
	int running;
	unsigned int dma_size;
	unsigned int period_size;
	unsigned int offset;
	unsigned int fmt;
	int ch, dac;
};

struct _snd_cmipci {
	snd_card_t *card;

	/* switches */
	int enable_midi;
	int enable_synth;

	struct pci_dev *pci;
	unsigned long dma_playback_size;
	unsigned long dma_capture_size;
	int irq;

	unsigned long iobase, iomidi, iosynth;
	struct resource *res_iobase;
	unsigned int ctrl;

	snd_pcm_t *pcm;		/* DAC/ADC PCM */

	cmipci_pcm_t playback, capture;

	/* external MIDI */
	snd_rawmidi_t *rmidi;

	spinlock_t reg_lock;
	snd_info_entry_t *proc_entry;
};


/* read/write operations for dword register */
inline static void snd_cmipci_write(cmipci_t *cm, unsigned int cmd, unsigned int data)
{
	outl(data, cm->iobase + cmd);
}
inline static unsigned int snd_cmipci_read(cmipci_t *cm, unsigned int cmd)
{
	return inl(cm->iobase + cmd);
}

/* read/write operations for word register */
inline static void snd_cmipci_write_w(cmipci_t *cm, unsigned int cmd, unsigned short data)
{
	outw(data, cm->iobase + cmd);
}
inline static unsigned short snd_cmipci_read_w(cmipci_t *cm, unsigned int cmd)
{
	return inw(cm->iobase + cmd);
}

/* bit operations for dword register */
static void snd_cmipci_set_bit(cmipci_t *cm, unsigned int cmd, unsigned int flag)
{
	unsigned int val;
	val = inl(cm->iobase + cmd);
	val |= flag;
	outl(val, cm->iobase + cmd);
}

static void snd_cmipci_clear_bit(cmipci_t *cm, unsigned int cmd, unsigned int flag)
{
	unsigned int val;
	val = inl(cm->iobase + cmd);
	val &= ~flag;
	outl(val, cm->iobase + cmd);
}


/*
 * PCM interface
 */

/*
 * calculate frequency
 */

static int rates[] = { 5512, 11025, 22050, 44100, 8000, 16000, 32000, 48000 };
#define RATES (sizeof(rates) / sizeof(rates[0]))

static const unsigned sample_size[] = { 1, 2, 2, 4 };
static const unsigned sample_shift[] = { 0, 1, 1, 2 };

static unsigned int snd_cmipci_rate_freq(unsigned int rate)
{
	int i;
	for (i = 0; i < RATES; i++) {
		if (rates[i] == rate)
			return i;
	}
	snd_BUG();
	return 0;
}

static int snd_cmipci_hw_params(snd_pcm_substream_t * substream,
				snd_pcm_hw_params_t * hw_params)
{
	cmipci_t *chip = snd_pcm_substream_chip(substream);
	return snd_pcm_lib_malloc_pci_pages(chip->pci, substream, params_buffer_bytes(hw_params));
}

static int snd_cmipci_hw_free(snd_pcm_substream_t * substream)
{
	cmipci_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_lib_free_pci_pages(chip->pci, substream);
	return 0;
}

/*
 * prepare playback/capture channel
 * channel to be used must have been set in rec->ch.
 */
static int snd_cmipci_pcm_prepare(cmipci_t *cm, cmipci_pcm_t *rec,
				 snd_pcm_substream_t *substream)
{
	unsigned long flags;
	unsigned int reg, freq, val;
	snd_pcm_runtime_t *runtime = substream->runtime;

	rec->fmt = 0;
	if (snd_pcm_format_width(runtime->format) == 16)
		rec->fmt |= 0x02;
	if (runtime->channels > 1)
		rec->fmt |= 0x01;
	rec->offset = runtime->dma_addr;
	/* buffer and period sizes in frame */
	rec->dma_size = runtime->buffer_size;
	rec->period_size = runtime->period_size;

	spin_lock_irqsave(&cm->reg_lock, flags);

	/* set buffer address */
	reg = rec->ch ? CM_REG_CH1_FRAME1 : CM_REG_CH0_FRAME1;
	snd_cmipci_write(cm, reg, rec->offset);
	/* program sample counts */
	reg = rec->ch ? CM_REG_CH1_FRAME2 : CM_REG_CH0_FRAME2;
	snd_cmipci_write_w(cm, reg, rec->dma_size - 1);
	snd_cmipci_write_w(cm, reg + 2, rec->period_size - 1);

	/* set adc/dac flag */
	val = rec->ch ? CM_CHADC1 : CM_CHADC0;
	if (rec->dac)
		cm->ctrl &= ~val;
	else
		cm->ctrl |= val;
	snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
	//snd_printd("cmipci: functrl0 = %08x\n", cm->ctrl);

	/* set sample rate */
	freq = snd_cmipci_rate_freq(runtime->rate);
	val = snd_cmipci_read(cm, CM_REG_FUNCTRL1);
	if (rec->dac) {
		val &= ~CM_DSFC_MASK;
		val |= (freq << CM_DSFC_SHIFT) & CM_DSFC_MASK;
	} else {
		val &= ~CM_ASFC_MASK;
		val |= (freq << CM_ASFC_SHIFT) & CM_ASFC_MASK;
	}
	snd_cmipci_write(cm, CM_REG_FUNCTRL1, val);
	//snd_printd("cmipci: functrl1 = %08x\n", val);

	/* set format */
	val = snd_cmipci_read(cm, CM_REG_CHFORMAT);
	if (rec->ch) {
		val &= ~CM_CH1FMT_MASK;
		val |= rec->fmt << CM_CH1FMT_SHIFT;
	} else {
		val &= ~CM_CH0FMT_MASK;
		val |= rec->fmt << CM_CH0FMT_SHIFT;
	}
	snd_cmipci_write(cm, CM_REG_CHFORMAT, val);
	//snd_printd("cmipci: chformat = %08x\n", val);

	rec->running = 0;
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	return 0;
}

/*
 * PCM trigger/stop
 */
static int snd_cmipci_pcm_trigger(cmipci_t *cm, cmipci_pcm_t *rec,
				 snd_pcm_substream_t *substream, int cmd)
{
	unsigned int inthld, chen, reset;
	int result = 0;

	if (rec->ch) {
		inthld = CM_CH1_INT_EN;
		chen = CM_CHEN1;
		reset = CM_RST_CH1;
	} else {
		inthld = CM_CH0_INT_EN;
		chen = CM_CHEN0;
		reset = CM_RST_CH0;
	}

	spin_lock(&cm->reg_lock);
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		rec->running = 1;
		/* set interrupt */
		snd_cmipci_set_bit(cm, CM_REG_INT_HLDCLR, inthld);
		cm->ctrl |= chen;
		/* enable channel */
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
		//snd_printd("cmipci: functrl0 = %08x\n", cm->ctrl);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		rec->running = 0;
		/* disable interrupt */
		snd_cmipci_clear_bit(cm, CM_REG_INT_HLDCLR, inthld);
		/* reset */
		cm->ctrl &= ~chen;
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl | reset);
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, cm->ctrl);
		break;
	default:
		result = -EINVAL;
		break;
	}
	spin_unlock(&cm->reg_lock);
	return result;
}

/*
 * return the current pointer
 */
static snd_pcm_uframes_t snd_cmipci_pcm_pointer(cmipci_t *cm, cmipci_pcm_t *rec,
					  snd_pcm_substream_t *substream)
{
	size_t ptr;
	unsigned int reg;
	if (!rec->running)
		return 0;
#if 1
	reg = rec->ch ? CM_REG_CH1_FRAME2 : CM_REG_CH0_FRAME2;
	ptr = rec->dma_size - (snd_cmipci_read_w(cm, reg) + 1);
	return ptr;
#else
	reg = rec->ch ? CM_REG_CH1_FRAME1 : CM_REG_CH0_FRAME1;
	ptr = snd_cmipci_read(cm, reg) - rec->offset;
	return bytes_to_frames(substream->runtime, ptr);
#endif
}

/*
 * playback
 */

static int snd_cmipci_playback_prepare(snd_pcm_substream_t *substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	return snd_cmipci_pcm_prepare(cm, &cm->playback, substream);
}

static int snd_cmipci_playback_trigger(snd_pcm_substream_t *substream,
				       int cmd)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	return snd_cmipci_pcm_trigger(cm, &cm->playback, substream, cmd);
}

static snd_pcm_uframes_t snd_cmipci_playback_pointer(snd_pcm_substream_t *substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	return snd_cmipci_pcm_pointer(cm, &cm->playback, substream);
}



/*
 * capture
 */

static int snd_cmipci_capture_prepare(snd_pcm_substream_t *substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	return snd_cmipci_pcm_prepare(cm, &cm->capture, substream);
}

static int snd_cmipci_capture_trigger(snd_pcm_substream_t *substream,
				     int cmd)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	return snd_cmipci_pcm_trigger(cm, &cm->capture, substream, cmd);
}

static snd_pcm_uframes_t snd_cmipci_capture_pointer(snd_pcm_substream_t *substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	return snd_cmipci_pcm_pointer(cm, &cm->capture, substream);
}

/*
 * interrupt handler
 */
static void snd_cmipci_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, dev_id, return);
	unsigned int status;
	
	/* fastpath out, to ease interrupt sharing */
	status = snd_cmipci_read(cm, CM_REG_INT_STATUS);
	if (!(status & CM_INTR))
		return;

	/* acknowledge interrupt */
	spin_lock(&cm->reg_lock);
	if (status & CM_CHINT0) {
		snd_cmipci_clear_bit(cm, CM_REG_INT_HLDCLR, CM_CH0_INT_EN);
		snd_cmipci_set_bit(cm, CM_REG_INT_HLDCLR, CM_CH0_INT_EN);
	}
	if (status & CM_CHINT1) {
		snd_cmipci_clear_bit(cm, CM_REG_INT_HLDCLR, CM_CH1_INT_EN);
		snd_cmipci_set_bit(cm, CM_REG_INT_HLDCLR, CM_CH1_INT_EN);
	}
	spin_unlock(&cm->reg_lock);

	if (cm->rmidi) {
		if ((status & CM_UARTINT) && cm->rmidi != NULL) {
			snd_mpu401_uart_interrupt(irq, cm->rmidi->private_data, regs);
		}
	}

	if (cm->pcm) {
		if ((status & CM_CHINT0) && cm->playback.running)
			snd_pcm_period_elapsed(cm->playback.substream);
		if ((status & CM_CHINT1) && cm->capture.running)
			snd_pcm_period_elapsed(cm->capture.substream);
	}
}

/*
 * mixer interface:
 * - CM8338/8738 has a compatible mixer interface with SB16, but
 *   lack of some elements like tone control, i/o gain and AGC.
 * - Access to native registers:
 *   - A 3D switch
 *   - Output mute switches
 */

static void snd_cmipci_mixer_write(cmipci_t *s, unsigned char idx, unsigned char data)
{
	outb(idx, s->iobase + CM_REG_SB16_ADDR);
	outb(data, s->iobase + CM_REG_SB16_DATA);
}

static unsigned char snd_cmipci_mixer_read(cmipci_t *s, unsigned char idx)
{
	unsigned char v;

	outb(idx, s->iobase + CM_REG_SB16_ADDR);
	v = inb(s->iobase + CM_REG_SB16_DATA);
	return v;
}

/*
 * general mixer element
 */
#define CMIPCI_DOUBLE(xname, xindex, left_reg, right_reg, left_shift, right_shift, mask, invert, stereo) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_cmipci_info_volume, \
  get: snd_cmipci_get_volume, put: snd_cmipci_put_volume, \
  private_value: (left_reg) | ((right_reg) << 8) | (left_shift << 16) | (right_shift << 19) | (mask << 24) | (invert << 22) | (stereo << 23)}

#define CMIPCI_VOL_STEREO(xname,xindex,reg,shift,mask) CMIPCI_DOUBLE(xname, xindex, reg, reg+1, shift, shift, mask, 0, 1)
#define CMIPCI_VOL_MONO(xname,xindex,reg,shift,mask) CMIPCI_DOUBLE(xname, xindex, reg, reg, shift, shift, mask, 0, 0)
#define CMIPCI_SW_STEREO(xname,xindex,lshift,rshift) CMIPCI_DOUBLE(xname, xindex, SB_DSP4_OUTPUT_SW, SB_DSP4_OUTPUT_SW, lshift, rshift, 1, 1, 1)

static int snd_cmipci_info_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_info_t * uinfo)
{
	int mask = (kcontrol->private_value >> 24) & 0xff;
	int stereo = (kcontrol->private_value >> 23) & 1;

	uinfo->type = mask == 1 ? SNDRV_CTL_ELEM_TYPE_BOOLEAN : SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = stereo + 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = mask;
	return 0;
}
 
static int snd_cmipci_get_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int left_reg = kcontrol->private_value & 0xff;
	int right_reg = (kcontrol->private_value >> 8) & 0xff;
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;
	int mask = (kcontrol->private_value >> 24) & 0xff;
	int invert = (kcontrol->private_value >> 22) & 1;
	int stereo = (kcontrol->private_value >> 23) & 1;
	int val;

	spin_lock_irqsave(&cm->reg_lock, flags);
	val = (snd_cmipci_mixer_read(cm, left_reg) >> left_shift) & mask;
	if (invert)
		val = mask - val;
	ucontrol->value.integer.value[0] = val;
	if (stereo) {
		val = (snd_cmipci_mixer_read(cm, right_reg) >> right_shift) & mask;
		if (invert)
			val = mask - val;
		 ucontrol->value.integer.value[1] = val;
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return 0;
}

static int snd_cmipci_put_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int left_reg = kcontrol->private_value & 0xff;
	int right_reg = (kcontrol->private_value >> 8) & 0xff;
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;
	int mask = (kcontrol->private_value >> 24) & 0xff;
	int invert = (kcontrol->private_value >> 22) & 1;
	int stereo = (kcontrol->private_value >> 23) & 1;
	int change;
	int left, right, oleft, oright;

	left = ucontrol->value.integer.value[0] & mask;
	if (invert)
		left = mask - left;
	left <<= left_shift;
	if (stereo) {
		right = ucontrol->value.integer.value[1] & mask;
		if (invert)
			right = mask - right;
		right <<= right_shift;
	} else
		right = 0;
	spin_lock_irqsave(&cm->reg_lock, flags);
	oleft = snd_cmipci_mixer_read(cm, left_reg);
	left = (oleft & ~(mask << left_shift)) | left;
	change = left != oleft;
	if (left_reg != right_reg)
		snd_cmipci_mixer_write(cm, left_reg, left);
	if (stereo) {
		if (left_reg != right_reg)
			oright = snd_cmipci_mixer_read(cm, right_reg);
		else
			oright = oleft;
		right = (oright & ~(mask << right_shift)) | right;
		change = (change || right != oright);
		snd_cmipci_mixer_write(cm, right_reg, right);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

/*
 * input route (left,right) -> (left,right)
 */
#define CMIPCI_INPUT_SW(xname, xindex, left_shift, right_shift) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_cmipci_info_input_sw, \
  get: snd_cmipci_get_input_sw, put: snd_cmipci_put_input_sw, \
  private_value: (left_shift << 16) | (right_shift << 24) }

static int snd_cmipci_info_input_sw(snd_kcontrol_t * kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	uinfo->count = 4;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;
	return 0;
}
 
static int snd_cmipci_get_input_sw(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int reg1 = SB_DSP4_INPUT_LEFT;
	int reg2 = SB_DSP4_INPUT_RIGHT;
	int left_shift = (kcontrol->private_value >> 16) & 0x0f;
	int right_shift = (kcontrol->private_value >> 24) & 0x0f;
	int val1, val2;

	spin_lock_irqsave(&cm->reg_lock, flags);
	val1 = snd_cmipci_mixer_read(cm, reg1);
	val2 = snd_cmipci_mixer_read(cm, reg2);
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	ucontrol->value.integer.value[0] = (val1 >> left_shift) & 1;
	ucontrol->value.integer.value[1] = (val2 >> left_shift) & 1;
	ucontrol->value.integer.value[2] = (val1 >> right_shift) & 1;
	ucontrol->value.integer.value[3] = (val2 >> right_shift) & 1;
	return 0;
}

static int snd_cmipci_put_input_sw(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int reg1 = SB_DSP4_INPUT_LEFT;
	int reg2 = SB_DSP4_INPUT_RIGHT;
	int left_shift = (kcontrol->private_value >> 16) & 0x0f;
	int right_shift = (kcontrol->private_value >> 24) & 0x0f;
	int change;
	int val1, val2, oval1, oval2;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oval1 = snd_cmipci_mixer_read(cm, reg1);
	oval2 = snd_cmipci_mixer_read(cm, reg2);
	val1 = oval1 & ~((1 << left_shift) | (1 << right_shift));
	val2 = oval2 & ~((1 << left_shift) | (1 << right_shift));
	val1 |= (ucontrol->value.integer.value[0] & 1) << left_shift;
	val2 |= (ucontrol->value.integer.value[1] & 1) << left_shift;
	val1 |= (ucontrol->value.integer.value[2] & 1) << right_shift;
	val2 |= (ucontrol->value.integer.value[3] & 1) << right_shift;
	change = val1 != oval1 || val2 != oval2;
	snd_cmipci_mixer_write(cm, reg1, val1);
	snd_cmipci_mixer_write(cm, reg2, val2);
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

/*
 * native mixer switches
 */

#define CMIPCI_MIXER_SW(xname, xindex, reg, lshift, rshift, invert) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_cmipci_info_mixer_sw, \
  get: snd_cmipci_get_mixer_sw, put: snd_cmipci_put_mixer_sw, \
  private_value: reg | (lshift << 16) | (rshift << 19) | (invert << 22)}

static int snd_cmipci_info_mixer_sw(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t *uinfo)
{
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;

	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	if (left_shift == right_shift)
		uinfo->count = 1;
	else
		uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;
	return 0;

}

static int snd_cmipci_get_mixer_sw(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	int reg = kcontrol->private_value & 0xff;
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;
	int invert = (kcontrol->private_value >> 22) & 1;
	unsigned long flags;
	unsigned char oreg, val;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = inb(cm->iobase + reg);
	val = (oreg >> left_shift) & 1;
	if (invert)
		val = 1 - val;
	ucontrol->value.integer.value[0] = val;
	if (left_shift != right_shift) {
		val = (oreg >> right_shift) & 1;
		if (invert)
			val = 1 - val;
		ucontrol->value.integer.value[1] = val;
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return 0;
}

static int snd_cmipci_put_mixer_sw(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	int reg = kcontrol->private_value & 0xff;
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;
	int invert = (kcontrol->private_value >> 22) & 1;
	unsigned long flags;
	unsigned char oreg, nreg, val;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = inb(cm->iobase + reg);
	val = ucontrol->value.integer.value[0] & 1;
	if (invert)
		val = 1 - val;
	if (val)
		nreg = oreg | (1 << left_shift);
	else
		nreg = oreg & ~(1 << left_shift);
	if (left_shift != right_shift) {
		val = ucontrol->value.integer.value[1] & 1;
		if (invert)
			val = 1 - val;
		if (val)
			nreg |= (1 << right_shift);
		else
			nreg &= ~(1 << right_shift);
	}
	outb(nreg, cm->iobase + reg);
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return  (nreg != oreg);
}

#define CMIPCI_MIXER_VOLUME(xname, xindex, reg, lshift, rshift, mask) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, index: xindex, \
  info: snd_cmipci_info_mixer_volume, \
  get: snd_cmipci_get_mixer_volume, put: snd_cmipci_put_mixer_volume, \
  private_value: reg | (lshift << 16) | (rshift << 19) | (mask << 24)}

static int snd_cmipci_info_mixer_volume(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t *uinfo)
{
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;
	int mask = (kcontrol->private_value >> 24) & 0xff;

	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	if (left_shift == right_shift)
		uinfo->count = 1;
	else
		uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = mask;
	return 0;

}

static int snd_cmipci_get_mixer_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	int reg = kcontrol->private_value & 0xff;
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;
	int mask = (kcontrol->private_value >> 24) & 0xff;
	unsigned long flags;
	unsigned char oreg, val;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = inb(cm->iobase + reg);
	val = (oreg >> left_shift) & mask;
	ucontrol->value.integer.value[0] = val;
	if (left_shift != right_shift) {
		val = (oreg >> right_shift) & mask;
		ucontrol->value.integer.value[1] = val;
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return 0;
}

static int snd_cmipci_put_mixer_volume(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	int reg = kcontrol->private_value & 0xff;
	int left_shift = (kcontrol->private_value >> 16) & 0x07;
	int right_shift = (kcontrol->private_value >> 19) & 0x07;
	int mask = (kcontrol->private_value >> 24) & 0xff;
	unsigned long flags;
	unsigned char oreg, nreg, val;

	spin_lock_irqsave(&cm->reg_lock, flags);
	oreg = inb(cm->iobase + reg);
	val = ucontrol->value.integer.value[0] & mask;
	nreg = oreg & ~(mask << left_shift);
	nreg |= (val << left_shift);
	if (left_shift != right_shift) {
		val = ucontrol->value.integer.value[1] & mask;
		nreg &= ~(mask << right_shift);
		nreg |= (val << right_shift);
	}
	outb(nreg, cm->iobase + reg);
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return  (nreg != oreg);
}



#define num_controls(ary) (sizeof(ary) / sizeof(snd_kcontrol_new_t))

static snd_kcontrol_new_t snd_cmipci_mixers[] = {
	CMIPCI_VOL_STEREO("Master Playback Volume", 0, SB_DSP4_MASTER_DEV, 3, 31),
	CMIPCI_MIXER_SW("3D Control - Switch", 0, CM_REG_MIXER1, CM_X3DEN_SHIFT, CM_X3DEN_SHIFT, 0),
	CMIPCI_VOL_STEREO("PCM Playback Volume", 0, SB_DSP4_PCM_DEV, 3, 31),
	CMIPCI_MIXER_SW("PCM Playback Switch", 0, CM_REG_MIXER1, CM_WSMUTE_SHIFT, CM_WSMUTE_SHIFT, 1),
	CMIPCI_MIXER_SW("PCM Capture Switch", 0, CM_REG_MIXER1, CM_WAVEINL_SHIFT, CM_WAVEINR_SHIFT, 0),
	CMIPCI_VOL_STEREO("Synth Playback Volume", 0, SB_DSP4_SYNTH_DEV, 3, 31),
	CMIPCI_MIXER_SW("Synth Playback Switch", 0, CM_REG_MIXER1, CM_FMMUTE_SHIFT, CM_FMMUTE_SHIFT, 1),
	CMIPCI_INPUT_SW("Synth Capture Route", 0, 6, 5),
	CMIPCI_VOL_STEREO("CD Playback Volume", 0, SB_DSP4_CD_DEV, 3, 31),
	CMIPCI_SW_STEREO("CD Playback Switch", 0, 2, 1),
	CMIPCI_INPUT_SW("CD Capture Route", 0, 2, 1),
	CMIPCI_VOL_STEREO("Line Playback Volume", 0, SB_DSP4_LINE_DEV, 3, 31),
	CMIPCI_SW_STEREO("Line Playback Switch", 0, 4, 3),
	CMIPCI_INPUT_SW("Line Capture Route", 0, 4, 3),
	CMIPCI_VOL_MONO("Mic Playback Volume", 0, SB_DSP4_MIC_DEV, 3, 31),
	CMIPCI_DOUBLE("Mic Playback Switch", 0, SB_DSP4_OUTPUT_SW, SB_DSP4_OUTPUT_SW, 0, 0, 1, 1, 0),
	CMIPCI_DOUBLE("Mic Capture Switch", 0, SB_DSP4_INPUT_LEFT, SB_DSP4_INPUT_RIGHT, 0, 0, 1, 1, 0),
	CMIPCI_VOL_MONO("PC Speaker Playback Volume", 0, SB_DSP4_SPEAKER_DEV, 6, 3),
	CMIPCI_MIXER_VOLUME("Aux Playback Volume", 0, CM_REG_AUX_VOL, 4, 0, 15),
	CMIPCI_MIXER_SW("Aux Playback Switch", 0, CM_REG_MIXER2, CM_VAUXLM, CM_VAUXRM, 1),
	CMIPCI_MIXER_SW("Aux Capture Switch", 0, CM_REG_MIXER2, CM_RAUXLEN, CM_RAUXREN, 0),
	CMIPCI_MIXER_SW("Mic Boost", 0, CM_REG_MIXER2, CM_MICGAINZ, CM_MICGAINZ, 0),
	CMIPCI_MIXER_VOLUME("Mic Capture Volume", 0, CM_REG_MIXER2, 1, 1, 7),
};

static int __init snd_cmipci_mixer_new(cmipci_t *cm)
{
	unsigned long flags;
	snd_card_t * card;
	int idx, err;

	snd_assert(cm != NULL && cm->card != NULL, return -EINVAL);

	card = cm->card;

	strcpy(card->mixerid, "CMedia PCI");
	strcpy(card->mixername, "CMedia PCI");

	spin_lock_irqsave(&cm->reg_lock, flags);
	snd_cmipci_mixer_write(cm, 0x00, 0x00);		/* mixer reset */
	spin_unlock_irqrestore(&cm->reg_lock, flags);

	for (idx = 0; idx < num_controls(snd_cmipci_mixers); idx++) {
		if ((err = snd_ctl_add(card, snd_ctl_new1(&snd_cmipci_mixers[idx], cm))) < 0)
			return err;
	}
	return 0;
}

/*
 */
static snd_pcm_hardware_t snd_cmipci_playback =
{
	info:			(SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_BLOCK_TRANSFER |
				 SNDRV_PCM_INFO_MMAP_VALID),
	formats:		SNDRV_PCM_FMTBIT_U8 | SNDRV_PCM_FMTBIT_S16_LE,
	rates:			SNDRV_PCM_RATE_5512 | SNDRV_PCM_RATE_8000_48000,
	rate_min:		5512,
	rate_max:		48000,
	channels_min:		1,
	channels_max:		2,
	buffer_bytes_max:	(128*1024),
	period_bytes_min:	4096, // ??
	period_bytes_max:	(128*1024),
	periods_min:		1,
	periods_max:		1024,
	fifo_size:		0,
};

static snd_pcm_hardware_t snd_cmipci_capture =
{
	info:			(SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_BLOCK_TRANSFER |
				 SNDRV_PCM_INFO_MMAP_VALID),
	formats:		SNDRV_PCM_FMTBIT_U8 | SNDRV_PCM_FMTBIT_S16_LE,
	rates:			SNDRV_PCM_RATE_5512 | SNDRV_PCM_RATE_8000_48000,
	rate_min:		5512,
	rate_max:		48000,
	channels_min:		1,
	channels_max:		2,
	buffer_bytes_max:	(128*1024),
	period_bytes_min:	64,
	period_bytes_max:	(128*1024),
	periods_min:		1,
	periods_max:		1024,
	fifo_size:		0,
};

static int snd_cmipci_playback_open(snd_pcm_substream_t * substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	cm->playback.substream = substream;
	cm->playback.ch = 0;
	cm->playback.dac = 1;
	runtime->hw = snd_cmipci_playback;
	snd_pcm_hw_constraint_minmax(runtime, SNDRV_PCM_HW_PARAM_BUFFER_SIZE, 0, 0x10000);
	return 0;
}

static int snd_cmipci_capture_open(snd_pcm_substream_t * substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	cm->capture.substream = substream;
	cm->capture.ch = 1;
	cm->capture.dac = 0;
	runtime->hw = snd_cmipci_capture;
	snd_pcm_hw_constraint_minmax(runtime, SNDRV_PCM_HW_PARAM_BUFFER_SIZE, 0, 0x10000);
	return 0;
}

static int snd_cmipci_playback_close(snd_pcm_substream_t * substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);

	cm->playback.substream = NULL;
	return 0;
}

static int snd_cmipci_capture_close(snd_pcm_substream_t * substream)
{
	cmipci_t *cm = snd_pcm_substream_chip(substream);

	cm->capture.substream = NULL;
	return 0;
}

static snd_pcm_ops_t snd_cmipci_playback_ops = {
	open:		snd_cmipci_playback_open,
	close:		snd_cmipci_playback_close,
	ioctl:		snd_pcm_lib_ioctl,
	hw_params:	snd_cmipci_hw_params,
	hw_free:	snd_cmipci_hw_free,
	prepare:	snd_cmipci_playback_prepare,
	trigger:	snd_cmipci_playback_trigger,
	pointer:	snd_cmipci_playback_pointer,
};

static snd_pcm_ops_t snd_cmipci_capture_ops = {
	open:		snd_cmipci_capture_open,
	close:		snd_cmipci_capture_close,
	ioctl:		snd_pcm_lib_ioctl,
	hw_params:	snd_cmipci_hw_params,
	hw_free:	snd_cmipci_hw_free,
	prepare:	snd_cmipci_capture_prepare,
	trigger:	snd_cmipci_capture_trigger,
	pointer:	snd_cmipci_capture_pointer,
};

static void snd_cmipci_pcm_free(snd_pcm_t *pcm)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, pcm->private_data, return);
	cm->pcm = NULL;
	snd_pcm_lib_preallocate_pci_free_for_all(cm->pci, pcm);
}

static int __init snd_cmipci_pcm_new(cmipci_t *cm, int device, snd_pcm_t **rpcm)
{
	snd_pcm_t *pcm;
	int err;

	if (rpcm)
		*rpcm = NULL;
	err = snd_pcm_new(cm->card, "C-Media PCI", device, 1, 1, &pcm);
	if (err < 0)
		return err;

	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, &snd_cmipci_playback_ops);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_cmipci_capture_ops);

	pcm->private_data = cm;
	pcm->private_free = snd_cmipci_pcm_free;
	pcm->info_flags = 0;
	strcpy(pcm->name, "C-Media PCI DAC/ADC");
	cm->pcm = pcm;

	snd_pcm_lib_preallocate_pci_pages_for_all(cm->pci, pcm, 64*1024, 128*1024);

	if (rpcm)
		*rpcm = pcm;
	return 0;
}


/*
 * universal switches
 */

typedef struct snd_cmipci_switch_args {
	int reg;
	unsigned int mask;
	unsigned int mask_on;
	int is_byte;
} snd_cmipci_switch_args_t;

static int snd_cmipci_uswitch_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;
	return 0;
}

static int snd_cmipci_uswitch_get(snd_kcontrol_t *kcontrol, snd_ctl_elem_value_t *ucontrol)
{
	unsigned long flags;
	unsigned int val;
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	snd_cmipci_switch_args_t *args = (snd_cmipci_switch_args_t*)kcontrol->private_value;
	snd_assert(args != NULL, return -EINVAL);

	spin_lock_irqsave(&cm->reg_lock, flags);
	if (args->is_byte)
		val = inb(cm->iobase + args->reg);
	else
		val = snd_cmipci_read(cm, args->reg);
	ucontrol->value.integer.value[0] = ((val & args->mask) == args->mask_on) ? 1 : 0;
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return 0;
}

static int snd_cmipci_uswitch_put(snd_kcontrol_t *kcontrol, snd_ctl_elem_value_t *ucontrol)
{
	unsigned long flags;
	unsigned int val;
	int change;
	cmipci_t *cm = snd_kcontrol_chip(kcontrol);
	snd_cmipci_switch_args_t *args = (snd_cmipci_switch_args_t*)kcontrol->private_value;
	snd_assert(args != NULL, return -EINVAL);

	spin_lock_irqsave(&cm->reg_lock, flags);
	if (args->is_byte)
		val = inb(cm->iobase + args->reg);
	else
		val = snd_cmipci_read(cm, args->reg);
	change = (val & args->mask) != (ucontrol->value.integer.value[0] ? args->mask : 0);
	if (change) {
		val &= ~args->mask;
		if (ucontrol->value.integer.value[0])
			val |= args->mask_on;
		else
			val |= (args->mask & ~args->mask_on);
		if (args->is_byte)
			outb((unsigned char)val, cm->iobase + args->reg);
		else
			snd_cmipci_write(cm, args->reg, val);
	}
	spin_unlock_irqrestore(&cm->reg_lock, flags);
	return change;
}

static snd_cmipci_switch_args_t joy_args = {
	CM_REG_FUNCTRL1, CM_JYSTK_EN, CM_JYSTK_EN, 0
};
static snd_cmipci_switch_args_t modem_args = {
	CM_REG_MISC_CTRL, CM_FLINKON|CM_FLINKOFF, CM_FLINKON, 0
};
static snd_cmipci_switch_args_t spdif_in_args = {
	CM_REG_FUNCTRL1, CM_SPDF_1, CM_SPDF_1, 0
};
static snd_cmipci_switch_args_t spdif_out_args = {
	CM_REG_LEGACY_CTRL, CM_ENSPDOUT|CM_SPDCOPYRHT|CM_DAC2SPDO,
	CM_ENSPDOUT|CM_SPDCOPYRHT|CM_DAC2SPDO, 0
};
static snd_cmipci_switch_args_t spdo_48k_args = {
	CM_REG_MISC_CTRL, CM_SPDF_AC97, CM_SPDF_AC97, 0
};
static snd_cmipci_switch_args_t spdif_5v_args = {
	CM_REG_MISC_CTRL, CM_SPDO5V, 0, 0
};
static snd_cmipci_switch_args_t spdif_loop_args = {
	CM_REG_FUNCTRL1, CM_SPDFLOOP, CM_SPDFLOOP, 0
};
static snd_cmipci_switch_args_t spdif_in_monitor_args = {
	CM_REG_MIXER1, CM_CDPLAY, CM_CDPLAY, 1
};
static snd_cmipci_switch_args_t spdif_in_phase_args = {
	CM_REG_MISC, 0x04, 0x04, 0
};
static snd_cmipci_switch_args_t fourch_args = {
	CM_REG_MISC_CTRL, CM_N4SPK3D, CM_N4SPK3D, 0
};
static snd_cmipci_switch_args_t rearout_args = {
	CM_REG_MIXER1, CM_SPK4, CM_SPK4, 1
};

static snd_kcontrol_new_t snd_cmipci_switches[] = {
	{ iface: SNDRV_CTL_ELEM_IFACE_CARD,
	  name: "Joystick Enable",
	  info: snd_cmipci_uswitch_info,
	  get: snd_cmipci_uswitch_get,
	  put: snd_cmipci_uswitch_put,
	  private_value: (unsigned long)&joy_args,
	},
	{ iface: SNDRV_CTL_ELEM_IFACE_CARD,
	  name: "Modem Enable",
	  info: snd_cmipci_uswitch_info,
	  get: snd_cmipci_uswitch_get,
	  put: snd_cmipci_uswitch_put,
	  private_value: (unsigned long)&modem_args,
	},
};

#define CMIPCI_SWITCH(xname,args) \
{ iface: SNDRV_CTL_ELEM_IFACE_MIXER, name: xname, info: snd_cmipci_uswitch_info,\
		 get: snd_cmipci_uswitch_get, put: snd_cmipci_uswitch_put,\
		 private_value: (unsigned long)(args) }

static snd_kcontrol_new_t snd_cm8738_switches[] = {
	CMIPCI_SWITCH("SPDIF In Record", &spdif_in_args),
	CMIPCI_SWITCH("SPDIF Out", &spdif_out_args),
	CMIPCI_SWITCH("SPDIF Out 48KHz", &spdo_48k_args),
	CMIPCI_SWITCH("SPDIF 5V", &spdif_5v_args),
	CMIPCI_SWITCH("SPDIF Loop", &spdif_loop_args),
	CMIPCI_SWITCH("SPDIF In Monitor", &spdif_in_monitor_args),
	CMIPCI_SWITCH("SPDIF In Phase Inverse", &spdif_in_phase_args),
	CMIPCI_SWITCH("Four Channel", &fourch_args),
	CMIPCI_SWITCH("Rear Out", &rearout_args),
};

static int snd_cmipci_switches_new(cmipci_t *cm)
{
	int idx;

	for (idx = 0; idx < num_controls(snd_cmipci_switches); idx++)
		snd_ctl_add(cm->card, snd_ctl_new1(&snd_cmipci_switches[idx], cm));
	if (cm->card->type == SNDRV_CARD_TYPE_CMI8738) {
		for (idx = 0; idx < num_controls(snd_cm8738_switches); idx++)
			snd_ctl_add(cm->card, snd_ctl_new1(&snd_cm8738_switches[idx], cm));
	}
	return 0;
}


/*
 * proc interface
 */

static void snd_cmipci_proc_read(snd_info_entry_t *entry, 
				 snd_info_buffer_t *buffer)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, entry->private_data, return);
	int i;
	
	snd_iprintf(buffer, "C-Media PCI\n\n");
	for (i = 0; i < 0x40; i++) {
		int v = inb(cm->iobase + i);
		if (i % 4 == 0)
			snd_iprintf(buffer, "%02x: ", i);
		snd_iprintf(buffer, "%02x", v);
		if (i % 4 == 3)
			snd_iprintf(buffer, "\n");
		else
			snd_iprintf(buffer, " ");
	}
}

static void __init snd_cmipci_proc_init(cmipci_t *cm)
{
	snd_info_entry_t *entry;

	if ((entry = snd_info_create_card_entry(cm->card, "cmipci", cm->card->proc_root)) != NULL) {
		entry->content = SNDRV_INFO_CONTENT_TEXT;
		entry->private_data = cm;
		entry->mode = S_IFREG | S_IRUGO | S_IWUSR;
		entry->c.text.read_size = 256;
		entry->c.text.read = snd_cmipci_proc_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	cm->proc_entry = entry;
}

static void snd_cmipci_proc_done(cmipci_t *cm)
{
	if (cm->proc_entry) {
		snd_info_unregister(cm->proc_entry);
		cm->proc_entry = NULL;
	}
}


#ifndef PCI_VENDOR_ID_CMEDIA
#define PCI_VENDOR_ID_CMEDIA         0x13F6
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8338A
#define PCI_DEVICE_ID_CMEDIA_CM8338A 0x0100
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8338B
#define PCI_DEVICE_ID_CMEDIA_CM8338B 0x0101
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8738
#define PCI_DEVICE_ID_CMEDIA_CM8738  0x0111
#endif
#ifndef PCI_DEVICE_ID_CMEDIA_CM8738B
#define PCI_DEVICE_ID_CMEDIA_CM8738B 0x0112
#endif

static struct pci_device_id snd_cmipci_ids[] __devinitdata = {
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8338A, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8338B, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8738, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_CMEDIA, PCI_DEVICE_ID_CMEDIA_CM8738B, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0,},
};


static int snd_cmipci_free(cmipci_t *cm)
{
	snd_cmipci_proc_done(cm);

	if (cm->irq >= 0) {
		snd_cmipci_write(cm, CM_REG_INT_HLDCLR, 0);  /* disable ints */
		snd_cmipci_write(cm, CM_REG_FUNCTRL0, 0); /* disable channels */

		/* reset mixer */
		snd_cmipci_mixer_write(cm, 0, 0);

		synchronize_irq();

		free_irq(cm->irq, (void *)cm);
	}
	if (cm->res_iobase)
		release_resource(cm->res_iobase);
	snd_magic_kfree(cm);
	return 0;
}

static int snd_cmipci_dev_free(snd_device_t *device)
{
	cmipci_t *cm = snd_magic_cast(cmipci_t, device->device_data, return -ENXIO);
	return snd_cmipci_free(cm);
}

static int __init snd_cmipci_create(snd_card_t *card,
				    struct pci_dev *pci,
				    int enable_midi,
				    unsigned long iomidi,
				    int enable_synth,
				    unsigned long iosynth,
				    cmipci_t **rcmipci)
{
	cmipci_t *cm;
	int err;
	static snd_device_ops_t ops = {
		dev_free:	snd_cmipci_dev_free,
	};
	unsigned int val;
	
	*rcmipci = NULL;

	if ((err = pci_enable_device(pci)) < 0)
		return err;

	cm = snd_magic_kcalloc(cmipci_t, 0, GFP_KERNEL);
	if (cm == NULL)
		return -ENOMEM;

	spin_lock_init(&cm->reg_lock);
	cm->card = card;
	cm->pci = pci;
	cm->enable_midi = enable_midi;
	cm->enable_synth = enable_synth;
	cm->irq = -1;
	cm->iobase = pci_resource_start(pci, 0);
	cm->iomidi = iomidi;
	cm->iosynth = iosynth;

	if ((cm->res_iobase = request_region(cm->iobase, CM_EXTENT_CODEC, "C-Media PCI")) == NULL) {
		snd_cmipci_free(cm);
		snd_printk("unable to grab ports 0x%lx-0x%lx\n", cm->iobase, cm->iobase + CM_EXTENT_CODEC - 1);
		return -EBUSY;
	}
	if (request_irq(pci->irq, snd_cmipci_interrupt, SA_INTERRUPT|SA_SHIRQ, "C-Media PCI", (void *)cm)) {
		snd_cmipci_free(cm);
		snd_printk("unable to grab IRQ %d\n", pci->irq);
		return -EBUSY;
	}
	cm->irq = pci->irq;

	cm->ctrl = CM_CHADC1;	/* FUNCNTRL0 */

	pci_set_master(cm->pci);

	/* initialize codec registers */
	snd_cmipci_write(cm, CM_REG_INT_HLDCLR, 0);	/* disable ints */
	snd_cmipci_write(cm, CM_REG_FUNCTRL0, 0);	/* disable channels */
	snd_cmipci_write(cm, CM_REG_FUNCTRL1, 0);

	snd_cmipci_write(cm, CM_REG_CHFORMAT, 0x00200000);

	/* XXX */
	outb(0xee, cm->iobase + 0x26);

	/* set MPU address */
	if (cm->enable_midi) {
		switch (cm->iomidi) {
		case 0x320: val = CM_VMPU_320; break;
		case 0x310: val = CM_VMPU_310; break;
		case 0x300: val = CM_VMPU_300; break;
		default: val = CM_VMPU_330; break; /* 0x330 */
		}
		snd_cmipci_write(cm, CM_REG_LEGACY_CTRL, val);
		/* enable UART */
		snd_cmipci_set_bit(cm, CM_REG_FUNCTRL1, CM_UART_EN);
	}

	/* set FM address */
	if (cm->enable_synth) {
		val = snd_cmipci_read(cm, CM_REG_LEGACY_CTRL) & ~CM_FMSEL_MASK;
		switch (cm->iosynth) {
		case 0x3E8: val |= CM_FMSEL_3E8; break;
		case 0x3E0: val |= CM_FMSEL_3E0; break;
		case 0x3C8: val |= CM_FMSEL_3C8; break;
		default: val |= CM_FMSEL_388; break; /* 0x388 */
		}
		snd_cmipci_write(cm, CM_REG_LEGACY_CTRL, val);
	}

	/* enable FM */
	if (cm->enable_synth)
		snd_cmipci_set_bit(cm, CM_REG_MISC_CTRL, CM_FM_EN);

	/* reset mixer */
	snd_cmipci_mixer_write(cm, 0, 0);

	/* create switches */
	snd_cmipci_switches_new(cm);

	synchronize_irq();

	snd_cmipci_proc_init(cm);

	if ((err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, cm, &ops)) < 0) {
		snd_cmipci_free(cm);
		return err;
	}

	*rcmipci = cm;
	return 0;
}

/*
 */

MODULE_DEVICE_TABLE(pci, snd_cmipci_ids);

static int __init snd_cmipci_probe(struct pci_dev *pci,
				  const struct pci_device_id *id)
{
	static int dev = 0;
	snd_card_t *card;
	cmipci_t *cm;
	int err;

	for (; dev < SNDRV_CARDS; dev++) {
		if (! snd_enable[dev]) {
			dev++;
			return -ENOENT;
		}
		break;
	}
	if (dev >= SNDRV_CARDS)
		return -ENODEV;

	card = snd_card_new(snd_index[dev], snd_id[dev], THIS_MODULE, 0);
	if (card == NULL)
		return -ENOMEM;
	
	if (pci->device == PCI_DEVICE_ID_CMEDIA_CM8738)
		card->type = SNDRV_CARD_TYPE_CMI8738;
	else
		card->type = SNDRV_CARD_TYPE_CMI8338;

	if ((err = snd_cmipci_create(card, pci,
				     snd_enable_midi[dev],
				     snd_mpu_port[dev],
				     snd_enable_fm[dev],
				     snd_fm_port[dev], &cm)) < 0) {
		snd_card_free(card);
		return err;
	}
	if ((err = snd_cmipci_mixer_new(cm)) < 0) {
		snd_card_free(card);
		return err;
	}
	if ((err = snd_cmipci_pcm_new(cm, 0, NULL)) < 0) {
		snd_card_free(card);
		return err;
	}
	if (cm->enable_midi) {
		if ((err = snd_mpu401_uart_new(card, 0, MPU401_HW_CMIPCI,
					       cm->iomidi, 0,
					       cm->irq, 0, &cm->rmidi)) < 0) {
			snd_card_free(card);
			return err;
		}
	}
	strcpy(card->abbreviation, "C-Media PCI");
	strcpy(card->shortname, "C-Media PCI");
	sprintf(card->longname, "%s %s at 0x%lx, irq %d",
		card->shortname,
		card->type == SNDRV_CARD_TYPE_CMI8738 ? "CM8738" : "CM8338",
		cm->iobase,
		cm->irq);

	snd_printd("%s is detected\n", card->longname);

	if ((err = snd_card_register(card)) < 0) {
		snd_card_free(card);
		return err;
	}
	PCI_SET_DRIVER_DATA(pci, card);
	dev++;
	return 0;

}

static void __exit snd_cmipci_remove(struct pci_dev *pci)
{
	snd_card_free(PCI_GET_DRIVER_DATA(pci));
	PCI_SET_DRIVER_DATA(pci, NULL);
}


static struct pci_driver driver = {
	name: "C-Media PCI",
	id_table: snd_cmipci_ids,
	probe: snd_cmipci_probe,
	remove: snd_cmipci_remove,
};
	
static int __init alsa_card_cmipci_init(void)
{
	int err;

	if ((err = pci_module_init(&driver)) < 0) {
#ifdef MODULE
		snd_printk("C-Media PCI soundcard not found or device busy\n");
#endif
		return err;
	}
	return 0;
}

static void __exit alsa_card_cmipci_exit(void)
{
	pci_unregister_driver(&driver);
}

module_init(alsa_card_cmipci_init)
module_exit(alsa_card_cmipci_exit)

#ifndef MODULE

/* format is: snd-card-cmipci=snd_enable,snd_index,snd_id,
			      snd_enable_midi,snd_enable_fm,
			      snd_mpu_port,snd_fm_port */

static int __init alsa_card_cmipci_setup(char *str)
{
	static unsigned __initdata nr_dev = 0;

	if (nr_dev >= SNDRV_CARDS)
		return 0;
	(void)(get_option(&str,&snd_enable[nr_dev]) == 2 &&
	       get_option(&str,&snd_index[nr_dev]) == 2 &&
	       get_id(&str,&snd_id[nr_dev]) == 2 &&
	       get_option(&str,&snd_enable_midi[nr_dev]) == 2 &&
	       get_option(&str,&snd_enable_fm[nr_dev]) == 2 &&
	       get_option(&str,(int *)&snd_mpu_port[nr_dev]) == 2 &&
	       get_option(&str,(int *)&snd_fm_port[nr_dev]) == 2);
	nr_dev++;
	return 1;
}

__setup("snd-card-cmipci=", alsa_card_cmipci_setup);

#endif /* ifndef MODULE */
