/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Memory allocation routines.
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../include/driver.h"
#include "../include/info.h"

/*
 *  memory allocation helpers and debug routines
 */

#ifdef CONFIG_SND_DEBUG

struct snd_alloc_track {
	unsigned long magic;
	void *caller;
	struct snd_alloc_track *prev, *next;
	size_t size;
};

static long snd_alloc_pages;
static long snd_alloc_kmalloc;
static long snd_alloc_vmalloc;
static struct snd_alloc_track *snd_alloc_kmalloc_last = NULL;
static struct snd_alloc_track *snd_alloc_vmalloc_last = NULL;
static spinlock_t snd_alloc_kmalloc_lock = SPIN_LOCK_UNLOCKED;
static spinlock_t snd_alloc_vmalloc_lock = SPIN_LOCK_UNLOCKED;
#define KMALLOC_MAGIC 0x87654321
#define VMALLOC_MAGIC 0x87654320
static snd_info_entry_t *snd_memory_info_entry;

void snd_memory_init(void)
{
	snd_alloc_pages = 0;
	snd_alloc_kmalloc = 0;
	snd_alloc_vmalloc = 0;
}

void snd_memory_done(void)
{
	struct snd_alloc_track *t;
	if (snd_alloc_pages > 0)
		snd_printk("Not freed snd_alloc_pages = %li\n", snd_alloc_pages);
	if (snd_alloc_kmalloc > 0)
		snd_printk("Not freed snd_alloc_kmalloc = %li\n", snd_alloc_kmalloc);
	if (snd_alloc_vmalloc > 0)
		snd_printk("Not freed snd_alloc_vmalloc = %li\n", snd_alloc_vmalloc);
	t = snd_alloc_kmalloc_last;
	while (t) {
		if (t->magic != KMALLOC_MAGIC) {
			snd_printk("Corrupted kmalloc\n");
			break;
		}
		snd_printk("kmalloc(%ld) from %p not freed\n", (long) t->size, t->caller);
		t = t->prev;
	}
	t = snd_alloc_vmalloc_last;
	while (t) {
		if (t->magic != VMALLOC_MAGIC) {
			snd_printk("Corrupted vmalloc\n");
			break;
		}
		snd_printk("vmalloc(%ld) from %p not freed\n", (long) t->size, t->caller);
		t = t->prev;
	}
}

void *__snd_kmalloc(size_t size, int flags, void *caller)
{
	unsigned long cpu_flags;
	struct snd_alloc_track *t;
	char *ptr;
	
	ptr = snd_wrapper_kmalloc(size + sizeof(struct snd_alloc_track), flags);
	if (ptr != NULL) {
		t = (struct snd_alloc_track *)ptr;
		t->magic = KMALLOC_MAGIC;
		t->caller = caller;
		spin_lock_irqsave(&snd_alloc_kmalloc_lock, cpu_flags);
		t->next = 0;
		t->prev = snd_alloc_kmalloc_last;
		if (t->prev)
			t->prev->next = t;
		snd_alloc_kmalloc_last = t;
		spin_unlock_irqrestore(&snd_alloc_kmalloc_lock, cpu_flags);
		t->size = size;
		snd_alloc_kmalloc += size;
		ptr += sizeof(struct snd_alloc_track);
	}
	return ptr;
}

#define _snd_kmalloc(size, flags) __snd_kmalloc((size), (flags), __builtin_return_address(0));
void *snd_hidden_kmalloc(size_t size, int flags)
{
	return _snd_kmalloc(size, flags);
}

void snd_hidden_kfree(const void *obj)
{
	unsigned long flags;
	struct snd_alloc_track *t;
	if (obj == NULL) {
		snd_printk("null kfree (called from %p)\n", __builtin_return_address(0));
		return;
	}
	t = (struct snd_alloc_track *)(obj - sizeof(struct snd_alloc_track));
	if (t->magic != KMALLOC_MAGIC) {
		snd_printk("bad kfree (called from %p)\n", __builtin_return_address(0));
		return;
	}
	spin_lock_irqsave(&snd_alloc_kmalloc_lock, flags);
	if (t->next)
		t->next->prev = t->prev;
	else
		snd_alloc_kmalloc_last = t->prev;
	if (t->prev)
		t->prev->next = t->next;
	spin_unlock_irqrestore(&snd_alloc_kmalloc_lock, flags);
	t->magic = 0;
	snd_alloc_kmalloc -= t->size;
	obj = t;
	snd_wrapper_kfree(obj);
}

void *_snd_magic_kcalloc(unsigned long magic, size_t size, int flags)
{
	unsigned long *ptr;
	ptr = _snd_kmalloc(size + sizeof(unsigned long), flags);
	if (ptr) {
		*ptr++ = magic;
		memset(ptr, 0, size);
	}
	return ptr;
}

void *_snd_magic_kmalloc(unsigned long magic, size_t size, int flags)
{
	unsigned long *ptr;
	ptr = _snd_kmalloc(size + sizeof(unsigned long), flags);
	if (ptr)
		*ptr++ = magic;
	return ptr;
}

void snd_magic_kfree(void *_ptr)
{
	unsigned long *ptr = _ptr;
	if (ptr == NULL) {
		snd_printk("null snd_magic_kfree (called from %p)\n", __builtin_return_address(0));
		return;
	}
	*--ptr = 0;
	{
		struct snd_alloc_track *t;
		t = (void *)ptr - sizeof(struct snd_alloc_track);
		if (t->magic != KMALLOC_MAGIC) {
			snd_printk("bad snd_magic_kfree (called from %p)\n", __builtin_return_address(0));
			return;
		}
	}
	snd_hidden_kfree(ptr);
	return;
}

void *snd_hidden_vmalloc(unsigned long size)
{
	char *ptr = snd_wrapper_vmalloc(size + sizeof(struct snd_alloc_track));
	if (ptr) {
		struct snd_alloc_track *t = (struct snd_alloc_track *)ptr;
		t->magic = VMALLOC_MAGIC;
		t->caller = __builtin_return_address(0);
		spin_lock(&snd_alloc_vmalloc_lock);
		t->next = 0;
		t->prev = snd_alloc_vmalloc_last;
		if (t->prev)
			t->prev->next = t;
		snd_alloc_vmalloc_last = t;
		spin_unlock(&snd_alloc_vmalloc_lock);
		t->size = size;
		snd_alloc_vmalloc += size;
		ptr += sizeof(struct snd_alloc_track);
	}
	return ptr;
}

void snd_hidden_vfree(void *obj)
{
	struct snd_alloc_track *t;
	if (obj == NULL) {
		snd_printk("null vfree (called from %p)\n", __builtin_return_address(0));
		return;
	}
	t = obj - sizeof(struct snd_alloc_track);
	if (t->magic != VMALLOC_MAGIC) {
		snd_printk("bad vfree (called from %p)\n", __builtin_return_address(0));
		return;
	}
	spin_lock(&snd_alloc_vmalloc_lock);
	if (t->next)
		t->next->prev = t->prev;
	else
		snd_alloc_vmalloc_last = t->prev;
	if (t->prev)
		t->prev->next = t->next;
	spin_unlock(&snd_alloc_vmalloc_lock);
	t->magic = 0;
	snd_alloc_vmalloc -= t->size;
	obj = t;
	snd_wrapper_vfree(obj);
}

static void snd_memory_info_read(snd_info_entry_t *entry, snd_info_buffer_t * buffer)
{
	long pages = snd_alloc_pages >> (PAGE_SHIFT-12);
	snd_iprintf(buffer, "pages  : %li bytes (%li pages per %likB)\n", pages * PAGE_SIZE, pages, PAGE_SIZE / 1024);
	snd_iprintf(buffer, "kmalloc: %li bytes\n", snd_alloc_kmalloc);
	snd_iprintf(buffer, "vmalloc: %li bytes\n", snd_alloc_vmalloc);
}

int __init snd_memory_info_init(void)
{
	snd_info_entry_t *entry;

	entry = snd_info_create_module_entry(THIS_MODULE, "meminfo", NULL);
	if (entry) {
		entry->content = SNDRV_INFO_CONTENT_TEXT;
		entry->c.text.read_size = 256;
		entry->c.text.read = snd_memory_info_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	snd_memory_info_entry = entry;
	return 0;
}

int __exit snd_memory_info_done(void)
{
	if (snd_memory_info_entry)
		snd_info_unregister(snd_memory_info_entry);
	return 0;
}

#else

#define _snd_kmalloc kmalloc

#endif /* CONFIG_SND_DEBUG */



void *snd_malloc_pages(unsigned long size, unsigned int dma_flags)
{
	int pg;
	void *res;

	snd_assert(size > 0, return NULL);
	snd_assert(dma_flags != 0, return NULL);
	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	if ((res = (void *) __get_free_pages(dma_flags, pg)) != NULL) {
		mem_map_t *page = virt_to_page(res);
		mem_map_t *last_page = page + (1 << pg);
		while (page < last_page)
			SetPageReserved(page++);
#ifdef CONFIG_SND_DEBUG
		snd_alloc_pages += 1 << pg;
#endif
	}
	return res;
}

void *snd_malloc_pages_fallback(unsigned long size, unsigned int dma_flags, unsigned long *res_size)
{
	void *res;

	snd_assert(size > 0, return NULL);
	snd_assert(res_size != NULL, return NULL);
	do {
		if ((res = snd_malloc_pages(size, dma_flags)) != NULL) {
			*res_size = size;
			return res;
		}
		size >>= 1;
	} while (size >= PAGE_SIZE);
	return NULL;
}

void snd_free_pages(void *ptr, unsigned long size)
{
	int pg;
	mem_map_t *page, *last_page;

	if (ptr == NULL)
		return;
	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	page = virt_to_page(ptr);
	last_page = page + (1 << pg);
	while (page < last_page)
		ClearPageReserved(page++);
	free_pages((unsigned long) ptr, pg);
#ifdef CONFIG_SND_DEBUG
	snd_alloc_pages -= 1 << pg;
#endif
}

#ifdef CONFIG_PCI

void *snd_malloc_pci_pages(struct pci_dev *pci,
			   unsigned long size,
			   dma_addr_t *dmaaddr)
{
	int pg;
	void *res;

	snd_assert(size > 0, return NULL);
	snd_assert(dmaaddr != NULL, return NULL);
	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	res = pci_alloc_consistent(pci, PAGE_SIZE * (1 << pg), dmaaddr);
	if (res != NULL) {
		mem_map_t *page = virt_to_page(res);
		mem_map_t *last_page = page + (1 << pg);
		while (page < last_page)
			SetPageReserved(page++);
#ifdef CONFIG_SND_DEBUG
		snd_alloc_pages += 1 << pg;
#endif
	}
	return res;
}

void *snd_malloc_pci_pages_fallback(struct pci_dev *pci, unsigned long size,
				    dma_addr_t *dmaaddr,
				    unsigned long *res_size)
{
	void *res;

	snd_assert(res_size != NULL, return NULL);
	do {
		if ((res = snd_malloc_pci_pages(pci, size, dmaaddr)) != NULL) {
			*res_size = size;
			return res;
		}
		size >>= 1;
	} while (size >= PAGE_SIZE);
	return NULL;
}

void snd_free_pci_pages(struct pci_dev *pci, unsigned long size, void *ptr, dma_addr_t dmaaddr)
{
	int pg;
	mem_map_t *page, *last_page;

	if (ptr == NULL)
		return;
	for (pg = 0; PAGE_SIZE * (1 << pg) < size; pg++);
	page = virt_to_page(ptr);
	last_page = page + (1 << pg);
	while (page < last_page)
		ClearPageReserved(page++);
	pci_free_consistent(pci, PAGE_SIZE * (1 << pg), ptr, dmaaddr);
#ifdef CONFIG_SND_DEBUG
	snd_alloc_pages -= 1 << pg;
#endif
}

#endif /* CONFIG_PCI */

void *snd_kcalloc(size_t size, int flags)
{
	void *ptr;
	
	ptr = _snd_kmalloc(size, flags);
	if (ptr)
		memset(ptr, 0, size);
	return ptr;
}

char *snd_kmalloc_strdup(const char *string, int flags)
{
	size_t len;
	char *ptr;

	if (!string)
		return NULL;
	len = strlen(string) + 1;
	ptr = _snd_kmalloc(len, flags);
	if (ptr)
		memcpy(ptr, string, len);
	return ptr;
}

int copy_to_user_fromio(void *dst, unsigned long src, size_t count)
{
#ifdef __i386_
	return copy_to_user(dst, (const void*) src, count);
#else
	char buf[1024];
	while (count) {
		size_t c = count;
		int err;
		if (c > sizeof(buf))
			c = sizeof(buf);
		memcpy_fromio(buf, src, c);
		err = copy_to_user(dst, buf, c);
		if (err)
			return err;
		count -= c;
		dst += c;
		src += c;
	}
	return 0;
#endif
}

int copy_from_user_toio(unsigned long dst, const void *src, size_t count)
{
#ifdef __i386_
	return copy_from_user((void*)dst, src, count);
#else
	char buf[1024];
	while (count) {
		size_t c = count;
		int err;
		if (c > sizeof(buf))
			c = sizeof(buf);
		err = copy_from_user(buf, src, c);
		if (err)
			return err;
		memcpy_toio(dst, buf, c);
		count -= c;
		dst += c;
		src += c;
	}
	return 0;
#endif
}
