/*
 *  The driver for the ForteMedia FM801 based soundcards
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SNDRV_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/pcm.h"
#include "../include/ac97_codec.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"
#define SNDRV_GET_ID
#include "../include/initval.h"

#define chip_t fm801_t

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("ForteMedia FM801");
MODULE_CLASSES("{sound}");
MODULE_DEVICES("{{ForteMedia,FM801}}");

static int snd_index[SNDRV_CARDS] = SNDRV_DEFAULT_IDX;	/* Index 0-MAX */
static char *snd_id[SNDRV_CARDS] = SNDRV_DEFAULT_STR;	/* ID for this card */
static int snd_enable[SNDRV_CARDS] = SNDRV_DEFAULT_ENABLE;	/* Enable this card */

MODULE_PARM(snd_index, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for the FM801 soundcard.");
MODULE_PARM_SYNTAX(snd_index, SNDRV_INDEX_DESC);
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SNDRV_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for the FM801 soundcard.");
MODULE_PARM_SYNTAX(snd_id, SNDRV_ID_DESC);
MODULE_PARM(snd_enable, "1-" __MODULE_STRING(SNDRV_CARDS) "i");
MODULE_PARM_DESC(snd_enable, "Enable FM801 soundcard.");
MODULE_PARM_SYNTAX(snd_enable, SNDRV_ENABLE_DESC);

/*
 *  Direct registers
 */

#define FM801_REG(chip, reg)	(chip->port + FM801_##reg)

#define FM801_PCM_VOL		0x00	/* PCM Output Volume */
#define FM801_FM_VOL		0x02	/* FM Output Volume */
#define FM801_I2S_VOL		0x04	/* I2S Volume */
#define FM801_REC_SRC		0x06	/* Resourd Source */
#define FM801_PLY_CTRL		0x08	/* Playback Control */
#define FM801_PLY_COUNT		0x0a	/* Playback Count */
#define FM801_PLY_BUF1		0x0c	/* Playback Bufer I */
#define FM801_PLY_BUF2		0x10	/* Playback Buffer II */
#define FM801_CAP_CTRL		0x14	/* Capture Control */
#define FM801_CAP_COUNT		0x16	/* Capture Count */
#define FM801_CAP_BUF1		0x18	/* Capture Buffer I */
#define FM801_CAP_BUF2		0x1c	/* Capture Buffer II */
#define FM801_CODEC_CTRL	0x22	/* Codec Control */
#define FM801_I2S_MODE		0x24	/* I2S Mode Control */
#define FM801_VOLUME		0x26	/* Volume Up/Down/Mute Status */
#define FM801_I2C_CTRL		0x29	/* I2C Control */
#define FM801_AC97_CMD		0x2a	/* AC'97 Command */
#define FM801_AC97_DATA		0x2c	/* AC'97 Data */
#define FM801_MPU401_DATA	0x30	/* MPU401 Data */
#define FM801_MPU401_CMD	0x31	/* MPU401 Command */
#define FM801_GPIO_CTRL		0x52	/* General Purpose I/O Control */
#define FM801_GEN_CTRL		0x54	/* General Control */
#define FM801_IRQ_MASK		0x56	/* Interrupt Mask */
#define FM801_IRQ_STATUS	0x5a	/* Interrupt Status */
#define FM801_OPL3_BANK0	0x68	/* OPL3 Status Read / Bank 0 Write */
#define FM801_OPL3_DATA0	0x69	/* OPL3 Data 0 Write */
#define FM801_OPL3_BANK1	0x6a	/* OPL3 Bank 1 Write */
#define FM801_OPL3_DATA1	0x6b	/* OPL3 Bank 1 Write */
#define FM801_POWERDOWN		0x70	/* Blocks Power Down Control */

#define FM801_AC97_ADDR		(0<<10)

/* playback and record control register bits */
#define FM801_BUF1_LAST		(1<<1)
#define FM801_BUF2_LAST		(1<<2)
#define FM801_START		(1<<5)
#define FM801_PAUSE		(1<<6)
#define FM801_IMMED_STOP	(1<<7)
#define FM801_RATE_SHIFT	8
#define FM801_RATE_MASK		(15 << FM801_RATE_SHIFT)
#define FM801_16BIT		(1<<14)
#define FM801_STEREO		(1<<15)

/* IRQ status bits */
#define FM801_IRQ_PLAYBACK	(1<<8)
#define FM801_IRQ_CAPTURE	(1<<9)
#define FM801_IRQ_VOLUME	(1<<14)
#define FM801_IRQ_MPU		(1<<15)
	
/*

 */

typedef struct _snd_fm801 fm801_t;

struct _snd_fm801 {
	int irq;

	unsigned long port;	/* I/O port number */
	struct resource *res_port;

	unsigned short ply_ctrl; /* playback control */
	unsigned short cap_ctrl; /* capture control */

	unsigned long ply_buffer;
	unsigned int ply_buf;
	unsigned int ply_count;
	unsigned int ply_size;
	unsigned int ply_pos;

	unsigned long cap_buffer;
	unsigned int cap_buf;
	unsigned int cap_count;
	unsigned int cap_size;
	unsigned int cap_pos;

	ac97_t *ac97;

	struct pci_dev *pci;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_rawmidi_t *rmidi;
	snd_pcm_substream_t *playback_substream;
	snd_pcm_substream_t *capture_substream;
	unsigned int p_dma_size;
	unsigned int c_dma_size;

	spinlock_t reg_lock;
	snd_info_entry_t *proc_entry;
};

static struct pci_device_id snd_fm801_ids[] __devinitdata = {
	{ 0x1319, 0x0801, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0, },   /* FM801 */
	{ 0, }
};

MODULE_DEVICE_TABLE(pci, snd_fm801_ids);

/*
 *  common I/O routines
 */

static void snd_fm801_codec_write(ac97_t *ac97,
				  unsigned short reg,
				  unsigned short val)
{
	fm801_t *chip = snd_magic_cast(fm801_t, ac97->private_data, return);
	int idx;

	/*
	 *  Wait until the codec interface is not ready..
	 */
	for (idx = 0; idx < 100; idx++) {
		if (!(inw(FM801_REG(chip, AC97_CMD)) & (1<<9)))
			goto ok1;
		udelay(10);
	}
	snd_printk("AC'97 interface is busy (1)\n");
	return;

 ok1:
	/* write data and address */
	outw(val, FM801_REG(chip, AC97_DATA));
	outw(reg | FM801_AC97_ADDR, FM801_REG(chip, AC97_CMD));
	/*
	 *  Wait until the write command is not completed..
         */
	for (idx = 0; idx < 1000; idx++) {
		if (!(inw(FM801_REG(chip, AC97_CMD)) & (1<<9)))
			return;
		udelay(10);
	}
	snd_printk("AC'97 interface is busy (2)\n");
}

static unsigned short snd_fm801_codec_read(ac97_t *ac97, unsigned short reg)
{
	fm801_t *chip = snd_magic_cast(fm801_t, ac97->private_data, return -ENXIO);
	int idx;

	/*
	 *  Wait until the codec interface is not ready..
	 */
	for (idx = 0; idx < 100; idx++) {
		if (!(inw(FM801_REG(chip, AC97_CMD)) & (1<<9)))
			goto ok1;
		udelay(10);
	}
	snd_printk("AC'97 interface is busy (1)\n");
	return 0;

 ok1:
	/* read command */
	outw(reg | FM801_AC97_ADDR | (1<<7), FM801_REG(chip, AC97_CMD));
	for (idx = 0; idx < 100; idx++) {
		if (!(inw(FM801_REG(chip, AC97_CMD)) & (1<<9)))
			goto ok2;
		udelay(10);
	}
	snd_printk("AC'97 interface is busy (2)\n");
	return 0;

 ok2:
	for (idx = 0; idx < 1000; idx++) {
		if (inw(FM801_REG(chip, AC97_CMD)) & (1<<8))
			goto ok3;
		udelay(10);
	}
	snd_printk("AC'97 interface is not valid (2)\n");
	return 0;

 ok3:
	return inw(FM801_REG(chip, AC97_DATA));
}

static unsigned int rates[] = {
  5500,  8000,  9600, 11025,
  16000, 19200, 22050, 32000,
  38400, 44100, 48000
};

#define RATES sizeof(rates) / sizeof(rates[0])

static snd_pcm_hw_constraint_list_t hw_constraints_rates = {
	count: RATES,
	list: rates,
	mask: 0,
};

/*
 *  Sample rate routines
 */

static unsigned short snd_fm801_rate_bits(int rate)
{
	unsigned int idx;

	for (idx = 0; idx < 11; idx++)
		if (rates[idx] == rate)
			return idx;
	snd_BUG();
	return RATES - 1;
}

/*
 *  PCM part
 */

static int snd_fm801_playback_trigger(snd_pcm_substream_t * substream,
				      int cmd)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	int result = 0;

	spin_lock(&chip->reg_lock);
	if (cmd == SNDRV_PCM_TRIGGER_START) {
		chip->ply_ctrl &= ~(FM801_BUF1_LAST |
				     FM801_BUF2_LAST |
				     FM801_PAUSE);
		chip->ply_ctrl |= FM801_START |
				   FM801_IMMED_STOP;
		outw(chip->ply_ctrl, FM801_REG(chip, PLY_CTRL));
	} else if (cmd == SNDRV_PCM_TRIGGER_STOP) {
		chip->ply_ctrl &= ~FM801_START;
		outw(chip->ply_ctrl, FM801_REG(chip, PLY_CTRL));
	} else {
		result = -EINVAL;
	}
	spin_unlock(&chip->reg_lock);
	return result;
}

static int snd_fm801_capture_trigger(snd_pcm_substream_t * substream,
				     int cmd)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	int result = 0;

	spin_lock(&chip->reg_lock);
	if (cmd == SNDRV_PCM_TRIGGER_START) {
		chip->cap_ctrl &= ~(FM801_BUF1_LAST |
				     FM801_BUF2_LAST |
				     FM801_PAUSE);
		chip->cap_ctrl |= FM801_START |
				   FM801_IMMED_STOP;
		outw(chip->cap_ctrl, FM801_REG(chip, CAP_CTRL));
	} else if (cmd == SNDRV_PCM_TRIGGER_STOP) {
		chip->cap_ctrl &= ~FM801_START;
		outw(chip->cap_ctrl, FM801_REG(chip, CAP_CTRL));
	} else {
		result = -EINVAL;
	}
	spin_unlock(&chip->reg_lock);
	return result;
}

static int snd_fm801_hw_params(snd_pcm_substream_t * substream,
			       snd_pcm_hw_params_t * hw_params)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	return snd_pcm_lib_malloc_pci_pages(chip->pci, substream, params_buffer_bytes(hw_params));
}

static int snd_fm801_hw_free(snd_pcm_substream_t * substream)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_lib_free_pci_pages(chip->pci, substream);
	return 0;
}

static int snd_fm801_playback_prepare(snd_pcm_substream_t * substream)
{
	unsigned long flags;
	fm801_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	chip->ply_size = snd_pcm_lib_buffer_bytes(substream);
	chip->ply_count = snd_pcm_lib_period_bytes(substream);
	spin_lock_irqsave(&chip->reg_lock, flags);
	chip->ply_ctrl &= ~(FM801_START | FM801_16BIT |
			     FM801_STEREO | FM801_RATE_MASK);
	if (snd_pcm_format_width(runtime->format) == 16)
		chip->ply_ctrl |= FM801_16BIT;
	if (runtime->channels > 1)
		chip->ply_ctrl |= FM801_STEREO;
	chip->ply_ctrl |= snd_fm801_rate_bits(runtime->rate) << FM801_RATE_SHIFT;
	chip->ply_buf = 0;
	outw(chip->ply_ctrl, FM801_REG(chip, PLY_CTRL));
	outw(chip->ply_count - 1, FM801_REG(chip, PLY_COUNT));
	chip->ply_buffer = runtime->dma_addr;
	chip->ply_pos = 0;
	outl(chip->ply_buffer, FM801_REG(chip, PLY_BUF1));
	outl(chip->ply_buffer + (chip->ply_count % chip->ply_size), FM801_REG(chip, PLY_BUF2));
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return 0;
}

static int snd_fm801_capture_prepare(snd_pcm_substream_t * substream)
{
	unsigned long flags;
	fm801_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	chip->cap_size = snd_pcm_lib_buffer_bytes(substream);
	chip->cap_count = snd_pcm_lib_period_bytes(substream);
	spin_lock_irqsave(&chip->reg_lock, flags);
	chip->cap_ctrl &= ~(FM801_START | FM801_16BIT |
			     FM801_STEREO | FM801_RATE_MASK);
	if (snd_pcm_format_width(runtime->format) == 16)
		chip->cap_ctrl |= FM801_16BIT;
	if (runtime->channels > 1)
		chip->cap_ctrl |= FM801_STEREO;
	chip->cap_ctrl |= snd_fm801_rate_bits(runtime->rate) << FM801_RATE_SHIFT;
	chip->cap_buf = 0;
	outw(chip->cap_ctrl, FM801_REG(chip, CAP_CTRL));
	outw(chip->cap_count - 1, FM801_REG(chip, CAP_COUNT));
	chip->cap_buffer = runtime->dma_addr;
	chip->cap_pos = 0;
	outl(chip->cap_buffer, FM801_REG(chip, CAP_BUF1));
	outl(chip->cap_buffer + (chip->cap_count % chip->cap_size), FM801_REG(chip, CAP_BUF2));
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return 0;
}

static snd_pcm_uframes_t snd_fm801_playback_pointer(snd_pcm_substream_t * substream)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	size_t ptr;

	if (!(chip->ply_ctrl & FM801_START))
		return 0;
	ptr = chip->ply_pos + (chip->ply_count - (inw(FM801_REG(chip, PLY_COUNT)) + 1));
	return bytes_to_frames(substream->runtime, ptr);
}

static snd_pcm_uframes_t snd_fm801_capture_pointer(snd_pcm_substream_t * substream)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	size_t ptr;

	if (!(chip->cap_ctrl & FM801_START))
		return 0;
	ptr = chip->cap_pos + (chip->cap_count - (inw(FM801_REG(chip, CAP_COUNT)) + 1));
	return bytes_to_frames(substream->runtime, ptr);
}

static void snd_fm801_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	fm801_t *chip = snd_magic_cast(fm801_t, dev_id, return);
	unsigned short status;
	unsigned int tmp;

	status = inw(FM801_REG(chip, IRQ_STATUS));
	if ((status & (FM801_IRQ_PLAYBACK|FM801_IRQ_CAPTURE|FM801_IRQ_MPU|FM801_IRQ_VOLUME)) == 0)
		return;
	if (chip->pcm && (status & FM801_IRQ_PLAYBACK)) {
		spin_lock(&chip->reg_lock);
		chip->ply_buf++;
		chip->ply_pos += chip->ply_count;
		chip->ply_pos %= chip->ply_size;
		tmp = chip->ply_pos + chip->ply_count;
		tmp %= chip->ply_size;
		outl(chip->ply_buffer + tmp,
				(chip->ply_buf & 1) ?
					FM801_REG(chip, PLY_BUF1) :
					FM801_REG(chip, PLY_BUF2));
		spin_unlock(&chip->reg_lock);
		snd_pcm_period_elapsed(chip->playback_substream);
	}
	if (chip->pcm && (status & FM801_IRQ_CAPTURE)) {
		spin_lock(&chip->reg_lock);
		chip->cap_buf++;
		chip->cap_pos += chip->cap_count;
		chip->cap_pos %= chip->cap_size;
		tmp = chip->cap_pos + chip->cap_count;
		tmp %= chip->cap_size;
		outl(chip->cap_buffer + tmp,
				(chip->cap_buf & 1) ?
					FM801_REG(chip, CAP_BUF1) :
					FM801_REG(chip, CAP_BUF2));
		spin_unlock(&chip->reg_lock);
		snd_pcm_period_elapsed(chip->capture_substream);
	}
	if ((status & FM801_IRQ_MPU) && chip->rmidi != NULL)
		snd_mpu401_uart_interrupt(irq, chip->rmidi->private_data, regs);
	if (status & FM801_IRQ_VOLUME) {
		/* TODO */
	}
	/* clear interrupt */
	outw((status & (FM801_IRQ_PLAYBACK|FM801_IRQ_CAPTURE|FM801_IRQ_MPU|FM801_IRQ_VOLUME)), FM801_REG(chip, IRQ_STATUS));
}

static snd_pcm_hardware_t snd_fm801_playback =
{
	info:			(SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_BLOCK_TRANSFER |
				 SNDRV_PCM_INFO_MMAP_VALID),
	formats:		SNDRV_PCM_FMTBIT_U8 | SNDRV_PCM_FMTBIT_S16_LE,
	rates:			SNDRV_PCM_RATE_KNOT | SNDRV_PCM_RATE_8000_48000,
	rate_min:		5500,
	rate_max:		48000,
	channels_min:		1,
	channels_max:		2,
	buffer_bytes_max:	(128*1024),
	period_bytes_min:	64,
	period_bytes_max:	(128*1024),
	periods_min:		1,
	periods_max:		1024,
	fifo_size:		0,
};

static snd_pcm_hardware_t snd_fm801_capture =
{
	info:			(SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
				 SNDRV_PCM_INFO_BLOCK_TRANSFER |
				 SNDRV_PCM_INFO_MMAP_VALID),
	formats:		SNDRV_PCM_FMTBIT_U8 | SNDRV_PCM_FMTBIT_S16_LE,
	rates:			SNDRV_PCM_RATE_KNOT | SNDRV_PCM_RATE_8000_48000,
	rate_min:		5500,
	rate_max:		48000,
	channels_min:		1,
	channels_max:		2,
	buffer_bytes_max:	(128*1024),
	period_bytes_min:	64,
	period_bytes_max:	(128*1024),
	periods_min:		1,
	periods_max:		1024,
	fifo_size:		0,
};

static int snd_fm801_playback_open(snd_pcm_substream_t * substream)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	chip->playback_substream = substream;
	runtime->hw = snd_fm801_playback;
	snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE, &hw_constraints_rates);
	return 0;
}

static int snd_fm801_capture_open(snd_pcm_substream_t * substream)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	chip->capture_substream = substream;
	runtime->hw = snd_fm801_capture;
	snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE, &hw_constraints_rates);
	return 0;
}

static int snd_fm801_playback_close(snd_pcm_substream_t * substream)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);

	chip->playback_substream = NULL;
	return 0;
}

static int snd_fm801_capture_close(snd_pcm_substream_t * substream)
{
	fm801_t *chip = snd_pcm_substream_chip(substream);

	chip->capture_substream = NULL;
	return 0;
}

static snd_pcm_ops_t snd_fm801_playback_ops = {
	open:		snd_fm801_playback_open,
	close:		snd_fm801_playback_close,
	ioctl:		snd_pcm_lib_ioctl,
	hw_params:	snd_fm801_hw_params,
	hw_free:	snd_fm801_hw_free,
	prepare:	snd_fm801_playback_prepare,
	trigger:	snd_fm801_playback_trigger,
	pointer:	snd_fm801_playback_pointer,
};

static snd_pcm_ops_t snd_fm801_capture_ops = {
	open:		snd_fm801_capture_open,
	close:		snd_fm801_capture_close,
	ioctl:		snd_pcm_lib_ioctl,
	hw_params:	snd_fm801_hw_params,
	hw_free:	snd_fm801_hw_free,
	prepare:	snd_fm801_capture_prepare,
	trigger:	snd_fm801_capture_trigger,
	pointer:	snd_fm801_capture_pointer,
};

static void snd_fm801_pcm_free(snd_pcm_t *pcm)
{
	fm801_t *chip = snd_magic_cast(fm801_t, pcm->private_data, return);
	chip->pcm = NULL;
	snd_pcm_lib_preallocate_pci_free_for_all(chip->pci, pcm);
}

static int __init snd_fm801_pcm(fm801_t *chip, int device, snd_pcm_t ** rpcm)
{
	snd_pcm_t *pcm;
	int err;

	if (rpcm)
		*rpcm = NULL;
	if ((err = snd_pcm_new(chip->card, "FM801", device, 1, 1, &pcm)) < 0)
		return err;

	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK, &snd_fm801_playback_ops);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_fm801_capture_ops);

	pcm->private_data = chip;
	pcm->private_free = snd_fm801_pcm_free;
	pcm->info_flags = 0;
	strcpy(pcm->name, "FM801");
	chip->pcm = pcm;

	snd_pcm_lib_preallocate_pci_pages_for_all(chip->pci, pcm, 64*1024, 128*1024);

	if (rpcm)
		*rpcm = pcm;
	return 0;
}

/*
 *  Mixer routines
 */

static void snd_fm801_mixer_free_ac97(ac97_t *ac97)
{
	fm801_t *chip = snd_magic_cast(fm801_t, ac97->private_data, return);
	chip->ac97 = NULL;
}

static int __init snd_fm801_mixer(fm801_t *chip)
{
	ac97_t ac97;
	int err;

	memset(&ac97, 0, sizeof(ac97));
	ac97.write = snd_fm801_codec_write;
	ac97.read = snd_fm801_codec_read;
	ac97.private_data = chip;
	ac97.private_free = snd_fm801_mixer_free_ac97;
	if ((err = snd_ac97_mixer(chip->card, &ac97, &chip->ac97)) < 0)
		return err;
	return 0;
}

/*
 *  initialization routines
 */

static int snd_fm801_free(fm801_t *chip)
{
	unsigned short cmdw;

	if (chip->irq < 0)
		goto __end_hw;

	/* interrupt setup - mask everything */
	cmdw = inw(FM801_REG(chip, IRQ_MASK));
	cmdw |= 0x00c3;
	outw(cmdw, FM801_REG(chip, IRQ_MASK));

      __end_hw:
	if (chip->res_port)
		release_resource(chip->res_port);
	if (chip->irq >= 0)
		free_irq(chip->irq, (void *)chip);

	snd_magic_kfree(chip);
	return 0;
}

static int snd_fm801_dev_free(snd_device_t *device)
{
	fm801_t *chip = snd_magic_cast(fm801_t, device->device_data, return -ENXIO);
	return snd_fm801_free(chip);
}

static int __init snd_fm801_create(snd_card_t * card,
				   struct pci_dev * pci,
				   fm801_t ** rchip)
{
	fm801_t *chip;
	unsigned short cmdw;
	int err;
	static snd_device_ops_t ops = {
		dev_free:	snd_fm801_dev_free,
	};
	
	*rchip = NULL;
	if ((err = pci_enable_device(pci)) < 0)
		return err;
	chip = snd_magic_kcalloc(fm801_t, 0, GFP_KERNEL);
	if (chip == NULL)
		return -ENOMEM;
	spin_lock_init(&chip->reg_lock);
	chip->card = card;
	chip->pci = pci;
	chip->irq = -1;
	chip->port = pci_resource_start(pci, 0);
	if ((chip->res_port = request_region(chip->port, 0x80, "FM801")) == NULL) {
		snd_fm801_free(chip);
		snd_printk("unable to grab region 0x%lx-0x%lx\n", chip->port, chip->port + 0x80 - 1);
		return -EBUSY;
	}
	if (request_irq(pci->irq, snd_fm801_interrupt, SA_INTERRUPT|SA_SHIRQ, "FM801", (void *)chip)) {
		snd_fm801_free(chip);
		snd_printk("unable to grab IRQ %d\n", chip->irq);
		return -EBUSY;
	}
	chip->irq = pci->irq;
	pci_set_master(pci);

	/* codec cold reset + AC'97 warm reset */
	outw((1<<5)|(1<<6), FM801_REG(chip, CODEC_CTRL));
	udelay(100);
	outw(0, FM801_REG(chip, CODEC_CTRL));
	mdelay(1);

	/* init volume */
	outw(0x0808, FM801_REG(chip, PCM_VOL));
	outw(0x0808, FM801_REG(chip, FM_VOL));
	outw(0x0808, FM801_REG(chip, I2S_VOL));

	/* interrupt setup - unmask MPU, PLAYBACK & CAPTURE */
	cmdw = inw(FM801_REG(chip, IRQ_MASK));
	cmdw &= ~0x0083;
	outw(cmdw, FM801_REG(chip, IRQ_MASK));

	/* interrupt clear */
	outw(FM801_IRQ_PLAYBACK|FM801_IRQ_CAPTURE|FM801_IRQ_MPU, FM801_REG(chip, IRQ_STATUS));

	if ((err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, chip, &ops)) < 0) {
		snd_fm801_free(chip);
		return err;
	}

	*rchip = chip;
	return 0;
}

static int __init snd_card_fm801_probe(struct pci_dev *pci,
				       const struct pci_device_id *id)
{
	static int dev = 0;
	snd_card_t *card;
	fm801_t *chip;
	opl3_t *opl3;
	int err;

	for ( ; dev < SNDRV_CARDS; dev++) {
		if (!snd_enable[dev]) {
			dev++;
			return -ENOENT;
		}
                break;
        }
        if (dev >= SNDRV_CARDS)
                return -ENODEV;

	card = snd_card_new(snd_index[dev], snd_id[dev], THIS_MODULE, 0);
	if (card == NULL)
		return -ENOMEM;
	card->type = SNDRV_CARD_TYPE_FM801;
	if ((err = snd_fm801_create(card, pci, &chip)) < 0) {
		snd_card_free(card);
		return err;
	}
	if ((err = snd_fm801_pcm(chip, 0, NULL)) < 0) {
		snd_card_free(card);
		return err;
	}
	if ((err = snd_fm801_mixer(chip)) < 0) {
		snd_card_free(card);
		return err;
	}
	if ((err = snd_mpu401_uart_new(card, 0, MPU401_HW_FM801,
				       FM801_REG(chip, MPU401_DATA), 1,
				       chip->irq, 0, &chip->rmidi)) < 0) {
		snd_card_free(card);
		return err;
	}
	if ((err = snd_opl3_create(card, FM801_REG(chip, OPL3_BANK0),
				   FM801_REG(chip, OPL3_BANK1),
				   OPL3_HW_OPL3_FM801, 1, &opl3)) < 0) {
		snd_card_free(card);
		return err;
	}
	if ((err = snd_opl3_hwdep_new(opl3, 0, 1, NULL)) < 0) {
		snd_card_free(card);
		return err;
	}
 
	strcpy(card->abbreviation, "FM801");
	strcpy(card->shortname, "ForteMedia FM801");
	sprintf(card->longname, "%s at 0x%lx, irq %i",
		card->shortname, chip->port, chip->irq);

	if ((err = snd_card_register(card)) < 0) {
		snd_card_free(card);
		return err;
	}
	PCI_SET_DRIVER_DATA(pci, card);
	dev++;
	return 0;
}

static void __exit snd_card_fm801_remove(struct pci_dev *pci)
{
	snd_card_free(PCI_GET_DRIVER_DATA(pci));
	PCI_SET_DRIVER_DATA(pci, NULL);
}

static struct pci_driver driver = {
	name: "FM801",
	id_table: snd_fm801_ids,
	probe: snd_card_fm801_probe,
	remove: snd_card_fm801_remove,
};
                                                                
static int __init alsa_card_fm801_init(void)
{
	int err;

	if ((err = pci_module_init(&driver)) < 0) {
#ifdef MODULE
		snd_printk("ForteMedia FM801 soundcard not found or device busy\n");
#endif
		return err;
	}
	return 0;
}

static void __exit alsa_card_fm801_exit(void)
{
	pci_unregister_driver(&driver);
}

module_init(alsa_card_fm801_init)
module_exit(alsa_card_fm801_exit)

#ifndef MODULE

/* format is: snd-card-fm801=snd_enable,snd_index,snd_id */

static int __init alsa_card_fm801_setup(char *str)
{
	static unsigned __initdata nr_dev = 0;

	if (nr_dev >= SNDRV_CARDS)
		return 0;
	(void)(get_option(&str,&snd_enable[nr_dev]) == 2 &&
	       get_option(&str,&snd_index[nr_dev]) == 2 &&
	       get_id(&str,&snd_id[nr_dev]) == 2);
	nr_dev++;
	return 1;
}

__setup("snd-card-fm801=", alsa_card_fm801_setup);

#endif /* ifndef MODULE */
