/*
 *   Generic i2c interface for linux
 *
 *   (c) 1998 Gerd Knorr <kraxel@cs.tu-berlin.de>
 *   Modified for the ALSA driver by Jaroslav Kysela <perex@suse.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SNDRV_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/i2c.h"

static int snd_i2c_bus_free(snd_i2c_bus_t *bus)
{
	snd_assert(bus != NULL, return -EINVAL);
	if (bus->private_free)
		bus->private_free(bus);
	snd_magic_kfree(bus);
	return 0;
}

static int snd_i2c_bus_dev_free(snd_device_t *device)
{
	snd_i2c_bus_t *bus = snd_magic_cast(snd_i2c_bus_t, device->device_data, return -ENXIO);
	return snd_i2c_bus_free(bus);
}

int snd_i2c_bus_create(snd_card_t *card, char *name, snd_i2c_bus_t **ri2c)
{
	struct snd_i2c_bus *bus;
	int err;
	static snd_device_ops_t ops = {
		dev_free:       snd_i2c_bus_dev_free,
	};

	*ri2c = NULL;
	bus = (snd_i2c_bus_t *)snd_magic_kcalloc(snd_i2c_bus_t, 0, GFP_KERNEL);
	if (bus == NULL)
		return -ENOMEM;
	spin_lock_init(&bus->lock);
	bus->card = card;
	strncpy(bus->name, name, sizeof(bus->name) - 1);
	if ((err = snd_device_new(card, SNDRV_DEV_LOWLEVEL, bus, &ops)) < 0) {
		snd_i2c_bus_free(bus);
		return err;
	}
	*ri2c = bus;
	return 0;
}

#define SNDRV_I2C_SET(bus,ctrl,data)  (bus->i2c_setlines(bus,ctrl,data))
#define SNDRV_I2C_GET(bus)            (bus->i2c_getdataline(bus))

void snd_i2c_reset(snd_i2c_bus_t *bus)
{
	SNDRV_I2C_SET(bus, 1, 1);
}

void snd_i2c_start(snd_i2c_bus_t *bus)
{
	SNDRV_I2C_SET(bus, 0, 1);
	SNDRV_I2C_SET(bus, 1, 1);
	SNDRV_I2C_SET(bus, 1, 0);
	SNDRV_I2C_SET(bus, 0, 0);
}

void snd_i2c_stop(snd_i2c_bus_t *bus)
{
	SNDRV_I2C_SET(bus, 0, 0);
	SNDRV_I2C_SET(bus, 1, 0);
	SNDRV_I2C_SET(bus, 1, 1);
}

void snd_i2c_one(snd_i2c_bus_t *bus)
{
	SNDRV_I2C_SET(bus, 0, 1);
	SNDRV_I2C_SET(bus, 1, 1);
	SNDRV_I2C_SET(bus, 0, 1);
}

void snd_i2c_zero(snd_i2c_bus_t *bus)
{
	SNDRV_I2C_SET(bus, 0, 0);
	SNDRV_I2C_SET(bus, 1, 0);
	SNDRV_I2C_SET(bus, 0, 0);
}

int snd_i2c_ack(snd_i2c_bus_t *bus)
{
	int ack;

	SNDRV_I2C_SET(bus, 0, 1);
	SNDRV_I2C_SET(bus, 1, 1);
	ack = SNDRV_I2C_GET(bus);
	SNDRV_I2C_SET(bus, 0, 1);
	return ack;
}

int snd_i2c_sendbyte(snd_i2c_bus_t *bus, unsigned char data, int wait_for_ack)
{
	int i, ack;

	SNDRV_I2C_SET(bus, 0, 0);
	for (i = 7; i >= 0; i--)
		(data & (1 << i)) ? snd_i2c_one(bus) : snd_i2c_zero(bus);
	if (wait_for_ack)
		udelay(wait_for_ack + 9);
	ack = snd_i2c_ack(bus);
	return ack;
}

unsigned char snd_i2c_readbyte(snd_i2c_bus_t *bus, int last)
{
	int i;
	unsigned char data = 0;

	SNDRV_I2C_SET(bus, 0, 1);
	for (i = 7; i >= 0; i--) {
		SNDRV_I2C_SET(bus, 1, 1);
		if (SNDRV_I2C_GET(bus))
			data |= (1 << i);
		SNDRV_I2C_SET(bus, 0, 1);
	}
	last ? snd_i2c_one(bus) : snd_i2c_zero(bus);
	return data;
}

int snd_i2c_read(snd_i2c_bus_t *bus, unsigned char addr)
{
	int ret;

	if (bus->i2c_read)
		return bus->i2c_read(bus, addr);

	snd_i2c_start(bus);
	snd_i2c_sendbyte(bus, addr, 0);
	ret = snd_i2c_readbyte(bus, 1);
	snd_i2c_stop(bus);
	return ret;
}

int snd_i2c_write(snd_i2c_bus_t *bus, unsigned char addr,
		  unsigned char data1, unsigned char data2, int both)
{
	int ack;

	if (bus->i2c_write)
		return bus->i2c_write(bus, addr, data1, data2, both);

	snd_i2c_start(bus);
	snd_i2c_sendbyte(bus, addr, 0);
	ack = snd_i2c_sendbyte(bus, data1, 0);
	if (both)
		ack = snd_i2c_sendbyte(bus, data2, 0);
	snd_i2c_stop(bus);
	return ack ? -1 : 0;
}

EXPORT_SYMBOL(snd_i2c_bus_create);
EXPORT_SYMBOL(snd_i2c_reset);
EXPORT_SYMBOL(snd_i2c_start);
EXPORT_SYMBOL(snd_i2c_stop);
EXPORT_SYMBOL(snd_i2c_one);
EXPORT_SYMBOL(snd_i2c_zero);
EXPORT_SYMBOL(snd_i2c_ack);
EXPORT_SYMBOL(snd_i2c_sendbyte);
EXPORT_SYMBOL(snd_i2c_readbyte);
EXPORT_SYMBOL(snd_i2c_read);
EXPORT_SYMBOL(snd_i2c_write);

static int __init alsa_i2c_init(void)
{
	return 0;
}

static void __exit alsa_i2c_exit(void)
{
}

module_init(alsa_i2c_init)
module_exit(alsa_i2c_exit)
