/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>,
 *                   Takashi Iwai <tiwai@suse.de>
 *                   Creative Labs, Inc.
 *  Routines for control of EMU10K1 chips / mixer routines
 *
 *  BUGS:
 *    --
 *
 *  TODO:
 *    --
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "../../include/emu10k1.h"

#define chip_t emu10k1_t

static const u32 bass_table[41][5] = {
	{ 0x3e4f844f, 0x84ed4cc3, 0x3cc69927, 0x7b03553a, 0xc4da8486 },
	{ 0x3e69a17a, 0x84c280fb, 0x3cd77cd4, 0x7b2f2a6f, 0xc4b08d1d },
	{ 0x3e82ff42, 0x849991d5, 0x3ce7466b, 0x7b5917c6, 0xc48863ee },
	{ 0x3e9bab3c, 0x847267f0, 0x3cf5ffe8, 0x7b813560, 0xc461f22c },
	{ 0x3eb3b275, 0x844ced29, 0x3d03b295, 0x7ba79a1c, 0xc43d223b },
	{ 0x3ecb2174, 0x84290c8b, 0x3d106714, 0x7bcc5ba3, 0xc419dfa5 },
	{ 0x3ee2044b, 0x8406b244, 0x3d1c2561, 0x7bef8e77, 0xc3f8170f },
	{ 0x3ef86698, 0x83e5cb96, 0x3d26f4d8, 0x7c114600, 0xc3d7b625 },
	{ 0x3f0e5390, 0x83c646c9, 0x3d30dc39, 0x7c319498, 0xc3b8ab97 },
	{ 0x3f23d60b, 0x83a81321, 0x3d39e1af, 0x7c508b9c, 0xc39ae704 },
	{ 0x3f38f884, 0x838b20d2, 0x3d420ad2, 0x7c6e3b75, 0xc37e58f1 },
	{ 0x3f4dc52c, 0x836f60ef, 0x3d495cab, 0x7c8ab3a6, 0xc362f2be },
	{ 0x3f6245e8, 0x8354c565, 0x3d4fdbb8, 0x7ca602d6, 0xc348a69b },
	{ 0x3f76845f, 0x833b40ec, 0x3d558bf0, 0x7cc036df, 0xc32f677c },
	{ 0x3f8a8a03, 0x8322c6fb, 0x3d5a70c4, 0x7cd95cd7, 0xc317290b },
	{ 0x3f9e6014, 0x830b4bc3, 0x3d5e8d25, 0x7cf1811a, 0xc2ffdfa5 },
	{ 0x3fb20fae, 0x82f4c420, 0x3d61e37f, 0x7d08af56, 0xc2e9804a },
	{ 0x3fc5a1cc, 0x82df2592, 0x3d6475c3, 0x7d1ef294, 0xc2d40096 },
	{ 0x3fd91f55, 0x82ca6632, 0x3d664564, 0x7d345541, 0xc2bf56b9 },
	{ 0x3fec9120, 0x82b67cac, 0x3d675356, 0x7d48e138, 0xc2ab796e },
	{ 0x401374c7, 0x8291088a, 0x3d672b93, 0x7d6f99c3, 0xc28601f2 },
	{ 0x4026f857, 0x827f6dd7, 0x3d65f559, 0x7d81d77c, 0xc27457a3 },
	{ 0x403a939f, 0x826e88c5, 0x3d63fc63, 0x7d9360d4, 0xc2635996 },
	{ 0x404e4faf, 0x825e5266, 0x3d613f32, 0x7da43d42, 0xc25300c6 },
	{ 0x406235ba, 0x824ec434, 0x3d5dbbc3, 0x7db473d7, 0xc243468e },
	{ 0x40764f1f, 0x823fd80c, 0x3d596f8f, 0x7dc40b44, 0xc23424a2 },
	{ 0x408aa576, 0x82318824, 0x3d545787, 0x7dd309e2, 0xc2259509 },
	{ 0x409f4296, 0x8223cf0b, 0x3d4e7012, 0x7de175b5, 0xc2179218 },
	{ 0x40b430a0, 0x8216a7a1, 0x3d47b505, 0x7def5475, 0xc20a1670 },
	{ 0x40c97a0a, 0x820a0d12, 0x3d4021a1, 0x7dfcab8d, 0xc1fd1cf5 },
	{ 0x40df29a6, 0x81fdfad6, 0x3d37b08d, 0x7e098028, 0xc1f0a0ca },
	{ 0x40f54ab1, 0x81f26ca9, 0x3d2e5bd1, 0x7e15d72b, 0xc1e49d52 },
	{ 0x410be8da, 0x81e75e89, 0x3d241cce, 0x7e21b544, 0xc1d90e24 },
	{ 0x41231051, 0x81dcccb3, 0x3d18ec37, 0x7e2d1ee6, 0xc1cdef10 },
	{ 0x413acdd0, 0x81d2b39e, 0x3d0cc20a, 0x7e38184e, 0xc1c33c13 },
	{ 0x41532ea7, 0x81c90ffb, 0x3cff9585, 0x7e42a58b, 0xc1b8f15a },
	{ 0x416c40cd, 0x81bfdeb2, 0x3cf15d21, 0x7e4cca7c, 0xc1af0b3f },
	{ 0x418612ea, 0x81b71cdc, 0x3ce20e85, 0x7e568ad3, 0xc1a58640 },
	{ 0x41a0b465, 0x81aec7c5, 0x3cd19e7c, 0x7e5fea1e, 0xc19c5f03 },
	{ 0x41bc3573, 0x81a6dcea, 0x3cc000e9, 0x7e68ebc2, 0xc1939250 }
};

static const u32 treble_table[41][5] = {
	{ 0x0125cba9, 0xfed5debd, 0x00599b6c, 0x0d2506da, 0xfa85b354 },
	{ 0x0142f67e, 0xfeb03163, 0x0066cd0f, 0x0d14c69d, 0xfa914473 },
	{ 0x016328bd, 0xfe860158, 0x0075b7f2, 0x0d03eb27, 0xfa9d32d2 },
	{ 0x0186b438, 0xfe56c982, 0x00869234, 0x0cf27048, 0xfaa97fca },
	{ 0x01adf358, 0xfe21f5fe, 0x00999842, 0x0ce051c2, 0xfab62ca5 },
	{ 0x01d949fa, 0xfde6e287, 0x00af0d8d, 0x0ccd8b4a, 0xfac33aa7 },
	{ 0x02092669, 0xfda4d8bf, 0x00c73d4c, 0x0cba1884, 0xfad0ab07 },
	{ 0x023e0268, 0xfd5b0e4a, 0x00e27b54, 0x0ca5f509, 0xfade7ef2 },
	{ 0x0278645c, 0xfd08a2b0, 0x01012509, 0x0c911c63, 0xfaecb788 },
	{ 0x02b8e091, 0xfcac9d1a, 0x0123a262, 0x0c7b8a14, 0xfafb55df },
	{ 0x03001a9a, 0xfc45e9ce, 0x014a6709, 0x0c65398f, 0xfb0a5aff },
	{ 0x034ec6d7, 0xfbd3576b, 0x0175f397, 0x0c4e2643, 0xfb19c7e4 },
	{ 0x03a5ac15, 0xfb5393ee, 0x01a6d6ed, 0x0c364b94, 0xfb299d7c },
	{ 0x0405a562, 0xfac52968, 0x01ddafae, 0x0c1da4e2, 0xfb39dca5 },
	{ 0x046fa3fe, 0xfa267a66, 0x021b2ddd, 0x0c042d8d, 0xfb4a8631 },
	{ 0x04e4b17f, 0xf975be0f, 0x0260149f, 0x0be9e0f2, 0xfb5b9ae0 },
	{ 0x0565f220, 0xf8b0fbe5, 0x02ad3c29, 0x0bceba73, 0xfb6d1b60 },
	{ 0x05f4a745, 0xf7d60722, 0x030393d4, 0x0bb2b578, 0xfb7f084d },
	{ 0x06923236, 0xf6e279bd, 0x03642465, 0x0b95cd75, 0xfb916233 },
	{ 0x07401713, 0xf5d3aef9, 0x03d01283, 0x0b77fded, 0xfba42984 },
	{ 0x08000000, 0xf4a6bd88, 0x0448a161, 0x0b594278, 0xfbb75e9f },
	{ 0x08d3c097, 0xf3587131, 0x04cf35a4, 0x0b3996c9, 0xfbcb01cb },
	{ 0x09bd59a2, 0xf1e543f9, 0x05655880, 0x0b18f6b2, 0xfbdf1333 },
	{ 0x0abefd0f, 0xf04956ca, 0x060cbb12, 0x0af75e2c, 0xfbf392e8 },
	{ 0x0bdb123e, 0xee806984, 0x06c739fe, 0x0ad4c962, 0xfc0880dd },
	{ 0x0d143a94, 0xec85d287, 0x0796e150, 0x0ab134b0, 0xfc1ddce5 },
	{ 0x0e6d5664, 0xea547598, 0x087df0a0, 0x0a8c9cb6, 0xfc33a6ad },
	{ 0x0fe98a2a, 0xe7e6ba35, 0x097edf83, 0x0a66fe5b, 0xfc49ddc2 },
	{ 0x118c4421, 0xe536813a, 0x0a9c6248, 0x0a4056d7, 0xfc608185 },
	{ 0x1359422e, 0xe23d19eb, 0x0bd96efb, 0x0a18a3bf, 0xfc77912c },
	{ 0x1554982b, 0xdef33645, 0x0d3942bd, 0x09efe312, 0xfc8f0bc1 },
	{ 0x1782b68a, 0xdb50deb1, 0x0ebf676d, 0x09c6133f, 0xfca6f019 },
	{ 0x19e8715d, 0xd74d64fd, 0x106fb999, 0x099b3337, 0xfcbf3cd6 },
	{ 0x1c8b07b8, 0xd2df56ab, 0x124e6ec8, 0x096f4274, 0xfcd7f060 },
	{ 0x1f702b6d, 0xcdfc6e92, 0x14601c10, 0x0942410b, 0xfcf108e5 },
	{ 0x229e0933, 0xc89985cd, 0x16a9bcfa, 0x09142fb5, 0xfd0a8451 },
	{ 0x261b5118, 0xc2aa8409, 0x1930bab6, 0x08e50fdc, 0xfd24604d },
	{ 0x29ef3f5d, 0xbc224f28, 0x1bfaf396, 0x08b4e3aa, 0xfd3e9a3b },
	{ 0x2e21a59b, 0xb4f2ba46, 0x1f0ec2d6, 0x0883ae15, 0xfd592f33 },
	{ 0x32baf44b, 0xad0c7429, 0x227308a3, 0x085172eb, 0xfd741bfd },
	{ 0x37c4448b, 0xa45ef51d, 0x262f3267, 0x081e36dc, 0xfd8f5d14 }
};

static const u32 db_table[101] = {
	0x00000000, 0x01571f82, 0x01674b41, 0x01783a1b, 0x0189f540,
	0x019c8651, 0x01aff763, 0x01c45306, 0x01d9a446, 0x01eff6b8,
	0x0207567a, 0x021fd03d, 0x0239714c, 0x02544792, 0x027061a1,
	0x028dcebb, 0x02ac9edc, 0x02cce2bf, 0x02eeabe8, 0x03120cb0,
	0x0337184e, 0x035de2df, 0x03868173, 0x03b10a18, 0x03dd93e9,
	0x040c3713, 0x043d0cea, 0x04702ff3, 0x04a5bbf2, 0x04ddcdfb,
	0x0518847f, 0x0555ff62, 0x05966005, 0x05d9c95d, 0x06206005,
	0x066a4a52, 0x06b7b067, 0x0708bc4c, 0x075d9a01, 0x07b6779d,
	0x08138561, 0x0874f5d5, 0x08dafde1, 0x0945d4ed, 0x09b5b4fd,
	0x0a2adad1, 0x0aa58605, 0x0b25f936, 0x0bac7a24, 0x0c3951d8,
	0x0ccccccc, 0x0d673b17, 0x0e08f093, 0x0eb24510, 0x0f639481,
	0x101d3f2d, 0x10dfa9e6, 0x11ab3e3f, 0x12806ac3, 0x135fa333,
	0x144960c5, 0x153e2266, 0x163e6cfe, 0x174acbb7, 0x1863d04d,
	0x198a1357, 0x1abe349f, 0x1c00db77, 0x1d52b712, 0x1eb47ee6,
	0x2026f30f, 0x21aadcb6, 0x23410e7e, 0x24ea64f9, 0x26a7c71d,
	0x287a26c4, 0x2a62812c, 0x2c61df84, 0x2e795779, 0x30aa0bcf,
	0x32f52cfe, 0x355bf9d8, 0x37dfc033, 0x3a81dda4, 0x3d43c038,
	0x4026e73c, 0x432ce40f, 0x46575af8, 0x49a8040f, 0x4d20ac2a,
	0x50c335d3, 0x54919a57, 0x588dead1, 0x5cba514a, 0x611911ea,
	0x65ac8c2f, 0x6a773c39, 0x6f7bbc23, 0x74bcc56c, 0x7a3d3272,
	0x7fffffff,
};

static int snd_emu10k1_mixer_digital_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 18;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 100;
	return 0;
}

static int snd_emu10k1_mixer_digital_get(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int i, j;
	
	spin_lock_irqsave(&emu->reg_lock, flags);
	for (i = 0; i < 9; i++)
		for (j = 0; j < 2; j++)
			ucontrol->value.integer.value[i * 2 + j] = emu->dig_vol[kcontrol->private_value % 6][i][j];
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	return 0;
}

static int snd_emu10k1_mixer_digital_put(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int nval, idx, i, j;
	int change = 0;
	u32 val;
	
	idx = kcontrol->private_value % 6;
	spin_lock_irqsave(&emu->reg_lock, flags);
	for (i = 0; i < 9; i++) {
		for (j = 0; j < 2; j++) {
			nval = ucontrol->value.integer.value[i * 2 + j];
			if (nval < 0)
				nval = 0;
			if (nval > 100)
				nval = 100;
			if (nval != emu->dig_vol[idx][i][j])
				change = 1;
			val = emu->dig_vol[idx][i][j] = nval;
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x14 + idx * 18 + i * 2 + j, 0, db_table[val]);
		}
	}
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	return change;
}

static snd_kcontrol_new_t snd_emu10k1_mixer_digital[6] = {
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "EMU10K1 Digital Volume - AC97 PCM",
	info: snd_emu10k1_mixer_digital_info,
	get: snd_emu10k1_mixer_digital_get,
	put: snd_emu10k1_mixer_digital_put,
	private_value: 0
},
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "EMU10K1 Digital Volume - TOSLink",
	info: snd_emu10k1_mixer_digital_info,
	get: snd_emu10k1_mixer_digital_get,
	put: snd_emu10k1_mixer_digital_put,
	private_value: 1
},
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "EMU10K1 Digital Volume - Headphone",
	info: snd_emu10k1_mixer_digital_info,
	get: snd_emu10k1_mixer_digital_get,
	put: snd_emu10k1_mixer_digital_put,
	private_value: 2
},
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "EMU10K1 Digital Volume - Rear",
	info: snd_emu10k1_mixer_digital_info,
	get: snd_emu10k1_mixer_digital_get,
	put: snd_emu10k1_mixer_digital_put,
	private_value: 3
},
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "EMU10K1 Digital Volume - ADC Capture",
	info: snd_emu10k1_mixer_digital_info,
	get: snd_emu10k1_mixer_digital_get,
	put: snd_emu10k1_mixer_digital_put,
	private_value: 4
},
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "EMU10K1 Digital Volume - MIC Capture",
	info: snd_emu10k1_mixer_digital_info,
	get: snd_emu10k1_mixer_digital_get,
	put: snd_emu10k1_mixer_digital_put,
	private_value: 5
}
};

static int snd_emu10k1_mixer_stereo_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 100;
	return 0;
}

static int snd_emu10k1_mixer_stereo_get(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	
	spin_lock_irqsave(&emu->reg_lock, flags);
	ucontrol->value.integer.value[0] = emu->stereo_vol[kcontrol->private_value][0];
	ucontrol->value.integer.value[1] = emu->stereo_vol[kcontrol->private_value][1];
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	return 0;
}

static int snd_emu10k1_mixer_stereo_put(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int nval, idx, j;
	int change = 0;
	u32 val;
	
	idx = kcontrol->private_value;
	spin_lock_irqsave(&emu->reg_lock, flags);
	for (j = 0; j < 2; j++) {
		nval = ucontrol->value.integer.value[j];
		if (nval < 0)
			nval = 0;
		if (nval > 100)
			nval = 100;
		if (nval != emu->stereo_vol[idx][j])
			change = 1;
		val = emu->stereo_vol[idx][j] = nval;
		snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x10 + idx * 2 + j, 0, db_table[val]);
	}
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	return change;
}

static snd_kcontrol_new_t snd_emu10k1_mixer_stereo[2] = {
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "Wave Playback Volume",
	info: snd_emu10k1_mixer_stereo_info,
	get: snd_emu10k1_mixer_stereo_get,
	put: snd_emu10k1_mixer_stereo_put,
	private_value: 0
},
{
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "Music Playback Volume",
	info: snd_emu10k1_mixer_stereo_info,
	get: snd_emu10k1_mixer_stereo_get,
	put: snd_emu10k1_mixer_stereo_put,
	private_value: 1
}
};

static int snd_emu10k1_mixer_tone_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 2;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 40;
	return 0;
}

static int snd_emu10k1_mixer_tone_get(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	
	spin_lock_irqsave(&emu->reg_lock, flags);
	ucontrol->value.integer.value[0] = emu->tone_control[kcontrol->private_value & 1][0];
	ucontrol->value.integer.value[1] = emu->tone_control[kcontrol->private_value & 1][1];
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	return 0;
}

static int snd_emu10k1_mixer_tone_put(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int nval1, nval2, idx, i;
	int change = 0;
	
	nval1 = ucontrol->value.integer.value[0] % 41;
	nval2 = ucontrol->value.integer.value[1] % 41;
	idx = kcontrol->private_value & 1;
	spin_lock_irqsave(&emu->reg_lock, flags);
	change = nval1 != emu->tone_control[idx][0] || nval2 != emu->tone_control[idx][1];
	emu->tone_control[idx][0] = nval1;
	emu->tone_control[idx][1] = nval2;
	if (emu->tone_control_active) {
		for (i = 0; i < 5; i++) {
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x80 + idx * 0x10 + i * 2 + 0, 0, idx == 0 ? bass_table[nval1][i] : treble_table[nval1][i]);
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x80 + idx * 0x10 + i * 2 + 1, 0, idx == 0 ? bass_table[nval2][i] : treble_table[nval2][i]);
		}
	}
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	return change;
}

static snd_kcontrol_new_t snd_emu10k1_mixer_bass = {
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "Tone Control - Bass",
	access: SNDRV_CTL_ELEM_ACCESS_READWRITE | SNDRV_CTL_ELEM_ACCESS_INACTIVE,
	info: snd_emu10k1_mixer_tone_info,
	get: snd_emu10k1_mixer_tone_get,
	put: snd_emu10k1_mixer_tone_put,
	private_value: 0
};

static snd_kcontrol_new_t snd_emu10k1_mixer_treble = {
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "Tone Control - Treble",
	access: SNDRV_CTL_ELEM_ACCESS_READWRITE | SNDRV_CTL_ELEM_ACCESS_INACTIVE,
	info: snd_emu10k1_mixer_tone_info,
	get: snd_emu10k1_mixer_tone_get,
	put: snd_emu10k1_mixer_tone_put,
	private_value: 1
};

static int snd_emu10k1_mixer_tone_activate_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
 	uinfo->count = 6;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;
	return 0;
}

static int snd_emu10k1_mixer_tone_activate_get(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int output;
	
	spin_lock_irqsave(&emu->reg_lock, flags);
	for (output = 0; output < 6; output++)
		ucontrol->value.integer.value[output] = (emu->tone_control_active & (1 << output)) ? 1 : 0;
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	return 0;
}

static int snd_emu10k1_mixer_tone_activate_put(snd_kcontrol_t * kcontrol, snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	unsigned int old_active;
	int output, i, tmp, err;
	int change = 0, change1 = 0;
	
	spin_lock_irqsave(&emu->reg_lock, flags);
	old_active = emu->tone_control_active;
	for (output = 0; output < 6; output++) {
		tmp = ucontrol->value.integer.value[output] ? 1 : 0;
		if (tmp != ((emu->tone_control_active >> output) & 1)) {
			if (tmp) {
				err = snd_emu10k1_fx8010_tone_control_activate(emu, output);
				if (err < 0) {
					snd_printk("unable to activate tone control for output %i (err = %i)\n", output, err);
					continue;
				}
			} else {
				err = snd_emu10k1_fx8010_tone_control_deactivate(emu, output);
				if (err < 0) {
					snd_printk("unable to deactivate tone control for output %i (err = %i)\n", output, err);
					continue;
				}
			}
			emu->tone_control_active &= ~(1 << output);
			emu->tone_control_active |= tmp << output;
			change = 1;
		}
	}
	/* change state of tone controls from inactive to active and versa-vice */
	if ((old_active == 0 && emu->tone_control_active) ||
	    (old_active && emu->tone_control_active == 0))
		change1 = 1;
	/* update tone control GPRs */
	if (emu->tone_control_active) {
		emu->tone_bass->access &= ~SNDRV_CTL_ELEM_ACCESS_INACTIVE;
		emu->tone_treble->access &= ~SNDRV_CTL_ELEM_ACCESS_INACTIVE;
		for (i = 0; i < 5; i++) {
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x80 + 0x00 + i * 2 + 0, 0, bass_table[emu->tone_control[0][0]][i]);
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x80 + 0x00 + i * 2 + 1, 0, bass_table[emu->tone_control[0][1]][i]);
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x80 + 0x10 + i * 2 + 0, 0, treble_table[emu->tone_control[1][0]][i]);
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + 0x80 + 0x10 + i * 2 + 1, 0, treble_table[emu->tone_control[1][1]][i]);
		}
	} else {
		emu->tone_bass->access |= SNDRV_CTL_ELEM_ACCESS_INACTIVE;
		emu->tone_treble->access |= SNDRV_CTL_ELEM_ACCESS_INACTIVE;
	}
	spin_unlock_irqrestore(&emu->reg_lock, flags);
	if (change1) {
		snd_ctl_notify(emu->card, SNDRV_CTL_EVENT_MASK_INFO, &emu->tone_bass->id);
		snd_ctl_notify(emu->card, SNDRV_CTL_EVENT_MASK_INFO, &emu->tone_treble->id);
	}
	return change;
}

static snd_kcontrol_new_t snd_emu10k1_mixer_tone_activate = {
	iface: SNDRV_CTL_ELEM_IFACE_MIXER,
	name: "Tone Control - Activate",
	info: snd_emu10k1_mixer_tone_activate_info,
	get: snd_emu10k1_mixer_tone_activate_get,
	put: snd_emu10k1_mixer_tone_activate_put,
};

static int snd_emu10k1_send_routing_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 3*4;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 15;
	return 0;
}

static int snd_emu10k1_send_routing_get(snd_kcontrol_t * kcontrol,
                                        snd_ctl_elem_value_t * ucontrol)
{
	unsigned long flags;
	emu10k1_pcm_mixer_t *mix = (emu10k1_pcm_mixer_t *)kcontrol->private_value;
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	int voice, idx;

	spin_lock_irqsave(&emu->reg_lock, flags);
	for (voice = 0; voice < 3; voice++)
		for (idx = 0; idx < 4; idx++)
		        ucontrol->value.integer.value[(voice * 4) + idx] = (mix->send_routing[voice] >> (idx * 4)) & 15;
	spin_unlock_irqrestore(&emu->reg_lock, flags);
        return 0;
}

static int snd_emu10k1_send_routing_put(snd_kcontrol_t * kcontrol,
                                        snd_ctl_elem_value_t * ucontrol)
{
	unsigned long flags;
	emu10k1_pcm_mixer_t *mix = (emu10k1_pcm_mixer_t *)kcontrol->private_value;
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	int change = 0, voice, idx, val;

	spin_lock_irqsave(&emu->reg_lock, flags);
	for (voice = 0; voice < 3; voice++)
		for (idx = 0; idx < 4; idx++) {
			val = ucontrol->value.integer.value[(voice * 4) + idx] & 15;
			if (((mix->send_routing[voice] >> (idx * 4)) & 15) != val) {
				mix->send_routing[voice] &= ~(15 << (idx * 4));
				mix->send_routing[voice] |= val << (idx * 4);
				change = 1;
			}
		}	
	if (change && mix->epcm) {
		if (mix->epcm->voices[0] && mix->epcm->voices[1]) {
			snd_emu10k1_ptr_write(emu, FXRT, mix->epcm->voices[0]->number, mix->send_routing[1] << 16);
			snd_emu10k1_ptr_write(emu, FXRT, mix->epcm->voices[0]->number, mix->send_routing[2] << 16);
		} else if (mix->epcm->voices[0]) {
			snd_emu10k1_ptr_write(emu, FXRT, mix->epcm->voices[1]->number, mix->send_routing[0] << 16);
		}
	}
	spin_unlock_irqrestore(&emu->reg_lock, flags);
        return change;
}

static snd_kcontrol_new_t snd_emu10k1_send_routing_control =
{
        iface:          SNDRV_CTL_ELEM_IFACE_MIXER,
        name:           "EMU10K1 PCM Send Routing",
        info:           snd_emu10k1_send_routing_info,
        get:            snd_emu10k1_send_routing_get,
        put:            snd_emu10k1_send_routing_put
};

static int snd_emu10k1_send_volume_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 3*4;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 255;
	return 0;
}

static int snd_emu10k1_send_volume_get(snd_kcontrol_t * kcontrol,
                                       snd_ctl_elem_value_t * ucontrol)
{
	unsigned long flags;
	emu10k1_pcm_mixer_t *mix = (emu10k1_pcm_mixer_t *)kcontrol->private_value;
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	int idx;

	spin_lock_irqsave(&emu->reg_lock, flags);
	for (idx = 0; idx < 3*4; idx++)
		ucontrol->value.integer.value[idx] = mix->send_volume[idx/4][idx%4];
	spin_unlock_irqrestore(&emu->reg_lock, flags);
        return 0;
}

static int snd_emu10k1_send_volume_put(snd_kcontrol_t * kcontrol,
                                       snd_ctl_elem_value_t * ucontrol)
{
	unsigned long flags;
	emu10k1_pcm_mixer_t *mix = (emu10k1_pcm_mixer_t *)kcontrol->private_value;
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	int change = 0, idx, val;

	spin_lock_irqsave(&emu->reg_lock, flags);
	for (idx = 0; idx < 3*4; idx++) {
		val = ucontrol->value.integer.value[idx] & 255;
		if (mix->send_volume[idx/4][idx%4] != val) {
			mix->send_volume[idx/4][idx%4] = val;
			change = 1;
		}
	}
	if (change && mix->epcm) {
		u32 voice;
		if (mix->epcm->voices[0] && mix->epcm->voices[1]) {
			voice = mix->epcm->voices[0]->number;
			snd_emu10k1_ptr_write(emu, PTRX_FXSENDAMOUNT_A, voice, mix->send_volume[1][0]);
			snd_emu10k1_ptr_write(emu, PTRX_FXSENDAMOUNT_B, voice, mix->send_volume[1][1]);
			snd_emu10k1_ptr_write(emu, PSST_FXSENDAMOUNT_C, voice, mix->send_volume[1][2]);
			snd_emu10k1_ptr_write(emu, DSL_FXSENDAMOUNT_D, voice, mix->send_volume[1][3]);
			voice = mix->epcm->voices[1]->number;
			snd_emu10k1_ptr_write(emu, PTRX_FXSENDAMOUNT_A, voice, mix->send_volume[2][0]);
			snd_emu10k1_ptr_write(emu, PTRX_FXSENDAMOUNT_B, voice, mix->send_volume[2][1]);
			snd_emu10k1_ptr_write(emu, PSST_FXSENDAMOUNT_C, voice, mix->send_volume[2][2]);
			snd_emu10k1_ptr_write(emu, DSL_FXSENDAMOUNT_D, voice, mix->send_volume[2][3]);
		} else if (mix->epcm->voices[0]) {
			voice = mix->epcm->voices[0]->number;
			snd_emu10k1_ptr_write(emu, PTRX_FXSENDAMOUNT_A, voice, mix->send_volume[0][0]);
			snd_emu10k1_ptr_write(emu, PTRX_FXSENDAMOUNT_B, voice, mix->send_volume[0][1]);
			snd_emu10k1_ptr_write(emu, PSST_FXSENDAMOUNT_C, voice, mix->send_volume[0][2]);
			snd_emu10k1_ptr_write(emu, DSL_FXSENDAMOUNT_D, voice, mix->send_volume[0][3]);
		}
	}
	spin_unlock_irqrestore(&emu->reg_lock, flags);
        return change;
}

static snd_kcontrol_new_t snd_emu10k1_send_volume_control =
{
        iface:          SNDRV_CTL_ELEM_IFACE_MIXER,
        name:           "EMU10K1 PCM Send Volume",
        info:           snd_emu10k1_send_volume_info,
        get:            snd_emu10k1_send_volume_get,
        put:            snd_emu10k1_send_volume_put
};

static int snd_emu10k1_attn_info(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 3;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 0xffff;
	return 0;
}

static int snd_emu10k1_attn_get(snd_kcontrol_t * kcontrol,
                                snd_ctl_elem_value_t * ucontrol)
{
	emu10k1_pcm_mixer_t *mix = (emu10k1_pcm_mixer_t *)kcontrol->private_value;
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int idx;

	spin_lock_irqsave(&emu->reg_lock, flags);
	for (idx = 0; idx < 3; idx++)
		ucontrol->value.integer.value[idx] = mix->attn[idx];
	spin_unlock_irqrestore(&emu->reg_lock, flags);
        return 0;
}

static int snd_emu10k1_attn_put(snd_kcontrol_t * kcontrol,
				snd_ctl_elem_value_t * ucontrol)
{
	unsigned long flags;
	emu10k1_pcm_mixer_t *mix = (emu10k1_pcm_mixer_t *)kcontrol->private_value;
	emu10k1_t *emu = snd_kcontrol_chip(kcontrol);
	int change = 0, idx, val;

	spin_lock_irqsave(&emu->reg_lock, flags);
	for (idx = 0; idx < 3; idx++) {
		val = ucontrol->value.integer.value[idx] & 0xffff;
		if (mix->attn[idx] != val) {
			mix->attn[idx] = val;
			change = 1;
		}
	}
	if (change && mix->epcm) {
		if (mix->epcm->voices[0] && mix->epcm->voices[1]) {
			snd_emu10k1_ptr_write(emu, VTFT_VOLUMETARGET, mix->epcm->voices[0]->number, mix->attn[1]);
			snd_emu10k1_ptr_write(emu, VTFT_VOLUMETARGET, mix->epcm->voices[1]->number, mix->attn[2]);
		} else if (mix->epcm->voices[0]) {
			snd_emu10k1_ptr_write(emu, VTFT_VOLUMETARGET, mix->epcm->voices[0]->number, mix->attn[0]);
		}
	}
	spin_unlock_irqrestore(&emu->reg_lock, flags);
        return change;
}

static snd_kcontrol_new_t snd_emu10k1_attn_control =
{
        iface:          SNDRV_CTL_ELEM_IFACE_MIXER,
        name:           "EMU10K1 PCM Volume",
        info:           snd_emu10k1_attn_info,
        get:            snd_emu10k1_attn_get,
        put:            snd_emu10k1_attn_put
};

static void snd_emu10k1_mixer_free_ac97(ac97_t *ac97)
{
	emu10k1_t *emu = snd_magic_cast(emu10k1_t, ac97->private_data, return);
	emu->ac97 = NULL;
}

int snd_emu10k1_mixer(emu10k1_t *emu)
{
	ac97_t ac97;
	int err, output, pcm;
	snd_ctl_elem_value_t ctl;
	snd_kcontrol_t *kctl;
	snd_card_t *card = emu->card;

	if (!emu->APS) {
		memset(&ac97, 0, sizeof(ac97));
		ac97.write = snd_emu10k1_ac97_write;
		ac97.read = snd_emu10k1_ac97_read;
		ac97.private_data = emu;
		ac97.private_free = snd_emu10k1_mixer_free_ac97;
		if ((err = snd_ac97_mixer(emu->card, &ac97, &emu->ac97)) < 0)
			return err;
	}

	emu->tone_control_active = 0;
	memset(&ctl, 0, sizeof(ctl));
	ctl.value.integer.value[0] = 20;
	ctl.value.integer.value[1] = 20;
	if ((err = snd_ctl_add(emu->card, kctl = snd_ctl_new1(&snd_emu10k1_mixer_bass, emu))) < 0)
		return err;
	kctl->put(emu->tone_bass = kctl, &ctl);
	if ((err = snd_ctl_add(emu->card, kctl = snd_ctl_new1(&snd_emu10k1_mixer_treble, emu))) < 0)
		return err;
	kctl->put(emu->tone_treble = kctl, &ctl);
	if ((err = snd_ctl_add(emu->card, snd_ctl_new1(&snd_emu10k1_mixer_tone_activate, emu))) < 0)
		return err;
	for (output = 0; output < 2; output++) {
		if ((err = snd_ctl_add(emu->card, kctl = snd_ctl_new1(&snd_emu10k1_mixer_stereo[output], emu))) < 0)
			return err;
		memset(&ctl.value.integer, 0, sizeof(ctl.value.integer));
		ctl.value.integer.value[0] =
		ctl.value.integer.value[1] = 100;
		kctl->put(kctl, &ctl);
	}
	for (output = 0; output < 6; output++) {
		if ((err = snd_ctl_add(emu->card, kctl = snd_ctl_new1(&snd_emu10k1_mixer_digital[output], emu))) < 0)
			return err;
		/* FIXME: add proper configuration for EMU APS boards */
		memset(&ctl.value.integer, 0, sizeof(ctl.value.integer));
		switch (output) {
		case EXTOUT_AC97_L/2:	/* AC'97 output */
		case EXTOUT_TOSLINK_L/2: /* TOSLink optical */
		case (EXTOUT_HEADPHONE_L/2)-1: /* Headphone */
			ctl.value.integer.value[FXBUS_PCM_LEFT] =
			ctl.value.integer.value[FXBUS_PCM_RIGHT] = 80;
			ctl.value.integer.value[EXTIN_SPDIF_CD_L+4] =
			ctl.value.integer.value[EXTIN_SPDIF_CD_R+4] = 80;
			ctl.value.integer.value[EXTIN_ZOOM_L+4] =
			ctl.value.integer.value[EXTIN_ZOOM_R+4] = 80;
			break;
		case (EXTOUT_REAR_L/2)-1: /* Rear channel */
			ctl.value.integer.value[FXBUS_PCM_LEFT_REAR] = 
			ctl.value.integer.value[FXBUS_PCM_RIGHT_REAR] = 80;
			break;
		case (EXTOUT_ADC_CAP_L/2)-1: /* ADC Capture */
			ctl.value.integer.value[EXTIN_AC97_L+4] =
			ctl.value.integer.value[EXTIN_AC97_R+4] = 100;
			ctl.value.integer.value[EXTIN_SPDIF_CD_L+4] =
			ctl.value.integer.value[EXTIN_SPDIF_CD_L+4] = 100;
			ctl.value.integer.value[EXTIN_ZOOM_L+4] =
			ctl.value.integer.value[EXTIN_ZOOM_R+4] = 100;
			ctl.value.integer.value[EXTIN_TOSLINK_L+4] =
			ctl.value.integer.value[EXTIN_TOSLINK_R+4] = 100;
			ctl.value.integer.value[EXTIN_LINE1_L+4] =
			ctl.value.integer.value[EXTIN_LINE1_R+4] = 100;
			ctl.value.integer.value[EXTIN_COAX_SPDIF_L+4] =
			ctl.value.integer.value[EXTIN_COAX_SPDIF_R+4] = 100;
			ctl.value.integer.value[EXTIN_LINE2_L+4] =
			ctl.value.integer.value[EXTIN_LINE2_R+4] = 100;
			break;
		case (EXTOUT_MIC_CAP/2)-1: /* MIC Capture */
			ctl.value.integer.value[EXTIN_ZOOM_L+4] =
			ctl.value.integer.value[EXTIN_ZOOM_R+4] = 100;
			break;
		}
		kctl->put(kctl, &ctl);
	}

	for (pcm = 0; pcm < 32; pcm++) {
		emu10k1_pcm_mixer_t *mix;
		
		mix = &emu->pcm_mixer[pcm];
		mix->epcm = NULL;

		if ((kctl = mix->ctl_send_routing = snd_ctl_new1(&snd_emu10k1_send_routing_control, emu)) == NULL)
			return -ENOMEM;
		kctl->private_value = (long)mix;
		kctl->id.index = pcm;
		if ((err = snd_ctl_add(card, kctl)))
			return err;
		mix->send_routing[0] = mix->send_routing[1] = mix->send_routing[2] = 0x3210;
		
		if ((kctl = mix->ctl_send_volume = snd_ctl_new1(&snd_emu10k1_send_volume_control, emu)) == NULL)
			return -ENOMEM;
		kctl->private_value = (long)mix;
		kctl->id.index = pcm;
		if ((err = snd_ctl_add(card, kctl)))
			return err;
		memset(&mix->send_volume, 0, sizeof(mix->send_volume));
		mix->send_volume[0][0] = mix->send_volume[0][1] =
		mix->send_volume[1][0] = mix->send_volume[2][1] = 255;
		
		if ((kctl = mix->ctl_attn = snd_ctl_new1(&snd_emu10k1_attn_control, emu)) == NULL)
			return -ENOMEM;
		kctl->private_value = (long)mix;
		kctl->id.index = pcm;
		if ((err = snd_ctl_add(card, kctl)))
			return err;
		mix->attn[0] = mix->attn[1] = mix->attn[2] = 0xffff;
	}

	return 0;
}
