/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *                   Creative Labs, Inc.
 *  Routines for effect processor FX8010
 *
 *  BUGS:
 *    --
 *
 *  TODO:
 *    --
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../include/driver.h"
#include "../../include/emu10k1.h"

/* CHANGES
 * 31.01.2000   Takashi Iwai
 *      - snd_emu10k1_init_efx() was recoded
 *      - Support for rear volume control
 */

/*************************************************************************
 * EMU10K1 effect manager
 *************************************************************************/

static int snd_emu10k1_fx8010_del(emu10k1_t *emu, snd_emu10k1_fx8010_t *iblock)
{
	list_del(&iblock->list);
	kfree(iblock);
	return 0;
}

static void snd_emu10k1_write_op(emu10k1_fx8010_code_t *icode,
				 u32 op, u32 r, u32 a, u32 x, u32 y)
{
	icode->code[icode->code_size  ][0] = ((x & 0x3ff) << 10) | (y & 0x3ff);
	icode->code[icode->code_size++][1] = ((op & 0x0f) << 20) | ((r & 0x3ff) << 10) | (a & 0x3ff);
}

#define OP(icode, op, r, a, x, y) \
	snd_emu10k1_write_op(icode, op, r, a, x, y)

static snd_emu10k1_fx8010_t *snd_emu10k1_look_for_iblock(emu10k1_t *emu, const char *name)
{
	snd_emu10k1_fx8010_t *iblock;
	struct list_head *list;

	list_for_each(list, &emu->fx8010) {
		iblock = list_entry(list, snd_emu10k1_fx8010_t, list);
		if (!strcmp(iblock->name, name))
			return iblock;
	}
	return NULL;
}

static void snd_emu10k1_gpr_poke(emu10k1_t *emu, emu10k1_fx8010_code_t *icode)
{
	int gpr;

	for (gpr = 0; gpr < 0x100; gpr++)
		if (test_bit(gpr, &icode->gpr_valid))
			snd_emu10k1_ptr_write(emu, FXGPREGBASE + gpr, 0, icode->gpr_map[gpr]);
}

static void snd_emu10k1_tram_poke(emu10k1_t *emu, emu10k1_fx8010_code_t *icode)
{
	int tram;

	for (tram = 0; tram < 0xa0; tram++)
		if (test_bit(tram, &icode->tram_valid)) {
			snd_emu10k1_ptr_write(emu, TANKMEMDATAREGBASE + tram, 0, icode->tram_data_map[tram]);
			snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + tram, 0, icode->tram_addr_map[tram]);
		}
}

static int snd_emu10k1_code_download(emu10k1_t *emu,
				     snd_emu10k1_fx8010_t *iblock_new,
				     emu10k1_fx8010_code_t *icode)
{
	snd_emu10k1_fx8010_t *iblock;
	struct list_head *list;
	u32 code_used[512/32], pc;

	memset(&code_used, 0, sizeof(code_used));
	list_for_each(list, &emu->fx8010) {
		iblock = list_entry(list, snd_emu10k1_fx8010_t, list);
		for (pc = 0; pc < iblock->code_size; pc++)
			set_bit(pc + iblock->code_start, &code_used);
	}
	for (pc = 0; pc < icode->code_size; pc++)
		if (test_bit(pc + icode->code_start, &code_used))
			return -EBUSY;
	/* stop FX processor - this may be dangerous, but it's better to miss
	   some samples than generate wrong ones - [jk] */
	snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg | EMU10K1_DBG_SINGLE_STEP);
	/* download instruction set to FX8010's memory */
	for (pc = 0; pc < icode->code_size; pc++) {
		snd_emu10k1_efx_write(emu, (icode->code_start + pc) * 2, icode->code[pc][0]);
		snd_emu10k1_efx_write(emu, (icode->code_start + pc) * 2 + 1, icode->code[pc][1]);
	}
	iblock_new->code_start = icode->code_start;
	iblock_new->code_size = icode->code_size;
	/* start FX processor */
	snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg);
	return 0;
}

static int snd_emu10k1_download_icode(emu10k1_t *emu, emu10k1_fx8010_code_t *icode)
{
	snd_emu10k1_fx8010_t *iblock;
	int err = 0;
	
	if (icode->code_start > 512)
		return -EINVAL;
	if (icode->code_size > 512)
		return -EINVAL;
	if (icode->code_start + icode->code_size > 512)
		return -EINVAL;
	iblock = (snd_emu10k1_fx8010_t *)snd_kcalloc(sizeof(snd_emu10k1_fx8010_t), GFP_KERNEL);
	if (iblock == NULL)
		return -ENOMEM;
	down(&emu->fx8010_lock);
	strncpy(iblock->name, icode->name, sizeof(iblock->name)-1);
	iblock->name[sizeof(iblock->name)-1] = '\0';
	if (snd_emu10k1_look_for_iblock(emu, iblock->name)) {
		err = -EALREADY;
		goto __error;
	}
	snd_emu10k1_gpr_poke(emu, icode);
	snd_emu10k1_tram_poke(emu, icode);
	if ((err = snd_emu10k1_code_download(emu, iblock, icode)) < 0)
		goto __error;
	list_add_tail(&iblock->list, &emu->fx8010);
	up(&emu->fx8010_lock);
	return 0;
      __error:
	up(&emu->fx8010_lock);
      	kfree(iblock);
      	return err;
}

static int snd_emu10k1_remove_icode(emu10k1_t *emu, const char *name)
{
	snd_emu10k1_fx8010_t *iblock;
	int res;
	u32 pc;

	down(&emu->fx8010_lock);
	iblock = snd_emu10k1_look_for_iblock(emu, name);
	if (iblock == NULL) {
		res = -ENOENT;
		goto __end;
	}
	/* stop FX processor - this may be dangerous, but it's better to miss
	   some samples than generate wrong ones - [jk] */
	snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg | EMU10K1_DBG_SINGLE_STEP);
	for (pc = 0; pc < iblock->code_size; pc++) {
		snd_emu10k1_efx_write(emu, (iblock->code_start + pc) * 2, 0x10040);
		snd_emu10k1_efx_write(emu, (iblock->code_start + pc) * 2 + 1, 0x610040);
	}
	/* start FX processor */
	snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg);
	/* free iblock */
	res = snd_emu10k1_fx8010_del(emu, iblock);
      __end:
	up(&emu->fx8010_lock);
	return res;
}

int snd_emu10k1_init_efx(emu10k1_t *emu)
{
	int err, i, j, k, l, d;
	emu10k1_fx8010_code_t *icode;

	/* stop FX processor */
	snd_emu10k1_ptr_write(emu, DBG, 0, (emu->dbg = 0) | EMU10K1_DBG_SINGLE_STEP);

	if ((icode = kmalloc(sizeof(emu10k1_fx8010_code_t), GFP_KERNEL)) == NULL)
		return -ENOMEM;

	/* clear instruction memory: */
	/* iACC3(C_0000000, C_00000000, C_00000000, C_00000000) */
	for (i = 0; i < 512; i++) {
		snd_emu10k1_efx_write(emu, i * 2, 0x10040);
		snd_emu10k1_efx_write(emu, i * 2 + 1, 0x610040);
	}

	/* clear free GPRs */
	for (i = 0; i < 256; i++)
		snd_emu10k1_ptr_write(emu, FXGPREGBASE + i, 0, 0);

	/* clear TRAM data & address lines */
	for (i = 0; i < 160; i++) {
		snd_emu10k1_ptr_write(emu, TANKMEMDATAREGBASE + i, 0, 0);
		snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + i, 0, 0);
	}

	memset(icode, 0, sizeof(*icode));
	strcpy(icode->name, "FXBUS sampler");
	/* GPR = FXBUS * 4 */
	OP(icode, iMACINT0, GPR(0), C_00000000, FXBUS(FXBUS_PCM_LEFT), C_00000004);
	OP(icode, iMACINT0, GPR(1), C_00000000, FXBUS(FXBUS_PCM_RIGHT), C_00000004);
	OP(icode, iMAC0, GPR(0), C_00000000, GPR(0), GPR(0x10));
	OP(icode, iMAC0, GPR(1), C_00000000, GPR(1), GPR(0x11));
	OP(icode, iMACINT0, GPR(2), C_00000000, FXBUS(FXBUS_MIDI_LEFT), C_00000004);
	OP(icode, iMACINT0, GPR(3), C_00000000, FXBUS(FXBUS_MIDI_RIGHT), C_00000004);
	OP(icode, iMAC0, GPR(0), GPR(0), GPR(2), GPR(0x12));
	OP(icode, iMAC0, GPR(1), GPR(1), GPR(3), GPR(0x13));
	OP(icode, iMACINT0, GPR(2), C_00000000, FXBUS(FXBUS_PCM_LEFT_REAR), C_00000004);
	OP(icode, iMACINT0, GPR(3), C_00000000, FXBUS(FXBUS_PCM_RIGHT_REAR), C_00000004);
	if ((err = snd_emu10k1_download_icode(emu, icode)) < 0) {
		kfree(icode);
		return err;
	}
	/* build output sampler and volume modifier */
	for (i = 0; i < 6; i++) {		/* output number */
		memset(icode, 0, sizeof(*icode));
		icode->code_start = 10 + i * 9 * 2;
		sprintf(icode->name, "Output %d sampler and volume modifier", i);
		for (j = 0; j < 2; j++) {	/* left/right */
			k = 0x14 + (i * 18) + j;
			d = 4 + i * 2 + j;
			OP(icode, iMAC0, GPR(d), C_00000000, GPR(k), GPR(j));
			OP(icode, iMAC0, GPR(d), GPR(d), GPR(k+2), GPR(j+2));
			for (l = 0; l < 7; l++)	/* input number */
				OP(icode, iMAC0, GPR(d), GPR(d), GPR(k+4+l*2), EXTIN(j+l*2));
		}
		if ((err = snd_emu10k1_download_icode(emu, icode)) < 0) {
			kfree(icode);
			return err;
		}
	}
	/* build output writer */
	for (i = 0; i < 6; i++) {		/* output number */
		memset(icode, 0, sizeof(*icode));
		icode->code_start = (0x200 - 6*2) + i*2;
		sprintf(icode->name, "Output %d writer", i);
		for (j = 0; j < 2; j++) {	/* left/right */
			d = 4 + i * 2 + j;
			k = i * 2 + j;
			if (i >= 2)		/* skip outputs 4 & 5 */
				k += 2;
			OP(icode, iACC3, EXTOUT(k), GPR(d), C_00000000, C_00000000);
		}
		if ((err = snd_emu10k1_download_icode(emu, icode)) < 0) {
			kfree(icode);
			return err;
		}
	}

#if 0	/* internal tram test */
	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 0, 0, TANKMEMADDRREG_CLEAR | TANKMEMADDRREG_WRITE | 0);
	snd_emu10k1_ptr_write(emu, TANKMEMDATAREGBASE + 0, 0, 0x100);
	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 1, 0, TANKMEMADDRREG_CLEAR | TANKMEMADDRREG_WRITE | 1);
	snd_emu10k1_ptr_write(emu, TANKMEMDATAREGBASE + 1, 0, 0x200);
	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 2, 0, TANKMEMADDRREG_CLEAR | TANKMEMADDRREG_WRITE | 2);
	snd_emu10k1_ptr_write(emu, TANKMEMDATAREGBASE + 2, 0, 0x300);
	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 3, 0, TANKMEMADDRREG_CLEAR | TANKMEMADDRREG_WRITE | 3);
	snd_emu10k1_ptr_write(emu, TANKMEMDATAREGBASE + 3, 0, 0x400);

	mdelay(1);

	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 0, 0, TANKMEMADDRREG_READ | 2);
	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 1, 0, TANKMEMADDRREG_READ | 3);

	mdelay(1);

	printk("(0) addr2 = 0x%x, addr3 = 0x%x\n",
		snd_emu10k1_ptr_read(emu, TANKMEMDATAREGBASE + 2, 0),
		snd_emu10k1_ptr_read(emu, TANKMEMDATAREGBASE + 3, 0));

	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 0, 0, TANKMEMADDRREG_READ | 0);
	snd_emu10k1_ptr_write(emu, TANKMEMADDRREGBASE + 1, 0, TANKMEMADDRREG_READ | 1);

	mdelay(1);

	printk("(1) addr0 = 0x%x, addr1 = 0x%x\n",
		snd_emu10k1_ptr_read(emu, TANKMEMDATAREGBASE + 0, 0),
		snd_emu10k1_ptr_read(emu, TANKMEMDATAREGBASE + 1, 0));
#endif

	/* free icode structure */
	kfree(icode);

	return 0;
}

void snd_emu10k1_free_efx(emu10k1_t *emu)
{
	snd_emu10k1_fx8010_t *iblock;

	/* stop processor */
	snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg = EMU10K1_DBG_SINGLE_STEP);
	/* free all instruction blocks */
	while (!list_empty(&emu->fx8010)) {	
		iblock = list_entry(emu->fx8010.next, snd_emu10k1_fx8010_t, list);
		snd_emu10k1_fx8010_del(emu, iblock);
	}
}

int snd_emu10k1_fx8010_tone_control_activate(emu10k1_t *emu, int output)
{
	int res, j, k, l, d;
	emu10k1_fx8010_code_t *icode;
	
	snd_runtime_check(output >= 0 && output < 6, return -EINVAL);

	if ((icode = kmalloc(sizeof(emu10k1_fx8010_code_t), GFP_KERNEL)) == NULL)
		return -ENOMEM;

	/* build tone control */
	memset(icode, 0, sizeof(*icode));
	icode->code_start = (0x200 - 6*2 - 6*13*2) + output * 13*2;
	sprintf(icode->name, "Output %d - Tone Control", output);
	for (j = 0; j < 2; j++) {	/* left/right */
		k = 0xa0 + (output * 8) + (j * 4);
		l = 0xd0 + (output * 8) + (j * 4);
		d = 4 + output * 2 + j;

		OP(icode, iMAC0, C_00000000, C_00000000, GPR(d), GPR(0x80 + j));
		OP(icode, iMACMV, GPR(k+1), GPR(k), GPR(k+1), GPR(0x84 + j));
		OP(icode, iMACMV, GPR(k), GPR(d), GPR(k), GPR(0x82 + j));
		OP(icode, iMACMV, GPR(k+3), GPR(k+2), GPR(k+3), GPR(0x88 + j));
		OP(icode, iMAC0, GPR(k+2), GPR_ACCU, GPR(k+2), GPR(0x86 + j));
		OP(icode, iACC3, GPR(k+2), GPR(k+2), GPR(k+2), C_00000000);

		OP(icode, iMAC0, C_00000000, C_00000000, GPR(k+2), GPR(0x90 + j));
		OP(icode, iMACMV, GPR(l+1), GPR(l), GPR(l+1), GPR(0x94 + j));
		OP(icode, iMACMV, GPR(l), GPR(k+2), GPR(l), GPR(0x92 + j));
		OP(icode, iMACMV, GPR(l+3), GPR(l+2), GPR(l+3), GPR(0x98 + j));
		OP(icode, iMAC0, GPR(l+2), GPR_ACCU, GPR(l+2), GPR(0x96 + j));
		OP(icode, iMACINT0, GPR(l+2), C_00000000, GPR(l+2), C_00000010);

		OP(icode, iACC3, GPR(d), GPR(l+2), C_00000000, C_00000000);
	}

	res = snd_emu10k1_download_icode(emu, icode);
	kfree(icode);
	return res;
}

int snd_emu10k1_fx8010_tone_control_deactivate(emu10k1_t *emu, int output)
{
	char name[64];

	snd_runtime_check(output >= 0 && output < 6, return -EINVAL);
	sprintf(name, "Output %d - Tone Control", output);
	return snd_emu10k1_remove_icode(emu, name);
}

static int snd_emu10k1_fx8010_tram_setup(emu10k1_t *emu, u32 size)
{
	u8 size_reg = 0;

	size *= 2;
	if (size != 0) {
		size = (size - 1) >> 14;

		while (size) {
			size >>= 1;
			size_reg++;
		}
		size = 0x4000 << size_reg;
	}
	if (emu->etram_size == size)
		return 0;
	spin_lock_irq(&emu->emu_lock);
	outl(HCFG_LOCKTANKCACHE | inl(emu->port + HCFG), emu->port + HCFG);
	spin_unlock_irq(&emu->emu_lock);	
	snd_emu10k1_ptr_write(emu, TCB, 0, 0);
	snd_emu10k1_ptr_write(emu, TCBS, 0, 0);
	if (emu->etram_pages != NULL) {
		snd_free_pci_pages(emu->pci, emu->etram_size * 2, emu->etram_pages, emu->etram_pages_dmaaddr);
		emu->etram_pages = NULL;
		emu->etram_size = 0;
	}

	if (emu->etram_size > 0) {
		emu->etram_pages = snd_malloc_pci_pages(emu->pci, size, &emu->etram_pages_dmaaddr);
		if (emu->etram_pages == NULL)
			return -ENOMEM;
		snd_emu10k1_ptr_write(emu, TCB, 0, emu->etram_pages_dmaaddr);
		snd_emu10k1_ptr_write(emu, TCBS, 0, size_reg);
		spin_lock_irq(&emu->emu_lock);
		outl(inl(emu->port + HCFG) & ~HCFG_LOCKTANKCACHE, emu->port + HCFG);
		spin_unlock_irq(&emu->emu_lock);	
	}

	emu->etram_size = size / 2;
	return 0;
}

static int snd_emu10k1_fx8010_open(snd_hwdep_t * hw, struct file *file)
{
	return 0;
}

static int snd_emu10k1_fx8010_ioctl(snd_hwdep_t * hw, struct file *file, unsigned int cmd, unsigned long arg)
{
	emu10k1_t *emu = snd_magic_cast(emu10k1_t, hw->private_data, return -ENXIO);
	emu10k1_fx8010_info_t info;
	emu10k1_fx8010_code_t *icode;
	emu10k1_fx8010_code_remove_t remove;
	unsigned int addr;
	int res;
	
	switch (cmd) {
	case SNDRV_EMU10K1_IOCTL_INFO:
		memset(&info, 0, sizeof(info));
		info.card = emu->APS ? EMU10K1_CARD_EMUAPS : EMU10K1_CARD_CREATIVE;
		info.internal_tram_size = emu->itram_size;
		info.external_tram_size = emu->etram_size;
		if (copy_to_user((void *)arg, &info, sizeof(info)))
			return -EFAULT;
		return 0;
	case SNDRV_EMU10K1_IOCTL_CODE:
		icode = (emu10k1_fx8010_code_t *)kmalloc(sizeof(*icode), GFP_KERNEL);
		if (icode == NULL)
			return -ENOMEM;
		if (copy_from_user(icode, (void *)arg, sizeof(*icode))) {
			kfree(icode);
			return -EFAULT;
		}
		res = snd_emu10k1_download_icode(emu, icode);
		kfree(icode);
		return res;
	case SNDRV_EMU10K1_IOCTL_CODE_REMOVE:
		if (copy_from_user(&remove, (void *)arg, sizeof(remove)))
			return -EFAULT;
		return snd_emu10k1_remove_icode(emu, remove.name);
	case SNDRV_EMU10K1_IOCTL_TRAM_SETUP:
		if (get_user(addr, (unsigned int *)arg))
			return -EFAULT;
		down(&emu->fx8010_lock);
		res = snd_emu10k1_fx8010_tram_setup(emu, addr);
		up(&emu->fx8010_lock);
		return res;
	case SNDRV_EMU10K1_IOCTL_STOP:
		snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg |= EMU10K1_DBG_SINGLE_STEP);
		return 0;
	case SNDRV_EMU10K1_IOCTL_CONTINUE:
		snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg = 0);
		return 0;
	case SNDRV_EMU10K1_IOCTL_ZERO_TRAM_COUNTER:
		snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg | EMU10K1_DBG_ZC);
		udelay(10);
		snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg);
		return 0;
	case SNDRV_EMU10K1_IOCTL_SINGLE_STEP:
		if (get_user(addr, (unsigned int *)arg))
			return -EFAULT;
		if (addr > 0x1ff)
			return -EINVAL;
		snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg |= EMU10K1_DBG_SINGLE_STEP | addr);
		udelay(10);
		snd_emu10k1_ptr_write(emu, DBG, 0, emu->dbg |= EMU10K1_DBG_SINGLE_STEP | EMU10K1_DBG_STEP | addr);
		return 0;
	case SNDRV_EMU10K1_IOCTL_DBG_READ:
		addr = snd_emu10k1_ptr_read(emu, DBG, 0);
		if (put_user(addr, (unsigned int *)arg))
			return -EFAULT;
		return 0;
	}
	return -ENXIO;
}

static int snd_emu10k1_fx8010_release(snd_hwdep_t * hw, struct file *file)
{
	return 0;
}

int snd_emu10k1_fx8010_new(emu10k1_t *emu, int device, snd_hwdep_t ** rhwdep)
{
	snd_hwdep_t *hw;
	int err;
	
	if (rhwdep)
		*rhwdep = NULL;
	if ((err = snd_hwdep_new(emu->card, "FX8010", device, &hw)) < 0)
		return err;
	strcpy(hw->name, "EMU10K1 (FX8010)");
	hw->type = SNDRV_HWDEP_TYPE_EMU10K1;
	hw->ops.open = snd_emu10k1_fx8010_open;
	hw->ops.ioctl = snd_emu10k1_fx8010_ioctl;
	hw->ops.release = snd_emu10k1_fx8010_release;
	hw->private_data = emu;
	if (rhwdep)
		*rhwdep = hw;
	return 0;
}
