/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of SoundBlaster cards
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/control.h"
#include "../../include/sb.h"

extern snd_kswitch_t snd_sb16_dma_switch;

int snd_sb16dsp_command(sbdsp_t * codec, unsigned char val)
{
	int i;

	for (i = 10000; i; i--)
		if ((inb(SBP(codec, STATUS)) & 0x80) == 0) {
			outb(val, SBP(codec, COMMAND));
			return 1;
		}
	snd_printd(__FUNCTION__ ": timeout (0x%x)\n", val);
	return 0;
}

int snd_sb16dsp_get_byte(sbdsp_t * codec)
{
	int i;

	for (i = 1000; i; i--)
		if (inb(SBP(codec, DATA_AVAIL)) & 0x80)
			return inb(SBP(codec, READ));
	snd_printd(__FUNCTION__ " failed: 0x%lx = 0x%x!!!\n", SBP(codec, DATA_AVAIL), inb(SBP(codec, DATA_AVAIL)));
	return -ENODEV;
}

int snd_sb16dsp_reset(sbdsp_t * codec)
{
	int i;

	outb(1, SBP(codec, RESET));
	udelay(10);
	outb(0, SBP(codec, RESET));
	udelay(30);
	for (i = 0; i < 1000 && !(inb(SBP(codec, DATA_AVAIL)) & 0x80); i++);
	if (inb(SBP(codec, READ)) != 0xaa) {
		snd_printd(__FUNCTION__ ": failed at 0x%lx!!!\n", codec->port);
		return -ENODEV;
	}
	return 0;
}

int snd_sb16dsp_version(snd_pcm_t * pcm)
{
	unsigned long flags;
	sbdsp_t *codec;
	int i;
	unsigned int result = -1;

	codec = snd_magic_cast(sbdsp_t, pcm->private_data, -ENXIO);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_sb16dsp_command(codec, SB_DSP_GET_VERSION);
	for (i = 100000; i; i--) {
		if (inb(SBP(codec, DATA_AVAIL)) & 0x80) {
			result = (short) inb(SBP(codec, READ)) << 8;
			break;
		}
	}
	for (i = 100000; i; i--) {
		if (inb(SBP(codec, DATA_AVAIL)) & 0x80) {
			result |= (short) inb(SBP(codec, READ));
			break;
		}
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

int snd_sb16dsp_probe(snd_pcm_t * pcm)
{
	unsigned long flags;
	sbdsp_t *codec;
	int version, hw;
	char *str;

	/*
	 *  initialization sequence
	 */

	codec = snd_magic_cast(sbdsp_t, pcm->private_data, -ENXIO);

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (snd_sb16dsp_reset(codec) < 0) {
		snd_printdd("SB: [0x%lx] reset failed... 0x%x\n", codec->port, inb(SBP(codec, READ)));
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return -ENODEV;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	version = snd_sb16dsp_version(pcm);
	if (version < 0)
		return -ENODEV;

	snd_printdd("SB [0x%lx]: DSP chip found, version = %i.%i\n", codec->port, version >> 8, version & 0xff);

	hw = SB_HW_AUTO;
	switch (version >> 8) {
	case 1:
	case 2:
	case 3:
		/* version 3.2 reported by <holger@marzen.de> */
		if (((version & 0xff) == 2) && (codec->hardware == SB_HW_ALS100)) {
			hw = SB_HW_ALS100;
		} else {
			snd_printk("SB [0x%lx]: DSP chip version %i.%i is not supported with the SB16 code\n", codec->port, version >> 8, version & 0xff);
			return -ENODEV;
		}
		break;
	case 4:
		hw = SB_HW_16;
		/* we use PnP to detect the ALS100 chip -- piccio */
		if (((version & 0xff) == 2) && (codec->hardware == SB_HW_ALS100))
			hw = SB_HW_ALS100;
		break;
	default:
		snd_printk("SB [0x%lx]: unknown DSP chip version %i.%i\n", codec->port, version >> 8, version & 0xff);
		return -ENODEV;
	}
	switch (codec->hardware = hw) {
	case SB_HW_16:
		str = "Sound Blaster 16";
		break;
	case SB_HW_ALS100:
		str = "ALS100 chip";
		break;
	default:
		str = "???";
	}
	strcpy(codec->name, str);
	sprintf(pcm->name, "DSP v%i.%i", version >> 8, version & 0xff);

	return 0;
}

int snd_sb16dsp_configure(snd_pcm_t * pcm)
{
	sbdsp_t *codec;
	unsigned long flags;
	unsigned char irqreg = 0, dmareg = 0, mpureg;
	unsigned char realirq, realdma, realmpureg;
	/* note: mpu register should be present only on SB16 Vibra soundcards */

	codec = snd_magic_cast(sbdsp_t, pcm->private_data, -ENXIO);
#if 0
	printk("codec -> irq = %i, codec -> dma8 = %i, codec -> dma16 = %i\n", codec->irq, codec->dma8, codec->dma16);
#endif
	spin_lock_irqsave(&codec->mixer.lock, flags);
	mpureg = snd_sb16mixer_read(&codec->mixer, SB_DSP4_MPUSETUP) & ~0x06;
	spin_unlock_irqrestore(&codec->mixer.lock, flags);
	switch (codec->irq) {
	case 2:
	case 9:
		irqreg |= SB_IRQSETUP_IRQ9;
		break;
	case 5:
		irqreg |= SB_IRQSETUP_IRQ5;
		break;
	case 7:
		irqreg |= SB_IRQSETUP_IRQ7;
		break;
	case 10:
		irqreg |= SB_IRQSETUP_IRQ10;
		break;
	default:
		return -EINVAL;
	}
	if (codec->dma8ptr) {
		switch (codec->dma8) {
		case 0:
			dmareg |= SB_DMASETUP_DMA0;
			break;
		case 1:
			dmareg |= SB_DMASETUP_DMA1;
			break;
		case 3:
			dmareg |= SB_DMASETUP_DMA3;
			break;
		default:
			return -EINVAL;
		}
	}
	if (codec->dma16ptr) {
		switch (codec->dma16) {
		case 5:
			dmareg |= SB_DMASETUP_DMA5;
			break;
		case 6:
			dmareg |= SB_DMASETUP_DMA6;
			break;
		case 7:
			dmareg |= SB_DMASETUP_DMA7;
			break;
		default:
			return -EINVAL;
		}
	}
	switch (codec->mpu_port) {
	case 0x300:
		mpureg |= 0x04;
		break;
	case 0x330:
		mpureg |= 0x00;
		break;
	default:
		mpureg |= 0x02;	/* disable MPU */
	}
	spin_lock_irqsave(&codec->mixer.lock, flags);

	snd_sb16mixer_write(&codec->mixer, SB_DSP4_IRQSETUP, irqreg);
	realirq = snd_sb16mixer_read(&codec->mixer, SB_DSP4_IRQSETUP);

	snd_sb16mixer_write(&codec->mixer, SB_DSP4_DMASETUP, dmareg);
	realdma = snd_sb16mixer_read(&codec->mixer, SB_DSP4_DMASETUP);

	snd_sb16mixer_write(&codec->mixer, SB_DSP4_MPUSETUP, mpureg);
	realmpureg = snd_sb16mixer_read(&codec->mixer, SB_DSP4_MPUSETUP);

	spin_unlock_irqrestore(&codec->mixer.lock, flags);
	if ((~realirq) & irqreg || (~realdma) & dmareg) {
		snd_printdd("SB16 [0x%x]: unable to set DMA & IRQ (PnP device?)\n", codec->port);
		snd_printdd("SB16 [0x%x]: irqreg=0x%x, dmareg=0x%x, mpureg = 0x%x\n", codec->port, realirq, realdma, realmpureg);
		snd_printdd("SB16 [0x%x]: new: irqreg=0x%x, dmareg=0x%x, mpureg = 0x%x\n", codec->port, irqreg, dmareg, mpureg);
#if 0		/* this test doesn't work for all SB16 PnP soundcards */
		return -ENODEV;
#endif
	}
	return 0;
}

void snd_sb16dsp_free(void *private_data)
{
	sbdsp_t *codec = snd_magic_cast(sbdsp_t, private_data, );
	snd_magic_kfree(codec);
}

int snd_sb16dsp_new_pcm(snd_card_t * card,
			int device,
			unsigned long port,
			snd_irq_t * irqptr,
			snd_dma_t * dma8ptr,
			snd_dma_t * dma16ptr,
			unsigned short hardware,
			snd_pcm_t ** rpcm)
{
	snd_pcm_t *pcm;
	sbdsp_t *codec;
	int err;

	if ((err = snd_pcm_new(card, "SB16 DSP", device, 1, 1, &pcm)) < 0)
		return err;
	codec = snd_magic_kcalloc(sbdsp_t, 0, GFP_KERNEL);
	if (codec == NULL)
		return -ENOMEM;
	spin_lock_init(&codec->reg_lock);
	spin_lock_init(&codec->open8_lock);
	spin_lock_init(&codec->open16_lock);
	spin_lock_init(&codec->midi_input_lock);
	spin_lock_init(&codec->mixer.lock);
	codec->pcm = pcm;
	codec->card = pcm->card;
	codec->port = port;
	codec->mixer.port = port;
	codec->irqptr = irqptr;
	codec->irq = irqptr->irq;
	codec->dma8ptr = dma8ptr;
	if (dma8ptr)
		codec->dma8 = dma8ptr->dma;
	codec->dma16ptr = dma16ptr;
	if (dma16ptr)
		codec->dma16 = dma16ptr->dma;
	codec->hardware = hardware;
	pcm->private_data = codec;
	pcm->private_free = snd_sb16dsp_free;
	strcpy(pcm->name, "Sound Blaster");
	if (snd_sb16dsp_probe(pcm) < 0) {
		snd_device_free(card, pcm);
		return -ENODEV;
	}
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE |
	  		  SND_PCM_INFO_DUPLEX | SND_PCM_INFO_DUPLEX_RATE;
	codec->force_mode16 = SB_MODE16_AUTO;

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_sb16_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_sb16_playback_close;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_sb16_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_sb16_capture_close;

	snd_control_switch_new(card, &snd_sb16_dma_switch, codec);
	*rpcm = pcm;
	return 0;
}

EXPORT_SYMBOL(snd_sb16dsp_command);
EXPORT_SYMBOL(snd_sb16dsp_get_byte);
EXPORT_SYMBOL(snd_sb16dsp_reset);
EXPORT_SYMBOL(snd_sb16dsp_new_pcm);
EXPORT_SYMBOL(snd_sb16dsp_configure);
  /* sb16.c */
EXPORT_SYMBOL(snd_sb16dsp_interrupt);
  /* mixer16.c */
EXPORT_SYMBOL(snd_sb16mixer_read);
EXPORT_SYMBOL(snd_sb16mixer_write);
EXPORT_SYMBOL(snd_sb16dsp_new_mixer);

/*
 *  INIT part
 */

static int __init alsa_sb16_init(void)
{
	return 0;
}

static void __exit alsa_sb16_exit(void)
{
}

module_init(alsa_sb16_init)
module_exit(alsa_sb16_exit)
