
/*
    card-azt2320.c - driver for Aztech Systems AZT2320 based soundcards.
    Copyright (C) 1999-2000 by Massimo Piccioni <dafastidio@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
    This driver should provide support for most Aztech AZT2320 based cards.
    Several AZT2316 chips are also supported/tested, but autoprobe doesn't
    work: all module option have to be set.

    No docs available for us at Aztech headquarters !!!   Unbelievable ...
    No other help obtained.

    Thanks to Rainer Wiesner <rainer.wiesner@01019freenet.de> for the WSS
    activation method (full-duplex audio!).
*/

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/initval.h"
#include "../include/cs4231.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"

MODULE_DESCRIPTION("\
Driver: Aztech Systems AZT2320\n\
Card: Aztech Systems PRO16V\n\
Card: Aztech Systems AZT2320\n\
Card: Aztech Systems AT3300\n\
Card: Aztech Systems AT3000\n\
Card: Packard Bell Sound III 336 AM/SP\n\
ISAPNP: AZT1008=AZT1008,AZT2001\n\
ISAPNP: AZT2320=AZT0001,AZT0002\n\
ISAPNP: AZT3002=AZT1004,AZT2001\n\
ISAPNP: AZT3000=AZT1003,AZT2001\n\
");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;		/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;		/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_wss_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_mpu_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_fm_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* Pnp setup */
int snd_mpu_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* Pnp setup */
int snd_dma1[SND_CARDS] = SND_DEFAULT_DMA;		/* PnP setup */
int snd_dma1_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
int snd_dma2[SND_CARDS] = SND_DEFAULT_DMA;		/* PnP setup */
int snd_dma2_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */

#ifdef MODULE_PARM
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for azt2320 based soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for azt2320 based soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for azt2320 driver. [PORT12]");
MODULE_PARM(snd_wss_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_wss_port, "WSS Port # for azt2320 driver. [PORT12]");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for azt2320 driver. [PORT12]");
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_fm_port, "FM port # for azt2320 driver. [PORT12]");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for azt2320 driver. [IRQ]");
MODULE_PARM(snd_mpu_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_irq, "MPU-401 IRQ # for azt2320 driver. [IRQ]");
MODULE_PARM(snd_dma1, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1, "1st DMA # for azt2320 driver. [DMA]");
MODULE_PARM(snd_dma1_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1_size, "1st DMA size for azt2320 driver. [DMA_SIZE]");
MODULE_PARM(snd_dma2, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2, "2nd DMA # for azt2320 driver. [DMA]");
MODULE_PARM(snd_dma2_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2_size, "2nd DMA size for azt2320 driver. [DMA_SIZE]");
#endif	/* MODULE_PARM */

static struct snd_card_azt2320 {
	snd_irq_t *irqptr;
	snd_irq_t *mpuirqptr;
	snd_dma_t *dma1ptr;
	snd_dma_t *dma2ptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;
#ifdef __ISAPNP__
	struct isapnp_dev *dev;
	struct isapnp_dev *devmpu;
#endif	/* __ISAPNP__ */
} *snd_card_azt2320_cards[SND_CARDS] = SND_DEFAULT_PTR;

#ifdef __ISAPNP__
static unsigned int snd_card_azt2320_pnpids[] = {
	/* PRO16V */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_DEVICE(0x1008),    /* device */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x1008),  /* audio */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x2001),  /* mpu401 */
	/* --- */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_DEVICE(0x2320),    /* device */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x0001),  /* audio */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x0002),  /* mpu401 */
	/* AT3300 */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_DEVICE(0x3002),    /* device */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x1004),  /* audio */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x2001),  /* mpu401 */
	/* Packard Bell Sound III 336 AM/SP */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_DEVICE(0x3000),    /* device */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x1002),  /* audio */
	(ISAPNP_VENDOR('A','Z','T')<<16)|ISAPNP_FUNCTION(0x2001),  /* mpu401 */
	0	/* end */
};
#endif	/* __ISAPNP__ */

#define	DRIVER_NAME	"snd-card-azt2320"

#define snd_azt2320_printk(args...)	snd_printk(__FILE__": " ##args)


static void snd_card_azt2320_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_azt2320_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static void snd_card_azt2320_interrupt(int irq, void *dev_id,
				       struct pt_regs *regs)
{
	struct snd_card_azt2320 *acard;
	cs4231_t *codec;
	snd_pcm_t *pcm;
	unsigned char status;

	if (!(acard = (struct snd_card_azt2320 *) dev_id))
		return;

	if (!(pcm = acard->pcm))
		return;

	codec = snd_magic_cast(cs4231_t, pcm->private_data, );
	if (codec == NULL)
		return;

	if ((status = inb(codec->port + 2) & 0x01))
		snd_cs4231_interrupt(pcm, status);
}

static void snd_card_azt2320_mpu_interrupt(int irq, void *dev_id,
					   struct pt_regs *regs)
{
	struct snd_card_azt2320 *acard;

	if (!(acard = (struct snd_card_azt2320 *) dev_id))
		return;

	snd_mpu401_uart_interrupt(acard->rmidi);
}

static int __init snd_card_azt2320_resources(int dev,
					     struct snd_card_azt2320 *acard,
					     snd_card_t *card)
{
	int error;
	static long no_alternatives[] = {-1};

	if (snd_register_ioport(card, snd_mpu_port[dev], 2,
			DRIVER_NAME" - MPU-401", NULL) < 0)
		snd_mpu_port[dev] = -1;
	else if (snd_register_interrupt(card,
			DRIVER_NAME" - MPU-401", snd_mpu_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_azt2320_mpu_interrupt,
			acard, no_alternatives, &acard->mpuirqptr) < 0) {
		snd_unregister_ioports(card);
		snd_mpu_port[dev] = -1;
	}

	if ((error = snd_register_ioport(card, snd_port[dev], 16,
			DRIVER_NAME" - DSP", NULL)) < 0)
		return error;
	if ((error = snd_register_ioport(card, snd_wss_port[dev], 4,
			DRIVER_NAME" - WSS", NULL)) < 0)
		return error;
	if ((error = snd_register_interrupt(card,
			DRIVER_NAME" - WSS", snd_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_azt2320_interrupt,
			acard, no_alternatives, &acard->irqptr)) < 0)
		return error;
	if ((error = snd_register_dma_channel(card,
			DRIVER_NAME" - WSS", snd_dma1[dev],
			SND_DMA_TYPE_ISA, snd_dma1_size[dev],
			no_alternatives, &acard->dma1ptr)) < 0)
		return error;
	if ((error = snd_register_dma_channel(card,
			DRIVER_NAME" - WSS", snd_dma2[dev],
			SND_DMA_TYPE_ISA, snd_dma2_size[dev],
			no_alternatives, &acard->dma2ptr)) < 0)
		return error;

	if (snd_register_ioport(card, snd_fm_port[dev], 8,
			DRIVER_NAME" - FM", NULL) < 0)
		snd_fm_port[dev] = -1;

	return 0;
}

#ifdef __ISAPNP__
static int __init snd_card_azt2320_isapnp(int dev, struct snd_card_azt2320 *acard)
{
	static int idx = 0;
	static struct isapnp_card *card = NULL;
        unsigned int tmp;
	struct isapnp_dev *pdev;

      __again:
	while ((tmp = snd_card_azt2320_pnpids[idx]) != 0) {
		if ((card = isapnp_find_card(tmp >> 16, tmp & 0xffff, card)))
			break;
		idx += 3;
	}

	if (card == NULL)
		return -ENODEV;

	tmp = snd_card_azt2320_pnpids[idx+1];
	acard->dev = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);
	if (acard->dev == NULL)
		goto __again;
	if (acard->dev->active)
		goto __again;		

	tmp = snd_card_azt2320_pnpids[idx+2];
	acard->devmpu = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);

	pdev = acard->dev;
	if (pdev->prepare(pdev) < 0)
		return -EAGAIN;

	if (snd_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_port[dev], 16);
	if (snd_fm_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[1], snd_fm_port[dev], 4);
	if (snd_wss_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[2], snd_wss_port[dev],
			4);
	if (snd_dma1[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[0], snd_dma1[dev],
			1);
	if (snd_dma2[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[1], snd_dma2[dev],
			1);
	if (snd_irq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_irq[dev], 1);

	if (pdev->activate(pdev) < 0) {
		snd_azt2320_printk("AUDIO isapnp configure failure\n");
		return -EBUSY;
	}

	snd_port[dev] = pdev->resource[0].start;
	snd_fm_port[dev] = pdev->resource[1].start;
	snd_wss_port[dev] = pdev->resource[2].start;
	snd_dma1[dev] = pdev->dma_resource[0].start;
	snd_dma2[dev] = pdev->dma_resource[1].start;
	snd_irq[dev] = pdev->irq_resource[0].start;

	pdev = acard->devmpu;
	if (pdev == NULL || pdev->activate(pdev) < 0) {
		snd_mpu_port[dev] = -1;
		return 0;
	}

	if (snd_mpu_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_mpu_port[dev],
			2);
	if (snd_mpu_irq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_mpu_irq[dev],
			1);

	if (pdev->activate(pdev) < 0) {
		/* not fatal error */
		snd_azt2320_printk("MPU-401 isapnp configure failure\n");
		snd_mpu_port[dev] = -1;
		acard->devmpu = NULL;
	} else {
		snd_mpu_port[dev] = pdev->resource[0].start;
		snd_mpu_irq[dev] = pdev->irq_resource[0].start;
	}

	return 0;
}

static void snd_card_azt2320_deactivate(struct snd_card_azt2320 *acard)
{
	if (acard->dev)
		acard->dev->deactivate(acard->dev);
	if (acard->devmpu)
		acard->devmpu->deactivate(acard->devmpu);
}
#endif	/* __ISAPNP__ */

/* same of snd_sb8dsp_command by Jaroslav Kysela */
static int __init snd_card_azt2320_command(unsigned short port, unsigned char val)
{
	int i;
	unsigned long limit;

	limit = jiffies + HZ / 10;
	for (i = 50000; i && (limit - jiffies) > 0; i--)
		if (!(inb(port + 0x0c) & 0x80)) {
			outb(val, port + 0x0c);
			return 0;
		}
	return -EBUSY;
}

static int __init snd_card_azt2320_enable_wss(unsigned long port)
{
	int error;

	if ((error = snd_card_azt2320_command(port, 0x09)))
		return error;
	if ((error = snd_card_azt2320_command(port, 0x00)))
		return error;

	mdelay(5);
	return 0;
}

static int __init snd_card_azt2320_probe(int dev, struct snd_card_azt2320 *acard)
{
	int error;
	snd_card_t *card;
	cs4231_t *codec;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;

	if (!(card = snd_card_new(snd_index[dev], snd_id[dev],
			snd_card_azt2320_use_inc, snd_card_azt2320_use_dec)))
		return -ENOMEM;
	card->type = SND_CARD_TYPE_CS4231;

#ifdef __ISAPNP__
	if ((error = snd_card_azt2320_isapnp(dev, acard)))
		goto __on_error;
#endif	/* __ISAPNP__ */

	if ((error = snd_card_azt2320_resources(dev, acard, card)))
		goto __on_error;

	if (snd_card_azt2320_enable_wss(snd_port[dev]))
		goto __on_error;

	if (snd_cs4231_new_pcm(card, 0, snd_wss_port[dev],
			acard->irqptr, acard->dma1ptr, acard->dma2ptr,
			CS4231_HW_DETECT, 0, &pcm) < 0)
		goto __on_error;
	codec = snd_magic_cast(cs4231_t, pcm->private_data, -ENXIO);

	if (snd_cs4231_new_mixer(pcm, 0, &mixer) < 0)
		goto __on_error;

	rmidi = NULL;
	if (snd_mpu_port[dev] > 0) {
		if (snd_mpu401_uart_new(card, 0, MPU401_HW_AZT2320,
				snd_mpu_port[dev], snd_mpu_irq[dev],
				&rmidi) < 0)
			snd_azt2320_printk("no MPU-401 device at 0x%x ???\n",
				snd_mpu_port[dev]);
	}

	synth = NULL;
	if (snd_fm_port[dev] > 0) {
		if (snd_opl3_new(card, 0,
				snd_fm_port[dev], snd_fm_port[dev] + 2,
				OPL3_HW_AUTO, 1, &synth) < 0)
			snd_azt2320_printk("no OPL device at 0x%x-0x%x ?\n",
				snd_fm_port[dev], snd_fm_port[dev] + 2);
	}

	strcpy(card->abbreviation, "AZT2320");
	strcpy(card->shortname, "Aztech AZT2320");
	sprintf(card->longname, "%s soundcard, WSS at 0x%lx, irq %li, dma %li&%li",
		card->shortname, codec->port, acard->irqptr->irq,
		acard->dma1ptr->dma, acard->dma2ptr->dma);

	if (!(error = snd_card_register(card))) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		acard->rmidi = rmidi;
		acard->synth = synth;
		return 0;
	}

      __on_error:
	snd_card_free(card);
	return error ? error : -ENXIO;
}

#ifdef MODULE

static void __exit snd_card_azt2320_free(int dev)
{
	struct snd_card_azt2320 *acard;

	acard = snd_card_azt2320_cards[dev];
	if (acard) {
		snd_card_azt2320_cards[dev] = NULL;
		snd_card_unregister(acard->card);
#ifdef __ISAPNP__
		snd_card_azt2320_deactivate(acard);
#endif	/* __ISAPNP__ */
		snd_kfree(acard);
	}
}

#endif

static int __init alsa_card_azt2320_init(void)
{
	int cards, dev;
	struct snd_card_azt2320 *acard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		acard = (struct snd_card_azt2320 *) snd_kcalloc(
			sizeof(struct snd_card_azt2320), GFP_KERNEL);
		if (acard == NULL)
			break;

		if (snd_card_azt2320_probe(dev, acard)) {
#ifdef __ISAPNP__
			snd_card_azt2320_deactivate(acard);
#endif	/* __ISAPNP__ */
			snd_kfree(acard);
			continue;
		}
		snd_card_azt2320_cards[cards++] = acard;
	}

#ifdef MODULE
	if (!cards)
		snd_azt2320_printk("no AZT2320 based soundcards found\n");
#endif
	return cards ? 0 : -ENODEV;
}

static void __exit alsa_card_azt2320_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_card_azt2320_free(dev);
}

module_init(alsa_card_azt2320_init)
module_exit(alsa_card_azt2320_exit)
