/*
 *  Generic driver for AD1848/AD1847/CS4248 chips (0.1 Alpha)
 *  Copyright (c) by Tugrul Galatali <galatalt@stuy.edu>,
 *                   Jaroslav Kysela <perex@suse.cz>
 *  Based on card-4232.c by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/ad1848.h"
#include "../include/initval.h"

MODULE_DESCRIPTION("\
Driver: AD1848/AD1847/CS4248\n\
Card: Analog Devices AD1848\n\
Card: Analog Devices Ad1847\n\
Card: Crystal Semiconductors CS4248\n\
");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;	/* PnP setup */
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;	/* 5,7,9,11,12,15 */
int snd_dma1[SND_CARDS] = SND_DEFAULT_DMA;	/* 0,1,3,5,6,7 */
int snd_dma1_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for AD1848 soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for AD1848 soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for AD1848 driver. [PORT12]");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for AD1848 driver. [IRQ]");
MODULE_PARM(snd_dma1, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1, "DMA1 # for AD1848 driver. [DMA]");
MODULE_PARM(snd_dma1_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1_size, "Size of DMA1 # for AD1848 driver. [DMA_SIZE]");

static struct snd_card_ad1848 {
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	ad1848_t *codec;
	unsigned short pcm_status_reg;
} *snd_card_ad1848_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_card_ad1848_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_ad1848_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static snd_pcm_t * __init snd_card_ad1848_detect(snd_card_t * card,
						 unsigned short port,
						 snd_irq_t * irqptr,
						 snd_dma_t * dma1ptr)
{
	snd_pcm_t *pcm;

	if (snd_register_ioport(card, port, 4, "AD1848", NULL) < 0)
		return NULL;

	if (snd_ad1848_new_pcm(card, 0, port, irqptr, dma1ptr, AD1848_HW_DETECT, &pcm) < 0) {
		snd_unregister_ioports(card);
		return NULL;
	}
	return pcm;
}

static void snd_card_ad1848_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_card_ad1848 *acard;
	unsigned char status;

	acard = (struct snd_card_ad1848 *) dev_id;

	if (acard == NULL || acard->pcm == NULL)
		return;

	status = inb(acard->pcm_status_reg);

	if (status & 0x01)
		snd_ad1848_interrupt(acard->pcm, status);
}

static int __init snd_card_ad1848_resources(int dev,
					    struct snd_card_ad1848 *acard,
					    snd_card_t * card)
{
	static int possible_irqs[] = {-1};
	static int possible_dmas[] = {-1};
	int err;

	if ((err = snd_register_interrupt(card, "AD1848",
					  snd_irq[dev], SND_IRQ_TYPE_ISA,
				       snd_card_ad1848_interrupt, acard,
				     possible_irqs, &acard->irqptr)) < 0)
		return err;

	if ((err = snd_register_dma_channel(card,
					 "AD1848", snd_dma1[dev],
				    SND_DMA_TYPE_ISA, snd_dma1_size[dev],
				    possible_dmas, &acard->dma1ptr)) < 0)
		return err;

	return 0;
}

static int __init snd_card_ad1848_probe(int dev, struct snd_card_ad1848 *acard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	ad1848_t *codec;

	card = snd_card_new(snd_index[dev], snd_id[dev],
		     snd_card_ad1848_use_inc, snd_card_ad1848_use_dec);
	if (card == NULL)
		return -ENOMEM;

	card->type = SND_CARD_TYPE_AD1848;
	if (snd_card_ad1848_resources(dev, acard, card) < 0) {
		snd_card_free(card);
		return -EBUSY;
	}
	pcm = snd_card_ad1848_detect(card, snd_port[dev],
				     acard->irqptr, acard->dma1ptr);

	if (pcm == NULL) {
		snd_card_free(card);
		return -ENODEV;
	}
	codec = snd_magic_cast(ad1848_t, pcm->private_data, -ENXIO);
	acard->codec = codec;
	acard->pcm_status_reg = codec->port + 2;

	if (snd_ad1848_new_mixer(pcm, 0, &mixer) < 0)
		goto __nodev;
	strcpy(card->abbreviation, "AD1848");
	strcpy(card->shortname, pcm->name);

	sprintf(card->longname, "%s at 0x%x, irq %i, dma %i",
		pcm->name,
		codec->port,
		acard->irqptr->irq,
		acard->dma1ptr->dma);

	if (!snd_card_register(card)) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		return 0;
	}

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

static int __init alsa_card_ad1848_init(void)
{
	int dev, cards;
	struct snd_card_ad1848 *acard;

	for (dev = cards = 0; dev < SND_CARDS && snd_port[dev] > 0; dev++) {
		acard = (struct snd_card_ad1848 *)
		    snd_kcalloc(sizeof(struct snd_card_ad1848), GFP_KERNEL);
		if (acard == NULL)
			continue;
		if (snd_card_ad1848_probe(dev, acard) < 0) {
			snd_kfree(acard);
			if (snd_port[dev] == SND_AUTO_PORT)
				break;
#ifdef MODULE
			snd_printk("AD1848 soundcard #%i not found at 0x%x or device busy\n", dev + 1, snd_port[dev]);
#endif
			continue;
		}
		snd_card_ad1848_cards[dev] = acard;
		cards++;
	}

	if (!cards) {
#ifdef MODULE
		snd_printk("AD1848 soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

static void __exit alsa_card_ad1848_exit(void)
{
	int idx;
	struct snd_card_ad1848 *acard;

	for (idx = 0; idx < SND_CARDS; idx++) {
		acard = snd_card_ad1848_cards[idx];
		if (acard) {
			snd_card_unregister(acard->card);
			snd_kfree(acard);
		}
	}
}

module_init(alsa_card_ad1848_init)
module_exit(alsa_card_ad1848_exit)
