/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of CS4231(A)/CS4232/InterWave & compatible chips
 *
 *  Bugs:
 *     - sometimes record brokes playback with WSS portion of 
 *       Yamaha OPL3-SA3 chip
 *     - CS4231 (GUS MAX) - still trouble with occasional noises
 *                        - broken initialization?
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/cs4231.h"
#include "../../include/ulaw.h"

#if 0
#define SND_DEBUG_MCE
#endif

/*
 *  Some variables
 */

static struct snd_stru_cs4231_freq snd_cs4231_freq[14] =
{
	{5, 5510, 0x00 | CS4231_XTAL2},
	{6, 6620, 0x0E | CS4231_XTAL2},
	{8, 8000, 0x00 | CS4231_XTAL1},
	{9, 9600, 0x0E | CS4231_XTAL1},
	{11, 11025, 0x02 | CS4231_XTAL2},
	{16, 16000, 0x02 | CS4231_XTAL1},
	{18, 18900, 0x04 | CS4231_XTAL2},
	{22, 22050, 0x06 | CS4231_XTAL2},
	{27, 27042, 0x04 | CS4231_XTAL1},
	{32, 32000, 0x06 | CS4231_XTAL1},
	{33, 33075, 0x0C | CS4231_XTAL2},
	{37, 37800, 0x08 | CS4231_XTAL2},
	{44, 44100, 0x0A | CS4231_XTAL2},
	{48, 48000, 0x0C | CS4231_XTAL1}
};

static unsigned char snd_cs4231_original_image[32] =
{
	0x00,			/* 00/00 - lic */
	0x00,			/* 01/01 - ric */
	0x80,			/* 02/02 - la1ic */
	0x80,			/* 03/03 - ra1ic */
	0x80,			/* 04/04 - la2ic */
	0x80,			/* 05/05 - ra2ic */
	0x80,			/* 06/06 - loc */
	0x80,			/* 07/07 - roc */
	0x20,			/* 08/08 - pdfr */
	CS4231_AUTOCALIB,	/* 09/09 - ic */
	0x00,			/* 0a/10 - pc */
	0x00,			/* 0b/11 - ti */
	CS4231_MODE2,		/* 0c/12 - mi */
	0x00,			/* 0d/13 - lbc */
	0x00,			/* 0e/14 - pbru */
	0x00,			/* 0f/15 - pbrl */
	0x80,			/* 10/16 - afei */
	0x01,			/* 11/17 - afeii */
	0x00,			/* 12/18 - llic */
	0x00,			/* 13/19 - rlic */
	0x00,			/* 14/20 - tlb */
	0x00,			/* 15/21 - thb */
	0x00,			/* 16/22 - la3mic/reserved */
	0x00,			/* 17/23 - ra3mic/reserved */
	0x00,			/* 18/24 - afs */
	0x00,			/* 19/25 - lamoc/version */
	0xc0,			/* 1a/26 - mioc */
	0x00,			/* 1b/27 - ramoc/reserved */
	0x20,			/* 1c/28 - cdfr */
	0x00,			/* 1d/29 - res4 */
	0x00,			/* 1e/30 - cbru */
	0x00,			/* 1f/31 - cbrl */
};

/*
 *  Basic I/O functions
 */

void snd_cs4231_outm(cs4231_t * codec, unsigned char reg,
		     unsigned char mask, unsigned char value)
{
	int timeout;
	unsigned char tmp;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
	     	udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_outm: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	if (codec->calibrate_mute) {
		codec->image[reg] &= mask;
		codec->image[reg] |= value;
	} else {
		outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
		mb();
		tmp = (codec->image[reg] & mask) | value;
		outb(tmp, CS4231P(codec, REG));
		codec->image[reg] = tmp;
		mb();
	}
}

static void snd_cs4231_dout(cs4231_t * codec, unsigned char reg, unsigned char value)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
	     	udelay(10);
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	outb(value, CS4231P(codec, REG));
	mb();
}

void snd_cs4231_out(cs4231_t * codec, unsigned char reg, unsigned char value)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
	     	udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_out: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	outb(value, CS4231P(codec, REG));
	codec->image[reg] = value;
	mb();
#if 0
	printk("codec out - reg 0x%x = 0x%x\n", codec->mce_bit | reg, value);
#endif
}

unsigned char snd_cs4231_in(cs4231_t * codec, unsigned char reg)
{
	int timeout;

	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
	     	udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("snd_cs4231_in: auto calibration time out - reg = 0x%x\n", reg);
#endif
	outb(codec->mce_bit | reg, CS4231P(codec, REGSEL));
	mb();
	return inb(CS4231P(codec, REG));
}

void snd_cs4236_ext_out(cs4231_t * codec, unsigned char reg, unsigned char val)
{
	outb(codec->mce_bit | 0x17, CS4231P(codec, REGSEL));
	outb(reg | (codec->image[CS4236_EXT_REG] & 0x01), CS4231P(codec, REG));
	outb(val, CS4231P(codec, REG));
	codec->eimage[CS4236_REG(reg)] = val;
#if 0
	printk("ext out : reg = 0x%x, val = 0x%x\n", reg, val);
#endif
}

unsigned char snd_cs4236_ext_in(cs4231_t * codec, unsigned char reg)
{
	outb(codec->mce_bit | 0x17, CS4231P(codec, REGSEL));
	outb(reg | (codec->image[CS4236_EXT_REG] & 0x01), CS4231P(codec, REG));
#if 1
	return inb(CS4231P(codec, REG));
#else
	{
		unsigned char res;
		res = inb(CS4231P(codec, REG));
		printk("ext in : reg = 0x%x, val = 0x%x\n", reg, res);
		return res;
	}
#endif
}

#ifdef CONFIG_SND_DEBUG

void snd_cs4231_debug(cs4231_t * codec)
{
	printk("CS4231 REGS:      INDEX = 0x%02x  ", inb(CS4231P(codec, REGSEL)));
	printk("                 STATUS = 0x%02x\n", inb(CS4231P(codec, STATUS)));
	printk("  0x00: left input      = 0x%02x  ", snd_cs4231_in(codec, 0x00));
	printk("  0x10: alt 1 (CFIG 2)  = 0x%02x\n", snd_cs4231_in(codec, 0x10));
	printk("  0x01: right input     = 0x%02x  ", snd_cs4231_in(codec, 0x01));
	printk("  0x11: alt 2 (CFIG 3)  = 0x%02x\n", snd_cs4231_in(codec, 0x11));
	printk("  0x02: GF1 left input  = 0x%02x  ", snd_cs4231_in(codec, 0x02));
	printk("  0x12: left line in    = 0x%02x\n", snd_cs4231_in(codec, 0x12));
	printk("  0x03: GF1 right input = 0x%02x  ", snd_cs4231_in(codec, 0x03));
	printk("  0x13: right line in   = 0x%02x\n", snd_cs4231_in(codec, 0x13));
	printk("  0x04: CD left input   = 0x%02x  ", snd_cs4231_in(codec, 0x04));
	printk("  0x14: timer low       = 0x%02x\n", snd_cs4231_in(codec, 0x14));
	printk("  0x05: CD right input  = 0x%02x  ", snd_cs4231_in(codec, 0x05));
	printk("  0x15: timer high      = 0x%02x\n", snd_cs4231_in(codec, 0x15));
	printk("  0x06: left output     = 0x%02x  ", snd_cs4231_in(codec, 0x06));
	printk("  0x16: left MIC (PnP)  = 0x%02x\n", snd_cs4231_in(codec, 0x16));
	printk("  0x07: right output    = 0x%02x  ", snd_cs4231_in(codec, 0x07));
	printk("  0x17: right MIC (PnP) = 0x%02x\n", snd_cs4231_in(codec, 0x17));
	printk("  0x08: playback format = 0x%02x  ", snd_cs4231_in(codec, 0x08));
	printk("  0x18: IRQ status      = 0x%02x\n", snd_cs4231_in(codec, 0x18));
	printk("  0x09: iface (CFIG 1)  = 0x%02x  ", snd_cs4231_in(codec, 0x09));
	printk("  0x19: left line out   = 0x%02x\n", snd_cs4231_in(codec, 0x19));
	printk("  0x0a: pin control     = 0x%02x  ", snd_cs4231_in(codec, 0x0a));
	printk("  0x1a: mono control    = 0x%02x\n", snd_cs4231_in(codec, 0x1a));
	printk("  0x0b: init & status   = 0x%02x  ", snd_cs4231_in(codec, 0x0b));
	printk("  0x1b: right line out  = 0x%02x\n", snd_cs4231_in(codec, 0x1b));
	printk("  0x0c: revision & mode = 0x%02x  ", snd_cs4231_in(codec, 0x0c));
	printk("  0x1c: record format   = 0x%02x\n", snd_cs4231_in(codec, 0x1c));
	printk("  0x0d: loopback        = 0x%02x  ", snd_cs4231_in(codec, 0x0d));
	printk("  0x1d: var freq (PnP)  = 0x%02x\n", snd_cs4231_in(codec, 0x1d));
	printk("  0x0e: ply upr count   = 0x%02x  ", snd_cs4231_in(codec, 0x0e));
	printk("  0x1e: ply lwr count   = 0x%02x\n", snd_cs4231_in(codec, 0x1e));
	printk("  0x0f: rec upr count   = 0x%02x  ", snd_cs4231_in(codec, 0x0f));
	printk("  0x1f: rec lwr count   = 0x%02x\n", snd_cs4231_in(codec, 0x1f));
}

#endif

/*
 *  CS4231 detection / MCE routines
 */

static void snd_cs4231_busy_wait(cs4231_t * codec)
{
	int timeout;

	/* huh.. looks like this sequence is proper for CS4231A chip (GUS MAX) */
	for (timeout = 5; timeout > 0; timeout--)
		inb(CS4231P(codec, REGSEL));
	/* end of cleanup sequence */
	for (timeout = 250;
	     timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT);
	     timeout--)
	     	udelay(10);
}

void snd_cs4231_mce_up(cs4231_t * codec)
{
	unsigned long flags;
	int timeout;

	spin_lock_irqsave(&codec->reg_lock, flags);
	for (timeout = 250; timeout > 0 && (inb(CS4231P(codec, REGSEL)) & CS4231_INIT); timeout--)
		udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("cs4231_mce_up - auto calibration time out (0)\n");
#endif
	codec->mce_bit |= CS4231_MCE;
	timeout = inb(CS4231P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_cs4231_up [0x%x]: serious init problem - codec still busy\n", codec->port);
	if (!(timeout & CS4231_MCE))
		outb(codec->mce_bit | (timeout & 0x1f), CS4231P(codec, REGSEL));
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

void snd_cs4231_mce_down(cs4231_t * codec)
{
	unsigned long flags;
	int timeout;
	signed long time;

	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_busy_wait(codec);
#if 0
	printk("(1) timeout = %i\n", timeout);
#endif
#ifdef CONFIG_SND_DEBUG
	if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
		snd_printk("cs4231_mce_down [0x%x] - auto calibration time out (0)\n", CS4231P(codec, REGSEL));
#endif
	codec->mce_bit &= ~CS4231_MCE;
	timeout = inb(CS4231P(codec, REGSEL));
	outb(codec->mce_bit | (timeout & 0x1f), CS4231P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_cs4231_down [0x%x]: serious init problem - codec still busy\n", codec->port);
	if ((timeout & CS4231_MCE) == 0 ||
	    !(codec->hardware & (CS4231_HW_CS4231_MASK | CS4231_HW_CS4232_MASK))) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return;
	}
	snd_cs4231_busy_wait(codec);

	/* calibration process */

	for (timeout = 500; timeout > 0 && (snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) == 0; timeout--)
		udelay(10);
	if ((snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) == 0) {
		snd_printd("cs4231_mce_down - auto calibration time out (1)\n");
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return;
	}
#if 0
	printk("(2) timeout = %i, jiffies = %li\n", timeout, jiffies);
#endif
	time = HZ / 4;
	while (snd_cs4231_in(codec, CS4231_TEST_INIT) & CS4231_CALIB_IN_PROGRESS) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		if (time <= 0) {
			snd_printk("cs4231_mce_down - auto calibration time out (2)\n");
			return;
		}
		set_current_state(TASK_INTERRUPTIBLE);
		time = schedule_timeout(time);
		spin_lock_irqsave(&codec->reg_lock, flags);
	}
#if 0
	printk("(3) jiffies = %li\n", jiffies);
#endif
	time = HZ / 10;
	while (inb(CS4231P(codec, REGSEL)) & CS4231_INIT) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		if (time <= 0) {
			snd_printk("cs4231_mce_down - auto calibration time out (3)\n");
			return;
		}
		set_current_state(TASK_INTERRUPTIBLE);		
		time = schedule_timeout(time);
		spin_lock_irqsave(&codec->reg_lock, flags);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
#if 0
	printk("(4) jiffies = %li\n", jiffies);
	snd_printk("cs4231_mce_down - exit = 0x%x\n", inb(CS4231P(codec, REGSEL)));
#endif
}

static unsigned int snd_cs4231_get_count(unsigned char format, unsigned int size)
{
	switch (format & 0xe0) {
	case CS4231_LINEAR_16:
	case CS4231_LINEAR_16_BIG:
		size >>= 1;
		break;
	case CS4231_ADPCM_16:
		return size >> 2;
	}
	if (format & CS4231_STEREO)
		size >>= 1;
	return size;
}

static int snd_cs4231_trigger(cs4231_t * codec, unsigned char what, int channel, int cmd)
{
	unsigned long flags;
	int result = 0;
	snd_pcm_subchn_t *psubchn = NULL;
	snd_pcm_subchn_t *csubchn = NULL;

#if 0
	printk("codec trigger!!! - what = %i, enable = %i, status = 0x%x\n", what, enable, inb(CS4231P(card, STATUS)));
#endif
	if (cmd == SND_PCM_TRIGGER_SYNC_GO) {
		psubchn = codec->playback_subchn;
		csubchn = codec->capture_subchn;
		cmd = SND_PCM_TRIGGER_GO;
		if (psubchn == NULL || csubchn == NULL ||
		    memcmp(&psubchn->runtime->sync_group, &csubchn->runtime->sync_group, sizeof(snd_pcm_sync_t))) {
		    	if (what & CS4231_PLAYBACK_ENABLE) {
				csubchn = NULL;
			} else {
				psubchn = NULL;
			}
		    	goto __standard;
		}
		if (*psubchn->runtime->status != SND_PCM_STATUS_PREPARED ||
		    *csubchn->runtime->status != SND_PCM_STATUS_PREPARED)
		    	return -EINVAL;
		if (!snd_pcm_playback_data(psubchn) || !snd_pcm_capture_empty(csubchn))
			return -EINVAL;
		if ((result = snd_pcm_channel_go_pre(psubchn, SND_PCM_TRIGGER_GO)) < 0)
			goto __cleanup;
		if ((result = snd_pcm_channel_go_pre(csubchn, SND_PCM_TRIGGER_GO)) < 0)
			goto __cleanup;
		result = 0;
		what = CS4231_PLAYBACK_ENABLE | CS4231_RECORD_ENABLE;
	}
      __standard:
	// printk("trigger - cmd = %i, what = 0x%x\n", cmd, what);
	spin_lock_irqsave(&codec->reg_lock, flags);
	if (cmd == SND_PCM_TRIGGER_GO) {
		if ((codec->image[CS4231_IFACE_CTRL] & what) != what)
			snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL] |= what);
	} else if (cmd == SND_PCM_TRIGGER_STOP) {
		if ((codec->image[CS4231_IFACE_CTRL] & what) != 0)
			snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL] &= ~what);
	} else {
		result = -EINVAL;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
      __cleanup:
	if (psubchn)
		snd_pcm_channel_go_post(psubchn, SND_PCM_TRIGGER_GO, result);
	if (csubchn)
		snd_pcm_channel_go_post(csubchn, SND_PCM_TRIGGER_GO, result);
#if 0
	snd_cs4231_debug(codec);
#endif
	return result;
}

/*
 *  CODEC I/O
 */

static unsigned char snd_cs4231_get_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_cs4231_freq[i].hertz)
			return snd_cs4231_freq[i].bits;
	return snd_cs4231_freq[13].bits;
}

static unsigned int snd_cs4231_get_real_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_cs4231_freq[i].hertz)
			return snd_cs4231_freq[i].rate;
	return snd_cs4231_freq[13].rate;
}

static unsigned int snd_cs4231_xrate(cs4231_t * codec,
                                     unsigned int rate)
{
	return snd_cs4231_get_real_freq(rate);
}

static int snd_cs4231_playback_ioctl(void *private_data,
				     snd_pcm_subchn_t * subchn,
				     unsigned int cmd,
                                     unsigned long *arg)
{
	int result;
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	
	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	if (cmd == SND_PCM_IOCTL1_PARAMS)
		subchn->runtime->format.rate = codec->set_playback_rate(codec, subchn->runtime->format.rate);
	return 0;
}

static int snd_cs4231_capture_ioctl(void *private_data,
				    snd_pcm_subchn_t * subchn,
				    unsigned int cmd,
                                    unsigned long *arg)
{
	return snd_cs4231_playback_ioctl(private_data, subchn, cmd, arg);
}

static unsigned char snd_cs4231_get_format(cs4231_t * codec,
				           int format,
                                           int voices)
{
	unsigned char rformat;

	rformat = CS4231_LINEAR_8;
	switch (format) {
	case SND_PCM_SFMT_MU_LAW:	rformat = CS4231_ULAW_8; break;
	case SND_PCM_SFMT_A_LAW:	rformat = CS4231_ALAW_8; break;
	case SND_PCM_SFMT_S16_LE:	rformat = CS4231_LINEAR_16; break;
	case SND_PCM_SFMT_S16_BE:	rformat = CS4231_LINEAR_16_BIG; break;
	case SND_PCM_SFMT_IMA_ADPCM:	rformat = CS4231_ADPCM_16; break;
	}
	if (voices > 1)
		rformat |= CS4231_STEREO;
#if 0
	snd_printk("snd_cs4231_get_format: 0x%x (mode=0x%x)\n", format, mode);
#endif
	return rformat;
}

static void snd_cs4231_calibrate_mute(cs4231_t * codec, int mute)
{
	unsigned long flags;

	mute = mute ? 1 : 0;
	spin_lock_irqsave(&codec->reg_lock, flags);
	if (codec->calibrate_mute == mute) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return;
	}
	if (!mute) {
		snd_cs4231_dout(codec, CS4231_LEFT_INPUT, codec->image[CS4231_LEFT_INPUT]);
		snd_cs4231_dout(codec, CS4231_RIGHT_INPUT, codec->image[CS4231_RIGHT_INPUT]);
		snd_cs4231_dout(codec, CS4231_LOOPBACK, codec->image[CS4231_LOOPBACK]);
	}
	snd_cs4231_dout(codec, CS4231_AUX1_LEFT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX1_LEFT_INPUT]);
	snd_cs4231_dout(codec, CS4231_AUX1_RIGHT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX1_RIGHT_INPUT]);
	snd_cs4231_dout(codec, CS4231_AUX2_LEFT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX2_LEFT_INPUT]);
	snd_cs4231_dout(codec, CS4231_AUX2_RIGHT_INPUT, mute ? 0x80 : codec->image[CS4231_AUX2_RIGHT_INPUT]);
	snd_cs4231_dout(codec, CS4231_LEFT_OUTPUT, mute ? 0x80 : codec->image[CS4231_LEFT_OUTPUT]);
	snd_cs4231_dout(codec, CS4231_RIGHT_OUTPUT, mute ? 0x80 : codec->image[CS4231_RIGHT_OUTPUT]);
	snd_cs4231_dout(codec, CS4231_LEFT_LINE_IN, mute ? 0x80 : codec->image[CS4231_LEFT_LINE_IN]);
	snd_cs4231_dout(codec, CS4231_RIGHT_LINE_IN, mute ? 0x80 : codec->image[CS4231_RIGHT_LINE_IN]);
	snd_cs4231_dout(codec, CS4231_MONO_CTRL, mute ? 0xc0 : codec->image[CS4231_MONO_CTRL]);
	if (codec->hardware == CS4231_HW_INTERWAVE) {
		snd_cs4231_dout(codec, CS4231_LEFT_MIC_INPUT, mute ? 0x80 : codec->image[CS4231_LEFT_MIC_INPUT]);
		snd_cs4231_dout(codec, CS4231_RIGHT_MIC_INPUT, mute ? 0x80 : codec->image[CS4231_RIGHT_MIC_INPUT]);		
		snd_cs4231_dout(codec, CS4231_LINE_LEFT_OUTPUT, mute ? 0x80 : codec->image[CS4231_LINE_LEFT_OUTPUT]);
		snd_cs4231_dout(codec, CS4231_LINE_RIGHT_OUTPUT, mute ? 0x80 : codec->image[CS4231_LINE_RIGHT_OUTPUT]);
	}
	codec->calibrate_mute = mute;
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

static void snd_cs4231_playback_format(cs4231_t * codec,
				       unsigned char pdfr)
{
	unsigned long flags;
	int full_calib = 1;

	down(&codec->mce_mutex);
	snd_cs4231_calibrate_mute(codec, 1);
	if (codec->hardware == CS4231_HW_CS4231A ||
	    (codec->hardware & CS4231_HW_CS4232_MASK)) {
		spin_lock_irqsave(&codec->reg_lock, flags);
		if ((codec->image[CS4231_PLAYBK_FORMAT] & 0x0f) == (pdfr & 0x0f)) {	/* rate is same? */
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] | 0x10);
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image[CS4231_PLAYBK_FORMAT] = pdfr);
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] &= ~0x10);
			full_calib = 0;
		}
		spin_unlock_irqrestore(&codec->reg_lock, flags);
	}
	if (full_calib) {
		snd_cs4231_mce_up(codec);
		spin_lock_irqsave(&codec->reg_lock, flags);
		if (codec->hardware != CS4231_HW_INTERWAVE && !codec->single_dma) {
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT,
					(codec->image[CS4231_IFACE_CTRL] & CS4231_RECORD_ENABLE) ?
					(pdfr & 0xf0) | (codec->image[CS4231_REC_FORMAT] & 0x0f) :
				        pdfr);
		} else {
			snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image[CS4231_PLAYBK_FORMAT] = pdfr);
		}
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		snd_cs4231_mce_down(codec);
	}
	snd_cs4231_calibrate_mute(codec, 0);
	up(&codec->mce_mutex);
}

static void snd_cs4231_capture_format(cs4231_t * codec,
                                      unsigned char cdfr)
{
	unsigned long flags;
	int full_calib = 1;

	down(&codec->mce_mutex);
	snd_cs4231_calibrate_mute(codec, 1);
	if (codec->hardware == CS4231_HW_CS4231A ||
	    (codec->hardware & CS4231_HW_CS4232_MASK)) {
		spin_lock_irqsave(&codec->reg_lock, flags);
		if ((codec->image[CS4231_PLAYBK_FORMAT] & 0x0f) == (cdfr & 0x0f) ||	/* rate is same? */
		    (codec->image[CS4231_IFACE_CTRL] & CS4231_PLAYBACK_ENABLE)) {
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] | 0x20);
			snd_cs4231_out(codec, CS4231_REC_FORMAT, codec->image[CS4231_REC_FORMAT] = cdfr);
			snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] &= ~0x20);
			full_calib = 0;
		}
		spin_unlock_irqrestore(&codec->reg_lock, flags);
	}
	if (full_calib) {
		snd_cs4231_mce_up(codec);
		spin_lock_irqsave(&codec->reg_lock, flags);
		if (codec->hardware != CS4231_HW_INTERWAVE) {
			if (!(codec->image[CS4231_IFACE_CTRL] & CS4231_PLAYBACK_ENABLE)) {
				snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT,
					       ((codec->single_dma ? cdfr : codec->image[CS4231_PLAYBK_FORMAT]) & 0xf0) |
					       (cdfr & 0x0f));
				spin_unlock_irqrestore(&codec->reg_lock, flags);
				snd_cs4231_mce_down(codec);
				snd_cs4231_mce_up(codec);
				spin_lock_irqsave(&codec->reg_lock, flags);
			}
		}
		snd_cs4231_out(codec, CS4231_REC_FORMAT, cdfr);
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		snd_cs4231_mce_down(codec);
	}
	snd_cs4231_calibrate_mute(codec, 0);
	up(&codec->mce_mutex);
}

/*
 *  Timer interface
 */

static unsigned long snd_cs4231_timer_resolution(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = snd_magic_cast(cs4231_t, timer->private_data, -ENXIO);
	if (codec->hardware & CS4231_HW_CS4236_MASK)
		return 14467;
	else
		return codec->image[CS4231_PLAYBK_FORMAT] & 1 ? 9969 : 9920;
}

static void snd_cs4231_timer_start(snd_timer_t * timer)
{
	unsigned long flags;
	unsigned int ticks;
	cs4231_t *codec;

	codec = snd_magic_cast(cs4231_t, timer->private_data, );
	spin_lock_irqsave(&codec->reg_lock, flags);
	ticks = timer->sticks;
	if ((codec->image[CS4231_ALT_FEATURE_1] & CS4231_TIMER_ENABLE) == 0 ||
	    (unsigned char)(ticks >> 8) != codec->image[CS4231_TIMER_HIGH] ||
	    (unsigned char)ticks != codec->image[CS4231_TIMER_LOW]) {
		snd_cs4231_out(codec, CS4231_TIMER_HIGH, codec->image[CS4231_TIMER_HIGH] = (unsigned char) (ticks >> 8));
		snd_cs4231_out(codec, CS4231_TIMER_LOW, codec->image[CS4231_TIMER_LOW] = (unsigned char) ticks);
		snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] | CS4231_TIMER_ENABLE);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

static void snd_cs4231_timer_stop(snd_timer_t * timer)
{
	unsigned long flags;
	cs4231_t *codec;

	codec = snd_magic_cast(cs4231_t, timer->private_data, );
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1] &= ~CS4231_TIMER_ENABLE);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

static void snd_cs4231_init(cs4231_t * codec)
{
	unsigned long flags;

	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (1)\n");
#endif
	snd_cs4231_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO |
			     CS4231_RECORD_ENABLE | CS4231_RECORD_PIO |
			     CS4231_CALIB_MODE);
	codec->image[CS4231_IFACE_CTRL] |= CS4231_AUTOCALIB;
	snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (2)\n");
#endif

	snd_cs4231_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_1, codec->image[CS4231_ALT_FEATURE_1]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (3) - afei = 0x%x\n", codec->image[CS4231_ALT_FEATURE_1]);
#endif

	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_ALT_FEATURE_2, codec->image[CS4231_ALT_FEATURE_2]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	snd_cs4231_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_PLAYBK_FORMAT, codec->image[CS4231_PLAYBK_FORMAT]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (4)\n");
#endif

	snd_cs4231_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_REC_FORMAT, codec->image[CS4231_REC_FORMAT]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_cs4231_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_cs4231_init: (5)\n");
#endif
}

static void snd_cs4231_open(cs4231_t * codec, unsigned int mode)
{
	unsigned long flags;

	down(&codec->open_mutex);
	if (codec->mode & CS4231_MODE_OPEN) {
		codec->mode |= mode;
		up(&codec->open_mutex);
		return;
	}
	/* ok. now enable and ack CODEC IRQ */
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_PLAYBACK_IRQ |
		       CS4231_RECORD_IRQ |
		       CS4231_TIMER_IRQ);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	codec->image[CS4231_PIN_CTRL] |= CS4231_IRQ_ENABLE;
	snd_cs4231_out(codec, CS4231_PIN_CTRL, codec->image[CS4231_PIN_CTRL]);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, CS4231_PLAYBACK_IRQ |
		       CS4231_RECORD_IRQ |
		       CS4231_TIMER_IRQ);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	codec->mode = mode;
	up(&codec->open_mutex);
}

static void snd_cs4231_close(cs4231_t * codec, unsigned int mode)
{
	unsigned long flags;

	down(&codec->open_mutex);
	codec->mode &= ~mode;
	if (codec->mode & CS4231_MODE_OPEN) {
		up(&codec->open_mutex);
		return;
	}
	snd_cs4231_calibrate_mute(codec, 1);

	/* disable IRQ */
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	codec->image[CS4231_PIN_CTRL] &= ~CS4231_IRQ_ENABLE;
	snd_cs4231_out(codec, CS4231_PIN_CTRL, codec->image[CS4231_PIN_CTRL]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	/* now disable record & playback */

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (codec->image[CS4231_IFACE_CTRL] & (CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO |
					       CS4231_RECORD_ENABLE | CS4231_RECORD_PIO)) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		snd_cs4231_mce_up(codec);
		spin_lock_irqsave(&codec->reg_lock, flags);
		codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO |
						     CS4231_RECORD_ENABLE | CS4231_RECORD_PIO);
		snd_cs4231_out(codec, CS4231_IFACE_CTRL, codec->image[CS4231_IFACE_CTRL]);
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		snd_cs4231_mce_down(codec);
		spin_lock_irqsave(&codec->reg_lock, flags);
	}

	/* clear IRQ again */
	snd_cs4231_out(codec, CS4231_IRQ_STATUS, 0);
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	outb(0, CS4231P(codec, STATUS));	/* clear IRQ */
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	snd_cs4231_calibrate_mute(codec, 0);

	codec->mode = 0;
	up(&codec->open_mutex);
}

/*
 *  timer open/close
 */

static int snd_cs4231_timer_open(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = snd_magic_cast(cs4231_t, timer->private_data, -ENXIO);
	snd_cs4231_open(codec, CS4231_MODE_TIMER);
	return 0;
}

static int snd_cs4231_timer_close(snd_timer_t * timer)
{
	cs4231_t *codec;

	codec = snd_magic_cast(cs4231_t, timer->private_data, -ENXIO);
	snd_cs4231_close(codec, CS4231_MODE_TIMER);
	return 0;
}

static struct snd_stru_timer_hardware snd_cs4231_timer =
{
	SND_TIMER_HW_AUTO,	/* flags */
	9945,			/* resolution in us * 1000 */
	65535,			/* high timer mark */
	snd_cs4231_timer_open,	/* open */
	snd_cs4231_timer_close,	/* close */
	snd_cs4231_timer_resolution, /* resolution */
	snd_cs4231_timer_start,	/* start */
	snd_cs4231_timer_stop,	/* stop */
};

/*
 *  ok.. exported functions..
 */

static int snd_cs4231_playback_trigger(void *private_data,
				       snd_pcm_subchn_t * subchn,
				       int cmd)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	return snd_cs4231_trigger(codec, CS4231_PLAYBACK_ENABLE, SND_PCM_CHANNEL_PLAYBACK, cmd);
}

static int snd_cs4231_capture_trigger(void *private_data,
				      snd_pcm_subchn_t * subchn,
				      int cmd)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	return snd_cs4231_trigger(codec, CS4231_RECORD_ENABLE, SND_PCM_CHANNEL_CAPTURE, cmd);
}

static int snd_cs4231_playback_prepare(void *private_data,
				       snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	unsigned long flags;
	unsigned char new_pdfr;
	unsigned int size = snd_pcm_lib_transfer_size(subchn);
	unsigned int count = snd_pcm_lib_transfer_fragment(subchn);

	codec->p_dma_size = size;
	new_pdfr = snd_cs4231_get_format(codec, runtime->format.format, runtime->format.voices) |
		   snd_cs4231_get_freq(runtime->format.rate);
	codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_PLAYBACK_ENABLE | CS4231_PLAYBACK_PIO);
	snd_dma_program(codec->dma1, runtime->dma_area->buf, size, DMA_MODE_WRITE | DMA_AUTOINIT);
	codec->set_playback_format(codec, new_pdfr);
	count = snd_cs4231_get_count(codec->image[CS4231_PLAYBK_FORMAT], count) - 1;
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_out(codec, CS4231_PLY_LWR_CNT, (unsigned char) count);
	snd_cs4231_out(codec, CS4231_PLY_UPR_CNT, (unsigned char) (count >> 8));
	spin_unlock_irqrestore(&codec->reg_lock, flags);
#if 0
	snd_cs4231_debug(codec);
#endif
	return 0;
}

static int snd_cs4231_capture_prepare(void *private_data,
				      snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	unsigned long flags;
	unsigned char new_cdfr;
	unsigned int size = snd_pcm_lib_transfer_size(subchn);
	unsigned int count = snd_pcm_lib_transfer_fragment(subchn);

	codec->c_dma_size = size;
	new_cdfr = snd_cs4231_get_format(codec, runtime->format.format, runtime->format.voices) |
		   snd_cs4231_get_freq(runtime->format.rate);
	codec->image[CS4231_IFACE_CTRL] &= ~(CS4231_RECORD_ENABLE | CS4231_RECORD_PIO);
	snd_dma_program(codec->dma2, runtime->dma_area->buf, size, DMA_MODE_READ | DMA_AUTOINIT);
	codec->set_capture_format(codec, new_cdfr);
	count = snd_cs4231_get_count(codec->image[CS4231_REC_FORMAT], count) - 1;
	spin_lock_irqsave(&codec->reg_lock, flags);
	if (codec->single_dma && codec->hardware != CS4231_HW_INTERWAVE) {
		snd_cs4231_out(codec, CS4231_PLY_LWR_CNT, (unsigned char) count);
		snd_cs4231_out(codec, CS4231_PLY_UPR_CNT, (unsigned char) (count >> 8));
	} else {
		snd_cs4231_out(codec, CS4231_REC_LWR_CNT, (unsigned char) count);
		snd_cs4231_out(codec, CS4231_REC_UPR_CNT, (unsigned char) (count >> 8));
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return 0;
}

static void snd_cs4231_overrange(cs4231_t *codec)
{
	unsigned long flags;
	unsigned char res;

	spin_lock_irqsave(&codec->reg_lock, flags);
	res = snd_cs4231_in(codec, CS4231_TEST_INIT);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	if (res & (0x08 | 0x02))	/* detect overrange only above 0dB; may be user selectable? */
		codec->capture_subchn->runtime->overrange++;
}

void snd_cs4231_interrupt(snd_pcm_t * pcm, unsigned char status)
{
	unsigned long flags;
	cs4231_t *codec;

	codec = snd_magic_cast(cs4231_t, pcm->private_data, );
	if (codec == NULL)		
		return;
	status = snd_cs4231_in(codec, CS4231_IRQ_STATUS);

	if (status & CS4231_TIMER_IRQ) {
		if (codec->timer)
			snd_timer_interrupt(codec->timer, codec->timer->sticks);
	}		
	if (codec->single_dma && codec->hardware != CS4231_HW_INTERWAVE) {
		if (status & CS4231_PLAYBACK_IRQ) {
			if (codec->mode & CS4231_MODE_PLAY)
				snd_pcm_transfer_done(codec->playback_subchn);
			if (codec->mode & CS4231_MODE_RECORD) {
				snd_cs4231_overrange(codec);
				snd_pcm_transfer_done(codec->capture_subchn);
			}
		}
	} else {
		if (status & CS4231_PLAYBACK_IRQ)
			snd_pcm_transfer_done(codec->playback_subchn);
		if (status & CS4231_RECORD_IRQ) {
			snd_cs4231_overrange(codec);
			snd_pcm_transfer_done(codec->capture_subchn);
		}
	}

	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_cs4231_outm(codec, CS4231_IRQ_STATUS, ~CS4231_ALL_IRQS | ~status, 0);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

static unsigned int snd_cs4231_playback_pointer(void *private_data,
						snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);

	if (!(codec->image[CS4231_IFACE_CTRL] & CS4231_PLAYBACK_ENABLE))
		return 0;
	return codec->p_dma_size - snd_dma_residue(codec->dma1);
}

static unsigned int snd_cs4231_capture_pointer(void *private_data,
					       snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	
	if (!(codec->image[CS4231_IFACE_CTRL] & CS4231_RECORD_ENABLE))
		return 0;
	return codec->c_dma_size - snd_dma_residue(codec->dma2);
}

/*

 */

static int snd_cs4231_probe(snd_pcm_t * pcm)
{
	unsigned long flags;
	cs4231_t *codec;
	int i, id, rev;
	unsigned char *ptr;
	unsigned int hw;

	codec = snd_magic_cast(cs4231_t, pcm->private_data, -ENXIO);
	snd_debug_check(codec == NULL, -EINVAL);

#if 0
	snd_cs4231_debug(codec);
#endif
	id = 0;
	for (i = 0; i < 50; i++) {
		mb();
		if (inb(CS4231P(codec, REGSEL)) & CS4231_INIT)
			udelay(2000);
		else {
			spin_lock_irqsave(&codec->reg_lock, flags);
			snd_cs4231_out(codec, CS4231_MISC_INFO, CS4231_MODE2);
			id = snd_cs4231_in(codec, CS4231_MISC_INFO) & 0x0f;
			spin_unlock_irqrestore(&codec->reg_lock, flags);
			if (id == 0x0a)
				break;	/* this is valid value */
		}
	}
	snd_printdd("cs4231: port = 0x%x, id = 0x%x\n", codec->port, id);
	if (id != 0x0a)
		return -ENODEV;	/* no valid device found */

	if (((hw = codec->hardware) & CS4231_HW_TYPE_MASK) == CS4231_HW_DETECT) {
		rev = snd_cs4231_in(codec, CS4231_VERSION) & 0xe7;
		snd_printdd("CS4231: VERSION (I25) = 0x%x\n", rev);
		if (rev == 0x80) {
			codec->hardware = CS4231_HW_CS4231;
		} else if (rev == 0xa0) {
			codec->hardware = CS4231_HW_CS4231A;
		} else if (rev == 0xa2) {
			codec->hardware = CS4231_HW_CS4232;
		} else if (rev == 0xb2) {
			codec->hardware = CS4231_HW_CS4232A;
		} else if (rev == 0x83) {	/* CS4236 -> down to CS4232 */
			codec->hardware = CS4231_HW_CS4232A;
			snd_printk("Original CS4236 chip is not fully supported!!!\n");
		} else if (rev == 0x03) {
			codec->hardware = CS4231_HW_CS4236;
		} else {
			snd_printk("cs4231: unknown CS chip with version 0x%x\n", rev);
			return -ENODEV;		/* unknown CS4231 chip? */
		}
	}
	spin_lock_irqsave(&codec->reg_lock, flags);
	inb(CS4231P(codec, STATUS));	/* clear any pendings IRQ */
	outb(0, CS4231P(codec, STATUS));
	mb();
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	codec->image[CS4231_MISC_INFO] = CS4231_MODE2;
	switch (codec->hardware) {
	case CS4231_HW_INTERWAVE:
		codec->image[CS4231_MISC_INFO] = CS4231_IW_MODE3;
		break;
	case CS4231_HW_CS4235:
	case CS4231_HW_CS4236:
	case CS4231_HW_CS4236B:
	case CS4231_HW_CS4237B:
	case CS4231_HW_CS4238B:
	case CS4231_HW_CS4239:
		if (hw == CS4231_HW_DETECT3)
			codec->image[CS4231_MISC_INFO] = CS4231_4236_MODE3;
		else
			codec->hardware = CS4231_HW_CS4232A;
		break;
	}

	codec->image[CS4231_IFACE_CTRL] =
	    (codec->image[CS4231_IFACE_CTRL] & ~CS4231_SINGLE_DMA) |
	    (codec->single_dma ? CS4231_SINGLE_DMA : 0);
	codec->image[CS4231_ALT_FEATURE_1] = 0x80;
	codec->image[CS4231_ALT_FEATURE_2] = codec->hardware == CS4231_HW_INTERWAVE ? 0xc2 : 0x01;
	ptr = (unsigned char *) &codec->image;
	snd_cs4231_mce_down(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	for (i = 0; i < 32; i++)	/* ok.. fill all CS4231 registers */
		snd_cs4231_out(codec, i, *ptr++);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_cs4231_mce_up(codec);
	snd_cs4231_mce_down(codec);

	mdelay(2);

	/* ok.. try check hardware version for CS4236+ chips */
	if ((hw & CS4231_HW_TYPE_MASK) == CS4231_HW_DETECT) {
		if (codec->hardware == CS4231_HW_CS4236) {
			rev = snd_cs4236_ext_in(codec, CS4236_VERSION);
			snd_cs4236_ext_out(codec, CS4236_VERSION, 0xff);
			id = snd_cs4236_ext_in(codec, CS4236_VERSION);
			snd_cs4236_ext_out(codec, CS4236_VERSION, rev);
			snd_printdd("CS4231: ext version; rev = 0x%x, id = 0x%x\n", rev, id);
			if ((id & 0x1f) == 0x1d) {	/* CS4235 */
				codec->hardware = CS4231_HW_CS4235;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
					break;
				default:
					snd_printk("cs4231: unknown CS4235 chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x0b) {	/* CS4236/B */
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
				case 7:
					codec->hardware = CS4231_HW_CS4236B;
					break;
				default:
					snd_printk("cs4231: unknown CS4236 chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x08) {	/* CS4237B */
				codec->hardware = CS4231_HW_CS4237B;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
				case 7:
					break;
				default:
					snd_printk("cs4231: unknown CS4237B chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x09) {	/* CS4238B */
				codec->hardware = CS4231_HW_CS4238B;
				switch (id >> 5) {
				case 5:
				case 6:
				case 7:
					break;
				default:
					snd_printk("cs4231: unknown CS4238B chip (enhanced version = 0x%x)\n", id);
				}
			} else if ((id & 0x1f) == 0x1e) {	/* CS4239 */
				codec->hardware = CS4231_HW_CS4239;
				switch (id >> 5) {
				case 4:
				case 5:
				case 6:
					break;
				default:
					snd_printk("cs4231: unknown CS4239 chip (enhanced version = 0x%x)\n", id);
				}
			} else {
				snd_printk("cs4231: unknown CS4236/CS423xB chip (enhanced version = 0x%x)\n", id);
			}
		}
	}
	return 0;		/* all things are ok.. */
}

/*

 */

static snd_pcm_hardware_t snd_cs4231_playback =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* formats */
	SND_PCM_RATE_KNOT | SND_PCM_RATE_8000_48000,
	5510,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	(128*1024),		/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	4,			/* transfer block size */
	snd_cs4231_playback_ioctl,
	snd_cs4231_playback_prepare,
	snd_cs4231_playback_trigger,
	snd_cs4231_playback_pointer,
};

static snd_pcm_hardware_t snd_cs4231_capture =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE | SND_PCM_FMT_S16_BE,	/* formats */
	SND_PCM_RATE_KNOT | SND_PCM_RATE_8000_48000,
	5510,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	(128*1024),		/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	4,			/* transfer block size */
	snd_cs4231_capture_ioctl,
	snd_cs4231_capture_prepare,
	snd_cs4231_capture_trigger,
	snd_cs4231_capture_pointer,
};

/*

 */

static int snd_cs4231_playback_open(void * private_data,
				    snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	int err;

	runtime->hw = (snd_pcm_hardware_t *)snd_kmalloc(sizeof(*runtime->hw), GFP_KERNEL);
	if (runtime->hw == NULL)
		return -ENOMEM;
	memcpy(runtime->hw, &snd_cs4231_playback, sizeof(*runtime->hw));
	runtime->hw_free = _snd_kfree;

	snd_pcm_set_sync(subchn);
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->me_playback);

	/* hardware bug in InterWave chipset */
	if (codec->hardware == CS4231_HW_INTERWAVE && codec->dma1 > 3)
	    	runtime->hw->formats &= ~SND_PCM_FMT_MU_LAW;
	
	/* hardware limitation of cheap chips */
	if (codec->hardware == CS4231_HW_CS4235 ||
	    codec->hardware == CS4231_HW_CS4239)
		runtime->hw->formats = SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE;

	if ((err = snd_pcm_dma_alloc(subchn, codec->dmaptr1, "CS4231 (playback)")) < 0)
		return err;
	snd_cs4231_open(codec, CS4231_MODE_PLAY);
	codec->playback_subchn = subchn;
	return 0;
}

static int snd_cs4231_capture_open(void *private_data,
				   snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	int err;

	runtime->hw = (snd_pcm_hardware_t *)snd_kmalloc(sizeof(*runtime->hw), GFP_KERNEL);
	if (runtime->hw == NULL)
		return -ENOMEM;
	memcpy(runtime->hw, &snd_cs4231_capture, sizeof(*runtime->hw));
	runtime->hw_free = _snd_kfree;

	snd_pcm_set_sync(subchn);
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->me_capture);

	/* hardware limitation of cheap chips */
	if (codec->hardware == CS4231_HW_CS4235 ||
	    codec->hardware == CS4231_HW_CS4239)
		runtime->hw->formats = SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE;

	if ((err = snd_pcm_dma_alloc(subchn, codec->dmaptr2, "CS4231 (capture)")) < 0)
		return err;
	snd_cs4231_open(codec, CS4231_MODE_RECORD);
	codec->capture_subchn = subchn;
	return 0;
}

static int snd_cs4231_playback_close(void *private_data,
				     snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);

	codec->playback_subchn = NULL;
	snd_cs4231_close(codec, CS4231_MODE_PLAY);
	snd_pcm_dma_free(subchn);
	return 0;
}

static int snd_cs4231_capture_close(void *private_data,
				    snd_pcm_subchn_t * subchn)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, -ENXIO);

	codec->capture_subchn = NULL;
	snd_cs4231_close(codec, CS4231_MODE_RECORD);
	snd_pcm_dma_free(subchn);
	return 0;
}

static void snd_cs4231_free(void *private_data)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, private_data, );
	if (codec->timer)
		snd_device_free(codec->card, codec->timer);
	snd_magic_kfree(codec);
}

int snd_cs4231_new_pcm(snd_card_t * card, int device,
	               unsigned short port,
		       snd_irq_t * irqptr,
		       snd_dma_t * dmaptr1,
		       snd_dma_t * dmaptr2,
		       unsigned short hardware,
		       int timer_dev,
		       snd_pcm_t ** rpcm)
{
	snd_pcm_t *pcm;
	cs4231_t *codec;
	snd_timer_t *timer;
	char *str;
	int err;

	*rpcm = NULL;
	if ((err = snd_pcm_new(card, "CS4231", device, 1, 1, &pcm)) < 0)
		return err;
	codec = snd_magic_kcalloc(cs4231_t, 0, GFP_KERNEL);
	if (codec == NULL) {
		snd_device_free(card, pcm);
		return -ENOMEM;
	}
	spin_lock_init(&codec->reg_lock);
	init_MUTEX(&codec->mce_mutex);
	init_MUTEX(&codec->open_mutex);
	codec->pcm = pcm;
	codec->card = pcm->card;
	codec->port = port;
	codec->irq = irqptr->irq;
	codec->irqptr = irqptr;
	codec->dmaptr1 = dmaptr1;
	codec->dma1 = dmaptr1->dma;
	codec->dmaptr2 = dmaptr2;
	codec->dma2 = dmaptr2->dma;
	if (codec->dma1 == codec->dma2)
		codec->single_dma = 1;
	codec->hardware = hardware;
	codec->set_playback_rate =
	    codec->set_capture_rate = snd_cs4231_xrate;
	codec->set_playback_format = snd_cs4231_playback_format;
	codec->set_capture_format = snd_cs4231_capture_format;
	memcpy(&codec->image, &snd_cs4231_original_image, sizeof(snd_cs4231_original_image));

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_cs4231_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_cs4231_playback_close;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_cs4231_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_cs4231_capture_close;
	
	/* global setup */
	pcm->private_data = codec;
	pcm->private_free = snd_cs4231_free;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE;
	if (!codec->single_dma)
		pcm->info_flags |= SND_PCM_INFO_DUPLEX;
	if (codec->hardware != CS4231_HW_INTERWAVE)
		pcm->info_flags |= SND_PCM_INFO_DUPLEX_RATE;
	if (snd_cs4231_probe(pcm) < 0) {
		snd_device_free(card, pcm);
		return -ENODEV;
	}
	switch (codec->hardware) {
	case CS4231_HW_CS4231:
		str = "CS4231";
		break;
	case CS4231_HW_CS4231A:
		str = "CS4231A";
		break;
	case CS4231_HW_CS4232:
		str = "CS4232";
		break;
	case CS4231_HW_CS4232A:
		str = "CS4232A";
		break;
	case CS4231_HW_CS4235:
		str = "CS4235";
		break;
	case CS4231_HW_CS4236B:
		str = "CS4236B";
		break;
	case CS4231_HW_CS4237B:
		str = "CS4237B";
		break;
	case CS4231_HW_CS4238B:
		str = "CS4238B";
		break;
	case CS4231_HW_CS4239:
		str = "CS4239";
		break;
	case CS4231_HW_INTERWAVE:
		str = "AMD InterWave";
		break;
	case CS4231_HW_OPL3SA:
		str = card->shortname;
		break;
	default:
		str = "???";
	}
	strcpy(pcm->name, str);
	snd_cs4231_init(codec);
	if ((err = snd_timer_new(card, "CS4231", timer_dev, &timer)) < 0) {
		snd_device_free(card, pcm);
		return err;
	}
	strcpy(timer->name, pcm->name);
	timer->private_data = codec;
	memcpy(&timer->hw, &snd_cs4231_timer, sizeof(snd_cs4231_timer));
	codec->timer = timer;
	*rpcm = pcm;
	return 0;
}

/*
 *  MIXER part
 */

static int snd_cs4231_mixer_get_recsrc(cs4231_t *codec, snd_kmixer_element_t *element)
{
	if (element == codec->me_mux_line)
		return CS4231_MIXS_LINE;
	if (element == codec->me_mux_aux1)
		return CS4231_MIXS_AUX1;
	if (element == codec->me_mux_mic)
		return CS4231_MIXS_MIC;
	if (element == codec->me_mux_mix)
		return CS4231_MIXS_ALL;
	return -EINVAL;
}

static snd_kmixer_element_t *snd_cs4231_mixer_get_recsrc_element(cs4231_t *codec, int value)
{
	value &= CS4231_MIXS_ALL;
	switch (value) {
	case CS4231_MIXS_LINE:
		return codec->me_mux_line;
	case CS4231_MIXS_AUX1:
		return codec->me_mux_aux1;
	case CS4231_MIXS_MIC:
		return codec->me_mux_mic;
	case CS4231_MIXS_ALL:
		return codec->me_mux_mix;
	}
	return NULL;
}

int snd_cs4231_mixer_mux(snd_kmixer_element_t *element, int w_flag, snd_kmixer_element_t **elements)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, element->private_data, -ENXIO);
	unsigned long flags;
	int change = 0;
	int left, right, nleft, nright;

	spin_lock_irqsave(&codec->reg_lock, flags);
	left = codec->image[CS4231_LEFT_INPUT] & CS4231_MIXS_ALL;
	right = codec->image[CS4231_RIGHT_INPUT] & CS4231_MIXS_ALL;
	if (w_flag) {
		nleft = snd_cs4231_mixer_get_recsrc(codec, elements[0]);
		nright = snd_cs4231_mixer_get_recsrc(codec, elements[1]);
		if (nleft >= 0 && nright >= 0) {
			change = nleft != left || nright != right;
			snd_cs4231_outm(codec, CS4231_LEFT_INPUT, (unsigned char)~CS4231_MIXS_ALL, (unsigned char)nleft);
			snd_cs4231_outm(codec, CS4231_RIGHT_INPUT, (unsigned char)~CS4231_MIXS_ALL, (unsigned char)nright);
		} else {
			change = -EINVAL;
		}
	} else {
		elements[0] = snd_cs4231_mixer_get_recsrc_element(codec, left);
		elements[1] = snd_cs4231_mixer_get_recsrc_element(codec, right);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;
}

int snd_cs4231_mixer_stereo_volume(snd_kmixer_element_t *element,
				   int w_flag, int *voices,
				   int max, int invert, int shift,
				   unsigned char left_reg,
				   unsigned char right_reg)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, element->private_data, -ENXIO);
	int change = 0, left, right;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	left = (codec->image[left_reg] >> shift) & max;
	right = (codec->image[right_reg] >> shift) & max;
	if (!w_flag) {
		if (invert) {
			voices[0] = max - left;
			voices[1] = max - right;
		} else {
			voices[0] = left;
			voices[1] = right;
		}
	} else {
		if (invert) {
			change = max - left != voices[0] || max - right != voices[1];
		} else {
			change = left != voices[0] || right != voices[1];
		}
		left = voices[0]; right = voices[1];
		if (invert) {
			left = max - left;
			right = max - right;
		}
		snd_cs4231_outm(codec, left_reg, ~(max << shift), left << shift);
		snd_cs4231_outm(codec, right_reg, ~(max << shift), right << shift);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

int snd_cs4231_mixer_mono_volume(snd_kmixer_element_t *element,
				 int w_flag, int *voices,
				 int max, int invert, int shift,
				 unsigned char reg)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, element->private_data, -ENXIO);
	int change = 0, val;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	val = (codec->image[reg] >> shift) & max;
	if (!w_flag) {
		voices[0] = invert ? max - val : val;
	} else {
		change = (invert ? max - val : val) != voices[0];
		val = voices[0];
		if (invert)
			val = max - val;
		snd_cs4231_outm(codec, reg, ~(max << shift), val << shift);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

int snd_cs4231_mixer_stereo_switch(snd_kmixer_element_t *element,
				   int w_flag, unsigned int *bitmap,
				   int bit, int invert,
				   unsigned char left_reg,
				   unsigned char right_reg)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, element->private_data, -ENXIO);
	int change = 0, left, right;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	left = (codec->image[left_reg] >> bit) & 1;
	right = (codec->image[right_reg] >> bit) & 1;
	if (!w_flag) {
		if (invert) {
			snd_mixer_set_bit(bitmap, 0, left ^ 1);
			snd_mixer_set_bit(bitmap, 1, right ^ 1);
		} else {
			snd_mixer_set_bit(bitmap, 0, left);
			snd_mixer_set_bit(bitmap, 1, right);
		}
	} else {
		if (invert) {
			change = (left ^ 1) != snd_mixer_get_bit(bitmap, 0) ||
			         (right ^ 1) != snd_mixer_get_bit(bitmap, 1);
		} else {
			change = left != snd_mixer_get_bit(bitmap, 0) ||
			         right != snd_mixer_get_bit(bitmap, 1);
		}
		left = snd_mixer_get_bit(bitmap, 0);
		right = snd_mixer_get_bit(bitmap, 1);
		if (invert) {
			left ^= 1;
			right ^= 1;
		}
		snd_cs4231_outm(codec, left_reg, ~(1 << bit), left << bit);
		snd_cs4231_outm(codec, right_reg, ~(1 << bit), right << bit);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

int snd_cs4231_mixer_mono_switch(snd_kmixer_element_t *element,
				 int w_flag, int *value,
				 int bit, int invert,
				 unsigned char reg)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, element->private_data, -ENXIO);
	int change = 0, val;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	val = (codec->image[reg] >> bit) & 1;
	if (!w_flag) {
		*value = invert ? val ^ 1 : val;
	} else {
		change = (invert ? val ^ 1 : val) != (*value != 0);
		val = *value != 0;
		if (invert)
			val ^= 1;
		snd_cs4231_outm(codec, reg, ~(1 << bit), val << bit);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

static int snd_cs4231_mixer_mic_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_stereo_volume(element,
					      w_flag, voices,
					      1, 0, 5,
					      CS4231_LEFT_INPUT,
					      CS4231_RIGHT_INPUT);
}

int snd_cs4231_mixer_line_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_stereo_volume(element,
					      w_flag, voices,
					      31, 1, 0,
					      CS4231_LEFT_LINE_IN,
					      CS4231_RIGHT_LINE_IN);
}

int snd_cs4231_mixer_line_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_cs4231_mixer_stereo_switch(element,
					      w_flag, bitmap,
					      7, 1,
					      CS4231_LEFT_LINE_IN,
					      CS4231_RIGHT_LINE_IN);
}

int snd_cs4231_mixer_aux1_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_stereo_volume(element,
					      w_flag, voices,
					      31, 1, 0,
					      CS4231_AUX1_LEFT_INPUT,
					      CS4231_AUX1_RIGHT_INPUT);
}

int snd_cs4231_mixer_aux1_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_cs4231_mixer_stereo_switch(element,
					      w_flag, bitmap,
					      7, 1,
					      CS4231_AUX1_LEFT_INPUT,
					      CS4231_AUX1_RIGHT_INPUT);
}

int snd_cs4231_mixer_aux2_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_stereo_volume(element,
					      w_flag, voices,
					      31, 1, 0,
					      CS4231_AUX2_LEFT_INPUT,
					      CS4231_AUX2_RIGHT_INPUT);
}

int snd_cs4231_mixer_aux2_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_cs4231_mixer_stereo_switch(element,
					      w_flag, bitmap,
					      7, 1,
					      CS4231_AUX2_LEFT_INPUT,
					      CS4231_AUX2_RIGHT_INPUT);
}

int snd_cs4231_mixer_monoin_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_mono_volume(element,
					    w_flag, voices,
					    15, 1, 0,
					    CS4231_MONO_CTRL);
}

int snd_cs4231_mixer_monoin_switch(snd_kmixer_element_t *element, int w_flag, int *value)
{
	return snd_cs4231_mixer_mono_switch(element,
					    w_flag, value,
					    7, 1,
					    CS4231_MONO_CTRL);
}

int snd_cs4231_mixer_monoout_switch(snd_kmixer_element_t *element, int w_flag, int *value)
{
	return snd_cs4231_mixer_mono_switch(element,
					    w_flag, value,
					    6, 1,
					    CS4231_MONO_CTRL);
}

int snd_cs4231_mixer_mono_bypass_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_cs4231_mixer_mono_switch(element,
					    w_flag, bitmap,
					    5, 0,
					    CS4231_MONO_CTRL);
}

int snd_cs4231_mixer_igain_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_stereo_volume(element,
					      w_flag, voices,
					      15, 0, 0,
					      CS4231_LEFT_INPUT,
					      CS4231_RIGHT_INPUT);
}

static int snd_cs4231_mixer_loopback_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_mono_volume(element,
					    w_flag, voices,
					    63, 1, 2,
					    CS4231_LOOPBACK);
}

static int snd_cs4231_mixer_loopback_switch(snd_kmixer_element_t *element, int w_flag, int *value)
{
	return snd_cs4231_mixer_mono_switch(element,
					    w_flag, value,
					    0, 0,
					    CS4231_LOOPBACK);
}

int snd_cs4231_mixer_dac_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_cs4231_mixer_stereo_volume(element,
					      w_flag, voices,
					      63, 1, 0,
					      CS4231_LEFT_OUTPUT,
					      CS4231_RIGHT_OUTPUT);
}

int snd_cs4231_mixer_dac_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_cs4231_mixer_stereo_switch(element,
					      w_flag, bitmap,
					      7, 1,
					      CS4231_LEFT_OUTPUT,
					      CS4231_RIGHT_OUTPUT);
}

int snd_cs4231_mixer_group_ctrl1(snd_kmixer_group_t * group,
				 snd_kmixer_file_t * file,
				 int w_flag,
				 snd_mixer_group_t * ugroup,
				 snd_mixer_volume1_control_t *volume1,
				 snd_kmixer_element_t *volume1_element,
				 int max,
				 snd_mixer_sw1_control_t *sw1,
				 snd_kmixer_element_t *sw1_element,
				 snd_kmixer_element_t *mux_in)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);
	int voices[2];
	snd_kmixer_element_t *elements[2];
	unsigned int bitmap;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = 0;
		ugroup->channels = SND_MIXER_CHN_MASK_STEREO;
		if (volume1) {
			ugroup->caps |= SND_MIXER_GRPCAP_VOLUME;
			volume1(volume1_element, 0, voices);
			ugroup->volume.names.front_left = voices[0];
			ugroup->volume.names.front_right = voices[1];
			ugroup->min = 0;
			ugroup->max = max;
		}
		if (sw1) {
			ugroup->caps |= SND_MIXER_GRPCAP_MUTE;
			sw1(sw1_element, 0, &bitmap);
			ugroup->mute = 0;
			if (!snd_mixer_get_bit(&bitmap, 0))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (!snd_mixer_get_bit(&bitmap, 1))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		}
		if (mux_in) {
			ugroup->caps |= SND_MIXER_GRPCAP_CAPTURE | SND_MIXER_GRPCAP_EXCL_CAPTURE;
			ugroup->capture_group = 1;
			snd_cs4231_mixer_mux(codec->me_mux, 0, elements);
			ugroup->capture = 0;
			if (elements[0] == mux_in)
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (elements[1] == mux_in)
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		}
	} else {
		if (volume1) {
			voices[0] = ugroup->volume.names.front_left & max;
			voices[1] = ugroup->volume.names.front_right & max;
			if (volume1(volume1_element, 1, voices) > 0) {
				snd_mixer_element_value_change(file, volume1_element, 0);
				change = 1;
			}
		}
		if (sw1) {
			bitmap = 0;
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_LEFT))
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_RIGHT))
				snd_mixer_set_bit(&bitmap, 1, 1);
			if (sw1(sw1_element, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, sw1_element, 0);
				change = 1;
			}
		}
		if (mux_in) {
			snd_cs4231_mixer_mux(codec->me_mux, 0, elements);
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_LEFT)
				elements[0] = mux_in;
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_RIGHT)
				elements[1] = mux_in;
			if (snd_cs4231_mixer_mux(codec->me_mux, 1, elements) > 0) {
				snd_mixer_element_value_change_all_file(file, codec->me_mux, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_cs4231_mixer_group_ctrl2(snd_kmixer_group_t * group,
				        snd_kmixer_file_t * file,
				        int w_flag,
				        snd_mixer_group_t * ugroup,
				        snd_mixer_volume1_control_t *volume1,
				        snd_kmixer_element_t *volume1_element,
				        int max,
				        snd_mixer_sw2_control_t *sw2,
				        snd_kmixer_element_t *sw2_element)
{
	int voices[2];
	int value;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = 0;
		ugroup->channels = SND_MIXER_CHN_MASK_MONO;
		if (volume1) {
			ugroup->caps |= SND_MIXER_GRPCAP_VOLUME;
			volume1(volume1_element, 0, voices);
			ugroup->volume.names.front_left = voices[0];
			ugroup->min = 0;
			ugroup->max = max;
		}
		if (sw2) {
			ugroup->caps |= SND_MIXER_GRPCAP_MUTE;
			sw2(sw2_element, 0, &value);
			ugroup->mute = 0;
			if (!value)
				ugroup->mute |= SND_MIXER_CHN_MASK_MONO;
		}
	} else {
		if (volume1) {
			voices[0] = ugroup->volume.names.front_left & max;
			if (volume1(volume1_element, 1, voices) > 0) {
				snd_mixer_element_value_change(file, volume1_element, 0);
				change = 1;
			}
		}
		if (sw2) {
			value = 0;
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_MONO))
				value = 1;
			if (sw2(sw2_element, 1, &value) > 0) {
				snd_mixer_element_value_change(file, sw2_element, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_cs4231_mixer_group_mic(snd_kmixer_group_t * group,
				      snd_kmixer_file_t * file,
				      int w_flag,
				      snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_mic_volume,
					    codec->me_vol_mic,
					    1,
					    NULL,
					    NULL,
					    codec->me_mux_mic);
}

static int snd_cs4231_mixer_group_line(snd_kmixer_group_t * group,
				       snd_kmixer_file_t * file,
				       int w_flag,
				       snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_line_volume,
					    codec->me_vol_line,
					    31,
					    snd_cs4231_mixer_line_switch,
					    codec->me_sw_line,
					    codec->me_mux_line);
}

static int snd_cs4231_mixer_group_aux1(snd_kmixer_group_t * group,
				       snd_kmixer_file_t * file,
				       int w_flag,
				       snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_aux1_volume,
					    codec->me_vol_aux1,
					    31,
					    snd_cs4231_mixer_aux1_switch,
					    codec->me_sw_aux1,
					    codec->me_mux_aux1);
}

static int snd_cs4231_mixer_group_aux2(snd_kmixer_group_t * group,
				       snd_kmixer_file_t * file,
				       int w_flag,
				       snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_aux2_volume,
					    codec->me_vol_aux2,
					    31,
					    snd_cs4231_mixer_aux2_switch,
					    codec->me_sw_aux2,
					    codec->me_mux_mix);
}

static int snd_cs4231_mixer_group_monoin(snd_kmixer_group_t * group,
					 snd_kmixer_file_t * file,
					 int w_flag,
					 snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl2(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_monoin_volume,
					    codec->me_vol_mono,
					    15,
					    snd_cs4231_mixer_monoin_switch,
					    codec->me_sw_mono);
}

static int snd_cs4231_mixer_group_igain(snd_kmixer_group_t * group,
					snd_kmixer_file_t * file,
					int w_flag,
					snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_igain_volume,
					    codec->me_vol_igain,
					    15,
					    NULL,
					    NULL,
					    NULL);
}

static int snd_cs4231_mixer_group_loop(snd_kmixer_group_t * group,
				       snd_kmixer_file_t * file,
				       int w_flag,
				       snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl2(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_loopback_volume,
					    codec->me_vol_loop,
					    63,
					    snd_cs4231_mixer_loopback_switch,
					    codec->me_sw_loop);
}

static int snd_cs4231_mixer_group_pcm(snd_kmixer_group_t * group,
				      snd_kmixer_file_t * file,
				      int w_flag,
				      snd_mixer_group_t * ugroup)
{
	cs4231_t *codec = snd_magic_cast(cs4231_t, group->private_data, -ENXIO);

	return snd_cs4231_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_cs4231_mixer_dac_volume,
					    codec->me_vol_pcm,
					    63,
					    snd_cs4231_mixer_dac_switch,
					    codec->me_sw_pcm,
					    NULL);
}

int snd_cs4231_new_mixer(snd_pcm_t * pcm, int device, snd_kmixer_t ** rmixer)
{
	cs4231_t *codec;
	snd_kmixer_t *mixer;
	snd_kmixer_group_t *group;
	int err;
	static struct snd_mixer_element_volume1_range mic_range[2] = {
		{0, 1, 0, 2000},
		{0, 1, 0, 2000}
	};
	static struct snd_mixer_element_volume1_range table4_range[2] = {
		{0, 15, 0, 2250},
		{0, 15, 0, 2250}
	};
	static struct snd_mixer_element_volume1_range table5_range[2] = {
		{0, 31, -3450, 1200},
		{0, 31, -3450, 1200}
	};
	static struct snd_mixer_element_volume1_range table6_range[2] = {
		{0, 63, -9450, 0},
		{0, 63, -9450, 0}
	};
	static struct snd_mixer_element_volume1_range table7_range[1] = {
		{0, 15, -4500, 0},
	};

	snd_debug_check(rmixer == NULL, -EINVAL);
	*rmixer = NULL;
	snd_debug_check(pcm == NULL || pcm->card == NULL, -EINVAL);
	codec = snd_magic_cast(cs4231_t, pcm->private_data, -ENXIO);
	if ((err = snd_mixer_new(pcm->card, pcm->id, device, &mixer)) < 0)
		return err;
	strcpy(mixer->name, pcm->name);

	/* build input MUX and accumulators */
	if ((codec->me_mux = snd_mixer_lib_mux1(mixer, SND_MIXER_ELEMENT_INPUT_MUX, 0, 0, 2, snd_cs4231_mixer_mux, codec)) == NULL)
		goto __error;
	if ((codec->me_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_accu, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_mix = codec->me_accu;
	if ((codec->me_out_master = snd_mixer_lib_io_stereo(mixer, SND_MIXER_OUT_MASTER, 0, SND_MIXER_ETYPE_OUTPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_accu, codec->me_out_master) < 0)
		goto __error;
	if ((codec->me_mono_accu = snd_mixer_lib_accu2(mixer, SND_MIXER_ELEMENT_MONO_OUT_ACCU, 0, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_accu, codec->me_mono_accu) < 0)
		goto __error;
	if ((codec->me_sw_master_mono = snd_mixer_lib_sw2(mixer, "Master Mono Switch", 0, snd_cs4231_mixer_monoout_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_mono_accu, codec->me_sw_master_mono) < 0)
		goto __error;
   	if ((codec->me_out_master_mono = snd_mixer_lib_io_mono(mixer, SND_MIXER_OUT_MASTER_MONO, 0, SND_MIXER_ETYPE_OUTPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_master_mono, codec->me_out_master_mono) < 0)
		goto __error;
	if ((codec->me_dig_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_DIGITAL_ACCU, 0, 0)) == NULL)
		goto __error;
	/* build MIC input */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_OSS_MIC, snd_cs4231_mixer_group_mic, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_mux) < 0)
		goto __error;
	if ((codec->me_in_mic = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_MIC, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((codec->me_vol_mic = snd_mixer_lib_volume1(mixer, "MIC Volume", 0, 2, mic_range, snd_cs4231_mixer_mic_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_mic) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_in_mic, codec->me_vol_mic) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_mic, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_mic = codec->me_vol_mic;
	/* build Line-In */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_OSS_LINE, snd_cs4231_mixer_group_line, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_mux) < 0)
		goto __error;
	if ((codec->me_in_line = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_in_line, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_line = codec->me_in_line;
	if ((codec->me_vol_line = snd_mixer_lib_volume1(mixer, "Line Input Volume", 0, 2, table5_range, snd_cs4231_mixer_line_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_line) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_in_line, codec->me_vol_line) < 0)
		goto __error;
	if ((codec->me_sw_line = snd_mixer_lib_sw1(mixer, "Line Input Switch", 0, 2, snd_cs4231_mixer_line_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_line) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_line, codec->me_sw_line) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_line, codec->me_accu) < 0)
		goto __error;
	/* build AUX1 */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_OSS_LINE1, snd_cs4231_mixer_group_aux1, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_mux) < 0)
		goto __error;
	if ((codec->me_in_aux1 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_in_aux1, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_aux1 = codec->me_in_aux1;
	if ((codec->me_vol_aux1 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 0, 2, table5_range, snd_cs4231_mixer_aux1_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_aux1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_in_aux1, codec->me_vol_aux1) < 0)
		goto __error;
	if ((codec->me_sw_aux1 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 0, 2, snd_cs4231_mixer_aux1_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_aux1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_aux1, codec->me_sw_aux1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_aux1, codec->me_accu) < 0)
		goto __error;
	/* build AUX2 */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_AUX, 1, SND_MIXER_OSS_LINE2, snd_cs4231_mixer_group_aux2, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_mux) < 0)
		goto __error;
	if ((codec->me_in_aux2 = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 1, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((codec->me_vol_aux2 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 1, 2, table5_range, snd_cs4231_mixer_aux2_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_aux2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_in_aux2, codec->me_vol_aux2) < 0)
		goto __error;
	if ((codec->me_sw_aux2 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 1, 2, snd_cs4231_mixer_aux2_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_aux2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_aux2, codec->me_sw_aux2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_aux2, codec->me_accu) < 0)
		goto __error;
	/* build MIN */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_MONO, 0, SND_MIXER_OSS_SPEAKER, snd_cs4231_mixer_group_monoin, codec)) == NULL)
		goto __error;
	if ((codec->me_in_mono = snd_mixer_lib_io_mono(mixer, SND_MIXER_IN_MONO, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((codec->me_vol_mono = snd_mixer_lib_volume1(mixer, "Mono Input Volume", 0, 1, table7_range, snd_cs4231_mixer_monoin_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_mono) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_in_mono, codec->me_vol_mono) < 0)
		goto __error;
	if ((codec->me_sw_mono = snd_mixer_lib_sw2(mixer, "Mono Input Switch", 0, snd_cs4231_mixer_monoin_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_mono) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_mono, codec->me_sw_mono) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_mono, codec->me_accu) < 0)
		goto __error;
	/* build input GAIN */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_GRP_IGAIN, 0, SND_MIXER_OSS_IGAIN, snd_cs4231_mixer_group_igain, codec)) == NULL)
		goto __error;
	if ((codec->me_vol_igain = snd_mixer_lib_volume1(mixer, "Input Gain Volume", 0, 2, table4_range, snd_cs4231_mixer_igain_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_igain) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_mux, codec->me_vol_igain) < 0)
		goto __error;
	/* build ADC */
	if ((codec->me_adc = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_ADC, 0, SND_MIXER_ETYPE_ADC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_igain, codec->me_adc) < 0)
		goto __error;
	/* build capture endpoint */
	if ((codec->me_capture = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE1, 1, &pcm->device)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_adc, codec->me_capture) < 0)
		goto __error;
	/* build loopback */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_GRP_DIGITAL_LOOPBACK, 0, SND_MIXER_OSS_DIGITAL1, snd_cs4231_mixer_group_loop, codec)) == NULL)
		goto __error;
	if ((codec->me_vol_loop = snd_mixer_lib_volume1(mixer, "Loopback Volume", 0, 1, table6_range, snd_cs4231_mixer_loopback_volume, codec)) == NULL)
		goto __error;	
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_loop) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_adc, codec->me_vol_loop) < 0)
		goto __error;
	if ((codec->me_sw_loop = snd_mixer_lib_sw2(mixer, "Loopback Switch", 0, snd_cs4231_mixer_loopback_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_loop) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_loop, codec->me_sw_loop) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_loop, codec->me_dig_accu) < 0)
		goto __error;
	/* build playback endpoint */
	if ((codec->me_playback = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_PLAYBACK, 0, SND_MIXER_ETYPE_PLAYBACK1, 1, &pcm->device)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_playback, codec->me_dig_accu) < 0)
		goto __error;
	/* build DAC */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_PCM, 0, SND_MIXER_OSS_PCM, snd_cs4231_mixer_group_pcm, codec)) == NULL)
		goto __error;
	if ((codec->me_dac = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_DAC, 0, SND_MIXER_ETYPE_DAC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_dig_accu, codec->me_dac) < 0)
		goto __error;
	if ((codec->me_vol_pcm = snd_mixer_lib_volume1(mixer, "PCM Volume", 0, 2, table6_range, snd_cs4231_mixer_dac_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_dac, codec->me_vol_pcm) < 0)
		goto __error;
	if ((codec->me_sw_pcm = snd_mixer_lib_sw1(mixer, "PCM Switch", 0, 2, snd_cs4231_mixer_dac_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_pcm, codec->me_sw_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_pcm, codec->me_accu) < 0)
		goto __error;

	mixer->private_data = codec;
	*rmixer = codec->mixer = mixer;
	return 0;
      __error:
      	snd_device_free(pcm->card, mixer);
      	return -ENOMEM;
}

EXPORT_SYMBOL(snd_cs4231_out);
EXPORT_SYMBOL(snd_cs4231_in);
EXPORT_SYMBOL(snd_cs4231_outm);
EXPORT_SYMBOL(snd_cs4236_ext_out);
EXPORT_SYMBOL(snd_cs4236_ext_in);
EXPORT_SYMBOL(snd_cs4231_mce_up);
EXPORT_SYMBOL(snd_cs4231_mce_down);
EXPORT_SYMBOL(snd_cs4231_interrupt);
EXPORT_SYMBOL(snd_cs4231_new_pcm);
EXPORT_SYMBOL(snd_cs4231_new_mixer);
EXPORT_SYMBOL(snd_cs4231_mixer_mux);
EXPORT_SYMBOL(snd_cs4231_mixer_stereo_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_mono_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_stereo_switch);
EXPORT_SYMBOL(snd_cs4231_mixer_mono_switch);
EXPORT_SYMBOL(snd_cs4231_mixer_line_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_line_switch);
EXPORT_SYMBOL(snd_cs4231_mixer_aux1_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_aux1_switch);
EXPORT_SYMBOL(snd_cs4231_mixer_aux2_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_aux2_switch);
EXPORT_SYMBOL(snd_cs4231_mixer_monoin_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_monoin_switch);
EXPORT_SYMBOL(snd_cs4231_mixer_mono_bypass_switch);
EXPORT_SYMBOL(snd_cs4231_mixer_igain_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_dac_volume);
EXPORT_SYMBOL(snd_cs4231_mixer_dac_switch);
  /* for interwave.c */
EXPORT_SYMBOL(snd_cs4231_mixer_group_ctrl1);

/*
 *  INIT part
 */

static int __init alsa_cs4231_init(void)
{
	return 0;
}

static void __exit alsa_cs4231_exit(void)
{
}

module_init(alsa_cs4231_init)
module_exit(alsa_cs4231_exit)
