/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of AD1848/AD1847/CS4248
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/ad1848.h"
#include "../../include/ulaw.h"

#if 0
#define SND_DEBUG_MCE
#endif

/*
 *  Some variables
 */

static struct snd_stru_ad1848_freq snd_ad1848_freq[14] =
{
	{5, 5510, 0x00 | AD1848_XTAL2},
	{6, 6620, 0x0E | AD1848_XTAL2},
	{8, 8000, 0x00 | AD1848_XTAL1},
	{9, 9600, 0x0E | AD1848_XTAL1},
	{11, 11025, 0x02 | AD1848_XTAL2},
	{16, 16000, 0x02 | AD1848_XTAL1},
	{18, 18900, 0x04 | AD1848_XTAL2},
	{22, 22050, 0x06 | AD1848_XTAL2},
	{27, 27042, 0x04 | AD1848_XTAL1},
	{32, 32000, 0x06 | AD1848_XTAL1},
	{33, 33075, 0x0C | AD1848_XTAL2},
	{37, 37800, 0x08 | AD1848_XTAL2},
	{44, 44100, 0x0A | AD1848_XTAL2},
	{48, 48000, 0x0C | AD1848_XTAL1}
};

static unsigned char snd_ad1848_original_image[16] =
{
	0x00,			/* 00 - lic */
	0x00,			/* 01 - ric */
	0x80,			/* 02 - la1ic */
	0x80,			/* 03 - ra1ic */
	0x80,			/* 04 - la2ic */
	0x80,			/* 05 - ra2ic */
	0x80,			/* 06 - loc */
	0x80,			/* 07 - roc */
	0x20,			/* 08 - dfr */
	AD1848_AUTOCALIB,	/* 09 - ic */
	0x00,			/* 0a - pc */
	0x00,			/* 0b - ti */
	0x00,			/* 0c - mi */
	0x00,			/* 0d - lbc */
	0x00,			/* 0e - dru */
	0x00,			/* 0f - drl */
};

/*
 *  Basic I/O functions
 */

static void snd_ad1848_outm(ad1848_t * codec, unsigned char reg,
			    unsigned char mask, unsigned char value)
{
	int timeout;
	unsigned char tmp;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("snd_ad1848_outm: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	if (codec->calibrate_mute) {
		codec->image[reg] &= mask;
		codec->image[reg] |= value;
	} else {
		outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
		mb();
		tmp = (codec->image[reg] & mask) | value;
		outb(tmp, AD1848P(codec, REG));
		codec->image[reg] = tmp;
		mb();
	}
}

void snd_ad1848_out(ad1848_t * codec,
			   unsigned char reg,
			   unsigned char value)
{
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("snd_ad1848_out: auto calibration time out - reg = 0x%x, value = 0x%x\n", reg, value);
#endif
	outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
	outb(codec->image[reg] = value, AD1848P(codec, REG));
	mb();
#if 0
	printk("codec out - reg 0x%x = 0x%x\n", codec->mce_bit | reg, value);
#endif
}

static void snd_ad1848_dout(ad1848_t * codec,
			    unsigned char reg,
			    unsigned char value)
{
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		udelay(100);
	outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
	outb(value, AD1848P(codec, REG));
	mb();
}

static unsigned char snd_ad1848_in(ad1848_t * codec, unsigned char reg)
{
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("snd_ad1848_in: auto calibration time out - reg = 0x%x\n", reg);
#endif
	outb(codec->mce_bit | reg, AD1848P(codec, REGSEL));
	mb();
	return inb(AD1848P(codec, REG));
}

#ifdef CONFIG_SND_DEBUG

void snd_ad1848_debug(ad1848_t * codec)
{
	printk("AD1848 REGS:      INDEX = 0x%02x  ", inb(AD1848P(codec, REGSEL)));
	printk("                 STATUS = 0x%02x\n", inb(AD1848P(codec, STATUS)));
	printk("  0x00: left input      = 0x%02x  ", snd_ad1848_in(codec, 0x00));
	printk("  0x08: playback format = 0x%02x\n", snd_ad1848_in(codec, 0x08));
	printk("  0x01: right input     = 0x%02x  ", snd_ad1848_in(codec, 0x01));
	printk("  0x09: iface (CFIG 1)  = 0x%02x\n", snd_ad1848_in(codec, 0x09));
	printk("  0x02: AUXA left       = 0x%02x  ", snd_ad1848_in(codec, 0x02));
	printk("  0x0a: pin control     = 0x%02x\n", snd_ad1848_in(codec, 0x0a));
	printk("  0x03: AUXA right      = 0x%02x  ", snd_ad1848_in(codec, 0x03));
	printk("  0x0b: init & status   = 0x%02x\n", snd_ad1848_in(codec, 0x0b));
	printk("  0x04: AUXB left       = 0x%02x  ", snd_ad1848_in(codec, 0x04));
	printk("  0x0c: revision & mode = 0x%02x\n", snd_ad1848_in(codec, 0x0c));
	printk("  0x05: AUXB right      = 0x%02x  ", snd_ad1848_in(codec, 0x05));
	printk("  0x0d: loopback        = 0x%02x\n", snd_ad1848_in(codec, 0x0d));
	printk("  0x06: left output     = 0x%02x  ", snd_ad1848_in(codec, 0x06));
	printk("  0x0e: data upr count  = 0x%02x\n", snd_ad1848_in(codec, 0x0e));
	printk("  0x07: right output    = 0x%02x  ", snd_ad1848_in(codec, 0x07));
	printk("  0x0f: data lwr count  = 0x%02x\n", snd_ad1848_in(codec, 0x0f));
}

#endif

/*
 *  AD1848 detection / MCE routines
 */

static void snd_ad1848_mce_up(ad1848_t * codec)
{
	unsigned long flags;
	int timeout;

	for (timeout = 250; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		udelay(100);
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("ad1848_mce_up - auto calibration time out (0)\n");
#endif
	spin_lock_irqsave(&codec->reg_lock, flags);
	codec->mce_bit |= AD1848_MCE;
	timeout = inb(AD1848P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_ad1848_up [0x%x]: serious init problem - codec still busy\n", codec->port);
	if (!(timeout & AD1848_MCE))
		outb(codec->mce_bit | (timeout & 0x1f), AD1848P(codec, REGSEL));
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

static void snd_ad1848_mce_down(ad1848_t * codec)
{
	unsigned long flags;
	int timeout;
	signed long time;

	spin_lock_irqsave(&codec->reg_lock, flags);
	for (timeout = 5; timeout > 0; timeout--)
		inb(AD1848P(codec, REGSEL));
	/* end of cleanup sequence */
	for (timeout = 12000; timeout > 0 && (inb(AD1848P(codec, REGSEL)) & AD1848_INIT); timeout--)
		udelay(100);
#if 0
	printk("(1) timeout = %i\n", timeout);
#endif
#ifdef CONFIG_SND_DEBUG
	if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
		snd_printk("ad1848_mce_down [0x%x] - auto calibration time out (0)\n", AD1848P(codec, REGSEL));
#endif
	codec->mce_bit &= ~AD1848_MCE;
	timeout = inb(AD1848P(codec, REGSEL));
	outb(codec->mce_bit | (timeout & 0x1f), AD1848P(codec, REGSEL));
	if (timeout == 0x80)
		snd_printk("snd_ad1848_down [0x%x]: serious init problem - codec still busy\n", codec->port);
	if ((timeout & AD1848_MCE) == 0) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return;
	}
	/* calibration process */

	for (timeout = 500; timeout > 0 && (snd_ad1848_in(codec, AD1848_TEST_INIT) & AD1848_CALIB_IN_PROGRESS) == 0; timeout--);
	if ((snd_ad1848_in(codec, AD1848_TEST_INIT) & AD1848_CALIB_IN_PROGRESS) == 0) {
		snd_printd("ad1848_mce_down - auto calibration time out (1)\n");
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return;
	}
#if 0
	printk("(2) timeout = %i, jiffies = %li\n", timeout, jiffies);
#endif
	time = HZ / 4;
	while (snd_ad1848_in(codec, AD1848_TEST_INIT) & AD1848_CALIB_IN_PROGRESS) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		if (time <= 0) {
			snd_printk("ad1848_mce_down - auto calibration time out (2)\n");
			return;
		}
		set_current_state(TASK_INTERRUPTIBLE);
		time = schedule_timeout(time);
		spin_lock_irqsave(&codec->reg_lock, flags);
	}
#if 0
	printk("(3) jiffies = %li\n", jiffies);
#endif
	time = HZ / 10;
	while (inb(AD1848P(codec, REGSEL)) & AD1848_INIT) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		if (time <= 0) {
			snd_printk("ad1848_mce_down - auto calibration time out (3)\n");
			return;
		}
		set_current_state(TASK_INTERRUPTIBLE);
		time = schedule_timeout(time);
		spin_lock_irqsave(&codec->reg_lock, flags);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
#if 0
	printk("(4) jiffies = %li\n", jiffies);
	snd_printk("ad1848_mce_down - exit = 0x%x\n", inb(AD1848P(codec, REGSEL)));
#endif
}

static unsigned int snd_ad1848_get_count(unsigned char format,
				         unsigned int size)
{
	switch (format & 0xe0) {
	case AD1848_LINEAR_16:
		size >>= 1;
		break;
	}
	if (format & AD1848_STEREO)
		size >>= 1;
	return size;
}

static int snd_ad1848_trigger(ad1848_t * codec, unsigned char what,
			      int channel, int cmd)
{
	unsigned long flags;
	int result = 0;

#if 0
	printk("codec trigger!!! - what = %i, enable = %i, status = 0x%x\n", what, enable, inb(AD1848P(card, STATUS)));
#endif
	spin_lock_irqsave(&codec->reg_lock, flags);
	if (cmd == SND_PCM_TRIGGER_GO) {
		if (codec->image[AD1848_IFACE_CTRL] & what) {
			spin_unlock_irqrestore(&codec->reg_lock, flags);
			return 0;
		}
		snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL] |= what);
	} else if (cmd == SND_PCM_TRIGGER_STOP) {
		if (!(codec->image[AD1848_IFACE_CTRL] & what)) {
			spin_unlock_irqrestore(&codec->reg_lock, flags);
			return 0;
		}
		snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL] &= ~what);
	} else {
		result = -EINVAL;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

/*
 *  CODEC I/O
 */

static unsigned char snd_ad1848_get_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_ad1848_freq[i].hertz)
			return snd_ad1848_freq[i].bits;
	return snd_ad1848_freq[13].bits;
}

static unsigned int snd_ad1848_get_real_freq(unsigned int freq)
{				/* freq in Hz */
	int i;

	freq /= 1000;
#if 0
	snd_printk("pcm_rate: %d\n", freq);
#endif
	if (freq > 48)
		freq = 48;
	for (i = 0; i < 14; i++)
		if (freq <= snd_ad1848_freq[i].hertz)
			return snd_ad1848_freq[i].rate;
	return snd_ad1848_freq[13].rate;
}

static int snd_ad1848_playback_ioctl(void *private_data,
				    snd_pcm_subchn_t * subchn,
				    unsigned int cmd, unsigned long *arg)
{
	int result;
	
	result = snd_pcm_lib_ioctl(private_data, subchn, cmd, arg);
	if (result < 0)
		return result;
	if (cmd == SND_PCM_IOCTL1_PARAMS)
		subchn->runtime->format.rate = snd_ad1848_get_real_freq(subchn->runtime->format.rate);
	return 0;
}

static int snd_ad1848_capture_ioctl(void *private_data,
				    snd_pcm_subchn_t * subchn,
				    unsigned int cmd, unsigned long *arg)
{
	return snd_ad1848_playback_ioctl(private_data, subchn, cmd, arg);
}

static unsigned char snd_ad1848_get_format(int format, int voices)
{
	unsigned char rformat;

	rformat = AD1848_LINEAR_8;
	switch (format) {
	case SND_PCM_SFMT_A_LAW:	rformat = AD1848_ALAW_8; break;
	case SND_PCM_SFMT_MU_LAW:	rformat = AD1848_ULAW_8; break;
	case SND_PCM_SFMT_S16_LE:	rformat = AD1848_LINEAR_16; break;
	}
	if (voices > 1)
		rformat |= AD1848_STEREO;
#if 0
	snd_printk("snd_ad1848_get_format: 0x%x (mode=0x%x)\n", format, mode);
#endif
	return rformat;
}

static void snd_ad1848_calibrate_mute(ad1848_t * codec, int mute)
{
	unsigned long flags;
	
	mute = mute ? 1 : 0;
	spin_lock_irqsave(&codec->reg_lock, flags);
	if (codec->calibrate_mute == mute) {
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return;
	}
	if (!mute) {
		snd_ad1848_dout(codec, AD1848_LEFT_INPUT, codec->image[AD1848_LEFT_INPUT]);
		snd_ad1848_dout(codec, AD1848_RIGHT_INPUT, codec->image[AD1848_RIGHT_INPUT]);
	}
	snd_ad1848_dout(codec, AD1848_AUX1_LEFT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX1_LEFT_INPUT]);
	snd_ad1848_dout(codec, AD1848_AUX1_RIGHT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX1_RIGHT_INPUT]);
	snd_ad1848_dout(codec, AD1848_AUX2_LEFT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX2_LEFT_INPUT]);
	snd_ad1848_dout(codec, AD1848_AUX2_RIGHT_INPUT, mute ? 0x80 : codec->image[AD1848_AUX2_RIGHT_INPUT]);
	snd_ad1848_dout(codec, AD1848_LEFT_OUTPUT, mute ? 0x80 : codec->image[AD1848_LEFT_OUTPUT]);
	snd_ad1848_dout(codec, AD1848_RIGHT_OUTPUT, mute ? 0x80 : codec->image[AD1848_RIGHT_OUTPUT]);
	codec->calibrate_mute = mute;
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

static void snd_ad1848_set_data_format(ad1848_t * codec, snd_pcm_subchn_t *subchn)
{
	snd_pcm_runtime_t *runtime;

	if (subchn == NULL) {
		codec->image[AD1848_DATA_FORMAT] = 0x20;
	} else {
		runtime = subchn->runtime;
		codec->image[AD1848_DATA_FORMAT] =
		    snd_ad1848_get_format(runtime->format.format, runtime->format.voices) |
		    snd_ad1848_get_freq(runtime->format.rate);
	}
#if 0
	snd_printk(">>> pmode = 0x%x, dfr = 0x%x\n", pchn->mode, codec->image[AD1848_DATA_FORMAT]);
#endif
}

static int snd_ad1848_open(ad1848_t * codec, unsigned int mode)
{
	unsigned long flags;

	down(&codec->open_mutex);
	if (codec->mode & AD1848_MODE_OPEN) {
		up(&codec->open_mutex);
		return -1;
	}
	snd_ad1848_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_ad1848_open: (1)\n");
#endif
	snd_ad1848_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_PLAYBACK_ENABLE | AD1848_PLAYBACK_PIO |
			     AD1848_CAPTURE_ENABLE | AD1848_CAPTURE_PIO |
			     AD1848_CALIB_MODE);
	codec->image[AD1848_IFACE_CTRL] |= AD1848_AUTOCALIB;
	snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_ad1848_open: (2)\n");
#endif

	snd_ad1848_set_data_format(codec, NULL);

	snd_ad1848_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_ad1848_out(codec, AD1848_DATA_FORMAT, codec->image[AD1848_DATA_FORMAT]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_mce_down(codec);

#ifdef SND_DEBUG_MCE
	snd_printk("snd_ad1848_open: (3)\n");
#endif

	/* ok. now enable and ack CODEC IRQ */
	spin_lock_irqsave(&codec->reg_lock, flags);
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	codec->image[AD1848_PIN_CTRL] |= AD1848_IRQ_ENABLE;
	snd_ad1848_out(codec, AD1848_PIN_CTRL, codec->image[AD1848_PIN_CTRL]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	codec->mode = mode;
	up(&codec->open_mutex);

	return 0;
}

static void snd_ad1848_close(ad1848_t * codec)
{
	unsigned long flags;

	down(&codec->open_mutex);
	if (!codec->mode) {
		up(&codec->open_mutex);
		return;
	}
	/* disable IRQ */
	spin_lock_irqsave(&codec->reg_lock, flags);
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	codec->image[AD1848_PIN_CTRL] &= ~AD1848_IRQ_ENABLE;
	snd_ad1848_out(codec, AD1848_PIN_CTRL, codec->image[AD1848_PIN_CTRL]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	/* now disable capture & playback */

	snd_ad1848_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_PLAYBACK_ENABLE | AD1848_PLAYBACK_PIO |
			     AD1848_CAPTURE_ENABLE | AD1848_CAPTURE_PIO);
	snd_ad1848_out(codec, AD1848_IFACE_CTRL, codec->image[AD1848_IFACE_CTRL]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_mce_down(codec);

	/* clear IRQ again */
	spin_lock_irqsave(&codec->reg_lock, flags);
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	outb(0, AD1848P(codec, STATUS));	/* clear IRQ */
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	codec->mode = 0;
	up(&codec->open_mutex);
}

/*
 *  ok.. exported functions..
 */

static int snd_ad1848_playback_trigger(void *private_data,
				       snd_pcm_subchn_t * subchn,
				       int cmd)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);
	return snd_ad1848_trigger(codec, AD1848_PLAYBACK_ENABLE, SND_PCM_CHANNEL_PLAYBACK, cmd);
}

static int snd_ad1848_capture_trigger(void *private_data,
				      snd_pcm_subchn_t * subchn,
				      int cmd)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);
	return snd_ad1848_trigger(codec, AD1848_CAPTURE_ENABLE, SND_PCM_CHANNEL_CAPTURE, cmd);
}

static int snd_ad1848_playback_prepare(void *private_data,
				       snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	unsigned long flags;
	unsigned int size = snd_pcm_lib_transfer_size(subchn);
	unsigned int count = snd_pcm_lib_transfer_fragment(subchn);

	// Note: offset is always 0 if AUTO DMA
	codec->dma_size = size;
	snd_ad1848_calibrate_mute(codec, 1);
	snd_ad1848_set_data_format(codec, subchn);
	// snd_printk("ad1848_start_playback - dma = %i, buffer = 0x%lx, size = 0x%x, offset = 0x%x, count = 0x%x\n", codec->dma1, (long) buffer, size, offset, count);
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_PLAYBACK_ENABLE | AD1848_PLAYBACK_PIO);
	snd_dma_program(codec->dma, runtime->dma_area->buf, size, DMA_MODE_WRITE | DMA_AUTOINIT);
	count = snd_ad1848_get_count(codec->image[AD1848_DATA_FORMAT], count) - 1;
	snd_ad1848_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_ad1848_out(codec, AD1848_DATA_FORMAT, codec->image[AD1848_DATA_FORMAT]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_mce_down(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_ad1848_out(codec, AD1848_DATA_LWR_CNT, (unsigned char) count);
	snd_ad1848_out(codec, AD1848_DATA_UPR_CNT, (unsigned char) (count >> 8));
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_calibrate_mute(codec, 0);
	// snd_ad1848_debug(codec);
	return 0;
}

static int snd_ad1848_capture_prepare(void *private_data,
				      snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	unsigned long flags;
	unsigned int size = snd_pcm_lib_transfer_size(subchn);
	unsigned int count = snd_pcm_lib_transfer_fragment(subchn);

	codec->dma_size = size;
	snd_ad1848_calibrate_mute(codec, 1);
	snd_ad1848_set_data_format(codec, subchn);
	// snd_printk("ad1848_start_capture: start - buffer = 0x%lx, offset = 0x%x, size = 0x%x, count = 0x%x\n", (long) buffer, offset, size, count);
	codec->image[AD1848_IFACE_CTRL] &= ~(AD1848_CAPTURE_ENABLE | AD1848_CAPTURE_PIO);
	snd_dma_program(codec->dma, runtime->dma_area->buf, size, DMA_MODE_READ | DMA_AUTOINIT);
	count = snd_ad1848_get_count(codec->image[AD1848_DATA_FORMAT], count) - 1;
	snd_ad1848_mce_up(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_ad1848_out(codec, AD1848_DATA_FORMAT, codec->image[AD1848_DATA_FORMAT]);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_mce_down(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_ad1848_out(codec, AD1848_DATA_LWR_CNT, (unsigned char) count);
	snd_ad1848_out(codec, AD1848_DATA_UPR_CNT, (unsigned char) (count >> 8));
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_calibrate_mute(codec, 0);
	return 0;
}

void snd_ad1848_interrupt(snd_pcm_t * pcm, unsigned char status)
{
	unsigned long flags;
	ad1848_t *codec = snd_magic_cast(ad1848_t, pcm->private_data, );

#if 0
	snd_printk("snd_ad1848_interrupt: status=0x%x\n", status);
#endif
	if (codec == NULL)
		return;

	if (codec->mode & AD1848_MODE_PLAY)
		snd_pcm_transfer_done(codec->playback_subchn);
	if (codec->mode & AD1848_MODE_CAPTURE)
		snd_pcm_transfer_done(codec->capture_subchn);
	spin_lock_irqsave(&codec->reg_lock, flags);
	outb(0, AD1848P(codec, STATUS));	/* clear global interrupt bit */
	spin_unlock_irqrestore(&codec->reg_lock, flags);
}

static unsigned int snd_ad1848_playback_pointer(void *private_data,
						snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);
	
	if (!(codec->image[AD1848_IFACE_CTRL] & AD1848_PLAYBACK_ENABLE))
		return 0;
	return codec->dma_size - snd_dma_residue(codec->dma);
}

static unsigned int snd_ad1848_capture_pointer(void *private_data,
					       snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);

	if (!(codec->image[AD1848_IFACE_CTRL] & AD1848_CAPTURE_ENABLE))
		return 0;
	return codec->dma_size - snd_dma_residue(codec->dma);
}

/*

 */

static int snd_ad1848_probe(snd_pcm_t * pcm)
{
	unsigned long flags;
	ad1848_t *codec;
	int i, id, rev, ad1847;
	unsigned char *ptr;

	codec = snd_magic_cast(ad1848_t, pcm->private_data, -ENXIO);
	snd_debug_check(codec == NULL, -EINVAL);

#if 0
	snd_ad1848_debug(codec);
#endif
	id = ad1847 = 0;
	for (i = 0; i < 1000; i++) {
		mb();
		if (inb(AD1848P(codec, REGSEL)) & AD1848_INIT)
			udelay(500);
		else {
			spin_lock_irqsave(&codec->reg_lock, flags);
			snd_ad1848_out(codec, AD1848_MISC_INFO, 0x00);
			snd_ad1848_out(codec, AD1848_LEFT_INPUT, 0xaa);
			snd_ad1848_out(codec, AD1848_RIGHT_INPUT, 0x45);
			rev = snd_ad1848_in(codec, AD1848_RIGHT_INPUT);
			if (rev == 0x65) {
				id = 1;
				ad1847 = 1;
				break;
			}
			if (snd_ad1848_in(codec, AD1848_LEFT_INPUT) == 0xaa && rev == 0x45) {
				id = 1;
				break;
			}
			spin_unlock_irqrestore(&codec->reg_lock, flags);
		}
	}
	if (id != 1)
		return -ENODEV;	/* no valid device found */
	if (codec->hardware == AD1848_HW_DETECT) {
		if (ad1847) {
			codec->hardware = AD1848_HW_AD1847;
			strcpy(pcm->name, "AD1847");
		} else {
			codec->hardware = AD1848_HW_AD1848;
			strcpy(pcm->name, "AD1848");
			rev = snd_ad1848_in(codec, AD1848_MISC_INFO);
			if (rev & 0x80) {
				codec->hardware = AD1848_HW_CS4248;
				strcpy(pcm->name, "CS4248");
			} else if (rev & 0x0a) {
				codec->hardware = AD1848_HW_CMI8330;
				strcpy(pcm->name, "CMI8330/C3D");
			}
		}
	}
	spin_lock_irqsave(&codec->reg_lock, flags);
	inb(AD1848P(codec, STATUS));	/* clear any pendings IRQ */
	outb(0, AD1848P(codec, STATUS));
	mb();
	spin_unlock_irqrestore(&codec->reg_lock, flags);

	codec->image[AD1848_MISC_INFO] = 0x00;
	codec->image[AD1848_IFACE_CTRL] =
	    (codec->image[AD1848_IFACE_CTRL] & ~AD1848_SINGLE_DMA) | AD1848_SINGLE_DMA;
	ptr = (unsigned char *) &codec->image;
	snd_ad1848_mce_down(codec);
	spin_lock_irqsave(&codec->reg_lock, flags);
	for (i = 0; i < 16; i++)	/* ok.. fill all AD1848 registers */
		snd_ad1848_out(codec, i, *ptr++);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	snd_ad1848_mce_up(codec);
	snd_ad1848_mce_down(codec);
	return 0;		/* all things are ok.. */
}

/*

 */

static snd_pcm_hardware_t snd_ad1848_playback =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* formats */
	SND_PCM_RATE_8000_48000, /* supported rates */
	5510,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	(128*1024),		/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	4,			/* transfer block size */
	snd_ad1848_playback_ioctl,
	snd_ad1848_playback_prepare,
	snd_ad1848_playback_trigger,
	snd_ad1848_playback_pointer
};

static snd_pcm_hardware_t snd_ad1848_capture =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_MU_LAW | SND_PCM_FMT_A_LAW |
	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,	/* formats */
	SND_PCM_RATE_8000_48000, /* supported rates */
	5510,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	(128*1024),		/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	4,			/* transfer block size */
	snd_ad1848_capture_ioctl,
	snd_ad1848_capture_prepare,
	snd_ad1848_capture_trigger,
	snd_ad1848_capture_pointer
};

/*

 */

static int snd_ad1848_playback_open(void *private_data,
				    snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	int err;

	if ((err = snd_pcm_dma_alloc(subchn, codec->dmaptr, "AD1848 (playback)")) < 0)
		return err;
	if (snd_ad1848_open(codec, AD1848_MODE_PLAY) < 0) {
		snd_pcm_dma_free(subchn);
		return -EAGAIN;
	}
	codec->playback_subchn = subchn;
	runtime->hw = &snd_ad1848_playback;
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->me_playback);
	return 0;
}

static int snd_ad1848_capture_open(void *private_data,
				   snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	int err;

	if ((err = snd_pcm_dma_alloc(subchn, codec->dmaptr, "AD1848 (capture)")) < 0)
		return err;
	if (snd_ad1848_open(codec, AD1848_MODE_CAPTURE) < 0) {
		snd_pcm_dma_free(subchn);
		return -EAGAIN;
	}
	codec->capture_subchn = subchn;
	runtime->hw = &snd_ad1848_capture;
	snd_pcm_set_mixer(subchn, codec->mixer->device, codec->me_capture);
	return 0;
}

static int snd_ad1848_playback_close(void *private_data,
				     snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);

	codec->mode &= ~AD1848_MODE_PLAY;
	codec->playback_subchn = NULL;
	snd_ad1848_close(codec);
	snd_pcm_dma_free(subchn);
	return 0;
}

static int snd_ad1848_capture_close(void *private_data,
				    snd_pcm_subchn_t * subchn)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, private_data, -ENXIO);

	codec->mode &= ~AD1848_MODE_CAPTURE;
	codec->capture_subchn = NULL;
	snd_ad1848_close(codec);
	snd_pcm_dma_free(subchn);
	return 0;
}

int snd_ad1848_new_pcm(snd_card_t * card, int device,
		       unsigned short port,
		       snd_irq_t * irqptr,
		       snd_dma_t * dmaptr,
		       unsigned short hardware,
		       snd_pcm_t ** rpcm)
{
	snd_pcm_t *pcm;
	ad1848_t *codec;
	char *str;
	int err;

	*rpcm = NULL;
	if ((err = snd_pcm_new(card, "AD1848", device, 1, 1, &pcm)) < 0)
		return err;
	codec = snd_magic_kcalloc(ad1848_t, 0, GFP_KERNEL);
	if (codec == NULL) {
		snd_device_free(card, pcm);
		return -ENOMEM;
	}
	spin_lock_init(&codec->reg_lock);
	init_MUTEX(&codec->open_mutex);
	codec->pcm = pcm;
	codec->card = pcm->card;
	codec->port = port;
	codec->irq = irqptr->irq;
	codec->irqptr = irqptr;
	codec->dmaptr = dmaptr;
	codec->dma = dmaptr->dma;
	codec->hardware = hardware;
	memcpy(&codec->image, &snd_ad1848_original_image, sizeof(snd_ad1848_original_image));

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_ad1848_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_ad1848_playback_close;

	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_ad1848_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_ad1848_capture_close;

	pcm->private_data = codec;
	pcm->private_free = _snd_magic_kfree;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE;
	switch (codec->hardware) {
	case AD1848_HW_AD1847:
		str = "AD1847";
		break;
	case AD1848_HW_AD1848:
		str = "AD1848";
		break;
	case AD1848_HW_CS4248:
		str = "CS4248";
		break;
	case AD1848_HW_CMI8330:
		str = "CMI8330/C3D";
		break;
	default:
		str = "???";
	}
	strcpy(pcm->name, str);
	if (snd_ad1848_probe(pcm) < 0) {
		snd_device_free(card, pcm);
		return -ENODEV;
	}
	*rpcm = pcm;
	return 0;
}

/*
 *  MIXER part
 */

static int snd_ad1848_mixer_get_recsrc(ad1848_t *codec, snd_kmixer_element_t *element)
{
	if (element == codec->me_mux_line1)
		return AD1848_MIXS_LINE1;
	if (element == codec->me_mux_aux1)
		return AD1848_MIXS_AUX1;
	if (element == codec->me_mux_line2)
		return AD1848_MIXS_LINE2;
	if (element == codec->me_mux_mix)
		return AD1848_MIXS_ALL;
	return -EINVAL;
}

static snd_kmixer_element_t *snd_ad1848_mixer_get_recsrc_element(ad1848_t *codec, int value)
{
	value &= AD1848_MIXS_ALL;
	switch (value) {
	case AD1848_MIXS_LINE1:
		return codec->me_mux_line1;
	case AD1848_MIXS_AUX1:
		return codec->me_mux_aux1;
	case AD1848_MIXS_LINE2:
		return codec->me_mux_line2;
	case AD1848_MIXS_ALL:
		return codec->me_mux_mix;
	}
	return NULL;
}

static int snd_ad1848_mixer_mux(snd_kmixer_element_t *element, int w_flag, snd_kmixer_element_t **elements)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, element->private_data, -ENXIO);
	unsigned long flags;
	int change = 0;
	int left, right, nleft, nright;

	spin_lock_irqsave(&codec->reg_lock, flags);
	left = codec->image[AD1848_LEFT_INPUT] & AD1848_MIXS_ALL;
	right = codec->image[AD1848_RIGHT_INPUT] & AD1848_MIXS_ALL;
	if (w_flag) {
		nleft = snd_ad1848_mixer_get_recsrc(codec, elements[0]);
		nright = snd_ad1848_mixer_get_recsrc(codec, elements[1]);
		if (nleft >= 0 && nright >= 0) {
			change = nleft != left || nright != right;
			snd_ad1848_outm(codec, AD1848_LEFT_INPUT, (unsigned char)~AD1848_MIXS_ALL, (unsigned char)nleft);
			snd_ad1848_outm(codec, AD1848_RIGHT_INPUT, (unsigned char)~AD1848_MIXS_ALL, (unsigned char)nright);
		} else {
			change = -EINVAL;
		}
	} else {
		elements[0] = snd_ad1848_mixer_get_recsrc_element(codec, left);
		elements[1] = snd_ad1848_mixer_get_recsrc_element(codec, right);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;
}

int snd_ad1848_mixer_stereo_volume(snd_kmixer_element_t *element,
				   int w_flag, int *voices,
				   int max, int invert, int shift,
				   unsigned char left_reg,
				   unsigned char right_reg)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, element->private_data, -ENXIO);
	int change = 0, left, right;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	left = (codec->image[left_reg] >> shift) & max;
	right = (codec->image[right_reg] >> shift) & max;
	if (!w_flag) {
		if (invert) {
			voices[0] = max - left;
			voices[1] = max - right;
		} else {
			voices[0] = left;
			voices[1] = right;
		}
	} else {
		if (invert) {
			change = max - left != voices[0] || max - right != voices[1];
		} else {
			change = left != voices[0] || right != voices[1];
		}
		left = voices[0]; right = voices[1];
		if (invert) {
			left = max - left;
			right = max - right;
		}
		snd_ad1848_outm(codec, left_reg, ~(max << shift), left << shift);
		snd_ad1848_outm(codec, right_reg, ~(max << shift), right << shift);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

int snd_ad1848_mixer_mono_volume(snd_kmixer_element_t *element,
				 int w_flag, int *voices,
				 int max, int invert, int shift,
				 unsigned char reg)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, element->private_data, -ENXIO);
	int change = 0, val;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	val = (codec->image[reg] >> shift) & max;
	if (!w_flag) {
		voices[0] = invert ? max - val : val;
	} else {
		change = (invert ? max - val : val) != voices[0];
		val = voices[0];
		if (invert)
			val = max - val;
		snd_ad1848_outm(codec, reg, ~(max << shift), val << shift);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

int snd_ad1848_mixer_stereo_switch(snd_kmixer_element_t *element,
				   int w_flag, unsigned int *bitmap,
				   int bit, int invert,
				   unsigned char left_reg,
				   unsigned char right_reg)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, element->private_data, -ENXIO);
	int change = 0, left, right;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	left = (codec->image[left_reg] >> bit) & 1;
	right = (codec->image[right_reg] >> bit) & 1;
	if (!w_flag) {
		if (invert) {
			snd_mixer_set_bit(bitmap, 0, left ^ 1);
			snd_mixer_set_bit(bitmap, 1, right ^ 1);
		} else {
			snd_mixer_set_bit(bitmap, 0, left);
			snd_mixer_set_bit(bitmap, 1, right);
		}
	} else {
		if (invert) {
			change = (left ^ 1) != snd_mixer_get_bit(bitmap, 0) ||
			         (right ^ 1) != snd_mixer_get_bit(bitmap, 1);
		} else {
			change = left != snd_mixer_get_bit(bitmap, 0) ||
			         right != snd_mixer_get_bit(bitmap, 1);
		}
		left = snd_mixer_get_bit(bitmap, 0);
		right = snd_mixer_get_bit(bitmap, 1);
		if (invert) {
			left ^= 1;
			right ^= 1;
		}
		snd_ad1848_outm(codec, left_reg, ~(1 << bit), left << bit);
		snd_ad1848_outm(codec, right_reg, ~(1 << bit), right << bit);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

int snd_ad1848_mixer_mono_switch(snd_kmixer_element_t *element,
				 int w_flag, int *value,
				 int bit, int invert,
				 unsigned char reg)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, element->private_data, -ENXIO);
	int change = 0, val;
	unsigned long flags;

	spin_lock_irqsave(&codec->reg_lock, flags);
	val = (codec->image[reg] >> bit) & 1;
	if (!w_flag) {
		*value = invert ? val ^ 1 : val;
	} else {
		change = (invert ? val ^ 1 : val) != (*value != 0);
		val = value != 0;
		if (invert)
			val ^= 1;
		snd_ad1848_outm(codec, reg, ~(1 << bit), val << bit);
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return change;	
}

static int snd_ad1848_mixer_aux1_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_ad1848_mixer_stereo_volume(element,
					      w_flag, voices,
					      31, 1, 0,
					      AD1848_AUX1_LEFT_INPUT,
					      AD1848_AUX1_RIGHT_INPUT);
}

static int snd_ad1848_mixer_aux1_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_ad1848_mixer_stereo_switch(element,
					      w_flag, bitmap,
					      7, 1,
					      AD1848_AUX1_LEFT_INPUT,
					      AD1848_AUX1_RIGHT_INPUT);
}

static int snd_ad1848_mixer_aux2_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_ad1848_mixer_stereo_volume(element,
					      w_flag, voices,
					      31, 1, 0,
					      AD1848_AUX2_LEFT_INPUT,
					      AD1848_AUX2_RIGHT_INPUT);
}

static int snd_ad1848_mixer_aux2_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_ad1848_mixer_stereo_switch(element,
					      w_flag, bitmap,
					      7, 1,
					      AD1848_AUX2_LEFT_INPUT,
					      AD1848_AUX2_RIGHT_INPUT);
}

static int snd_ad1848_mixer_igain_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_ad1848_mixer_stereo_volume(element,
					      w_flag, voices,
					      15, 0, 0,
					      AD1848_LEFT_INPUT,
					      AD1848_RIGHT_INPUT);
}

static int snd_ad1848_mixer_loopback_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_ad1848_mixer_mono_volume(element,
					    w_flag, voices,
					    63, 1, 2,
					    AD1848_LOOPBACK);
}

static int snd_ad1848_mixer_loopback_switch(snd_kmixer_element_t *element, int w_flag, int *value)
{
	return snd_ad1848_mixer_mono_switch(element,
					    w_flag, value,
					    0, 0,
					    AD1848_LOOPBACK);
}

static int snd_ad1848_mixer_dac_volume(snd_kmixer_element_t *element, int w_flag, int *voices)
{
	return snd_ad1848_mixer_stereo_volume(element,
					      w_flag, voices,
					      63, 1, 0,
					      AD1848_LEFT_OUTPUT,
					      AD1848_RIGHT_OUTPUT);
}

static int snd_ad1848_mixer_dac_switch(snd_kmixer_element_t *element, int w_flag, unsigned int *bitmap)
{
	return snd_ad1848_mixer_stereo_switch(element,
					      w_flag, bitmap,
					      7, 1,
					      AD1848_LEFT_OUTPUT,
					      AD1848_RIGHT_OUTPUT);
}

int snd_ad1848_mixer_group_ctrl1(snd_kmixer_group_t * group,
				 snd_kmixer_file_t * file,
				 int w_flag,
				 snd_mixer_group_t * ugroup,
				 snd_mixer_volume1_control_t *volume1,
				 snd_kmixer_element_t *volume1_element,
				 int max,
				 snd_mixer_sw1_control_t *sw1,
				 snd_kmixer_element_t *sw1_element,
				 snd_kmixer_element_t *mux_in)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, group->private_data, -ENXIO);
	int voices[2];
	snd_kmixer_element_t *elements[2];
	unsigned int bitmap;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = 0;
		ugroup->channels = SND_MIXER_CHN_MASK_STEREO;
		if (volume1) {
			ugroup->caps |= SND_MIXER_GRPCAP_VOLUME;
			volume1(volume1_element, 0, voices);
			ugroup->volume.names.front_left = voices[0];
			ugroup->volume.names.front_right = voices[1];
			ugroup->min = 0;
			ugroup->max = max;
		}
		if (sw1) {
			ugroup->caps |= SND_MIXER_GRPCAP_MUTE;
			sw1(sw1_element, 0, &bitmap);
			ugroup->mute = 0;
			if (!snd_mixer_get_bit(&bitmap, 0))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (!snd_mixer_get_bit(&bitmap, 1))
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		}
		if (mux_in) {
			ugroup->caps |= SND_MIXER_GRPCAP_CAPTURE | SND_MIXER_GRPCAP_EXCL_CAPTURE;
			ugroup->capture_group = 1;
			snd_ad1848_mixer_mux(codec->me_mux, 0, elements);
			ugroup->capture = 0;
			if (elements[0] == mux_in)
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_LEFT;
			if (elements[1] == mux_in)
				ugroup->capture |= SND_MIXER_CHN_MASK_FRONT_RIGHT;
		}
	} else {
		if (volume1) {
			voices[0] = ugroup->volume.names.front_left & max;
			voices[1] = ugroup->volume.names.front_right & max;
			if (volume1(volume1_element, 1, voices) > 0) {
				snd_mixer_element_value_change(file, volume1_element, 0);
				change = 1;
			}
		}
		if (sw1) {
			bitmap = 0;
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_LEFT))
				snd_mixer_set_bit(&bitmap, 0, 1);
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_FRONT_RIGHT))
				snd_mixer_set_bit(&bitmap, 1, 1);
			if (sw1(sw1_element, 1, &bitmap) > 0) {
				snd_mixer_element_value_change(file, sw1_element, 0);
				change = 1;
			}
		}
		if (mux_in) {
			snd_ad1848_mixer_mux(codec->me_mux, 0, elements);
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_LEFT)
				elements[0] = mux_in;
			if (ugroup->capture & SND_MIXER_CHN_MASK_FRONT_RIGHT)
				elements[1] = mux_in;
			if (snd_ad1848_mixer_mux(codec->me_mux, 1, elements) > 0) {
				snd_mixer_element_value_change_all_file(file, codec->me_mux, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_ad1848_mixer_group_ctrl2(snd_kmixer_group_t * group,
				        snd_kmixer_file_t * file,
				        int w_flag,
				        snd_mixer_group_t * ugroup,
				        snd_mixer_volume1_control_t *volume1,
				        snd_kmixer_element_t *volume1_element,
				        int max,
				        snd_mixer_sw2_control_t *sw2,
				        snd_kmixer_element_t *sw2_element)
{
	int voices[2];
	int value;
	int change = 0;

	if (!w_flag) {
		ugroup->caps = 0;
		ugroup->channels = SND_MIXER_CHN_MASK_MONO;
		if (volume1) {
			ugroup->caps |= SND_MIXER_GRPCAP_VOLUME;
			volume1(volume1_element, 0, voices);
			ugroup->volume.names.front_left = voices[0];
			ugroup->min = 0;
			ugroup->max = max;
		}
		if (sw2) {
			ugroup->caps |= SND_MIXER_GRPCAP_MUTE;
			sw2(sw2_element, 0, &value);
			ugroup->mute = 0;
			if (!value)
				ugroup->mute |= SND_MIXER_CHN_MASK_FRONT_LEFT;
		}
	} else {
		if (volume1) {
			voices[0] = ugroup->volume.names.front_left & max;
			if (volume1(volume1_element, 1, voices) > 0) {
				snd_mixer_element_value_change(file, volume1_element, 0);
				change = 1;
			}
		}
		if (sw2) {
			value = 0;
			if (!(ugroup->mute & SND_MIXER_CHN_MASK_MONO))
				value = 1;
			if (sw2(sw2_element, 1, &value) > 0) {
				snd_mixer_element_value_change(file, sw2_element, 0);
				change = 1;
			}
		}
	}
	return change;
}

static int snd_ad1848_mixer_group_aux1(snd_kmixer_group_t * group,
				       snd_kmixer_file_t * file,
				       int w_flag,
				       snd_mixer_group_t * ugroup)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, group->private_data, -ENXIO);

	return snd_ad1848_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_ad1848_mixer_aux1_volume,
					    codec->me_vol_aux1,
					    31,
					    snd_ad1848_mixer_aux1_switch,
					    codec->me_sw_aux1,
					    codec->me_mux_aux1);
}

static int snd_ad1848_mixer_group_aux2(snd_kmixer_group_t * group,
				       snd_kmixer_file_t * file,
				       int w_flag,
				       snd_mixer_group_t * ugroup)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, group->private_data, -ENXIO);

	return snd_ad1848_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_ad1848_mixer_aux2_volume,
					    codec->me_vol_aux2,
					    31,
					    snd_ad1848_mixer_aux2_switch,
					    codec->me_sw_aux2,
					    codec->me_mux_mix);
}

static int snd_ad1848_mixer_group_igain(snd_kmixer_group_t * group,
					snd_kmixer_file_t * file,
					int w_flag,
					snd_mixer_group_t * ugroup)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, group->private_data, -ENXIO);

	return snd_ad1848_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_ad1848_mixer_igain_volume,
					    codec->me_vol_igain,
					    15,
					    NULL,
					    NULL,
					    NULL);
}

static int snd_ad1848_mixer_group_loop(snd_kmixer_group_t * group,
				       snd_kmixer_file_t * file,
				       int w_flag,
				       snd_mixer_group_t * ugroup)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, group->private_data, -ENXIO);

	return snd_ad1848_mixer_group_ctrl2(group, file, w_flag, ugroup,
					    snd_ad1848_mixer_loopback_volume,
					    codec->me_vol_loop,
					    63,
					    snd_ad1848_mixer_loopback_switch,
					    codec->me_sw_loop);
}

static int snd_ad1848_mixer_group_pcm(snd_kmixer_group_t * group,
				      snd_kmixer_file_t * file,
				      int w_flag,
				      snd_mixer_group_t * ugroup)
{
	ad1848_t *codec = snd_magic_cast(ad1848_t, group->private_data, -ENXIO);

	return snd_ad1848_mixer_group_ctrl1(group, file, w_flag, ugroup,
					    snd_ad1848_mixer_dac_volume,
					    codec->me_vol_pcm,
					    63,
					    snd_ad1848_mixer_dac_switch,
					    codec->me_sw_pcm,
					    NULL);
}

int snd_ad1848_new_mixer(snd_pcm_t * pcm, int device, snd_kmixer_t ** rmixer)
{
	ad1848_t *codec;
	snd_kmixer_t *mixer;
	snd_kmixer_element_t *element;
	snd_kmixer_group_t *group;
	int err;
	static struct snd_mixer_element_volume1_range igain_range[2] = {
		{0, 15, 0, 2250},
		{0, 15, 0, 2250}
	};
	static struct snd_mixer_element_volume1_range aux_range[2] = {
		{0, 31, -3450, 1200},
		{0, 31, -3450, 1200}
	};
	static struct snd_mixer_element_volume1_range pcm_range[2] = {
		{0, 63, -9450, 0},
		{0, 63, -9450, 0}
	};

	snd_debug_check(rmixer == NULL, -EINVAL);
	*rmixer = NULL;
	snd_debug_check(pcm == NULL || pcm->card == NULL, -EINVAL);
	codec = snd_magic_cast(ad1848_t, pcm->private_data, -ENXIO);
	snd_debug_check(codec == NULL, -EINVAL);
	if ((err = snd_mixer_new(pcm->card, pcm->id, device, &mixer)) < 0)
		return err;
	strcpy(mixer->name, pcm->name);
	
	/* build input MUX and accumulators */
	if ((codec->me_mux = snd_mixer_lib_mux1(mixer, SND_MIXER_ELEMENT_INPUT_MUX, 0, 0, 2, snd_ad1848_mixer_mux, codec)) == NULL)
		goto __error;
	if ((codec->me_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_OUTPUT_ACCU, 0, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_accu, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_mix = codec->me_accu;
	if ((codec->me_dig_accu = snd_mixer_lib_accu1(mixer, SND_MIXER_ELEMENT_DIGITAL_ACCU, 0, 0)) == NULL)
		goto __error;
	/* build Line1 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_LINE, 0)) == NULL)
		goto __error;
	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_line1 = element;
	/* build Line2 */
	if ((group = snd_mixer_lib_group(mixer, SND_MIXER_IN_LINE, 1)) == NULL)
		goto __error;
	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_LINE, 1, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_line2 = element;
	/* build AUX1 */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_OSS_LINE1, snd_ad1848_mixer_group_aux1, codec)) == NULL)
		goto __error;
	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 0, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_mux) < 0)
		goto __error;
	codec->me_mux_aux1 = element;
	if ((codec->me_vol_aux1 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 0, 2, aux_range, (snd_mixer_volume1_control_t *)snd_ad1848_mixer_aux1_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_aux1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_vol_aux1) < 0)
		goto __error;
	if ((codec->me_sw_aux1 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 0, 2, snd_ad1848_mixer_aux1_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_aux1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_aux1, codec->me_sw_aux1) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_aux1, codec->me_accu) < 0)
		goto __error;
	/* build AUX2 */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_AUX, 1, SND_MIXER_OSS_LINE2, snd_ad1848_mixer_group_aux2, codec)) == NULL)
		goto __error;
	if ((element = snd_mixer_lib_io_stereo(mixer, SND_MIXER_IN_AUX, 1, SND_MIXER_ETYPE_INPUT, 0)) == NULL)
		goto __error;
	if ((codec->me_vol_aux2 = snd_mixer_lib_volume1(mixer, "Aux Input Volume", 1, 2, aux_range, snd_ad1848_mixer_aux2_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_aux2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_vol_aux2) < 0)
		goto __error;
	if ((codec->me_sw_aux2 = snd_mixer_lib_sw1(mixer, "Aux Input Switch", 1, 2, snd_ad1848_mixer_aux2_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_aux2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_aux2, codec->me_sw_aux2) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_aux2, codec->me_accu) < 0)
		goto __error;
	/* build input GAIN */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_GRP_IGAIN, 0, SND_MIXER_OSS_IGAIN, snd_ad1848_mixer_group_igain, codec)) == NULL)
		goto __error;
	if ((codec->me_vol_igain = snd_mixer_lib_volume1(mixer, "Input Gain Volume", 0, 2, igain_range, snd_ad1848_mixer_igain_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_igain) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_mux, codec->me_vol_igain) < 0)
		goto __error;
	/* build ADC */
	if ((element = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_ADC, 0, SND_MIXER_ETYPE_ADC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_igain, element) < 0)
		goto __error;
	/* build capture endpoint */
	if ((codec->me_capture = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_CAPTURE, 0, SND_MIXER_ETYPE_CAPTURE1, 1, &pcm->device)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_capture) < 0)
		goto __error;
	/* build loopback */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_GRP_DIGITAL_LOOPBACK, 0, SND_MIXER_OSS_UNKNOWN, snd_ad1848_mixer_group_loop, codec)) == NULL)
		goto __error;
	if ((codec->me_vol_loop = snd_mixer_lib_volume1(mixer, "Loopback Volume", 0, 1, pcm_range, snd_ad1848_mixer_loopback_volume, codec)) == NULL)
		goto __error;	
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_loop) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_vol_loop) < 0)
		goto __error;
	if ((codec->me_sw_loop = snd_mixer_lib_sw2(mixer, "Loopback Switch", 0, snd_ad1848_mixer_loopback_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_loop) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_loop, codec->me_sw_loop) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_loop, codec->me_dig_accu) < 0)
		goto __error;
	/* build playback endpoint */
	if ((codec->me_playback = snd_mixer_lib_pcm1(mixer, SND_MIXER_ELEMENT_PLAYBACK, 0, SND_MIXER_ETYPE_PLAYBACK1, 1, &pcm->device)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_playback, codec->me_dig_accu) < 0)
		goto __error;
	/* build DAC */
	if ((group = snd_mixer_lib_group_ctrl(mixer, SND_MIXER_IN_PCM, 0, SND_MIXER_OSS_PCM, snd_ad1848_mixer_group_pcm, codec)) == NULL)
		goto __error;
	if ((element = snd_mixer_lib_converter(mixer, SND_MIXER_ELEMENT_DAC, 0, SND_MIXER_ETYPE_DAC, 16)) == NULL)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_dig_accu, element) < 0)
		goto __error;
	if ((codec->me_vol_pcm = snd_mixer_lib_volume1(mixer, "PCM Volume", 0, 2, pcm_range, snd_ad1848_mixer_dac_volume, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_vol_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, element, codec->me_vol_pcm) < 0)
		goto __error;
	if ((codec->me_sw_pcm = snd_mixer_lib_sw1(mixer, "PCM Switch", 0, 2, snd_ad1848_mixer_dac_switch, codec)) == NULL)
		goto __error;
	if (snd_mixer_group_element_add(mixer, group, codec->me_sw_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_vol_pcm, codec->me_sw_pcm) < 0)
		goto __error;
	if (snd_mixer_element_route_add(mixer, codec->me_sw_pcm, codec->me_accu) < 0)
		goto __error;	

	codec->mixer = mixer;
	*rmixer = mixer;
	return 0;

      __error:
      	snd_device_free(pcm->card, mixer);
      	return -ENOMEM;
}

EXPORT_SYMBOL(snd_ad1848_out);
EXPORT_SYMBOL(snd_ad1848_interrupt);
EXPORT_SYMBOL(snd_ad1848_new_pcm);
EXPORT_SYMBOL(snd_ad1848_new_mixer);
EXPORT_SYMBOL(snd_ad1848_mixer_stereo_volume);
EXPORT_SYMBOL(snd_ad1848_mixer_mono_volume);
EXPORT_SYMBOL(snd_ad1848_mixer_stereo_switch);
EXPORT_SYMBOL(snd_ad1848_mixer_mono_switch);

/*
 *  INIT part
 */

static int __init alsa_ad1848_init(void)
{
	return 0;
}

static void __exit alsa_ad1848_exit(void)
{
}

module_init(alsa_ad1848_init)
module_exit(alsa_ad1848_exit)
