/*
 *  Advanced Linux Sound Architecture
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../include/driver.h"

#ifdef CONFIG_SND_OSSEMUL

#if !defined(CONFIG_SOUND) && !defined(CONFIG_SOUND_MODULE)
#error "Enable the OSS soundcore multiplexer (CONFIG_SOUND) in the kernel."
#endif

#include "../include/minors.h"
#include "../include/info.h"
#include <linux/sound.h>

#if 0
#define DEBUG_ACCESS( d, m ) printk( ">>" d "[%i] - begin\n", m )
#define DEBUG_RETURN( d, m, f ) \
	do { long result = f; \
	     printk( ">>" d "[%i] - end (result = %li)\n", m, result ); \
	     return result; } while( 0 )
#else
#define DEBUG_ACCESS( d, m )	/* nothing */
#define DEBUG_RETURN( d, m, f ) return(f)
#endif

#define SND_OS_MINORS		256

static int snd_oss_minors_device[SND_OS_MINORS] = {
	0,		/* SND_MINOR_OSS_MIXER */
	-1,		/* SND_MINOR_OSS_SEQUENCER */
	0,		/* SND_MINOR_OSS_MIDI */
	0,		/* SND_MINOR_OSS_PCM_8 */
	0,		/* SND_MINOR_OSS_AUDIO */
	0,		/* SND_MINOR_OSS_PCM_16 */
	-1,		/* SND_MINOR_OSS_SNDSTAT */
	-1,		/* SND_MINOR_OSS_RESERVED7 */
	-1,		/* SND_MINOR_OSS_MUSIC */
	0,		/* SND_MINOR_OSS_DMMIDI */
	0,		/* SND_MINOR_OSS_DMFM */
	1,		/* SND_MINOR_OSS_MIXER1 */
	1, 		/* SND_MINOR_OSS_PCM1 */
	1,		/* SND_MINOR_OSS_MIDI1 */
	1,		/* SND_MINOR_OSS_DMMIDI1 */
	-1,		/* SND_MINOR_OSS_RESERVED15 */
};
static snd_minor_t *snd_oss_minors[SND_OS_MINORS] = {[0 ... (SND_OS_MINORS - 1)] = NULL};

static DECLARE_MUTEX(sound_oss_mutex);

static inline snd_minor_t *snd_oss_verify_card(unsigned short minor)
{
	int dev;

	dev = SND_MINOR_OSS_DEVICE(minor);
	if (dev != SND_MINOR_OSS_SNDSTAT &&
	    dev != SND_MINOR_OSS_SEQUENCER &&
	    dev != SND_MINOR_OSS_MUSIC) {
		if (!(snd_cards_bitmap & (1 << SND_MINOR_OSS_CARD(minor))))
			return NULL;
	}
	return snd_oss_minors[minor];
}

static inline snd_minor_t *snd_oss_verify_card_open(unsigned short minor)
{
	int dev;
	snd_minor_t *mptr;

	dev = SND_MINOR_OSS_DEVICE(minor);
	if (dev != SND_MINOR_OSS_SNDSTAT &&
	    dev != SND_MINOR_OSS_SEQUENCER &&
	    dev != SND_MINOR_OSS_MUSIC) {
		if (!(snd_cards_bitmap & (1 << SND_MINOR_OSS_CARD(minor))))
			return NULL;
	}
	mptr = snd_oss_minors[minor];
	return mptr;
}

static loff_t snd_oss_lseek(struct file *file, loff_t offset, int orig)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_lseek_t *ptr;

	DEBUG_ACCESS("lseek", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->lseek) == NULL)
		return -ESPIPE;
	DEBUG_RETURN("lseek", minor, ptr(file, offset, orig));
}

static ssize_t snd_oss_read(struct file *file,
			    char *buf, size_t count,
			    loff_t * offset)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_read_t *ptr;

	DEBUG_ACCESS("read", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->read) == NULL)
		return -ENXIO;
	DEBUG_RETURN("read", minor, ptr(file, buf, count));
}

static ssize_t snd_oss_write(struct file *file,
			     const char *buf, size_t count,
			     loff_t * offset)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_write_t *ptr;

	DEBUG_ACCESS("write", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->write) == NULL)
		return -ENXIO;
	DEBUG_RETURN("write", minor, ptr(file, buf, count));
}

static int snd_oss_open(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_open_t *ptr;

	DEBUG_ACCESS("open", minor);
	if ((mptr = snd_oss_verify_card_open(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->open) == NULL)
		return -ENXIO;
	DEBUG_RETURN("open", minor, ptr(minor, SND_MINOR_OSS_CARD(minor),
		     snd_oss_minors_device[SND_MINOR_OSS_DEVICE(minor)], file));
}

static int snd_oss_release(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_release_t *ptr;

	DEBUG_ACCESS("release", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->release) == NULL)
		return -ENXIO;
	DEBUG_RETURN("release", minor, ptr(minor, SND_MINOR_OSS_CARD(minor),
		     snd_oss_minors_device[SND_MINOR_OSS_DEVICE(minor)], file));
}

static unsigned int snd_oss_poll(struct file *file, poll_table * wait)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_poll_t *ptr;

	DEBUG_ACCESS("poll", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return 0;
	if ((ptr = mptr->poll) == NULL)
		return 0;
	DEBUG_RETURN("poll", minor, ptr(file, wait));
}

static int snd_oss_ioctl(struct inode *inode, struct file *file,
			 unsigned int cmd, unsigned long arg)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_ioctl_t *ptr;

	DEBUG_ACCESS("ioctl", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->ioctl) == NULL)
		return -ENXIO;
	DEBUG_RETURN("ioctl", minor, ptr(file, cmd, arg));
}

static int snd_oss_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct inode *inode = file->f_dentry->d_inode;
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_mmap_t *ptr;

	DEBUG_ACCESS("mmap", minor);
	if ((mptr = snd_oss_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->mmap) == NULL)
		return -ENXIO;
	DEBUG_RETURN("mmap", minor, ptr(inode, file, vma));
}

struct file_operations snd_oss_fops =
{
	llseek:		snd_oss_lseek,
	read:		snd_oss_read,
	write:		snd_oss_write,
	poll:		snd_oss_poll,
	ioctl:		snd_oss_ioctl,
	mmap:		snd_oss_mmap,
	open:		snd_oss_open,
	release:	snd_oss_release,
};

static int snd_kernel_minor(int type, snd_card_t * card, int dev)
{
	int minor;

	switch (type) {
	case SND_OSS_DEVICE_TYPE_MIXER:
		snd_debug_check(card == NULL || dev > 1, -EINVAL);
		minor = SND_MINOR_OSS(card->number, (dev ? SND_MINOR_OSS_MIXER1 : SND_MINOR_OSS_MIXER));
		break;
	case SND_OSS_DEVICE_TYPE_SEQUENCER:
		minor = SND_MINOR_OSS_SEQUENCER;
		break;
	case SND_OSS_DEVICE_TYPE_MUSIC:
		minor = SND_MINOR_OSS_MUSIC;
		break;
	case SND_OSS_DEVICE_TYPE_PCM:
		snd_debug_check(card == NULL || dev > 1, -EINVAL);
		minor = SND_MINOR_OSS(card->number, (dev ? SND_MINOR_OSS_PCM1 : SND_MINOR_OSS_PCM));
		break;
	case SND_OSS_DEVICE_TYPE_MIDI:
		snd_debug_check(card == NULL || dev > 1, -EINVAL);
		minor = SND_MINOR_OSS(card->number, (dev ? SND_MINOR_OSS_MIDI1 : SND_MINOR_OSS_MIDI));
		break;
	case SND_OSS_DEVICE_TYPE_DMFM:
		minor = SND_MINOR_OSS_DMFM;
		break;
	case SND_OSS_DEVICE_TYPE_SNDSTAT:
		minor = SND_MINOR_OSS_SNDSTAT;
		break;
	default:
		return -EINVAL;
	}
	snd_debug_check(minor < 0 || minor > SND_OS_MINORS, -EINVAL);
	return minor;
}

int snd_register_oss_device(int type, snd_card_t * card, int dev, snd_minor_t * reg, const char *name)
{
	int minor = snd_kernel_minor(type, card, dev);
	int cidx = SND_MINOR_OSS_CARD(minor);
	int err;
	int track1 = -1, track2 = -1, track3 = -1;
	int register1 = 1, register2 = -1, register3 = -1;

	if (minor < 0)
		return minor;
	down(&sound_oss_mutex);
	if (snd_oss_minors[minor]) {
		up(&sound_oss_mutex);
		return -EBUSY;
	}
	if (SND_MINOR_OSS_DEVICE(minor) == SND_MINOR_OSS_PCM) {
		if (snd_oss_minors[minor+1] ||
		    snd_oss_minors[minor+2]) {
			up(&sound_oss_mutex);
			return -EBUSY;
		}
		snd_oss_minors[track2 = minor+1] =
		snd_oss_minors[track3 = minor+2] = reg;
	} else if (SND_MINOR_OSS_DEVICE(minor) == SND_MINOR_OSS_MIDI) {
		track2 = SND_MINOR_OSS(cidx, SND_MINOR_OSS_DMMIDI);
		if (snd_oss_minors[track2]) {
			up(&sound_oss_mutex);
			return -EBUSY;
		}
		snd_oss_minors[track2] = reg;
	} else if (SND_MINOR_OSS_DEVICE(minor) == SND_MINOR_OSS_MIDI1) {
		track2 = SND_MINOR_OSS(cidx, SND_MINOR_OSS_DMMIDI1);
		if (snd_oss_minors[track2]) {
			up(&sound_oss_mutex);
			return -EBUSY;
		}
		snd_oss_minors[track2] = reg;
	}
	snd_oss_minors[track1 = minor] = reg;
	if (SND_MINOR_OSS_DEVICE(minor) == SND_MINOR_OSS_PCM) {
		if (register_sound_dsp(&snd_oss_fops, card->number)<0)
			goto __end;
	} else {
		if (track1 >= 0 && (err = register_sound_special(&snd_oss_fops, track1)) < 0)
			goto __end;
		register1 = track1;
		if (track2 >= 0 && register_sound_special(&snd_oss_fops, track2) < 0)
			goto __end;
		register2 = track2;
		if (track3 >= 0 && register_sound_special(&snd_oss_fops, track3) < 0)
			goto __end;
		register3 = track3;
	}
	up(&sound_oss_mutex);
	return 0;

      __end:
      	if (track1 >= 0)
      		snd_oss_minors[track1] = NULL;
      	if (track2 >= 0)
      		snd_oss_minors[track2] = NULL;
      	if (track3 >= 0)
      		snd_oss_minors[track3] = NULL;
	up(&sound_oss_mutex);
      	return -EBUSY;
}

int snd_unregister_oss_device(int type, snd_card_t * card, int dev)
{
	int minor = snd_kernel_minor(type, card, dev);
	int cidx = SND_MINOR_OSS_CARD(minor);

	if (minor < 0)
		return minor;
	down(&sound_oss_mutex);
	if (snd_oss_minors[minor] == NULL) {
		up(&sound_oss_mutex);
		return -EINVAL;
	}
	if (SND_MINOR_OSS_DEVICE(minor) == SND_MINOR_OSS_PCM) {
		snd_oss_minors[minor + 1] =
		snd_oss_minors[minor + 2] = NULL;
		unregister_sound_dsp(minor);
	} else if (SND_MINOR_OSS_DEVICE(minor) == SND_MINOR_OSS_MIDI) {
		int m = SND_MINOR_OSS(cidx, SND_MINOR_OSS_DMMIDI);
		snd_oss_minors[m] = NULL;
		unregister_sound_special(m);
	} else if (SND_MINOR_OSS_DEVICE(minor) == SND_MINOR_OSS_MIDI1) {
		int m = SND_MINOR_OSS(cidx, SND_MINOR_OSS_DMMIDI1);
		snd_oss_minors[m] = NULL;
		unregister_sound_special(m);
	}
	snd_oss_minors[minor] = NULL;
	if (SND_MINOR_OSS_DEVICE(minor) != SND_MINOR_OSS_PCM)
		unregister_sound_special(minor);
	up(&sound_oss_mutex);
	return 0;
}

/*
 *  INFO PART
 */

static snd_info_entry_t *snd_minor_info_oss_entry = NULL;

static void snd_minor_info_oss_read(snd_info_buffer_t * buffer, void *private_data)
{
	int idx, dev;
	snd_minor_t *mptr;

	down(&sound_oss_mutex);
	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		if ((mptr = snd_oss_minors[idx]) == NULL)
			continue;
		dev = SND_MINOR_OSS_DEVICE(idx);
	        if (dev != SND_MINOR_OSS_SNDSTAT &&
		    dev != SND_MINOR_OSS_SEQUENCER &&
		    dev != SND_MINOR_OSS_MUSIC)
			snd_iprintf(buffer, "%3i: [%i-%2i]: %s\n", idx, idx >> 4, dev, mptr->comment);
		else
			snd_iprintf(buffer, "%3i:       : %s\n", idx, mptr->comment);
	}
	up(&sound_oss_mutex);
}

int snd_minor_info_oss_init(void)
{
	snd_info_entry_t *oss_entry;

	oss_entry = snd_info_create_entry(NULL, "oss-devices");
	if (oss_entry) {
		oss_entry->t.text.read_size = PAGE_SIZE;
		oss_entry->t.text.read = snd_minor_info_oss_read;
		if (snd_info_register(oss_entry) < 0) {
			snd_info_free_entry(oss_entry);
			oss_entry = NULL;
		}
	}
	snd_minor_info_oss_entry = oss_entry;
	return 0;
}

int snd_minor_info_oss_done(void)
{
	if (snd_minor_info_oss_entry)
		snd_info_unregister(snd_minor_info_oss_entry);
	return 0;
}

int snd_oss_init_module(void)
{
	return 0;
}

void snd_oss_cleanup_module(void)
{
	int idx;
	snd_minor_t *minor;

	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		minor = snd_oss_minors[idx];
		if (minor)
			snd_printk("OSS device minor %i not unregistered!!!\n", idx);
	}
}

#endif /* CONFIG_SND_OSSEMUL */
