/*
 *  Routines for the universal switch interface
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../include/driver.h"
#include "../include/switch.h"
#include "../include/control.h"

static int snd_switch_register_list(snd_card_t *card, snd_kswitch_list_t * list);
static int snd_switch_unregister_list(snd_card_t *card, snd_kswitch_list_t * list);

int snd_switch_prepare(snd_card_t *card,
		       snd_kswitch_list_t *list, void *desc,
		       int iface, int device, int channel)
{
	list->count = 0;
	list->switches = NULL;
	init_MUTEX(&list->lock);
	list->desc = desc;
	list->iface = iface;
	list->device = device;
	list->channel = channel;
	list->prev = list->next = NULL;
	return snd_switch_register_list(card, list);
}

snd_kswitch_t *snd_switch_new(snd_kswitch_t * ksw)
{
	snd_kswitch_t *nsw;

	nsw = (snd_kswitch_t *) snd_kmalloc(sizeof(snd_kswitch_t), GFP_KERNEL);
	if (nsw == NULL)
		return NULL;
	memcpy(nsw, ksw, sizeof(snd_kswitch_t));
	return nsw;
}

void snd_switch_free_one(snd_kswitch_t * ksw)
{
	if (ksw) {
		if (ksw->private_free)
			ksw->private_free(ksw->private_data);
		snd_kfree(ksw);
	}
}

int snd_switch_add(snd_kswitch_list_t *list, snd_kswitch_t *ksw)
{
	snd_kswitch_t **x;

	snd_debug_check(list == NULL || ksw == NULL, -ENXIO);
	down(&list->lock);
	x = (snd_kswitch_t **)
			snd_kmalloc((list->count + 1) * sizeof(void *), GFP_KERNEL);
	if (x == NULL) {
		up(&list->lock);
		return -ENOMEM;
	}
	if (list->switches) {
		memcpy(x, list->switches, (list->count + 1) * sizeof(void *));
		snd_kfree(list->switches);
	}
	list->switches = x;
	list->switches[list->count++] = ksw;
	up(&list->lock);
	return 0;
}

int snd_switch_remove(snd_kswitch_list_t * list, snd_kswitch_t * ksw)
{
	snd_debug_check(list == NULL || ksw == NULL, -ENXIO);
	return -EINVAL;	/* TODO */
}

int snd_switch_free(snd_card_t *card, snd_kswitch_list_t * list)
{
	snd_kswitch_t **x;
	int err, idx, count;

	if ((err = snd_switch_unregister_list(card, list)) < 0)
		return err;
	down(&list->lock);
	if ((x = list->switches) != NULL) {
		count = list->count;
		list->switches = NULL;
		list->count = 0;
		up(&list->lock);
		for (idx = 0; idx < count; idx++)
			snd_switch_free_one(x[idx]);
		snd_kfree(x);
	} else {
		up(&list->lock);
	}
	return 0;
}

void snd_switch_lock(snd_card_t *card, int xup)
{
	if (!xup) {
		down(&card->lists_lock);
	} else {
		up(&card->lists_lock);
	}
}

static int snd_switch_register_list(snd_card_t *card, snd_kswitch_list_t * list)
{
	if (list == NULL)
		return -EINVAL;
	snd_switch_lock(card, 0);
	list->next = NULL;
	if (card->last_list) {
		list->prev = card->last_list;
		card->last_list->next = list;
		card->last_list = list;
	} else {
		list->prev = NULL;
		card->first_list = card->last_list = list;
	}
	snd_switch_lock(card, 1);
	return 0;
}

static int snd_switch_unregister_list(snd_card_t *card, snd_kswitch_list_t * list)
{
	if (list == NULL)
		return -EINVAL;
	snd_switch_lock(card, 0);
	if (card->first_list == list) {
		card->first_list = list->next;
		if (card->first_list == NULL)
			card->last_list = NULL;
	} else if (card->last_list == list) {
		card->last_list = list->prev;
		card->last_list->next = NULL;
	} else {
		list->prev->next = list->next;
		list->next->prev = list->prev;
	}
	snd_switch_lock(card, 1);
	return 0;
}

int snd_switch_list(snd_card_t *card, snd_switch_list_t *_list)
{
	int idx;
	snd_kswitch_list_t *list;
	snd_switch_list_t xlist;
	snd_switch_list_item_t item;
	snd_kswitch_t *ksw;

	if (copy_from_user(&xlist, _list, sizeof(xlist)))
		return -EFAULT;
	if (xlist.switches_size > 0) {
		if (xlist.pswitches == NULL)
			return -EINVAL;
	}
	snd_switch_lock(card, 0);
	for (list = card->first_list; list != NULL; list = list->next) {
		if (list->iface == xlist.iface && list->device == xlist.device && list->channel == xlist.channel)
			break;
	}
	if (list == NULL) {
		snd_switch_lock(card, 1);
		return -ENOENT;
	}
	down(&list->lock);
	for (idx = 0; idx < list->count && idx < xlist.switches_size; idx++) {
		ksw = list->switches[idx];
		memcpy(item.name, ksw->name, sizeof(item.name));
		if (copy_to_user(&xlist.pswitches[idx], &item, sizeof(item))) {
			snd_switch_lock(card, 1);
			return -EFAULT;
		}
	}
	xlist.switches = idx;
	xlist.switches_over = list->count - idx;
	up(&list->lock);
	snd_switch_lock(card, 1);
	if (copy_to_user(_list, &xlist, sizeof(*_list)))
		return -EFAULT;
	return 0;
}

int snd_switch_read1(snd_card_t *card, snd_switch_t *_switch)
{
	snd_kswitch_list_t *list;
	snd_switch_t sw, sw1;
	snd_kswitch_t *ksw, **x;
	int idx, err = -ENXIO;

	if (copy_from_user(&sw, _switch, sizeof(sw)))
		return -EFAULT;
	for (list = card->first_list; list != NULL; list = list->next) {
		if (list->iface == sw.iface && list->device == sw.device && list->channel == sw.channel)
			break;
	}
	if (list == NULL)
		return -ENOENT;
	x = list->switches;
	down(&list->lock);
	for (idx = 0; idx < list->count; idx++) {
		ksw = x[idx];
		if (!strcmp(ksw->name, sw.name)) {
			memcpy(&sw1, &sw, sizeof(sw1));
			memset(&sw, 0, sizeof(sw));
			sw.iface = sw1.iface;
			sw.device = sw1.device;
			sw.channel = sw1.channel;
			memcpy(sw.name, sw1.name, sizeof(sw.name));
			err = ksw->get(list->desc, ksw, &sw);
			break;
		}
	}
	up(&list->lock);
	if (!err)
		if (copy_to_user(_switch, &sw, sizeof(sw)))
			return -EFAULT;
	return err;	
}

int snd_switch_read(snd_card_t *card, snd_switch_t *_switch)
{
	int result;

	snd_switch_lock(card, 0);
	result = snd_switch_read1(card, _switch);
	snd_switch_lock(card, 1);
	return result;
}

static int snd_switch_valid_request(snd_switch_t *sw)
{
	if (sw->iface == SND_CTL_IFACE_CONTROL) {
		static char *root_only[] = {
			SND_CTL_SW_JOYSTICK,
			SND_CTL_SW_JOYSTICK_ADDRESS,
			NULL
		};
		char **root_str = root_only;
		while (*root_str) {
			if (!strcmp(sw->name, *root_str)) {
				if (!suser())
					return -EPERM;
                        }
                        root_str++;
                }
        }
	return 0;
}

int snd_switch_write1(snd_control_t *control, snd_switch_t *_switch)
{
	snd_card_t *card = control->card;
	snd_kswitch_list_t *list;
	snd_switch_t sw;
	snd_kswitch_t *ksw, **x;
	int idx, err = -ENXIO;

	if (copy_from_user(&sw, _switch, sizeof(sw)))
		return -EFAULT;
	for (list = card->first_list; list != NULL; list = list->next) {
		if (list->iface == sw.iface && list->device == sw.device && list->channel == sw.channel)
			break;
	}
	if (list == NULL)
		return -ENOENT;
	down(&list->lock);
	x = list->switches;
	for (idx = 0; idx < list->count; idx++) {
		ksw = x[idx];
		if (!strcmp(ksw->name, sw.name)) {
			err = snd_switch_valid_request(&sw);
			if (err == 0)
				err = ksw->set(list->desc, ksw, &sw);
			break;
		}
	}
	up(&list->lock);
	if (!err)
		if (copy_to_user(_switch, &sw, sizeof(sw)))
			return -EFAULT;
	if (err > 0)	/* change */
		snd_control_notify_switch_value_change(control,
						       sw.iface,
						       sw.device,
						       sw.channel,
						       sw.name,
						       0);
	return err;	
}

int snd_switch_write(snd_control_t *control, snd_switch_t *_switch)
{
	int result;

	snd_switch_lock(control->card, 0);
	result = snd_switch_write1(control, _switch);
	snd_switch_lock(control->card, 1);
	return result;
}

int snd_switch_count(snd_kswitch_list_t * list)
{
	return list->count;
}
