/*
 *  Advanced Linux Sound Architecture
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/minors.h"
#include "../include/info.h"
#include "../include/control.h"
#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif
#ifdef CONFIG_DEVFS_FS
#include <linux/devfs_fs_kernel.h>
#endif

#if 0
#define DEBUG_ACCESS( d, m ) printk( ">>" d "[%i] - begin\n", m )
#define DEBUG_RETURN( d, m, f ) \
	do { long result = f; \
	     printk( ">>" d "[%i] - end (result = %li)\n", m, result ); \
	     return result; } while( 0 )
#else
#define DEBUG_ACCESS( d, m )	/* nothing */
#define DEBUG_RETURN( d, m, f ) return(f)
#endif

int snd_major = CONFIG_SND_MAJOR;
int snd_cards_limit = SND_CARDS, snd_ecards_limit;
int snd_device_mode = S_IFCHR | S_IRUGO | S_IWUGO;
int snd_device_gid = 0;
int snd_device_uid = 0;
MODULE_AUTHOR("Jaroslav Kysela <perex@suse.cz>");
MODULE_DESCRIPTION("Advanced Linux Sound Architecture driver for soundcards.");
MODULE_SUPPORTED_DEVICE("sound");
MODULE_PARM(snd_major, "i");
MODULE_PARM_DESC(snd_major, "Major # for sound driver. (116 by default)");
MODULE_PARM(snd_cards_limit, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_cards_limit, "Count of soundcards installed in the system.");
MODULE_PARM(snd_device_mode, "i");
MODULE_PARM_DESC(snd_device_mode, "Device file permission mask for sound dynamic device filesystem.");
MODULE_PARM(snd_device_gid, "i");
MODULE_PARM_DESC(snd_device_gid, "Device file GID for sound dynamic device filesystem.");
MODULE_PARM(snd_device_uid, "i");
MODULE_PARM_DESC(snd_device_uid, "Device file UID for sound dynamic device filesystem.");

#define SND_OS_MINORS		256

static int snd_minors_device[SND_MINOR_DEVICES] = {[0 ... (SND_MINOR_DEVICES - 1)] = -1};
static snd_minor_t *snd_minors[SND_OS_MINORS] = {[0 ... (SND_OS_MINORS - 1)] = NULL};

static DECLARE_MUTEX(sound_mutex);

#ifdef CONFIG_DEVFS_FS
static devfs_handle_t devfs_handle = NULL;
#endif

static inline snd_minor_t *snd_verify_card(unsigned short minor)
{
	int dev;

	dev = SND_MINOR_DEVICE(minor);
	if (dev != SND_MINOR_SEQUENCER) {
		if (!(snd_cards_bitmap & (1 << (SND_MINOR_CARD(minor)))))
			return NULL;
	}
	return snd_minors[minor];
}

#ifdef CONFIG_KMOD

void snd_request_card(int card)
{
	char str[32];

	if (snd_cards_bitmap & (1 << card))
		return;
	if (card < 0 || card >= snd_ecards_limit)
		return;
	sprintf(str, "snd-card-%i", card);
	request_module(str);
}

static void snd_request_other(int minor)
{
	char *str;

	switch (minor) {
	case SND_MINOR_SEQUENCER:	str = "snd-seq";	break;
	case SND_MINOR_TIMER:		str = "snd-timer";	break;
	default:			return;
	}
	request_module(str);
}

#endif				/* request_module support */

static inline snd_minor_t *snd_verify_card_open(unsigned short minor)
{
	int dev = SND_MINOR_DEVICE(minor);
	int card = SND_MINOR_CARD(minor);

	if (dev != SND_MINOR_SEQUENCER) {
	  
		if (!(snd_cards_bitmap & (1 << card))) {
#ifdef CONFIG_KMOD
			snd_request_card(card);
			if (!(snd_cards_bitmap & (1 << card)))
#endif
				return NULL;
		}
	} else {
#ifdef CONFIG_KMOD
		snd_request_other(minor);
#endif
	}
	return snd_minors[minor];
}

static loff_t snd_lseek(struct file *file, loff_t offset, int orig)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_lseek_t *ptr;

	DEBUG_ACCESS("lseek", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->lseek) == NULL)
		return -ESPIPE;
	DEBUG_RETURN("lseek", minor, ptr(file, offset, orig));
}

static ssize_t snd_read(struct file *file,
                        char *buf, size_t count,
                        loff_t * offset)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_read_t *ptr;

	DEBUG_ACCESS("read", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->read) == NULL)
		return -ENXIO;
	DEBUG_RETURN("read", minor, ptr(file, buf, count));
}

static ssize_t snd_write(struct file *file,
			 const char *buf, size_t count,
			 loff_t * offset)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_write_t *ptr;

	DEBUG_ACCESS("write", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->write) == NULL)
		return -ENXIO;
	DEBUG_RETURN("write", minor, ptr(file, buf, count));
}

#if KERNEL_VERSION(2, 3, 44) <= LINUX_VERSION_CODE
static ssize_t snd_readv(struct file *file,
			 const struct iovec *vector, unsigned long count,
			 loff_t * offset)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_readv_t *ptr;

	DEBUG_ACCESS("readv", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->readv) == NULL)
		return -ENXIO;
	DEBUG_RETURN("readv", minor, ptr(file, vector, count));
}

static ssize_t snd_writev(struct file *file,
			  const struct iovec *vector, unsigned long count,
			  loff_t * offset)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_writev_t *ptr;

	DEBUG_ACCESS("writev", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->writev) == NULL)
		return -ENXIO;
	DEBUG_RETURN("writev", minor, ptr(file, vector, count));
}
#endif

static int snd_open(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_open_t *ptr;

	DEBUG_ACCESS("open", minor);
	if ((mptr = snd_verify_card_open(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->open) == NULL)
		return -ENXIO;
	DEBUG_RETURN("open", minor, ptr(minor + 256, SND_MINOR_CARD(minor), snd_minors_device[SND_MINOR_DEVICE(minor)], file));
}

static int snd_release(struct inode *inode, struct file *file)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_release_t *ptr;

	DEBUG_ACCESS("release", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->release) == NULL)
		return -ENXIO;
	DEBUG_RETURN("release", minor, ptr(minor + 256, SND_MINOR_CARD(minor), snd_minors_device[SND_MINOR_DEVICE(minor)], file));
}

static unsigned int snd_poll(struct file *file, poll_table * wait)
{
	unsigned short minor = MINOR(file->f_dentry->d_inode->i_rdev);
	snd_minor_t *mptr;
	snd_poll_t *ptr;

	DEBUG_ACCESS("poll", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return 0;
	if ((ptr = mptr->poll) == NULL)
		return 0;
	DEBUG_RETURN("poll", minor, ptr(file, wait));
}

static int snd_ioctl(struct inode *inode, struct file *file,
		     unsigned int cmd, unsigned long arg)
{
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_ioctl_t *ptr;

	DEBUG_ACCESS("ioctl", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
#ifndef READV_WRITEV_FILE_OPERATIONS
	if (arg == SND_IOCTL_READV ||
	    arg == SND_IOCTL_WRITEV) {
		snd_v_args_t rargs;
		struct iovec iovstack[UIO_FASTIOV];
		struct iovec *iov = iovstack;
		int ret;
		if (arg == SND_IOCTL_READV) {
			if (mptr->readv == NULL)
				return -ENXIO;
		} else if (mptr->writev == NULL)
			return -ENXIO;
		
		if (copy_from_user(&rargs, (snd_v_args_t *) arg, sizeof(rargs)))
			return -EFAULT;
		if (rargs.count > UIO_MAXIOV)
			return -EINVAL;
		if (rargs.count > UIO_FASTIOV) {
			iov = snd_kmalloc(rargs.count * sizeof(*iov), GFP_KERNEL);
			if (!iov)
				return -ENOMEM;
		}
		if (copy_from_user(iov, rargs.vector, rargs.count * sizeof(*iov))) {
			ret = -EFAULT;
			goto out;
		}
		if (arg == SND_IOCTL_READV)
			ret = mptr->readv(file, iov, rargs.count);
		else
			ret = mptr->writev(file, iov, rargs.count);
	out:
		if (iov != iovstack)
			snd_kfree(iov);
		return ret;
			
	}
#endif
	if ((ptr = mptr->ioctl) == NULL)
		return -ENXIO;
	DEBUG_RETURN("ioctl", minor, ptr(file, cmd, arg));
}

static int snd_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct inode *inode = file->f_dentry->d_inode;
	unsigned short minor = MINOR(inode->i_rdev);
	snd_minor_t *mptr;
	snd_mmap_t *ptr;

	DEBUG_ACCESS("mmap", minor);
	if ((mptr = snd_verify_card(minor)) == NULL)
		return -ENODEV;
	if ((ptr = mptr->mmap) == NULL)
		return -ENXIO;
	DEBUG_RETURN("mmap", minor, ptr(inode, file, vma));
}

struct file_operations snd_fops =
{
	llseek:		snd_lseek,
	read:		snd_read,
	write:		snd_write,
	poll:		snd_poll,
	ioctl:		snd_ioctl,
	mmap:		snd_mmap,
	open:		snd_open,
	release:	snd_release,
#if KERNEL_VERSION(2, 3, 44) <= LINUX_VERSION_CODE
	readv:		snd_readv,
	writev:		snd_writev
#endif
};

static void snd_init_minors(void)
{
	int idx;

	for (idx = 0; idx < SND_MINOR_DEVICES; idx++)
		snd_minors_device[idx] = -1;
	for (idx = 0; idx < SND_MINOR_HWDEPS; idx++)
		snd_minors_device[idx + SND_MINOR_HWDEP] = idx;
	for (idx = 0; idx < SND_MINOR_MIXERS; idx++)
		snd_minors_device[idx + SND_MINOR_MIXER] = idx;
	for (idx = 0; idx < SND_MINOR_RAWMIDIS; idx++)
		snd_minors_device[idx + SND_MINOR_RAWMIDI] = idx;
	for (idx = 0; idx < SND_MINOR_PCMS; idx++) {
		snd_minors_device[idx + SND_MINOR_PCM_PLAYBACK] = idx;
		snd_minors_device[idx + SND_MINOR_PCM_CAPTURE] = idx;
	}
}

static int snd_kernel_minor(int type, snd_card_t * card, int dev)
{
	int minor;

	switch (type) {
	case SND_DEVICE_TYPE_SEQUENCER:
	case SND_DEVICE_TYPE_TIMER:
		minor = type;
		break;
	case SND_DEVICE_TYPE_CONTROL:
		snd_debug_check(card == NULL, -EINVAL);
		minor = SND_MINOR(card->number, type);
		break;
	case SND_DEVICE_TYPE_HWDEP:
	case SND_DEVICE_TYPE_MIXER:
	case SND_DEVICE_TYPE_RAWMIDI:
	case SND_DEVICE_TYPE_PCM_PLAYBACK:
	case SND_DEVICE_TYPE_PCM_CAPTURE:
		snd_debug_check(card == NULL, -EINVAL);
		minor = SND_MINOR(card->number, type + dev);
		break;
	default:
		return -EINVAL;
	}
	snd_debug_check(minor < 0 || minor > SND_OS_MINORS, -EINVAL);
	return minor;
}

int snd_register_device(int type, snd_card_t * card, int dev, snd_minor_t * reg, const char *name)
{
	int minor = snd_kernel_minor(type, card, dev);
	snd_minor_t *preg;

	if (minor < 0)
		return minor;
	preg = (snd_minor_t *)snd_kmalloc(sizeof(snd_minor_t), GFP_KERNEL);
	if (preg == NULL)
		return -ENOMEM;
	memcpy(preg, reg, sizeof(snd_minor_t));
	preg->dev = NULL;
	down(&sound_mutex);
	if (snd_minors[minor]) {
		up(&sound_mutex);
		snd_kfree(preg);
		return -EBUSY;
	}
	if (name)
		preg->dev = snd_info_create_device(name, minor, 0);
	snd_minors[minor] = preg;
	up(&sound_mutex);
	return 0;
}

int snd_unregister_device(int type, snd_card_t * card, int dev)
{
	int minor = snd_kernel_minor(type, card, dev);
	snd_minor_t *mptr;

	if (minor < 0)
		return minor;
	down(&sound_mutex);
	if ((mptr = snd_minors[minor]) == NULL) {
		up(&sound_mutex);
		return -EINVAL;
	}
	if (mptr->dev)
		snd_info_free_device(mptr->dev);
	snd_minors[minor] = NULL;
	up(&sound_mutex);
	snd_kfree(mptr);
	return 0;
}

/*
 *  INFO PART
 */

static snd_info_entry_t *snd_minor_info_entry = NULL;

static void snd_minor_info_read(snd_info_buffer_t * buffer, void *private_data)
{
	int idx, device;
	snd_minor_t *mptr;

	down(&sound_mutex);
	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		int dev;
		if ((mptr = snd_minors[idx]) == NULL)
			continue;
		dev = SND_MINOR_DEVICE(idx);
		if (dev != SND_MINOR_SEQUENCER) {
			if ((device = snd_minors_device[dev]) >= 0)
				snd_iprintf(buffer, "%3i: [%i-%2i]: %s\n", idx, SND_MINOR_CARD(idx), device, mptr->comment);
			else
				snd_iprintf(buffer, "%3i: [%i]   : %s\n", idx, SND_MINOR_CARD(idx), mptr->comment);
		} else {
			snd_iprintf(buffer, "%3i:       : %s\n", idx, mptr->comment);
		}
	}
	up(&sound_mutex);
}

int __init snd_minor_info_init(void)
{
	snd_info_entry_t *entry;

	entry = snd_info_create_entry(NULL, "devices");
	if (entry) {
		entry->t.text.read_size = PAGE_SIZE;
		entry->t.text.read = snd_minor_info_read;
		if (snd_info_register(entry) < 0) {
			snd_info_free_entry(entry);
			entry = NULL;
		}
	}
	snd_minor_info_entry = entry;
	return 0;
}

int __exit snd_minor_info_done(void)
{
	if (snd_minor_info_entry)
		snd_info_unregister(snd_minor_info_entry);
	return 0;
}

/*
 *  INIT PART
 */

static int __init alsa_sound_init(void)
{
#ifdef CONFIG_DEVFS_FS
	short controlnum;
	char controlname[24];
#endif
#ifdef CONFIG_SND_OSSEMUL
	int err;
#endif
	snd_ecards_limit = snd_cards_limit;
	snd_init_minors();
#ifdef CONFIG_SND_OSSEMUL
	if ((err = snd_oss_init_module()) < 0)
		return err;
#endif
#ifdef CONFIG_DEVFS_FS
	devfs_handle = devfs_mk_dir(NULL, "snd", 0, NULL);
	if (devfs_register_chrdev(snd_major, "alsa", &snd_fops)) {
#else
	if (register_chrdev(snd_major, "alsa", &snd_fops)) {
#endif
		snd_printk("unable to register native major device number %d\n", snd_major);
#ifdef CONFIG_SND_OSSEMUL
		snd_oss_cleanup_module();
#endif
		return -EIO;
	}
	snd_memory_init();
	snd_driver_init();
	if (snd_info_init() < 0) {
		snd_memory_done();
#ifdef CONFIG_SND_OSSEMUL
		snd_oss_cleanup_module();
#endif
		return -ENOMEM;
	}
#ifdef CONFIG_SND_OSSEMUL
	snd_info_minor_register();
#endif
#ifdef CONFIG_DEVFS_FS
	for (controlnum = 0; controlnum < snd_cards_limit; controlnum++) {
		sprintf(controlname, "snd/controlC%d", controlnum);
		devfs_register(NULL, controlname, 0, DEVFS_FL_DEFAULT,
				snd_major, controlnum<<5, snd_device_mode | S_IFCHR,
				0, 0, &snd_fops, NULL);
	}
#endif
#ifndef MODULE
	{
		int idx, ok = 0;
	
		printk("Advanced Linux Sound Architecture Driver Version " CONFIG_SND_VERSION ".\n");
		for (idx = 0; idx < SND_CARDS; idx++)
			if (snd_cards[idx] != NULL) {
				printk("  #%i: %s\n", idx, snd_cards[idx]->longname);
				ok++;
			}
		if (ok == 0)
			printk("  No soundcards found.\n");
	}
#endif
	return 0;
}

static void __exit alsa_sound_exit(void)
{
	int idx;
	snd_minor_t *minor;
#ifdef CONFIG_DEVFS_FS
	devfs_handle_t master;
	char controlname[24];
	short controlnum;

	for (controlnum = 0; controlnum < snd_cards_limit; controlnum++) {
		sprintf(controlname, "snd/controlC%d", controlnum);
		master = devfs_find_handle(NULL, controlname, 0, 0, 0, DEVFS_SPECIAL_CHR, 0);
		devfs_unregister(master);
	}
#endif
	
	for (idx = 0; idx < SND_OS_MINORS; idx++) {
		minor = snd_minors[idx];
		if (minor) {
			snd_minors[idx] = NULL;
			if (minor->dev)
				snd_info_free_device(minor->dev);
			snd_kfree(minor);
		}
	}
#ifdef CONFIG_SND_OSSEMUL
	snd_info_minor_unregister();
	snd_oss_cleanup_module();
#endif
	snd_info_done();
	snd_memory_done();
#ifdef CONFIG_DEVFS_FS
	if (devfs_unregister_chrdev(snd_major, "alsa") != 0)
#else
	if (unregister_chrdev(snd_major, "alsa") != 0)
#endif
		snd_printk("unable to unregister major device number %d\n", snd_major);
#ifdef CONFIG_DEVFS_FS
	devfs_unregister(devfs_handle);
#endif
}

module_init(alsa_sound_init)
module_exit(alsa_sound_exit)

  /* sound.c */
EXPORT_SYMBOL(snd_ecards_limit);
#if defined(CONFIG_KMOD)
EXPORT_SYMBOL(snd_request_card);
#endif
EXPORT_SYMBOL(snd_register_device);
EXPORT_SYMBOL(snd_unregister_device);
#if defined(CONFIG_SND_OSSEMUL)
EXPORT_SYMBOL(snd_register_oss_device);
EXPORT_SYMBOL(snd_unregister_oss_device);
#endif
  /* memory.c */
EXPORT_SYMBOL(snd_kmalloc);
EXPORT_SYMBOL(snd_kcalloc);
EXPORT_SYMBOL(_snd_kfree);
#ifdef CONFIG_SND_DEBUG
EXPORT_SYMBOL(_snd_magic_kmalloc);
EXPORT_SYMBOL(_snd_magic_kcalloc);
EXPORT_SYMBOL(_snd_magic_kfree);
#endif
EXPORT_SYMBOL(snd_kmalloc_strdup);
EXPORT_SYMBOL(snd_vmalloc);
EXPORT_SYMBOL(snd_vfree);
EXPORT_SYMBOL(snd_malloc_pages);
EXPORT_SYMBOL(snd_free_pages);
EXPORT_SYMBOL(snd_dma_malloc);
EXPORT_SYMBOL(snd_dma_free);
  /* vma.c */
EXPORT_SYMBOL(snd_vma_add);
EXPORT_SYMBOL(snd_vma_disconnect);
  /* init.c */
EXPORT_SYMBOL(snd_cards_count);
EXPORT_SYMBOL(snd_cards_bitmap);
EXPORT_SYMBOL(snd_cards);
EXPORT_SYMBOL(snd_card_new);
EXPORT_SYMBOL(snd_card_free);
EXPORT_SYMBOL(snd_card_register);
EXPORT_SYMBOL(snd_card_unregister);
EXPORT_SYMBOL(snd_check_ioport);
EXPORT_SYMBOL(snd_register_ioport);
EXPORT_SYMBOL(snd_unregister_ioport);
EXPORT_SYMBOL(snd_unregister_ioports);
EXPORT_SYMBOL(snd_register_dma_channel);
EXPORT_SYMBOL(snd_unregister_dma_channels);
EXPORT_SYMBOL(snd_register_interrupt);
EXPORT_SYMBOL(snd_unregister_interrupts);
  /* device.c */
EXPORT_SYMBOL(snd_device_new);
EXPORT_SYMBOL(snd_device_free);
EXPORT_SYMBOL(snd_device_register);
EXPORT_SYMBOL(snd_device_unregister);
EXPORT_SYMBOL(snd_device_free_all);
  /* misc.c */
EXPORT_SYMBOL(snd_dma_program);
EXPORT_SYMBOL(snd_dma_disable);
EXPORT_SYMBOL(snd_dma_residue);
  /* info.c */
EXPORT_SYMBOL(snd_iprintf);
EXPORT_SYMBOL(snd_info_get_line);
EXPORT_SYMBOL(snd_info_get_str);
EXPORT_SYMBOL(snd_info_create_entry);
EXPORT_SYMBOL(snd_info_free_entry);
EXPORT_SYMBOL(snd_info_create_device);
EXPORT_SYMBOL(snd_info_free_device);
EXPORT_SYMBOL(snd_info_register);
EXPORT_SYMBOL(snd_info_unregister);
  /* info_oss.c */
#if defined(CONFIG_SND_OSSEMUL)
EXPORT_SYMBOL(snd_oss_info_register);
#endif
  /* switch.c */
EXPORT_SYMBOL(snd_switch_lock);
EXPORT_SYMBOL(snd_switch_prepare);
EXPORT_SYMBOL(snd_switch_new);
EXPORT_SYMBOL(snd_switch_free_one);
EXPORT_SYMBOL(snd_switch_add);
EXPORT_SYMBOL(snd_switch_remove);
EXPORT_SYMBOL(snd_switch_free);
EXPORT_SYMBOL(snd_switch_count);
  /* control.c */
EXPORT_SYMBOL(snd_control_busy);
EXPORT_SYMBOL(snd_control_notify_structure_change);
EXPORT_SYMBOL(snd_control_notify_value_change);
EXPORT_SYMBOL(snd_control_notify_switch_change);
EXPORT_SYMBOL(snd_control_notify_switch_value_change);
EXPORT_SYMBOL(snd_control_register_ioctl);
EXPORT_SYMBOL(snd_control_unregister_ioctl);
EXPORT_SYMBOL(snd_control_switch_add);
EXPORT_SYMBOL(snd_control_switch_remove);
EXPORT_SYMBOL(snd_control_switch_new);
EXPORT_SYMBOL(snd_control_switch_change);
  /* misc.c */
EXPORT_SYMBOL(snd_task_name);
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
EXPORT_SYMBOL(snd_pci_compat_match_device);
EXPORT_SYMBOL(snd_pci_compat_register_driver);
EXPORT_SYMBOL(snd_pci_compat_unregister_driver);
EXPORT_SYMBOL(snd_pci_compat_get_size);
EXPORT_SYMBOL(snd_pci_compat_get_flags);
EXPORT_SYMBOL(snd_pci_compat_set_power_state);
EXPORT_SYMBOL(snd_pci_compat_enable_device);
EXPORT_SYMBOL(snd_pci_compat_find_capability);
EXPORT_SYMBOL(snd_pci_compat_get_driver_data);
EXPORT_SYMBOL(snd_pci_compat_set_driver_data);
#endif
