/*
 *  Driver for generic ESS AudioDrive ESx688 soundcards
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/es1688.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"
#include "../include/initval.h"

MODULE_DESCRIPTION("\
Driver: ESS AudioDrive ESx688\n\
");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;	/* 0x220,0x240,0x260 */
int snd_mpu_port[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = -1};
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;	/* 5,7,9,10 */
int snd_mpu_irq[SND_CARDS] = SND_DEFAULT_IRQ;	/* 5,7,9,10 */
int snd_dma8[SND_CARDS] = SND_DEFAULT_DMA;	/* 0,1,3 */
int snd_dma8_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for ESx688 soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for ESx688 soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for ESx688 driver. [PORT]");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for ESx688 driver. [PORT]");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for ESx688 driver. [IRQ]");
MODULE_PARM(snd_mpu_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_irq, "MPU-401 IRQ # for ESx688 driver. [IRQ]");
MODULE_PARM(snd_dma8, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma8, "8-bit DMA # for ESx688 driver. [DMA8]");
MODULE_PARM(snd_dma8_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma8_size, "Size of 8-bit DMA # for ESx688 driver. [DMA8_SIZE]");

static struct snd_audiodrive {
	snd_irq_t *irqptr;
	snd_irq_t *mpuirqptr;
	snd_dma_t *dma8ptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;
} *snd_audiodrive_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_audiodrive_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_audiodrive_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static snd_pcm_t * __init snd_audiodrive_detect(snd_card_t * card,
						unsigned short port,
						unsigned short mpu_port,
						snd_irq_t * irqptr,
						snd_irq_t * mpu_irqptr,
						snd_dma_t * dma8ptr)
{
	snd_pcm_t *pcm;
	es1688_t *codec;
	int err;

	if (snd_register_ioport(card, port, 16, "ESx688", NULL) < 0)
		return NULL;
	if (mpu_irqptr) {
		if (snd_register_ioport(card, mpu_port, 2, "ESx688 - MPU-401", NULL) < 0) {
			snd_unregister_ioports(card);
			snd_printk("ports for ESx688 MPU-401 are already used, try another\n");
			return NULL;
		}
	}
	err = snd_es1688_new_pcm(card, 0, port, mpu_port,
				 irqptr, mpu_irqptr,
				 dma8ptr, ES1688_HW_AUTO, &pcm);
	if (err < 0) {
		snd_unregister_ioports(card);
		return NULL;
	}
	codec = snd_magic_cast(es1688_t, pcm->private_data, NULL);
	if (codec->hardware == ES1688_HW_688) {
		if (mpu_irqptr) {
			snd_printk("fatal: ES688 chip doesn't have MPU401\n");
			snd_device_free(card, pcm);
			snd_unregister_ioports(card);
			return NULL;
		}
	}
	return pcm;
}

static void snd_audiodrive_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_audiodrive *acard;

	acard = (struct snd_audiodrive *) dev_id;
	if (acard == NULL || acard->pcm == NULL)
		return;
	snd_es1688_interrupt(acard->pcm);
}

static void snd_audiodrive_midi_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_audiodrive *acard;

	acard = (struct snd_audiodrive *) dev_id;
	if (acard == NULL || acard->rmidi == NULL)
		return;
	snd_mpu401_uart_interrupt(acard->rmidi);
}

static int __init snd_audiodrive_resources(int dev,
					   struct snd_audiodrive *acard,
					   snd_card_t * card)
{
	static int possible_irqs[] = {5, 9, 10, 7, -1};
	static int possible_dmas[] = {1, 3, 0, -1};
	int err;

	if ((err = snd_register_interrupt(card, "ESx688", snd_irq[dev],
				SND_IRQ_TYPE_ISA, snd_audiodrive_interrupt,
				acard, possible_irqs, &acard->irqptr)) < 0)
		return err;
	acard->mpuirqptr = NULL;
	if (snd_mpu_port[dev] >= 0x300 && snd_mpu_port[dev] <= 0x330 &&
	    (snd_mpu_port[dev] & 0x00f) == 0) {
		if (snd_register_interrupt(card, "ESx688 - MPU401",
				snd_mpu_irq[dev], SND_IRQ_TYPE_ISA,
				snd_audiodrive_midi_interrupt, acard,
				possible_irqs, &acard->mpuirqptr) < 0) {
			snd_printk("warning: can't allocate IRQ for MPU401, midi port is disabled\n");
		}
	} else {
		acard->mpuirqptr = NULL;
	}
	if ((err = snd_register_dma_channel(card, "ESx688",
			snd_dma8[dev], SND_DMA_TYPE_ISA,
			snd_dma8_size[dev], possible_dmas, &acard->dma8ptr)) < 0) {
		return err;
	}
	return 0;
}

static int __init snd_audiodrive_probe(int dev, struct snd_audiodrive *acard)
{
	static int possible_ports[] = {0x220, 0x240, 0x260, -1};
	int *ports = possible_ports;
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *rmidi = NULL;
	snd_hwdep_t *synth = NULL;
	es1688_t *codec;
	int err;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			 snd_audiodrive_use_inc, snd_audiodrive_use_dec);
	if (card == NULL)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_ESS_ES1688;
	if (snd_audiodrive_resources(dev, acard, card) < 0) {
		snd_card_free(card);
		return -EBUSY;
	}
	pcm = NULL;
	if (snd_port[dev] == SND_AUTO_PORT) {
		for (ports = possible_ports; *ports >= 0; ports++) {
			pcm = snd_audiodrive_detect(card, *ports,
						   snd_mpu_port[dev],
						   acard->irqptr,
						   acard->mpuirqptr,
						   acard->dma8ptr);
			if (pcm)
				break;
		}
		if (pcm == NULL) {
			snd_card_free(card);
			return -ENODEV;
		}
	} else {
		pcm = snd_audiodrive_detect(card, snd_port[dev],
					    snd_mpu_port[dev], acard->irqptr,
					    acard->mpuirqptr, acard->dma8ptr);
		if (pcm == NULL) {
			snd_card_free(card);
			return -ENODEV;
		}
	}

	codec = snd_magic_cast(es1688_t, pcm->private_data, -ENXIO);

	err = snd_es1688_new_mixer(pcm, 0, &mixer);
	if (err < 0)
		goto __nodev;

	snd_opl3_new(card, 0, codec->port, codec->port + 2, OPL3_HW_OPL3, -1, &synth);

	if (acard->mpuirqptr) {
		err = snd_mpu401_uart_new(card, 0, MPU401_HW_ES1688,
					  codec->mpu_port,
					  acard->mpuirqptr->irq,
					  &rmidi);
		if (err < 0)
			goto __nodev;
	}
	if (snd_es1688_init(pcm, 1) < 0)
		goto __nodev;
	strcpy(card->abbreviation, "AudioDrive");
	strcpy(card->shortname, pcm->name);
	sprintf(card->longname, "%s at 0x%x, irq %i, dma %i",
		pcm->name,
		codec->port,
		acard->irqptr->irq,
		acard->dma8ptr->dma);
	if (!snd_card_register(card)) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		acard->rmidi = rmidi;
		acard->synth = synth;
		return 0;
	}

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

static int __init alsa_card_es1688_init(void)
{
	int dev, cards;
	struct snd_audiodrive *acard;

	for (dev = cards = 0; dev < SND_CARDS && snd_port[dev] > 0; dev++) {
		acard = (struct snd_audiodrive *)
				snd_kcalloc(sizeof(struct snd_audiodrive), GFP_KERNEL);
		if (acard == NULL)
			continue;
		if (snd_audiodrive_probe(dev, acard) < 0) {
			snd_kfree(acard);
			if (snd_port[dev] == SND_AUTO_PORT)
				break;
#ifdef MODULE
			snd_printk("ESS AudioDrive ES1688 soundcard #%i not found at 0x%x or device busy\n", dev + 1, snd_port[dev]);
#endif
			continue;
		}
		snd_audiodrive_cards[dev] = acard;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		snd_printk("ESS AudioDrive ES1688 soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

static void __exit alsa_card_es1688_exit(void)
{
	int idx;
	struct snd_audiodrive *acard;

	for (idx = 0; idx < SND_CARDS; idx++) {
		acard = snd_audiodrive_cards[idx];
		if (acard) {
			snd_card_unregister(acard->card);
			snd_kfree(acard);
		}
	}
}

module_init(alsa_card_es1688_init)
module_exit(alsa_card_es1688_exit)
