
/*
    card-ad1816a.c - driver for ADI SoundPort AD1816A based soundcards.
    Copyright (C) 2000 by Massimo Piccioni <dafastidio@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/initval.h"
#include "../include/ad1816a.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"

MODULE_DESCRIPTION("\
Driver: AD1816A, AD1815\n\
Card: ADI SoundPort AD1816A\n\
Card: ADI SoundComm AD1815\n\
Card: Highscreen Sound-Boostar 16 3D\n\
ISAPNP: MDK1605=ADS7180,ADS7181\n\
ISAPNP: ADS7150=ADS7150,ADS7151\n\
ISAPNP: ADS7181=ADS7180,ADS7181\n\
ISAPNP: TER1411=ADS7180,ADS7181\n\
");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;		/* Index 1-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;		/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_mpu_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_fm_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* Pnp setup */
int snd_mpu_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* Pnp setup */
int snd_dma1[SND_CARDS] = SND_DEFAULT_DMA;		/* PnP setup */
int snd_dma1_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
int snd_dma2[SND_CARDS] = SND_DEFAULT_DMA;		/* PnP setup */
int snd_dma2_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */

#ifdef MODULE_PARM
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for ad1816a based soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for ad1816a based soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for ad1816a driver. [PORT12]");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for ad1816a driver. [PORT12]");
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_fm_port, "FM port # for ad1816a driver. [PORT12]");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for ad1816a driver. [IRQ]");
MODULE_PARM(snd_mpu_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_irq, "MPU-401 IRQ # for ad1816a driver. [IRQ]");
MODULE_PARM(snd_dma1, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1, "1st DMA # for ad1816a driver. [DMA]");
MODULE_PARM(snd_dma1_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma1_size, "1st DMA size for ad1816a driver. [DMA_SIZE]");
MODULE_PARM(snd_dma2, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2, "2nd DMA # for ad1816a driver. [DMA]");
MODULE_PARM(snd_dma2_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma2_size, "2nd DMA size for ad1816a driver. [DMA_SIZE]");
#endif	/* MODULE_PARM */

static struct snd_card_ad1816a {
	snd_irq_t *irqptr;
	snd_irq_t *mpuirqptr;
	snd_dma_t *dma1ptr;
	snd_dma_t *dma2ptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;
#ifdef __ISAPNP__
	struct isapnp_dev *dev;
	struct isapnp_dev *devmpu;
#endif	/* __ISAPNP__ */
} *snd_card_ad1816a_cards[SND_CARDS] = SND_DEFAULT_PTR;

#ifdef __ISAPNP__
static unsigned int snd_card_ad1816a_pnpids[] = {
	/* Highscreen Sound-Boostar 16 3D */
	(ISAPNP_VENDOR('M','D','K')<<16)|ISAPNP_DEVICE(0x1605),    /* device */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7180),  /* audio */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7181),  /* mpu401 */
	/* Analog Devices AD1815 */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_DEVICE(0x7150),    /* device */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7150),  /* audio */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7151),  /* mpu401 */
	/* Analog Devices AD1816A - added by Kenneth Platz <kxp@atl.hp.com> */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_DEVICE(0x7181),    /* device */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7180),  /* audio */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7181),  /* mpu401 */
	/* Analog Devices AD1816A - Terratec Base 64 */
	(ISAPNP_VENDOR('T','E','R')<<16)|ISAPNP_DEVICE(0x1411),    /* device */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7180),  /* audio */
	(ISAPNP_VENDOR('A','D','S')<<16)|ISAPNP_FUNCTION(0x7181),  /* mpu401 */
	0	/* end */
};
#endif	/* __ISAPNP__ */

#define	DRIVER_NAME	"snd-card-ad1816a"

#define snd_card_ad1816a_printk(args...)	snd_printk(__FILE__": " ##args)


static void snd_card_ad1816a_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_ad1816a_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static void snd_card_ad1816a_interrupt(int irq, void *dev_id,
				       struct pt_regs *regs)
{
	struct snd_card_ad1816a *acard;
	ad1816a_t *codec;
	snd_pcm_t *pcm;
	unsigned char status = 0;

	if (!(acard = (struct snd_card_ad1816a *) dev_id))
		return;

	if (!(pcm = acard->pcm))
		return;

	codec = snd_magic_cast(ad1816a_t, pcm->private_data, );
	if (!codec)
		return;

	snd_ad1816a_interrupt(pcm, status);
}

static void snd_card_ad1816a_mpu_interrupt(int irq, void *dev_id,
					   struct pt_regs *regs)
{
	struct snd_card_ad1816a *acard;

	if (!(acard = (struct snd_card_ad1816a *) dev_id))
		return;

	snd_mpu401_uart_interrupt(acard->rmidi);
}

static int __init snd_card_ad1816a_resources(int dev,
					     struct snd_card_ad1816a *acard,
					     snd_card_t *card)
{
	int error;
	static int no_alternatives[] = {-1};

	if (snd_register_ioport(card, snd_mpu_port[dev], 2,
			DRIVER_NAME" - MPU-401", NULL) < 0)
		snd_mpu_port[dev] = -1;
	else if (snd_register_interrupt(card,
			DRIVER_NAME" - MPU-401", snd_mpu_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_ad1816a_mpu_interrupt,
			acard, no_alternatives, &acard->mpuirqptr) < 0) {
		snd_unregister_ioports(card);
		snd_mpu_port[dev] = -1;
	}

	if ((error = snd_register_ioport(card, snd_port[dev], 16,
			DRIVER_NAME" - SS", NULL)) < 0)
		return error;
	if ((error = snd_register_interrupt(card,
			DRIVER_NAME" - SS", snd_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_ad1816a_interrupt,
			acard, no_alternatives, &acard->irqptr)) < 0)
		return error;
	if ((error = snd_register_dma_channel(card,
			DRIVER_NAME" - SS", snd_dma1[dev],
			SND_DMA_TYPE_ISA, snd_dma1_size[dev],
			no_alternatives, &acard->dma1ptr)) < 0)
		return error;
	if ((error = snd_register_dma_channel(card,
			DRIVER_NAME" - SS", snd_dma2[dev],
			SND_DMA_TYPE_ISA, snd_dma2_size[dev],
			no_alternatives, &acard->dma2ptr)) < 0)
		return error;

	if (snd_register_ioport(card, snd_fm_port[dev], 8,
			DRIVER_NAME" - FM", NULL) < 0)
		snd_fm_port[dev] = -1;

	return 0;
}

#ifdef __ISAPNP__
static int __init snd_card_ad1816a_isapnp(int dev,
					  struct snd_card_ad1816a *acard)
{
	static int idx = 0;
	static struct isapnp_card *card = NULL;
        unsigned int tmp;
	struct isapnp_dev *pdev;

      __again:
	while ((tmp = snd_card_ad1816a_pnpids[idx]) != 0) {
		if ((card = isapnp_find_card(tmp >> 16, tmp & 0xffff, card)))
			break;
		idx += 3;
	}

	if (card == NULL)
		return -ENODEV;

	tmp = snd_card_ad1816a_pnpids[idx+1];
	acard->dev = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);
	if (!acard->dev || acard->dev->active)
		goto __again;		

	tmp = snd_card_ad1816a_pnpids[idx+2];
	acard->devmpu = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);

	pdev = acard->dev;
	if (pdev->prepare(pdev) < 0)
		return -EAGAIN;

	if (snd_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[2], snd_port[dev], 16);
	if (snd_fm_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[1], snd_fm_port[dev], 4);
	if (snd_dma1[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[0], snd_dma1[dev],
			1);
	if (snd_dma2[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[1], snd_dma2[dev],
			1);
	if (snd_irq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_irq[dev], 1);

	if (pdev->activate(pdev) < 0) {
		snd_card_ad1816a_printk("AUDIO isapnp configure failure\n");
		return -EBUSY;
	}

	snd_port[dev] = pdev->resource[2].start;
	snd_fm_port[dev] = pdev->resource[1].start;
	snd_dma1[dev] = pdev->dma_resource[0].start;
	snd_dma2[dev] = pdev->dma_resource[1].start;
	snd_irq[dev] = pdev->irq_resource[0].start;

	pdev = acard->devmpu;
	if (pdev == NULL || pdev->activate(pdev) < 0) {
		snd_mpu_port[dev] = -1;
		return 0;
	}

	if (snd_mpu_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_mpu_port[dev],
			2);
	if (snd_mpu_irq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_mpu_irq[dev],
			1);

	if (pdev->activate(pdev) < 0) {
		/* not fatal error */
		snd_card_ad1816a_printk("MPU-401 isapnp configure failure\n");
		snd_mpu_port[dev] = -1;
		acard->devmpu = NULL;
	} else {
		snd_mpu_port[dev] = pdev->resource[0].start;
		snd_mpu_irq[dev] = pdev->irq_resource[0].start;
	}

	return 0;
}

static void snd_card_ad1816a_deactivate(struct snd_card_ad1816a *acard)
{
	if (acard->dev)
		acard->dev->deactivate(acard->dev);
	if (acard->devmpu)
		acard->devmpu->deactivate(acard->devmpu);
}
#endif	/* __ISAPNP__ */

static int __init snd_card_ad1816a_probe(int dev,
					 struct snd_card_ad1816a *acard)
{
	int error;
	snd_card_t *card;
	ad1816a_t *codec;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;

	if (!(card = snd_card_new(snd_index[dev], snd_id[dev],
			snd_card_ad1816a_use_inc, snd_card_ad1816a_use_dec)))
		return -ENOMEM;
	card->type = SND_CARD_TYPE_AD1816A;

#ifdef __ISAPNP__
	if ((error = snd_card_ad1816a_isapnp(dev, acard)))
		goto __on_error;
#else
	snd_card_ad1816a_printk("you have to enable PnP support.\n");
	return -ENOSYS;
#endif	/* __ISAPNP__ */

	if ((error = snd_card_ad1816a_resources(dev, acard, card)))
		goto __on_error;

	if (snd_ad1816a_new_pcm(card, 0, snd_port[dev],
			acard->irqptr, acard->dma1ptr, acard->dma2ptr,
			&pcm) < 0)
		goto __on_error;

	codec = snd_magic_cast(ad1816a_t, pcm->private_data, -ENXIO);

	if (snd_ad1816a_new_mixer(pcm, 0, &mixer) < 0)
		goto __on_error;

	rmidi = NULL;
	if (snd_mpu_port[dev] > 0) {
		if (snd_mpu401_uart_new(card, 0, MPU401_HW_MPU401,
				snd_mpu_port[dev], snd_mpu_irq[dev],
				&rmidi) < 0)
			snd_card_ad1816a_printk("no MPU-401 device at 0x%x.\n",
				snd_mpu_port[dev]);
	}

	synth = NULL;
	if (snd_fm_port[dev] > 0) {
		if (snd_opl3_new(card, 0,
				 snd_fm_port[dev], snd_fm_port[dev] + 2,
				 OPL3_HW_AUTO, 0, &synth) < 0)
			snd_card_ad1816a_printk("no OPL device at 0x%x-0x%x.\n",
				snd_fm_port[dev], snd_fm_port[dev] + 2);
	}

	strcpy(card->abbreviation, "AD1816A");
	strcpy(card->shortname, "ADI SoundPort AD1816A");
	sprintf(card->longname, "%s soundcard, SS at 0x%x, irq %i, dma %i&%i",
		card->shortname, codec->port, acard->irqptr->irq,
		acard->dma1ptr->dma, acard->dma2ptr->dma);

	if (!(error = snd_card_register(card))) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		acard->rmidi = rmidi;
		acard->synth = synth;
		return 0;
	}

      __on_error:
	snd_card_free(card);
	return error ? error : -ENXIO;
}

static void __exit snd_card_ad1816a_free(int dev)
{
	struct snd_card_ad1816a *acard;

	acard = snd_card_ad1816a_cards[dev];
	if (acard) {
		snd_card_ad1816a_cards[dev] = NULL;
		snd_card_unregister(acard->card);
#ifdef __ISAPNP__
		snd_card_ad1816a_deactivate(acard);
#endif	/* __ISAPNP__ */
		snd_kfree(acard);
	}
}

static int __init alsa_card_ad1816a_init(void)
{
	int cards, dev;
	struct snd_card_ad1816a *acard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		acard = (struct snd_card_ad1816a *) snd_kcalloc(
			sizeof(struct snd_card_ad1816a), GFP_KERNEL);
		if (acard == NULL)
			break;

		if (snd_card_ad1816a_probe(dev, acard)) {
#ifdef __ISAPNP__
			snd_card_ad1816a_deactivate(acard);
#endif	/* __ISAPNP__ */
			snd_kfree(acard);
			continue;
		}
		snd_card_ad1816a_cards[cards++] = acard;
	}

#ifdef MODULE
	if (!cards)
		snd_card_ad1816a_printk("no AD1816A based soundcards found.\n");
#endif	/* MODULE */
	return cards ? 0 : -ENODEV;
}

static void __exit alsa_card_ad1816a_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_card_ad1816a_free(dev);
}

module_init(alsa_card_ad1816a_init)
module_exit(alsa_card_ad1816a_exit)
