/*
 *  ALSA sequencer Memory Manager
 *  Copyright (c) 1998 by Frank van de Pol <frank@vande-pol.demon.nl>
 *                        Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include "../../include/driver.h"

#include "../../include/seq_kernel.h"
#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_info.h"
#include "seq_lock.h"

/* define to keep statistics on successful allocations (for debugging) */
#define TRACK_SUCCESS

/* semaphore in struct file record */
#define semaphore_of(fp)	((fp)->f_dentry->d_inode->i_sem)

/* counter for tracking memory leaks for 'external' data */
static int ext_alloc = 0;
static int ext_alloc_max = 0;
static unsigned long ext_alloc_bytes = 0;
static unsigned long ext_alloc_max_bytes = 0;
static unsigned long ext_alloc_biggest_alloc = 0;

/* keep track of failures */
static int ext_alloc_failures = 0;

#ifdef TRACK_SUCCESS
/* keep track of successfull allocations */
static int ext_alloc_success = 0;
#endif

inline static int snd_seq_output_ok(pool_t *pool)
{
	return pool->available >= pool->room;
}


/* release this cell, free extended data if available */
void snd_seq_cell_free(snd_seq_event_cell_t * cell)
{
	unsigned long flags;
	void *ptr;
	unsigned int len;
	pool_t *pool;

	if (cell == NULL)
		return;
	pool = cell->pool;
	if (cell->pool == NULL) {
		snd_printd("oops: snd_seq_cell called with pool == NULL?\n");
		return;
	}
	spin_lock_irqsave(&pool->lock, flags);
	if (pool->free != NULL) {
		/* normal situation */
		cell->ptr_l = pool->free;	/* chain in old element */
		cell->ptr_r = NULL;
		pool->free = cell;
	} else {
		/* first element */
		pool->free = cell;
		cell->ptr_l = NULL;
		cell->ptr_r = NULL;
	}
	pool->available++;
	if ((cell->event.flags & SND_SEQ_EVENT_LENGTH_MASK) == SND_SEQ_EVENT_LENGTH_VARIABLE) {
		ptr = cell->event.data.ext.ptr;
		len = cell->event.data.ext.len;
	} else {
		ptr = NULL;
		len = 0;
	}
	cell->event.flags = 0;	/* for sure */
	spin_unlock_irqrestore(&pool->lock, flags);
	if (ptr && len >= 0)
		snd_seq_ext_free(ptr, len);

	/* if the client is sleeping, wake it up */
	spin_lock_irqsave(&pool->output_sleep_lock, flags);
	if (waitqueue_active(&pool->output_sleep)) {
		/* has enough space now? */
		if (snd_seq_output_ok(pool))
			wake_up(&pool->output_sleep);
	}
	spin_unlock_irqrestore(&pool->output_sleep_lock, flags);
}


/* return pointer to cell. NULL on failure */
snd_seq_event_cell_t *snd_seq_cell_alloc(pool_t *pool)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;

	if (pool == NULL)
		return NULL;
	spin_lock_irqsave(&pool->lock, flags);
	if (pool->ptr == NULL) {	/* not initialized */
		pool->event_alloc_nopool++;
		spin_unlock_irqrestore(&pool->lock, flags);
		return NULL;
	}
	if (pool->free != NULL) {
		cell = pool->free;
		pool->free = cell->ptr_l;
		pool->available--;
		if (pool->available < pool->min_available)
			pool->min_available = pool->available;

		/* clear cell pointers */
		cell->ptr_l = NULL;
		cell->ptr_r = NULL;
	} else {
		/* no element available... */
		/*snd_printd("Seq: cell_alloc failed: no cells available\n");*/
		cell = NULL;
	}
	if (cell == NULL)
		pool->event_alloc_failures++;
	else
		pool->event_alloc_success++;
	spin_unlock_irqrestore(&pool->lock, flags);

	return cell;
}


/* convert from LENGTH_VARUSR type to LENGTH_VARIABLE type */
int snd_seq_event_convert_varlen(snd_seq_event_t *event, int atomic)
{
	char *ptr;

	if ((event->flags & SND_SEQ_EVENT_LENGTH_MASK) != SND_SEQ_EVENT_LENGTH_VARUSR)
		return 0;

	if (event->data.ext.len < 0 || event->data.ext.len >= SND_SEQ_MAX_EVENT_LEN)
		return -EINVAL;

	/* allocate kernel buffer */
	ptr = snd_seq_ext_malloc(event->data.ext.len, atomic);
	if (ptr == NULL)
		return -ENOMEM;

	/* copy data to kernel space */
	if (copy_from_user(ptr, event->data.ext.ptr, event->data.ext.len)) {
		snd_seq_ext_free(event->data.ext.ptr, event->data.ext.len);
		return -EFAULT;
	}

	event->flags &= ~SND_SEQ_EVENT_LENGTH_MASK;
	event->flags |= SND_SEQ_EVENT_LENGTH_VARIABLE;
	event->data.ext.ptr = ptr;

	return 0;
}

/* duplicate event, NULL on failure */
int snd_seq_event_dup(pool_t *pool, snd_seq_event_t *event, snd_seq_event_cell_t **cellp, int atomic)
{
	int err;
	snd_seq_event_cell_t *new_cell;

	*cellp = NULL;

	snd_debug_check(event == NULL, -EINVAL);

	new_cell = snd_seq_cell_alloc(pool);
	if (new_cell == NULL)
		return -EAGAIN;

	memcpy(&new_cell->event, event, sizeof(snd_seq_event_t));
	switch (event->flags & SND_SEQ_EVENT_LENGTH_MASK) {
	case SND_SEQ_EVENT_LENGTH_VARIABLE:
		/* increment reference counter */
		snd_seq_ext_ref(event->data.ext.ptr);
 		break;
	case SND_SEQ_EVENT_LENGTH_VARUSR:
		/* convert type to LENGTH_VARIABLE */
		err = snd_seq_event_convert_varlen(&new_cell->event, atomic);
		if (err) {
			snd_seq_cell_free(new_cell);
			return err;
		}
		break;
	}
	*cellp = new_cell;
	return 0;
}


/* duplicate a cell with blocking */
int snd_seq_event_dup_blocking(pool_t *pool, snd_seq_event_t *event, snd_seq_event_cell_t **cellp, struct file *file)
{
	snd_seq_event_cell_t *cell;
	unsigned long flags;
	int err = 0;

	*cellp = cell = NULL;
	spin_lock_irqsave(&pool->output_sleep_lock, flags);
	if (pool->available > 0) {
		err = snd_seq_event_dup(pool, event, &cell, 0); /* non-atomic */
		if (err != -EAGAIN)
			goto __error_exit;
	}

	/* sleep until pool has enough room */
	while (cell == NULL) {
		/* change semaphore to allow other clients
		   to access device file */
		if (file)
			up(&semaphore_of(file));

		snd_seq_sleep_in_lock(&pool->output_sleep, &pool->output_sleep_lock);

		/* restore semaphore again */
		if (file)
			down(&semaphore_of(file));

		/* interrupted? */
		if (signal_pending(current)) {
			err = -EINTR;
			break;
		}

		/* try again.. */
		err = snd_seq_event_dup(pool, event, &cell, 0);
		if (err != -EAGAIN)
			break;
	}
__error_exit:
	spin_unlock_irqrestore(&pool->output_sleep_lock, flags);
	*cellp = cell;
	return err;
}


/* poll wait */
int snd_seq_pool_poll_wait(pool_t *pool, struct file *file, poll_table *wait)
{
	poll_wait(file, &pool->output_sleep, wait);
	return snd_seq_output_ok(pool);
}


/* return number of unused (free) cells */
int snd_seq_unused_cells(pool_t *pool)
{
	if (pool == NULL)
		return 0;
	return (pool->available);
}


/* return total number of allocated cells */
int snd_seq_total_cells(pool_t *pool)
{
	if (pool == NULL)
		return 0;
	return (pool->total_elements);
}


/* allocate room specified number of events */
int snd_seq_pool_init(pool_t *pool)
{
	int cell;
	snd_seq_event_cell_t *ptr, *cellptr;
	unsigned long flags;
	int events;

	snd_debug_check(pool == NULL, -EINVAL);
	if (pool->ptr)			/* should be atomic? */
		return 0;

	events = pool->size;
	/* alloc memory */
	ptr = snd_kcalloc(sizeof(snd_seq_event_cell_t) * events, GFP_KERNEL);
	if (ptr) {

		/* add the new cell's to the free cell list by calling the free()
		   function for each one */
		for (cell = 0; cell < events; cell++) {
			cellptr = &ptr[cell];
			cellptr->pool = pool;
			snd_seq_cell_free(cellptr);
		}
		/* init statistics */
		spin_lock_irqsave(&pool->lock, flags);
		if (!pool->min_available)
			pool->min_available = pool->available;
		pool->total_elements = events;
		pool->ptr = ptr;
		spin_unlock_irqrestore(&pool->lock, flags);
	} else {
		snd_printd("Seq: malloc for sequencer events failed\n");
		return -ENOMEM;
	}
	return 0;
}

/* remove events */
int snd_seq_pool_done(pool_t *pool)
{
	unsigned long flags;
	snd_seq_event_cell_t *ptr;	
	unsigned int total_elements;

	snd_debug_check(pool == NULL, -EINVAL);
	spin_lock_irqsave(&pool->lock, flags);
	if (pool->available != pool->total_elements) {
		spin_unlock_irqrestore(&pool->lock, flags);
		snd_printd("seq: snd_seq_pool_done() - pool isn't free!!\n");
		return -EBUSY;
	}
	ptr = pool->ptr;
	total_elements = pool->total_elements;
	pool->ptr = NULL;
	pool->free = NULL;
	pool->total_elements = 0;
	pool->available = 0;
	spin_unlock_irqrestore(&pool->lock, flags);
	if (ptr == NULL)
		return 0;
	snd_kfree(ptr);
	return 0;
}

/* init new memory pool */
pool_t *snd_seq_pool_new(int poolsize)
{
	pool_t *pool;

	/* create pool block */
	pool = snd_kcalloc(sizeof(pool_t), GFP_KERNEL);
	if (pool == NULL) {
		snd_printd("seq: malloc failed for pool\n");
		return NULL;
	}
	pool->lock = SPIN_LOCK_UNLOCKED;
	pool->ptr = NULL;
	pool->free = NULL;
	pool->total_elements = 0;
	pool->available = 0;
	init_waitqueue_head(&pool->output_sleep);
	pool->output_sleep_lock = SPIN_LOCK_UNLOCKED;
	
	pool->size = poolsize;
	pool->room = (poolsize + 1) / 2;

	/* init statistics */
	pool->min_available = pool->available;
	return pool;
}

/* remove memory pool */
int snd_seq_pool_delete(pool_t **ppool)
{
	pool_t *pool = *ppool;

	*ppool = NULL;
	if (pool == NULL)
		return 0;
	if (snd_seq_pool_done(pool) == -EBUSY) {
		pool->available = pool->total_elements;
		snd_seq_pool_done(pool);
	}
	snd_kfree(pool);
	return 0;
}

/* initialize sequencer memory */
void __init snd_sequencer_memory_init(void)
{
}

/* release sequencer memory */
void __exit snd_sequencer_memory_done(void)
{
	if (ext_alloc > 0) {
		snd_printd("seq: memory leak alert, still %d blocks of external data allocated\n", ext_alloc);
	}
}


/* wrapper for allocating and freeing 'external' data (eg. sysex, meta
   events etc.) for now it is just passed to the kmalloc and kfree
   calls, but in a later stadium a different allocation method could be
   used. */
/* ADDITION: reference counter is allocated at the head of each block
   followed by available space -- Iwai */
void *snd_seq_ext_malloc(unsigned long size, int atomic)
{
	int *refp;

	refp = snd_kmalloc(size + sizeof(int), atomic ? GFP_ATOMIC : GFP_KERNEL);
	if (refp) {
		*refp = 1; /* initialize reference counter */
		ext_alloc++;
		ext_alloc_bytes += size;
		if (ext_alloc > ext_alloc_max)
			ext_alloc_max = ext_alloc;
		if (ext_alloc_bytes > ext_alloc_max_bytes)
			ext_alloc_max_bytes = ext_alloc_bytes;
		if (size > ext_alloc_biggest_alloc)
			ext_alloc_biggest_alloc = size;
#ifdef TRACK_SUCCESS
		ext_alloc_success++;
#endif
		return (void*)(refp + 1);
	} else {
		ext_alloc_failures++;
		return NULL;
	}
}

void snd_seq_ext_ref(void *obj)
 {
	if (obj) {
		int *refp = (int*)obj - 1;
		(*refp)++;
	}
}

void snd_seq_ext_free(void *obj, unsigned long size)
{
	if (obj) {
		int *refp = (int*)obj - 1;
		(*refp)--;
		if (*refp <= 0) {
			ext_alloc--;
			ext_alloc_bytes -= size;

			if (ext_alloc < 0) {
				snd_printk("seq: whoops, more ext data free()s than malloc()...\n");
			}
			snd_kfree(refp);
		}
	}
}


/* exported to seq_clientmgr.c */
void snd_seq_info_pool(snd_info_buffer_t * buffer, pool_t *pool, char *space)
{
	if (pool == NULL)
		return;
	snd_iprintf(buffer, "%sPool size          : %d\n", space, snd_seq_total_cells(pool));
	snd_iprintf(buffer, "%sAvailable cells    : %d\n", space, snd_seq_unused_cells(pool));
	snd_iprintf(buffer, "%sCells in use       : %d\n", space, snd_seq_total_cells(pool) - snd_seq_unused_cells(pool));
	snd_iprintf(buffer, "%sPeak cells in use  : %d\n", space, snd_seq_total_cells(pool) - pool->min_available);
	snd_iprintf(buffer, "%sAlloc success      : %d\n", space, pool->event_alloc_success);
	snd_iprintf(buffer, "%sAlloc failures     : %d\n", space, pool->event_alloc_failures);
	snd_iprintf(buffer, "%sAlloc no-pool      : %d\n", space, pool->event_alloc_nopool);
}

/* exported to seq_info.c */
void snd_seq_info_memory_read(snd_info_buffer_t * buffer, void *private_data)
{
	/*
	int c;
	queue_t *q;

	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);
		if (q == NULL)
			continue;
		snd_iprintf(buffer, "Event pool %i\n", c);
		snd_seq_info_pool(buffer, q->pool, "  ");
		snd_iprintf(buffer, "\n");
	}
	*/
	snd_iprintf(buffer, "External data\n");
	snd_iprintf(buffer, "  Blocks in use      : %d\n", ext_alloc);
	snd_iprintf(buffer, "  Bytes in use       : %d\n", ext_alloc_bytes);
	snd_iprintf(buffer, "  Peak blocks in use : %d\n", ext_alloc_max);
	snd_iprintf(buffer, "  Peak bytes in use  : %d\n", ext_alloc_max_bytes);
	snd_iprintf(buffer, "  Largest allocation : %d\n", ext_alloc_biggest_alloc);
#ifdef TRACK_SUCCESS
	snd_iprintf(buffer, "  Alloc success      : %d\n", ext_alloc_success);
#endif
	snd_iprintf(buffer, "  Alloc failures     : %d\n", ext_alloc_failures);
}
