/*
 *  synth callback routines for Emu10k1
 *
 *  Copyright (C) 2000 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "emu10k1_synth_local.h"

/* voice status */
enum {
	V_OFF=0, V_RELEASED, V_PLAYING, V_END
};

/* Keeps track of what we are finding */
typedef struct best_voice {
	unsigned int time;
	int voice;
} best_voice_t;

/*
 * prototypes
 */
static void lookup_voices(snd_emux_t *emu, emu10k1_t *hw, best_voice_t *best, int active_only);
static snd_emux_voice_t *get_voice(snd_emux_t *emu, snd_emux_port_t *port);
static void start_voice(snd_emux_voice_t *vp);
static void trigger_voice(snd_emux_voice_t *vp);
static void release_voice(snd_emux_voice_t *vp);
static void update_voice(snd_emux_voice_t *vp, int update);
static void terminate_voice(snd_emux_voice_t *vp);
static void free_voice(snd_emux_voice_t *vp);
static void use_inc(void *hw);
static void use_dec(void *hw);

static void set_fmmod(emu10k1_t *hw, snd_emux_voice_t *vp);
static void set_fm2frq2(emu10k1_t *hw, snd_emux_voice_t *vp);
static void set_filterQ(emu10k1_t *hw, snd_emux_voice_t *vp);

/*
 * Ensure a value is between two points
 * macro evaluates its args more than once, so changed to upper-case.
 */
#define LIMITVALUE(x, a, b) do { if ((x) < (a)) (x) = (a); else if ((x) > (b)) (x) = (b); } while (0)
#define LIMITMAX(x, a) do {if ((x) > (a)) (x) = (a); } while (0)


/*
 * set up operators
 */
static snd_emux_operators_t emu10k1_ops = {
	get_voice,	/* get_voice */
	start_voice,	/* prepare */
	trigger_voice,	/* trigger */
	release_voice,	/* release */
	update_voice,	/* update */
	terminate_voice,/* terminate */
	free_voice,	/* free_voice */
	NULL,		/* reset */
	use_inc,	/* use_inc */
	use_dec,	/* use_dec */
	snd_emu10k1_sample_new,	/* sample_new */
	snd_emu10k1_sample_free,	/* sample_new */
	NULL,		/* sample_reset */
	NULL,		/* load_fx */
	NULL,		/* sysex */
#ifdef CONFIG_SND_OSSEMUL
	NULL,		/* oss_ioctl */
#endif
};

void
snd_emu10k1_ops_setup(snd_emux_t *emu)
{
	memcpy(&emu->ops, &emu10k1_ops, sizeof(snd_emux_operators_t));
}


/*
 * get more voice
 */
int
snd_emu10k1_synth_get_voice(emu10k1_t *hw)
{
	snd_emux_t *emu;
	snd_emux_voice_t *vp;
	best_voice_t best[V_END];
	unsigned long flags;
	int i;

	emu = snd_magic_cast(snd_emux_t, hw->synth, -EINVAL);

	spin_lock_irqsave(&emu->voice_lock, flags);
	lookup_voices(emu, hw, best, 1); /* no OFF voices */
	for (i = 0; i < V_END; i++) {
		if (best[i].voice >= 0) {
			int ch;
			vp = &emu->voices[best[i].voice];
			if ((ch = vp->ch) < 0) {
				//printk("synth_get_voice: ch < 0 (%d) ??", i);
				continue;
			}
			/* don't call snd_emu10k1_voice_free here -
			   it causes lock up */
			vp->emu->num_voices--;
			vp->ch = -1;
			vp->state = SND_EMUX_ST_OFF;
			spin_unlock_irqrestore(&emu->voice_lock, flags);
			return ch;
		}
	}
	spin_unlock_irqrestore(&emu->voice_lock, flags);

	/* not found */
	return -ENOMEM;
}


/*
 * turn off the voice (not terminated)
 */
static void
release_voice(snd_emux_voice_t *vp)
{
	int dcysusv;
	emu10k1_t *hw;
	
	hw = snd_magic_cast(emu10k1_t, vp->hw, );
	dcysusv = 0x8000 | (unsigned char)vp->reg.parm.modrelease;
	snd_emu10k1_ptr_write(hw, DCYSUSM, vp->ch, dcysusv);
	dcysusv = 0x8000 | (unsigned char)vp->reg.parm.volrelease | ENV_ON;
	snd_emu10k1_ptr_write(hw, DCYSUSV, vp->ch, dcysusv);
}


/*
 * terminate the voice and release it to system
 */
static void
terminate_voice(snd_emux_voice_t *vp)
{
	emu10k1_t *hw;
	
	hw = snd_magic_cast(emu10k1_t, vp->hw, );
	snd_emu10k1_ptr_write(hw, IFATN, vp->ch, 0xffff);
	snd_emu10k1_ptr_write(hw, DCYSUSV, vp->ch, ENV_OFF);
	snd_emu10k1_ptr_write(hw, VTFT, vp->ch, 0xffff);
	snd_emu10k1_ptr_write(hw, PTRX, vp->ch, 0);
	snd_emu10k1_ptr_write(hw, CVCF, vp->ch, 0xffff);
}

static void
free_voice(snd_emux_voice_t *vp)
{
	emu10k1_t *hw;
	
	hw = snd_magic_cast(emu10k1_t, vp->hw, );
	if (vp->ch >= 0) {
		snd_emu10k1_voice_free(hw, &hw->voices[vp->ch]);
		vp->emu->num_voices--;
		vp->ch = -1;
	}
}


/*
 * update registers
 */
static void
update_voice(snd_emux_voice_t *vp, int update)
{
	emu10k1_t *hw;
	
	hw = snd_magic_cast(emu10k1_t, vp->hw, );
	if (update & SND_EMUX_UPDATE_VOLUME)
		snd_emu10k1_ptr_write(hw, IFATN_ATTENUATION, vp->ch, vp->avol);
	if (update & SND_EMUX_UPDATE_PITCH)
		snd_emu10k1_ptr_write(hw, IP, vp->ch, vp->apitch);
	if (update & SND_EMUX_UPDATE_PAN) {
		snd_emu10k1_ptr_write(hw, PSST_FXSENDAMOUNT_C, vp->ch, vp->apan);
		snd_emu10k1_ptr_write(hw, PTRX_FXSENDAMOUNT_B, vp->ch, vp->aaux);
	}
	if (update & SND_EMUX_UPDATE_FMMOD)
		set_fmmod(hw, vp);
	if (update & SND_EMUX_UPDATE_TREMFREQ)
		snd_emu10k1_ptr_write(hw, TREMFRQ, vp->ch, vp->reg.parm.tremfrq);
	if (update & SND_EMUX_UPDATE_FM2FRQ2)
		set_fm2frq2(hw, vp);
	if (update & SND_EMUX_UPDATE_Q)
		set_filterQ(hw, vp);
}


/*
 * look up voice table - get the best voice in order of preference
 */
static void
lookup_voices(snd_emux_t *emu, emu10k1_t *hw, best_voice_t *best, int active_only)
{
	snd_emux_voice_t *vp;
	best_voice_t *bp;
	int  i;

	for (i = 0; i < V_END; i++) {
		best[i].time = (unsigned int)-1; /* XXX MAX_?INT really */;
		best[i].voice = -1;
	}

	/*
	 * Go through them all and get a best one to use.
	 * NOTE: could also look at volume and pick the quietest one.
	 */
	for (i = 0; i < emu->max_voices; i++) {
		int state, val;

		vp = &emu->voices[i];
		state = vp->state;
		if (state == SND_EMUX_ST_OFF) {
			if (vp->ch < 0) {
				if (active_only)
					continue;
				bp = best + V_OFF;
			} else {
				//printk("off but active %d\n", vp->ch);
				continue;
			}
		}
		else if (state == SND_EMUX_ST_RELEASED) {
			bp = best + V_RELEASED;
			val = snd_emu10k1_ptr_read(hw, CVCF_CURRENTVOL, vp->ch);
			if (! val)
				bp = best + V_OFF;
		}
		else if (state & SND_EMUX_ST_ON)
			bp = best + V_PLAYING;
		else
			continue;

		/* check if sample is finished playing (non-looping only) */
		if (bp != best + V_OFF &&
		    (vp->reg.sample_mode & SND_SFNT_SAMPLE_SINGLESHOT)) {
			val = snd_emu10k1_ptr_read(hw, CCCA_CURRADDR, vp->ch);
			if (val >= vp->reg.loopstart)
				bp = best + V_OFF;
		}

		if (vp->time < bp->time) {
			bp->time = vp->time;
			bp->voice = i;
		}
	}
}

/*
 * get an empty voice - if all voices are used, try to clear the best voice
 */
static snd_emux_voice_t *
get_voice(snd_emux_t *emu, snd_emux_port_t *port)
{
	emu10k1_t *hw;
	snd_emux_voice_t *vp;
	best_voice_t best[V_END];
	int i;

	hw = snd_magic_cast(emu10k1_t, emu->hw, NULL);

	lookup_voices(emu, hw, best, 0);
	for (i = 0; i < V_END; i++) {
		if (best[i].voice >= 0) {
			vp = &emu->voices[best[i].voice];
			if (vp->ch < 0) {
				/* allocate a voice */
				emu10k1_voice_t *hwvoice;
				if (snd_emu10k1_voice_alloc(hw, EMU10K1_SYNTH, 0, &hwvoice) < 0 || hwvoice == NULL)
					continue;
				vp->ch = hwvoice->number;
				emu->num_voices++;
			}
			return vp;
		}
	}

	/* not found */
	return NULL;
}

/*
 * prepare envelopes and LFOs
 */
static void
start_voice(snd_emux_voice_t *vp)
{
	unsigned int temp;
	int ch;
	int addr;
	soundfont_voice_parm_block_t *parm;
	snd_midi_channel_t *chan;
	emu10k1_t *hw;
	
	hw = snd_magic_cast(emu10k1_t, vp->hw, );
	ch = vp->ch;
	snd_debug_check(ch < 0,);
	chan = vp->chan;
	parm = (soundfont_voice_parm_block_t*)&vp->reg.parm;

	/* channel to be silent and idle */
	snd_emu10k1_ptr_write(hw, DCYSUSV, ch, 0x0080);
	snd_emu10k1_ptr_write(hw, VTFT, ch, 0x0000FFFF);
	snd_emu10k1_ptr_write(hw, CVCF, ch, 0x0000FFFF);
	snd_emu10k1_ptr_write(hw, PTRX, ch, 0);
	snd_emu10k1_ptr_write(hw, CPF, ch, 0);

	/* set pitch offset */
	snd_emu10k1_ptr_write(hw, IP, vp->ch, vp->apitch);

	/* set envelope parameters */
	snd_emu10k1_ptr_write(hw, ENVVAL, ch, vp->reg.parm.moddelay);
	snd_emu10k1_ptr_write(hw, ATKHLDM, ch, vp->reg.parm.modatkhld);
	snd_emu10k1_ptr_write(hw, DCYSUSM, ch, vp->reg.parm.moddcysus);
	snd_emu10k1_ptr_write(hw, ENVVOL, ch, vp->reg.parm.voldelay);
	snd_emu10k1_ptr_write(hw, ATKHLDV, ch, vp->reg.parm.volatkhld);
	/* decay/sustain parameter for volume envelope is used
	   for triggerg the voice */

	/* cutoff and volume */
	temp = (unsigned int)vp->acutoff << 8 | (unsigned char)vp->avol;
	snd_emu10k1_ptr_write(hw, IFATN, vp->ch, temp);

	/* modulation envelope heights */
	snd_emu10k1_ptr_write(hw, PEFE, ch, vp->reg.parm.pefe);

	/* lfo1/2 delay */
	snd_emu10k1_ptr_write(hw, LFOVAL1, ch, vp->reg.parm.lfo1delay);
	snd_emu10k1_ptr_write(hw, LFOVAL2, ch, vp->reg.parm.lfo2delay);

	/* lfo1 pitch & cutoff shift */
	set_fmmod(hw, vp);
	/* lfo1 volume & freq */
	snd_emu10k1_ptr_write(hw, TREMFRQ, vp->ch, vp->reg.parm.tremfrq);
	/* lfo2 pitch & freq */
	set_fm2frq2(hw, vp);
	/* pan & loop start */
	temp = (unsigned int)vp->reg.loopstart - 1;
	temp |= ((unsigned int)vp->apan<<24);
	snd_emu10k1_ptr_write(hw, PSST, vp->ch, temp);


	/* chorus & loop end (chorus 8bit, MSB) */
	addr = vp->reg.loopend - 1;
	temp = vp->reg.parm.chorus;
	temp += (int)chan->control[SND_MCTL_E3_CHORUS_DEPTH] * 9 / 10;
	LIMITVALUE(temp, 0, 255);
	temp = (temp <<24) | (unsigned int)addr;
	snd_emu10k1_ptr_write(hw, DSL, ch, temp);

	/* clear filter delay memory */
	snd_emu10k1_ptr_write(hw, Z1, ch, 0);
	snd_emu10k1_ptr_write(hw, Z2, ch, 0);

	/* invalidate maps */
	temp = ((unsigned int)virt_to_phys(hw->silent_page) << 1) | MAP_PTI_MASK;
	snd_emu10k1_ptr_write(hw, MAPA, ch, temp);
	snd_emu10k1_ptr_write(hw, MAPB, ch, temp);
#if 0
	/* cache */
	{
		unsigned int val, sample;
		val = 32;
		if (vp->reg.sample_mode & SND_SFNT_SAMPLE_8BITS)
			sample = 0x80808080;
		else {
			sample = 0;
			val *= 2;
		}

		/* cache */
		snd_emu10k1_ptr_write(hw, CCR, ch, 0x1c << 16);
		snd_emu10k1_ptr_write(hw, CDE, ch, sample);
		snd_emu10k1_ptr_write(hw, CDF, ch, sample);

		/* invalidate maps */
		temp = ((unsigned int)virt_to_phys(hw->silent_page) << 1) | MAP_PTI_MASK;
		snd_emu10k1_ptr_write(hw, MAPA, ch, temp);
		snd_emu10k1_ptr_write(hw, MAPB, ch, temp);
		
		/* fill cache */
		val -= 4;
		val <<= 25;
		val |= 0x1c << 16;
		snd_emu10k1_ptr_write(hw, CCR, ch, val);
	}
#endif

	/* Q & current address (Q 4bit value, MSB) */
	addr = vp->reg.start - 1;
	temp = vp->reg.parm.filterQ;
	temp = (temp<<28) | (unsigned int)addr | CCCA_INTERPROM_0;
	if (vp->reg.sample_mode & SND_SFNT_SAMPLE_8BITS)
		temp |= CCCA_8BITSELECT;
	snd_emu10k1_ptr_write(hw, CCCA, ch, temp);

	/* reset volume */
	temp = ((unsigned int)vp->vtarget<<16) | vp->ftarget;
	snd_emu10k1_ptr_write(hw, VTFT, ch, temp);
	snd_emu10k1_ptr_write(hw, CVCF, ch, temp);
}

/*
 * Start envelope
 */
static void
trigger_voice(snd_emux_voice_t *vp)
{
	unsigned int temp;
	emu10k1_t *hw;
	
	hw = snd_magic_cast(emu10k1_t, vp->hw, );

	/* set reverb and pitch target */
	temp = vp->reg.parm.reverb;
	temp += (int)vp->chan->control[SND_MCTL_E1_REVERB_DEPTH] * 9 / 10;
	LIMITVALUE(temp, 0, 255);
	temp = (temp << 8) | ((unsigned int)vp->ptarget << 16) | vp->aaux;
	snd_emu10k1_ptr_write(hw, PTRX, vp->ch, temp);

	/* pitch target */
	/*temp = (unsigned int)vp->ptarget << 16;*/
	snd_emu10k1_ptr_write(hw, CPF, vp->ch, 0);
	snd_emu10k1_ptr_write(hw, CPF_CURRENTPITCH, vp->ch, IP_TO_CP(vp->apitch) >> 16);

	/* trigger voice */
	/*snd_emu10k1_voice_clear_loop_stop(hw, vp->ch);*/
	snd_emu10k1_ptr_write(hw, DCYSUSV, vp->ch, vp->reg.parm.voldcysus|ENV_ON);
}

#define MOD_SENSE 18

/* set lfo1 modulation height and cutoff */
static void
set_fmmod(emu10k1_t *hw, snd_emux_voice_t *vp)
{
	unsigned short fmmod;
	short pitch;
	unsigned char cutoff;
	int modulation;

	pitch = (char)(vp->reg.parm.fmmod>>8);
	cutoff = (vp->reg.parm.fmmod & 0xff);
	modulation = vp->chan->gm_modulation + vp->chan->midi_pressure;
	pitch += (MOD_SENSE * modulation) / 1200;
	LIMITVALUE(pitch, -128, 127);
	fmmod = ((unsigned char)pitch<<8) | cutoff;
	snd_emu10k1_ptr_write(hw, FMMOD, vp->ch, fmmod);
}

/* set lfo2 pitch & frequency */
static void
set_fm2frq2(emu10k1_t *hw, snd_emux_voice_t *vp)
{
	unsigned short fm2frq2;
	short pitch;
	unsigned char freq;
	int modulation;

	pitch = (char)(vp->reg.parm.fm2frq2>>8);
	freq = vp->reg.parm.fm2frq2 & 0xff;
	modulation = vp->chan->gm_modulation + vp->chan->midi_pressure;
	pitch += (MOD_SENSE * modulation) / 1200;
	LIMITVALUE(pitch, -128, 127);
	fm2frq2 = ((unsigned char)pitch<<8) | freq;
	snd_emu10k1_ptr_write(hw, FM2FRQ2, vp->ch, fm2frq2);
}

/* set filterQ */
static void
set_filterQ(emu10k1_t *hw, snd_emux_voice_t *vp)
{
	unsigned int val;
	val = snd_emu10k1_ptr_read(hw, CCCA, vp->ch) & ~CCCA_RESONANCE;
	val |= (vp->reg.parm.filterQ << 28);
	snd_emu10k1_ptr_write(hw, CCCA, vp->ch, val);
}

/*
 * increment use counter
 */
static void
use_inc(void *emu)
{
	MOD_INC_USE_COUNT;
}

/*
 * decrement use counter
 */
static void
use_dec(void *emu)
{
	MOD_DEC_USE_COUNT;
}

