/*
 *  Initialization & resource registration routines
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../include/driver.h"
#include "../include/control.h"
#include "../include/info.h"

int snd_cards_count;
unsigned int snd_cards_bitmap;		/* valid */
static unsigned int snd_cards_lock;	/* locked for registering/using */
snd_card_t *snd_cards[SND_CARDS];
static spinlock_t snd_card_lock = SPIN_LOCK_UNLOCKED;

void __init snd_driver_init(void)
{
	snd_cards_count = 0;
	snd_cards_bitmap = 0;
	memset(&snd_cards, 0, sizeof(snd_cards));
}

snd_card_t *snd_card_new(int idx, char *xid,
			 void (*use_inc) (snd_card_t * card),
			 void (*use_dec) (snd_card_t * card))
{
	unsigned long flags;
	snd_card_t *card;

	MOD_INC_USE_COUNT;
	if (use_inc == NULL || use_dec == NULL) {
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	card = (snd_card_t *) snd_kcalloc(sizeof(snd_card_t), GFP_KERNEL);
	if (card == NULL) {
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	if (xid) {
		if (!snd_info_check_reserved_words(xid)) {
			snd_kfree(card);
			MOD_DEC_USE_COUNT;
			return NULL;
		}
		strncpy(card->id, xid, sizeof(card->id) - 1);
	}
	spin_lock_irqsave(&snd_card_lock, flags);
	if (idx < 0) {
		for (idx = 0; idx < snd_ecards_limit; idx++)
			if (!(snd_cards_lock & (1 << idx)))
				break;
	} else {
		if (idx >= 0 && idx < snd_ecards_limit)
			if (snd_cards_lock & (1 << idx))
				idx = -1;	/* invalid */
	}
	if (idx < 0 || idx >= snd_ecards_limit) {
		spin_unlock_irqrestore(&snd_card_lock, flags);
		snd_kfree(card);
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	snd_cards_lock |= 1 << idx;		/* lock it */
	spin_unlock_irqrestore(&snd_card_lock, flags);
	card->number = idx;
	if (!card->id[0])
		sprintf(card->id, "card%i", card->number + 1);
	card->use_inc = use_inc;
	card->use_dec = use_dec;
	init_MUTEX(&card->control);
	snd_switch_prepare(&card->switches);
	/* control interface cannot be accessed from user space until */
	/* snd_cards_bitmask and snd_cards are set by snd_card_register */
	if (snd_control_register(card) < 0) {
		snd_printd("unable to register control minors\n");
		/* Not fatal error */
	}
	if (snd_info_card_register(card) < 0) {
		snd_printd("unable to register card info\n");
		/* Not fatal error */
	}
#if 0
	snd_printk("%i: card = 0x%lx\n", card->number, (long) card);
#endif
	return card;
}

int snd_card_free(snd_card_t * card)
{
	unsigned long flags;

	snd_debug_check(card == NULL, -ENXIO);
	if (snd_device_free_all(card) < 0) {
		snd_printk("unable to free all devices\n");
		/* Fatal, but this situation should never occur */
	}
	if (snd_info_card_unregister(card) < 0) {
		snd_printk("unable to unregister card info\n");
		/* Not fatal error */
	}
	if (snd_control_unregister(card) < 0) {
		snd_printk("unable to unregister control minors\n");
		/* Not fatal error */
	}
	snd_switch_free(&card->switches);
	spin_lock_irqsave(&snd_card_lock, flags);
	snd_cards_lock &= ~(1 << card->number);
	spin_unlock_irqrestore(&snd_card_lock, flags);
	snd_unregister_interrupts(card);
	snd_unregister_dma_channels(card);
	snd_unregister_ioports(card);
	if (card->private_free)
		card->private_free(card->private_data);
	snd_kfree(card);
	MOD_DEC_USE_COUNT;
	return 0;
}

int snd_card_register(snd_card_t * card)
{
	unsigned long flags;
	int err;

	snd_debug_check(card == NULL, -ENXIO);
	if ((err = snd_device_register_all(card)) < 0)
		return err;
	spin_lock_irqsave(&snd_card_lock, flags);
	snd_cards_bitmap |= 1 << card->number;
	snd_cards[card->number] = card;
	snd_cards_count++;
	spin_unlock_irqrestore(&snd_card_lock, flags);
	return 0;
}

int snd_card_unregister(snd_card_t * card)
{
	unsigned long flags;

	snd_debug_check(card == NULL, -ENXIO);
	spin_lock_irqsave(&snd_card_lock, flags);
	snd_cards_bitmap &= ~(1 << card->number);
	snd_cards[card->number] = NULL;
	snd_cards_count--;
	spin_unlock_irqrestore(&snd_card_lock, flags);
	return snd_card_free(card);
}

static snd_info_entry_t *snd_card_info_entry = NULL;

static void snd_card_info_read(snd_info_buffer_t * buffer, void *private_data)
{
	unsigned long flags;
	int idx, count;
	snd_card_t *card;

	for (idx = count = 0; idx < SND_CARDS; idx++) {
		spin_lock_irqsave(&snd_card_lock, flags);
		if (snd_cards_bitmap & (1 << idx)) {
			count++;
			card = snd_cards[idx];
			snd_iprintf(buffer, "%i [%-15s]: %s - %s\n",
					idx,
					card->id,
					card->abbreviation,
					card->shortname);
			snd_iprintf(buffer, "                     %s\n",
					card->longname);
		}
		spin_unlock_irqrestore(&snd_card_lock, flags);
	}
	if (!count) {
		snd_iprintf(buffer, "--- no soundcards ---\n");
	}
}

#ifdef CONFIG_SND_OSSEMUL

void snd_card_info_read_oss(snd_info_buffer_t * buffer)
{
	unsigned long flags;
	int idx, count;
	snd_card_t *card;

	for (idx = count = 0; idx < SND_CARDS; idx++) {
		spin_lock_irqsave(&snd_card_lock, flags);
		if (snd_cards_bitmap & (1 << idx)) {
			count++;
			card = snd_cards[idx];
			snd_iprintf(buffer, "%s\n", card->longname);
		}
		spin_unlock_irqrestore(&snd_card_lock, flags);
	}
	if (!count) {
		snd_iprintf(buffer, "--- no soundcards ---\n");
	}
}

#endif

int __init snd_card_info_init(void)
{
	snd_info_entry_t *entry;

	entry = snd_info_create_entry(NULL, "cards");
	if (entry == NULL)
		return -ENOMEM;
	entry->t.text.read_size = PAGE_SIZE;
	entry->t.text.read = snd_card_info_read;
	if (snd_info_register(entry) < 0) {
		snd_info_free_entry(entry);
		return -ENOMEM;
	}
	snd_card_info_entry = entry;
	return 0;
}

int __exit snd_card_info_done(void)
{
	if (snd_card_info_entry)
		snd_info_unregister(snd_card_info_entry);
	return 0;
}

/*
 *  I/O port registration
 */

int snd_check_ioport(snd_card_t * card, int port, int size)
{
	snd_debug_check(card == NULL, -ENXIO);
	if (check_region(port, size)) {
		snd_printdd("check - ports 0x%x-0x%x busy\n", port, port + size - 1);
		return -EBUSY;
	}
	return 0;
}

int snd_register_ioport(snd_card_t * card, int port, int size, char *name,
                        snd_port_t ** rport)
{
	char *xname;
	snd_port_t *pport, *pport1;

	if (rport)
		*rport = NULL;
	snd_debug_check(card == NULL, -ENXIO);
	xname = snd_kmalloc_strdup(name, GFP_KERNEL);
	pport = (snd_port_t *) snd_kcalloc(sizeof(snd_port_t), GFP_KERNEL);
	if (pport == NULL) {
		snd_kfree(xname);
		return -ENOMEM;
	}
#ifdef NEW_RESOURCE
	pport->res = request_region(port, size, xname);
	if(pport->res == NULL) {
#else
	if (check_region(port, size)) {
#endif
		snd_printdd("requested ports 0x%x-0x%x busy\n",
						port, port + size - 1);
		snd_kfree(pport);
		snd_kfree(xname);
		return -EBUSY;
	}
#ifndef NEW_RESOURCE
	request_region(port, size, xname);
#endif
	pport->port = port;
	pport->size = size;
	pport->name = xname;
	if (card->ports == NULL) {
		card->ports = pport;
	} else {
		for (pport1 = card->ports; pport1->next != NULL; pport1 = pport1->next);
		pport1->next = pport;
	}
	if (rport)
		*rport = pport;
	return 0;
}

int snd_unregister_ioport(snd_card_t * card, snd_port_t * port)
{
	snd_port_t *prev;

	if (port == card->ports) {
		card->ports = port->next;
#ifdef NEW_RESOURCE
		release_resource(port->res);
#else
		release_region(port->port, port->size);
#endif
		snd_kfree(port);
		return 0;
	}
	for (prev = card->ports; prev && prev->next != port; prev = prev->next);
	snd_debug_check(prev == NULL, -ENXIO);
	prev->next = port->next;
#ifdef NEW_RESOURCE
	release_resource(port->res);
#else
	release_region(port->port, port->size);
#endif
	snd_kfree(port);
	return 0;
}

int snd_unregister_ioports(snd_card_t * card)
{
	snd_port_t *pport, *pportnext;

	snd_debug_check(card == NULL, -ENXIO);
	pport = card->ports;
	card->ports = NULL;
	while (pport != NULL) {
		pportnext = pport->next;
#ifdef NEW_RESOURCE
		release_resource(pport->res);
#else
		release_region(pport->port, pport->size);
#endif
		snd_kfree(pport->name);
		snd_kfree(pport);
		pport = pportnext;
	}
	return 0;
}

/*
 *  DMA channel registration
 */

int snd_register_dma_channel(snd_card_t * card, char *name, int number,
			     int type, int rsize, int *possible_numbers,
			     snd_dma_t ** rdma)
{
	snd_dma_t *dmaptr, *dmaptr1;
	int addressbits = 24;

	snd_debug_check(card == NULL, -ENXIO);
	snd_debug_check(type < 0 || type > SND_DMA_TYPE_HARDWARE, -EINVAL);
	if (rsize == SND_AUTO_DMA_SIZE) {
		switch (type) {
		case SND_DMA_TYPE_ISA:
		case SND_DMA_TYPE_PCI_16MB:
			addressbits = 24;
			rsize = 128;
			break;
		case SND_DMA_TYPE_PCI:
			addressbits = 32;
			rsize = 512;
			break;
		case SND_DMA_TYPE_HARDWARE:
			addressbits = 64;
			rsize = 1024;
			break;
		default:
			return -EINVAL;
		}
	}
	snd_debug_check(rsize < 4 || rsize > 1024, -EINVAL);
	if (type == SND_DMA_TYPE_ISA) {
		snd_debug_check(rsize > 128, -EINVAL);
	}
	if (type == SND_DMA_TYPE_ISA) {
		snd_debug_check(possible_numbers == NULL, -EINVAL);
		if (number == SND_AUTO_DMA) {
			for (; (number = *possible_numbers) >= 0;
			    				possible_numbers++) {
				if (request_dma(number, "snd test"))
					continue;
				free_dma(number);
				break;
			}
			if (number < 0) {
				snd_printdd("dma register: AUTO DMA failed\n");
				return -ENOMEM;
			}
		} else {
			if (*possible_numbers >= 0) {
				for (; *possible_numbers >= 0; possible_numbers++)
					if (*possible_numbers == number)
						break;
				if (*possible_numbers < 0) {
					snd_printdd("dma register: DMA invalid\n");
					return -EINVAL;
				}				
			}
		}
		if (number < 4 && rsize > 64)
			rsize = 64;
		snd_debug_check(number < 0 || number > 7, -EINVAL);
	} else {
		snd_debug_check(possible_numbers, -EINVAL);
		snd_debug_check(number < 0, -EINVAL);
	}
	dmaptr = (snd_dma_t *) snd_kcalloc(sizeof(snd_dma_t), GFP_KERNEL);
	if (dmaptr == NULL)
		return -ENOMEM;
	dmaptr->type = type;
	dmaptr->dma = number;
	dmaptr->addressbits = addressbits;
	dmaptr->rsize = rsize * 1024;
	dmaptr->name = snd_kmalloc_strdup(name, GFP_KERNEL);
	init_MUTEX(&dmaptr->mutex);
	if (type == SND_DMA_TYPE_ISA) {
		if (request_dma(number, dmaptr->name)) {
			snd_printk("unable to grab DMA %i for %s\n",
						number, dmaptr->name);
			snd_kfree(dmaptr->name);
			snd_kfree(dmaptr);
			return -EBUSY;
		}
	}
	if (card->dmas == NULL) {
		card->dmas = dmaptr;
	} else {
		for (dmaptr1 = card->dmas; dmaptr1->next != NULL; dmaptr1 = dmaptr1->next);
		dmaptr1->next = dmaptr;
	}
	*rdma = dmaptr;
	return 0;
}

int snd_unregister_dma_channels(snd_card_t * card)
{
	snd_dma_t *dmaptr, *dmaptrnext;

	snd_debug_check(card == NULL, -ENXIO);
	dmaptr = card->dmas;
	card->dmas = NULL;
	while (dmaptr) {
		dmaptrnext = dmaptr->next;
		if (dmaptr->type == SND_DMA_TYPE_ISA) {
			snd_dma_disable(dmaptr->dma);
			free_dma(dmaptr->dma);
		}
		snd_kfree(dmaptr->name);
		snd_kfree(dmaptr);
		dmaptr = dmaptrnext;
	}
	return 0;
}

/*
 *  IRQ channel registration
 */

int snd_register_interrupt(snd_card_t * card, char *name, int number,
			   int type, snd_irq_handler_t * handler,
			   void *dev_id, int *possible_numbers,
			   snd_irq_t ** rirq)
{
	int sa_flags;
	char *xname;
	snd_irq_t *irqptr, *irqtmp;

	snd_debug_check(rirq == NULL, -ENXIO);
	*rirq = NULL;
	snd_debug_check(card == NULL, -ENXIO);		
	snd_debug_check(type < 0 || type > SND_IRQ_TYPE_PCI, -EINVAL);
	sa_flags = SA_INTERRUPT;
	if (type != SND_IRQ_TYPE_ISA)
		sa_flags = SA_SHIRQ;
	if (type == SND_IRQ_TYPE_ISA) {
		snd_debug_check(possible_numbers == NULL, -EINVAL);
		if (number == SND_AUTO_IRQ) {
			for (; (number = *possible_numbers) >= 0;
							possible_numbers++) {
				if (request_irq(number, handler, sa_flags,
				                "snd test", dev_id))
					continue;
				free_irq(number, dev_id);
				break;
			}
			if (number < 0) {
				snd_printdd("irq register: AUTO IRQ failed\n");
				return -ENOMEM;
			}
		} else {
			if (*possible_numbers >= 0) {
				for (; *possible_numbers >= 0;
							possible_numbers++)
					if (*possible_numbers == number)
						break;
				if (*possible_numbers < 0) {
					snd_printdd("irq register: IRQ invalid\n");
					return -EINVAL;
				}
			}
		}
	} else {
		snd_debug_check(possible_numbers, -EINVAL);
	}
#if __SMP__ || !defined(__i386__)
	snd_debug_check(number < 0 || number > 63, -EINVAL);
#else
	snd_debug_check(number < 0 || number > 15, -EINVAL);
#endif
	xname = snd_kmalloc_strdup(name, GFP_KERNEL);
	if (xname == NULL)
		return -ENOMEM;
	irqptr = (snd_irq_t *) snd_kcalloc(sizeof(snd_irq_t), GFP_KERNEL);
	if (irqptr == NULL) {
		snd_kfree(xname);
		return -ENOMEM;
	}
	if (request_irq(number, handler, sa_flags, xname, dev_id)) {
		snd_printk("unable to grab IRQ %i for %s\n", number, xname);
		snd_kfree(xname);
		snd_kfree(irqptr);
		return -EBUSY;
	}
	irqptr->type = type;
	irqptr->irq = number;
	irqptr->name = xname;
	irqptr->dev_id = dev_id;
	if (card->irqs == NULL) {
		card->irqs = irqptr;
	} else {
		for (irqtmp = card->irqs; irqtmp->next != NULL; irqtmp = irqtmp->next);
		irqtmp->next = irqptr;
	}
	*rirq = irqptr;
	return 0;
}

int snd_unregister_interrupts(snd_card_t * card)
{
	snd_irq_t *irqptr, *irqnext;

	snd_debug_check(card == NULL, -ENXIO);
	irqptr = card->irqs;
	card->irqs = NULL;
	while (irqptr != NULL) {
		irqnext = irqptr->next;
		free_irq(irqptr->irq, irqptr->dev_id);
		snd_kfree(irqptr->name);
		snd_kfree(irqptr);
		irqptr = irqnext;
	}
	return 0;
}
