/*
 *  Dummy soundcard for virtual rawmidi devices
 *
 *  Copyright (c) 2000 by Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
 * VIRTUAL RAW MIDI DEVICE CARDS
 *
 * This dummy card contains up to 4 virtual rawmidi devices.
 * They are not real rawmidi devices but just associated with sequencer
 * clients, so that any input/output sources can be connected as a raw
 * MIDI device arbitrary.
 * Also, multiple access is allowed to a single rawmidi device.
 *
 * Typical usage is like following:
 * - Load snd-card-virmidi module.
 *	# modprobe snd-card-virmidi snd_index=2
 *   Then, sequencer clients 72:0 to 75:0 will be created, which are
 *   mapped from /dev/snd/midiC1D0 to /dev/snd/midiC1D3, respectively.
 *
 * - Connect input/output via aconnect.
 *	% aconnect 64:0 72:0	# keyboard input redirection 64:0 -> 72:0
 *	% aconnect 72:0 65:0	# output device redirection 72:0 -> 65:0
 *
 * - Run application using a midi device (eg. /dev/snd/midiC1D0)
 */

#define SND_MAIN_OBJECT_FILE
#define __SND_OSS_COMPAT__
#include "../include/driver.h"
#include "../include/seq_virmidi.h"
#include "../include/initval.h"

MODULE_DESCRIPTION("\n\
Driver: Dummy soundcard for virtual rawmidi devices\n\
ALWAYS\n\
");

#define MAX_MIDI_DEVICES	4

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_enable[SND_CARDS] = {1, [1 ... (SND_CARDS - 1)] = 0};
int snd_midi_devs[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 4};
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for virmidi soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for virmidi soundcard.");
MODULE_PARM(snd_enable, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_enable, "Enable this soundcard. [BOOL]");
MODULE_PARM(snd_midi_devs, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_midi_devs, "MIDI devices # (0-4) [range=0-4]");

typedef struct snd_card_virmidi {
	snd_card_t *card;
	snd_virmidi_dev_t *midi[MAX_MIDI_DEVICES];
} snd_card_virmidi_t;

snd_card_virmidi_t *snd_card_virmidi_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_card_virmidi_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_virmidi_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static void snd_card_virmidi_free(snd_card_virmidi_t *vmidi)
{
	if (vmidi == NULL)
		return;
	if (vmidi->card != NULL)
		snd_card_unregister(vmidi->card);
	snd_kfree(vmidi);
}

static int __init snd_card_virmidi_probe(int dev, struct snd_card_virmidi *vmidi)
{
	snd_card_t *card;
	int idx;

	if (!snd_enable[dev]) {
		snd_card_virmidi_free(vmidi);
		return -ENODEV;
	}
	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_card_virmidi_use_inc, snd_card_virmidi_use_dec);
	if (card == NULL) {
		snd_card_virmidi_free(vmidi);
		return -ENOMEM;
	}
	card->type = SND_CARD_TYPE_VIRMIDI;
	vmidi->card = card;

	if (snd_midi_devs[dev] > MAX_MIDI_DEVICES) {
		snd_printk("Card-VirMIDI #%i not found or device busy\n", dev + 1);
		snd_midi_devs[dev] = MAX_MIDI_DEVICES;
	}
	for (idx = 0; idx < snd_midi_devs[dev]; idx++) {
		snd_virmidi_dev_t *rdev;
		if (snd_virmidi_new(card, idx, &rdev) < 0 || rdev == NULL)
			goto __nodev;
		vmidi->midi[idx] = rdev;
		strcpy(rdev->name, "Virtual Raw MIDI");
		strcpy(rdev->id, "virmidi");
		rdev->seq_mode = SND_VIRMIDI_SEQ_DISPATCH;
		rdev->use_inc = (snd_virmidi_use_t)snd_card_virmidi_use_inc;
		rdev->use_dec = (snd_virmidi_use_t)snd_card_virmidi_use_dec;
	}
	
	strcpy(card->abbreviation, "VirMIDI");
	strcpy(card->shortname, "VirMIDI");
	sprintf(card->longname, "Virtual MIDI Card %i", dev + 1);
	if (!snd_card_register(card))
		return 0;
      __nodev:
	snd_card_virmidi_free(vmidi);
	return -ENXIO;
}

#ifdef MODULE
int __init init_module(void)
#else
int __init alsa_card_virmidi_init(void)
#endif
{
	int dev, cards;
	snd_card_virmidi_t *vmidi;

	for (dev = cards = 0; dev < SND_CARDS && snd_enable[dev]; dev++) {
		vmidi = snd_kcalloc(sizeof(snd_card_virmidi_t), GFP_KERNEL);
		if (vmidi == NULL)
			continue;
		if (snd_card_virmidi_probe(dev, vmidi) < 0) {
#ifdef MODULE
			snd_printk("Card-VirMIDI #%i not found or device busy\n", dev + 1);
#endif
			break;
		}
		snd_card_virmidi_cards[dev] = vmidi;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		snd_printk("Card-VirMIDI soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

#ifdef MODULE

void __exit cleanup_module(void)
{
	int idx;

	for (idx = 0; idx < SND_CARDS; idx++) {
		snd_card_virmidi_free(snd_card_virmidi_cards[idx]);
	}
}

#endif
