/* 
 * Driver for NeoMagic 256AV and 256ZX chipsets.
 * Copyright (c) 2000 by Takashi Iwai <tiwai@suse.de>
 *
 * Based on nm256_audio.c OSS driver in linux kernel.
 * The original author of OSS nm256 driver wishes to remain anonymous,
 * so I just put my acknoledgment to him/her here.
 * The original author's web page is found at
 *	http://www.uglx.org/sony.html
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
  
#define SND_MAIN_OBJECT_FILE

#include "../include/driver.h"
#include "../include/info.h"
#include "../include/control.h"
#include "../include/mixer.h"
#include "../include/pcm.h"
#include "../include/ac97_codec.h"
#include "../include/initval.h"

#ifndef __i386__
/* FIXME: I/O space is not accessible via pointers on all architectures */
#warning "This driver does not work properly on this architecture"
#endif

#define CARD_NAME "NeoMagic 256AV/ZX"
#define DRIVER_NAME "NM256"

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("\
Driver: NM256\n\
Card: NeoMagic 256AV\n\
Card: NeoMagic 256ZX\n\
PCI: 0x10c8=0x8005\n\
PCI: 0x10c8=0x8006\n\
");
MODULE_LICENSE("GPL");

/*
 * some compile conditions.
 */

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_enable[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 1}; /* all enabled */
int snd_playback_bufsize[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 16};
int snd_capture_bufsize[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 16};
int snd_force_ac97[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 0}; /* disabled as default */
int snd_ignore_signature[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 0}; /* disabled as default */
int snd_buffer_top[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 0}; /* not specified */
int snd_use_cache[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 0}; /* disabled */
int snd_vaio_hack[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 0}; /* disabled */

MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for " CARD_NAME " soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for " CARD_NAME " soundcard.");
MODULE_PARM(snd_enable, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_enable, "Enable this soundcard. [BOOL]");
MODULE_PARM(snd_playback_bufsize, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_playback_bufsize, "DAC frame size in kB for " CARD_NAME " soundcard.");
MODULE_PARM(snd_capture_bufsize, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_capture_bufsize, "ADC frame size in kB for " CARD_NAME " soundcard.");
MODULE_PARM(snd_force_ac97, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_force_ac97, "Force to use AC97 codec for " CARD_NAME " soundcard. [BOOL]");
MODULE_PARM(snd_ignore_signature, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_ignore_signature, "Ignore signature search for " CARD_NAME " soundcard. [BOOL]");
MODULE_PARM(snd_buffer_top, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_buffer_top, "Set the top address of audio buffer for " CARD_NAME " soundcard.");
MODULE_PARM(snd_use_cache, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_use_cache, "Enable the cache for coefficient table access. [BOOL]");
MODULE_PARM(snd_vaio_hack, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_vaio_hack, "Enable workaround for Sony VAIO notebooks. [BOOL]");

/*
 * directions
 */

#define NM_PLAYBACK	0
#define NM_CAPTURE	1


/*
 * hw definitions
 */

/* The BIOS signature. */
#define NM_SIGNATURE 0x4e4d0000
/* Signature mask. */
#define NM_SIG_MASK 0xffff0000

/* Size of the second memory area. */
#define NM_PORT2_SIZE 4096

/* The base offset of the mixer in the second memory area. */
#define NM_MIXER_OFFSET 0x600

/* The maximum size of a coefficient entry. */
#define NM_MAX_COEFFICIENT 0x3000

/* The interrupt register. */
#define NM_INT_REG 0xa04
/* And its bits. */
#define NM_PLAYBACK_INT 0x40
#define NM_RECORD_INT 0x100
#define NM_MISC_INT_1 0x4000
#define NM_MISC_INT_2 0x1
#define NM_ACK_INT(chip, X) snd_nm256_writew(chip, NM_INT_REG, (X) << 1)

/* The AV's "mixer ready" status bit and location. */
#define NM_MIXER_STATUS_OFFSET 0xa04
#define NM_MIXER_READY_MASK 0x0800
#define NM_MIXER_PRESENCE 0xa06
#define NM_PRESENCE_MASK 0x0050
#define NM_PRESENCE_VALUE 0x0040

/*
 * For the ZX.  It uses the same interrupt register, but it holds 32
 * bits instead of 16.
 */
#define NM2_PLAYBACK_INT 0x10000
#define NM2_RECORD_INT 0x80000
#define NM2_MISC_INT_1 0x8
#define NM2_MISC_INT_2 0x2
#define NM2_ACK_INT(chip, X) snd_nm256_writel(chip, NM_INT_REG, (X))

/* The ZX's "mixer ready" status bit and location. */
#define NM2_MIXER_STATUS_OFFSET 0xa06
#define NM2_MIXER_READY_MASK 0x0800

/* The playback registers start from here. */
#define NM_PLAYBACK_REG_OFFSET 0x0
/* The record registers start from here. */
#define NM_RECORD_REG_OFFSET 0x200

/* The rate register is located 2 bytes from the start of the register area. */
#define NM_RATE_REG_OFFSET 2

/* Mono/stereo flag, number of bits on playback, and rate mask. */
#define NM_RATE_STEREO 1
#define NM_RATE_BITS_16 2
#define NM_RATE_MASK 0xf0

/* Playback enable register. */
#define NM_PLAYBACK_ENABLE_REG (NM_PLAYBACK_REG_OFFSET + 0x1)
#define NM_PLAYBACK_ENABLE_FLAG 1
#define NM_PLAYBACK_ONESHOT 2
#define NM_PLAYBACK_FREERUN 4

/* Mutes the audio output. */
#define NM_AUDIO_MUTE_REG (NM_PLAYBACK_REG_OFFSET + 0x18)
#define NM_AUDIO_MUTE_LEFT 0x8000
#define NM_AUDIO_MUTE_RIGHT 0x0080

/* Recording enable register. */
#define NM_RECORD_ENABLE_REG (NM_RECORD_REG_OFFSET + 0)
#define NM_RECORD_ENABLE_FLAG 1
#define NM_RECORD_FREERUN 2

/* coefficient buffer pointer */
#define NM_COEFF_START_OFFSET	0x1c
#define NM_COEFF_END_OFFSET	0x20

/* DMA buffer offsets */
#define NM_RBUFFER_START (NM_RECORD_REG_OFFSET + 0x4)
#define NM_RBUFFER_END   (NM_RECORD_REG_OFFSET + 0x10)
#define NM_RBUFFER_WMARK (NM_RECORD_REG_OFFSET + 0xc)
#define NM_RBUFFER_CURRP (NM_RECORD_REG_OFFSET + 0x8)

#define NM_PBUFFER_START (NM_PLAYBACK_REG_OFFSET + 0x4)
#define NM_PBUFFER_END   (NM_PLAYBACK_REG_OFFSET + 0x14)
#define NM_PBUFFER_WMARK (NM_PLAYBACK_REG_OFFSET + 0xc)
#define NM_PBUFFER_CURRP (NM_PLAYBACK_REG_OFFSET + 0x8)

/*
 * type definitions
 */

typedef struct snd_nm256 nm256_t;
typedef struct snd_nm256_dma nm256_dma_t;

struct snd_nm256_dma {

	nm256_t *chip;
	snd_pcm_subchn_t *substream;
	int running;
	
	u32 buf;	/* offset from chip->buffer */
	int bufsize;	/* buffer size in kb */

	int dma_size;	/* in bytes */
	int frag_size;	/* in bytes */
	int frags;	/* # of fragments */
	int shift;	/* bit shifts */
	int cur_frag;	/* current fragment # */

	snd_dma_area_t area;	/* statically assigned DMA area */

};

struct snd_nm256 {
	
	snd_card_t *card;

	void *cport;			/* control port */
#if 0 // resource
	struct resource *res_cport;
#endif
	unsigned long cport_addr;	/* physical address */

	void *buffer;			/* buffer */
#if 0 // resource
	struct resource *res_buffer;
#endif
	unsigned long buffer_addr;	/* buffer phyiscal address */

	u32 buffer_start;		/* start offset from pci resource 0 */
	u32 buffer_end;			/* end offset */
	u32 buffer_size;

	u32 all_coeff_buf;
	u32 coeff_buf[2];		/* coefficient buffer */

	int coeffs_current;
	int use_cache;

	int mixer_base;
	int mixer_status_offset;
	int mixer_status_mask;

	snd_irq_t *irqptr;
	void (*interrupt)(int, void *, struct pt_regs *);
	int badintrcount;

	nm256_dma_t dma[2];

	snd_kmixer_t *mixer;
	ac97_t *ac97;

	snd_pcm_t *pcm;

	struct pci_dev *pci;

#ifdef CONFIG_PM
	struct pm_dev *pm_dev;
#endif

	spinlock_t reg_lock;
};


/*
 * include coefficient table
 */
#include "nm256_coef.c"


/*
 * PCI ids
 */

#ifndef PCI_VENDOR_ID_NEOMAGIC
#define PCI_VENDOR_ID_NEOMEGIC 0x10c8
#endif
#ifndef PCI_DEVICE_ID_NEOMAGIC_NM256AV_AUDIO
#define PCI_DEVICE_ID_NEOMAGIC_NM256AV_AUDIO 0x8005
#endif
#ifndef PCI_DEVICE_ID_NEOMAGIC_NM256ZX_AUDIO
#define PCI_DEVICE_ID_NEOMAGIC_NM256ZX_AUDIO 0x8006
#endif


static struct pci_device_id snd_nm256_ids[] __devinitdata = {
	{PCI_VENDOR_ID_NEOMAGIC, PCI_DEVICE_ID_NEOMAGIC_NM256AV_AUDIO, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{PCI_VENDOR_ID_NEOMAGIC, PCI_DEVICE_ID_NEOMAGIC_NM256ZX_AUDIO, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0,},
};

inline static unsigned long snd_dma_size(int size, int min, int max)
{
	if (size < min) {
		snd_printk("too small dma size %d, using %d\n", size, min);
		size = min;
	} else if (size > max) {
		snd_printk("too large dma size %d, using %d\n", size, max);
		size = max;
	}
	return (unsigned long)size * 1024;
}

static void snd_nm256_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_nm256_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}


/*
 * lowlvel stuffs
 */

inline static u8
snd_nm256_readb(nm256_t *chip, int offset)
{
	return readb(chip->cport + offset);
}

inline static u16
snd_nm256_readw(nm256_t *chip, int offset)
{
	return readw(chip->cport + offset);
}

inline static u32
snd_nm256_readl(nm256_t *chip, int offset)
{
	return readl(chip->cport + offset);
}

inline static void
snd_nm256_writeb(nm256_t *chip, int offset, u8 val)
{
	writeb(val, chip->cport + offset);
}

inline static void
snd_nm256_writew(nm256_t *chip, int offset, u16 val)
{
	writew(val, chip->cport + offset);
}

inline static void
snd_nm256_writel(nm256_t *chip, int offset, u32 val)
{
	writel(val, chip->cport + offset);
}

inline static void
snd_nm256_write_buffer(nm256_t *chip, void *src, int offset, int size)
{
	offset -= chip->buffer_start;
#ifdef SND_CONFIG_DEBUG
	if (offset < 0 || offset >= chip->buffer_size) {
		printk("nm256: write_buffer invalid offset = %d size = %d\n", offset, size);
		return;
	}
#endif
	memcpy_toio(chip->buffer + offset, src, size);
}

/*
 * coefficient handlers -- what a magic!
 */

static u16
snd_nm256_get_start_offset(int which)
{
	u16 offset = 0;
	while (which-- > 0)
		offset += coefficient_sizes[which];
	return offset;
}

static void
snd_nm256_load_one_coefficient(nm256_t *chip, int capture, u32 port, int which)
{
	u32 coeff_buf = chip->coeff_buf[capture];
	u16 offset = snd_nm256_get_start_offset(which);
	u16 size = coefficient_sizes[which];

	snd_nm256_write_buffer(chip, coefficients + offset, coeff_buf, size);
	snd_nm256_writel(chip, port, coeff_buf);
	/* ???  Record seems to behave differently than playback.  */
	if (! capture)
		size--;
	snd_nm256_writel(chip, port + 4, coeff_buf + size);
}

static void
snd_nm256_load_coefficient(nm256_t *chip, int capture, int number)
{
	/* The enable register for the specified engine.  */
	u32 poffset = capture ? NM_RECORD_ENABLE_REG : NM_PLAYBACK_ENABLE_REG;
	u32 addr = NM_COEFF_START_OFFSET;

	addr += capture ? NM_RECORD_REG_OFFSET : NM_PLAYBACK_REG_OFFSET;

	if (snd_nm256_readb(chip, poffset) & 1) {
		snd_printd("NM256: Engine was enabled while loading coefficients!\n");
		return;
	}

	/* The recording engine uses coefficient values 8-15.  */
	number &= 7;
	if (capture)
		number += 8;

	if (! chip->use_cache) {
		snd_nm256_load_one_coefficient(chip, capture, addr, number);
		return;
	}
	if (! chip->coeffs_current) {
		snd_nm256_write_buffer(chip, coefficients, chip->all_coeff_buf,
				       NM_TOTAL_COEFF_COUNT * 4);
		chip->coeffs_current = 1;
	} else {
		u32 base = chip->all_coeff_buf;
		u32 offset = snd_nm256_get_start_offset(number);
		u32 end_offset = offset + coefficient_sizes[number];
		snd_nm256_writel(chip, addr, base + offset);
		if (! capture)
			end_offset--;
		snd_nm256_writel(chip, addr + 4, base + end_offset);
	}
}


/* The actual rates supported by the card. */
static int samplerates[8] = {
	8000, 11025, 16000, 22050, 24000, 32000, 44100, 48000,
};

/*
 * return the index of the target rate
 */
static int
snd_nm256_fixed_rate(int rate)
{
	int i;

	if (rate < 8000)
		return 0;
	if (rate > 48000)
		return 7;

	for (i = 0; i < 7; i++) {
		if (rate < (samplerates[i] + samplerates[i + 1]) / 2)
			break;
	}
	return i;
}

/*
 * set sample rate and format
 */
static void
snd_nm256_set_format(nm256_t *chip, nm256_dma_t *s, snd_pcm_subchn_t *subs)
{
	snd_pcm_runtime_t *runtime = subs->runtime;
	int rate_index = snd_nm256_fixed_rate(runtime->format.rate);
	unsigned char ratebits = (rate_index << 4) & NM_RATE_MASK;

	s->shift = 0;
	if (snd_pcm_format_width(runtime->format.format) == 16) {
		ratebits |= NM_RATE_BITS_16;
		s->shift++;
	}
	if (runtime->format.voices > 1) {
		ratebits |= NM_RATE_STEREO;
		s->shift++;
	}

	runtime->format.rate = samplerates[rate_index];

#if 0	
	if (s->running)
		return;
#endif

	switch (subs->channel) {
	case SND_PCM_CHANNEL_PLAYBACK:
		snd_nm256_load_coefficient(chip, 0, rate_index); /* 0 = playback */
		snd_nm256_writeb(chip,
				 NM_PLAYBACK_REG_OFFSET + NM_RATE_REG_OFFSET,
				 ratebits);
		break;
	case SND_PCM_CHANNEL_CAPTURE:
		snd_nm256_load_coefficient(chip, 1, rate_index); /* 1 = record */
		snd_nm256_writeb(chip,
				 NM_RECORD_REG_OFFSET + NM_RATE_REG_OFFSET,
				 ratebits);
		break;
	}
}

/*
 * start / stop
 */

/* update the watermark (current fragment) */
static void snd_nm256_playback_mark(nm256_t *chip, nm256_dma_t *s)
{
	s->cur_frag++;
	s->cur_frag %= s->frags;
	snd_nm256_writel(chip, NM_PBUFFER_WMARK,
			 s->buf + s->cur_frag * s->frag_size);
}

static void
snd_nm256_playback_start(nm256_t *chip, nm256_dma_t *s, snd_pcm_subchn_t *subs)
{
	/* program buffer pointers */
	snd_nm256_writel(chip, NM_PBUFFER_START, s->buf);
	snd_nm256_writel(chip, NM_PBUFFER_END, s->buf + s->dma_size - (1 << s->shift));
	snd_nm256_writel(chip, NM_PBUFFER_CURRP, s->buf);
	snd_nm256_playback_mark(chip, s);

	/* Enable playback engine and interrupts. */
	snd_nm256_writeb(chip, NM_PLAYBACK_ENABLE_REG,
			 NM_PLAYBACK_ENABLE_FLAG | NM_PLAYBACK_FREERUN);
	/* Enable both channels. */
	snd_nm256_writew(chip, NM_AUDIO_MUTE_REG, 0x0);
}

/* update the watermark (current fragment) */
static void snd_nm256_capture_mark(nm256_t *chip, nm256_dma_t *s)
{
	s->cur_frag++;
	s->cur_frag %= s->frags;
	snd_nm256_writel(chip, NM_RBUFFER_WMARK,
			 s->buf + s->cur_frag * s->frag_size);
}

static void
snd_nm256_capture_start(nm256_t *chip, nm256_dma_t *s, snd_pcm_subchn_t *subs)
{
	/* program buffer pointers */
	snd_nm256_writel(chip, NM_RBUFFER_START, s->buf);
	snd_nm256_writel(chip, NM_RBUFFER_END, s->buf + s->dma_size);
	snd_nm256_writel(chip, NM_RBUFFER_CURRP, s->buf);
	snd_nm256_capture_mark(chip, s);

	/* Enable playback engine and interrupts. */
	snd_nm256_writeb(chip, NM_RECORD_ENABLE_REG,
			 NM_RECORD_ENABLE_FLAG | NM_RECORD_FREERUN);
}

/* Stop the play engine. */
static void
snd_nm256_playback_stop(nm256_t *chip)
{
	/* Shut off sound from both channels. */
	snd_nm256_writew(chip, NM_AUDIO_MUTE_REG,
			 NM_AUDIO_MUTE_LEFT | NM_AUDIO_MUTE_RIGHT);
	/* Disable play engine. */
	snd_nm256_writeb(chip, NM_PLAYBACK_ENABLE_REG, 0);
}

static void
snd_nm256_capture_stop(nm256_t *chip)
{
	/* Disable recording engine. */
	snd_nm256_writeb(chip, NM_RECORD_ENABLE_REG, 0);
}

static int
snd_nm256_playback_trigger(void *private_data,
			   snd_pcm_subchn_t *subs, int cmd)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	nm256_dma_t *s = (nm256_dma_t*)subs->runtime->private_data;
	unsigned long flags;
	int err = 0;

	snd_debug_check(s == NULL, -ENXIO);

	spin_lock_irqsave(&chip->reg_lock, flags);
	switch (cmd) {
	case SND_PCM_TRIGGER_GO:
		if (! s->running)
			snd_nm256_playback_start(chip, s, subs);
		s->running = 1;
		break;
	case SND_PCM_TRIGGER_STOP:
		if (s->running)
			snd_nm256_playback_stop(chip);
		s->running = 0;
		break;
	default:
		err = -EINVAL;
		break;
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return err;
}

static int
snd_nm256_capture_trigger(void *private_data,
			  snd_pcm_subchn_t *subs, int cmd)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	nm256_dma_t *s = (nm256_dma_t*)subs->runtime->private_data;
	unsigned long flags;
	int err = 0;

	snd_debug_check(s == NULL, -ENXIO);

	spin_lock_irqsave(&chip->reg_lock, flags);
	switch (cmd) {
	case SND_PCM_TRIGGER_GO:
		if (! s->running)
			snd_nm256_capture_start(chip, s, subs);
		s->running = 1;
		break;
	case SND_PCM_TRIGGER_STOP:
		if (s->running)
			snd_nm256_capture_stop(chip);
		s->running = 0;
		break;
	default:
		err = -EINVAL;
		break;
	}
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	return err;
}


/*
 * calculate fragment size
 */
static int snd_nm256_set_frag_size(nm256_t *chip, nm256_dma_t *s,
				   snd_pcm_subchn_t *subs)
{
	s->dma_size = snd_pcm_lib_transfer_size(subs);
	if (s->dma_size > s->bufsize)
		s->dma_size = s->bufsize;

	if (subs->runtime->mode == SND_PCM_MODE_BLOCK) {
		int align;
		s->frag_size = snd_pcm_lib_transfer_fragment(subs);
		snd_debug_check(s->frag_size <= 0, -EINVAL);
		if (s->frag_size > s->dma_size / 2)
			s->frag_size = s->dma_size / 2;
		if ((align = s->dma_size % s->frag_size) != 0)
			s->frag_size -= align;
		subs->runtime->buf.block.frag_size = s->frag_size;
		subs->runtime->frags = s->dma_size / s->frag_size;
	} else {
		subs->runtime->buf.stream.queue_size = s->dma_size;
		s->frag_size = s->dma_size / 2; /* dummy.. */
		subs->runtime->frags = 1;
	}
	s->frags = s->dma_size / s->frag_size;
	snd_debug_check(s->frags <= 0, -EINVAL);
	return snd_pcm_lib_set_buffer_size(subs, s->dma_size);
}

/*
 * prepare playback/capture channel
 */
static int snd_nm256_pcm_prepare(void *private_data, snd_pcm_subchn_t *subs)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subs->runtime;
	nm256_dma_t *s = (nm256_dma_t*)runtime->private_data;
	unsigned long flags;
	int err;

	snd_debug_check(s == NULL, -ENXIO);
	if ((err = snd_nm256_set_frag_size(chip, s, subs)) < 0)
		return err;

	spin_lock_irqsave(&chip->reg_lock, flags);
	s->running = 0;
	s->cur_frag = 0;
	snd_nm256_set_format(chip, s, subs);
	spin_unlock_irqrestore(&chip->reg_lock, flags);

	return 0;
}


/*
 * ioctl
 */
static int
snd_nm256_pcm_ioctl(void *private_data, snd_pcm_subchn_t *subs,
		    unsigned int cmd, unsigned long *arg)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subs->runtime;
	nm256_dma_t *s = (nm256_dma_t*)runtime->private_data;
	int err;

	snd_debug_check(s == NULL, -ENXIO);
	err = snd_pcm_lib_ioctl(private_data, subs, cmd, arg);
	if (err < 0)
		return err;
	if (cmd == SND_PCM_IOCTL1_PARAMS)
		return snd_nm256_set_frag_size(chip, s, subs);
	return 0;
}


/*
 * get the current pointer
 */

static unsigned int
snd_nm256_playback_pointer(void *private_data, snd_pcm_subchn_t * subs)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	nm256_dma_t *s = (nm256_dma_t*)subs->runtime->private_data;
	unsigned long curp;

	snd_debug_check(s == NULL, 0);
	curp = snd_nm256_readl(chip, NM_PBUFFER_CURRP) - (unsigned long)s->buf;
	return (unsigned int)(curp % s->dma_size);	
}

static unsigned int
snd_nm256_capture_pointer(void *private_data, snd_pcm_subchn_t * subs)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	nm256_dma_t *s = (nm256_dma_t*)subs->runtime->private_data;
	unsigned long curp;

	snd_debug_check(s == NULL, 0);
	curp = snd_nm256_readl(chip, NM_RBUFFER_CURRP) - (unsigned long)s->buf;
	return (unsigned int)(curp % s->dma_size);	
}


/*
 * update playback/capture watermarks
 */

/* spinlock held! */
static void
snd_nm256_playback_update(nm256_t *chip)
{
	nm256_dma_t *s;

	s = &chip->dma[NM_PLAYBACK];
	if (s->running && s->substream) {
		if (s->substream->runtime->mode == SND_PCM_MODE_BLOCK) {
			spin_unlock(&chip->reg_lock);
			snd_pcm_transfer_done(s->substream);
			spin_lock(&chip->reg_lock);
		}
		snd_nm256_playback_mark(chip, s);
	}
}

/* spinlock held! */
static void
snd_nm256_capture_update(nm256_t *chip)
{
	nm256_dma_t *s;

	s = &chip->dma[NM_CAPTURE];
	if (s->running && s->substream) {
		if (s->substream->runtime->mode == SND_PCM_MODE_BLOCK) {
			spin_unlock(&chip->reg_lock);
			snd_pcm_transfer_done(s->substream);
			spin_lock(&chip->reg_lock);
		}
		snd_nm256_capture_mark(chip, s);
	}
}

/*
 */

static snd_pcm_hardware_t snd_nm256_playback =
{
	chninfo:	(SND_PCM_CHNINFO_STREAM | SND_PCM_CHNINFO_BLOCK
			 /*| SND_PCM_CHNINFO_MMAP
			 | SND_PCM_CHNINFO_MMAP_VALID*/
			 | SND_PCM_CHNINFO_INTERLEAVE
			 | SND_PCM_CHNINFO_BLOCK_TRANSFER),
	formats:	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_8000_48000,
	min_rate:	8000,
	max_rate:	48000,
	min_voices:	1,
	max_voices:	2,
	min_fragment_size:	256,
	max_fragment_size:	128 * 1024,
	fragment_align:	255,
	ioctl:		snd_nm256_pcm_ioctl,
	prepare:	snd_nm256_pcm_prepare,
	trigger:	snd_nm256_playback_trigger,
	pointer:	snd_nm256_playback_pointer,
};

static snd_pcm_hardware_t snd_nm256_capture =
{
	chninfo:	(SND_PCM_CHNINFO_STREAM | SND_PCM_CHNINFO_BLOCK
			 /* | SND_PCM_CHNINFO_MMAP
			 | SND_PCM_CHNINFO_MMAP_VALID */
			 | SND_PCM_CHNINFO_INTERLEAVE
			 | SND_PCM_CHNINFO_BLOCK_TRANSFER),
	formats:	SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE,
	rates:		SND_PCM_RATE_8000_48000,
	min_rate:	8000,
	max_rate:	48000,
	min_voices:	1,
	max_voices:	2,
	min_fragment_size:	256,
	max_fragment_size:	128 * 1024,
	fragment_align:	255,
	ioctl:		snd_nm256_pcm_ioctl,
	prepare:	snd_nm256_pcm_prepare,
	trigger:	snd_nm256_capture_trigger,
	pointer:	snd_nm256_capture_pointer,
};


/*
 * playback
 */
static int
snd_nm256_playback_open(void *private_data,
			snd_pcm_subchn_t *subs)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	nm256_dma_t *s;
	int err;

	s = &chip->dma[NM_PLAYBACK];
	s->running = 0;
	/*
	 * we don't create dma are stuff here, since the ring buffer is
	 * statically assigned.  Just pass the given area.
	 */
	if ((err = snd_pcm_dma_setup(subs, &s->area)) < 0)
		return err;

	subs->runtime->private_data = s;
	s->substream = subs;

	subs->runtime->hw = &snd_nm256_playback;
	snd_pcm_set_sync(subs);

	return 0;
}

static int
snd_nm256_playback_close(void *private_data,
			 snd_pcm_subchn_t *subs)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	if (chip->dma[NM_PLAYBACK].running) {
		snd_nm256_playback_stop(chip);
		chip->dma[NM_PLAYBACK].running = 0;
	}
	return 0;
}


/*
 */
static int
snd_nm256_capture_open(void *private_data,
		       snd_pcm_subchn_t *subs)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	nm256_dma_t *s;
	int err;

	s = &chip->dma[NM_CAPTURE];
	s->running = 0;
	if ((err = snd_pcm_dma_setup(subs, &s->area)) < 0)
		return err;

	subs->runtime->private_data = s;
	s->substream = subs;

	subs->runtime->hw = &snd_nm256_capture;
	snd_pcm_set_sync(subs);

	return 0;
}

static int
snd_nm256_capture_close(void *private_data,
			snd_pcm_subchn_t *subs)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	if (chip->dma[NM_CAPTURE].running) {
		snd_nm256_capture_stop(chip);
		chip->dma[NM_CAPTURE].running = 0;
	}
	return 0;
}


/*
 * create a pcm instance
 */

static int __init
snd_nm256_pcm(nm256_t *chip, int device)
{
	snd_pcm_t *pcm;
	int i, err;

	for (i = 0; i < 2; i++) {
		nm256_dma_t *s = &chip->dma[i];
		s->area.size = s->bufsize;
		s->area.buf = chip->buffer +  s->buf - chip->buffer_start;
	}

	err = snd_pcm_new(chip->card, chip->card->abbreviation, device,
			  1, 1, &pcm);
	if (err < 0)
		return err;

	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = chip;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_nm256_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_nm256_playback_close;

	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = chip;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_nm256_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_nm256_capture_close;

	pcm->private_data = chip;
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE
		| SND_PCM_INFO_DUPLEX;
	strcpy(pcm->name, chip->card->abbreviation);
	chip->pcm = pcm;

	return 0;
}


/* 
 * Initialize the hardware. 
 */
static void
snd_nm256_init_chip(nm256_t *chip)
{
	unsigned long flags;

	spin_lock_irqsave(&chip->reg_lock, flags);
	/* Reset everything. */
	snd_nm256_writeb(chip, 0x0, 0x11);
	snd_nm256_writew(chip, 0x214, 0);
	/* stop sounds.. */
	//snd_nm256_playback_stop(chip);
	//snd_nm256_capture_stop(chip);
	spin_unlock_irqrestore(&chip->reg_lock, flags);
}


inline static void
snd_nm256_intr_check(nm256_t *chip)
{
	if (chip->badintrcount++ > 1000) {
		/*
		 * I'm not sure if the best thing is to stop the card from
		 * playing or just release the interrupt (after all, we're in
		 * a bad situation, so doing fancy stuff may not be such a good
		 * idea).
		 *
		 * I worry about the card engine continuing to play noise
		 * over and over, however--that could become a very
		 * obnoxious problem.  And we know that when this usually
		 * happens things are fairly safe, it just means the user's
		 * inserted a PCMCIA card and someone's spamming us with IRQ 9s.
		 */
		if (chip->dma[NM_PLAYBACK].running)
			snd_nm256_playback_stop(chip);
		if (chip->dma[NM_CAPTURE].running)
			snd_nm256_capture_stop(chip);
		chip->badintrcount = 0;
	}
}

/* 
 * Handle a potential interrupt for the device referred to by DEV_ID. 
 *
 * I don't like the cut-n-paste job here either between the two routines,
 * but there are sufficient differences between the two interrupt handlers
 * that parameterizing it isn't all that great either.  (Could use a macro,
 * I suppose...yucky bleah.)
 */

static void
snd_nm256_interrupt(int irq, void *dev_id, struct pt_regs *dummy)
{
	nm256_t *chip = snd_magic_cast(nm256_t, dev_id, );
	u16 status;
	u8 cbyte;

	status = snd_nm256_readw(chip, NM_INT_REG);

	/* Not ours. */
	if (status == 0) {
		snd_nm256_intr_check(chip);
		return;
	}

	chip->badintrcount = 0;

	/* Rather boring; check for individual interrupts and process them. */

	spin_lock(&chip->reg_lock);
	if (status & NM_PLAYBACK_INT) {
		status &= ~NM_PLAYBACK_INT;
		NM_ACK_INT(chip, NM_PLAYBACK_INT);
		snd_nm256_playback_update(chip);
	}

	if (status & NM_RECORD_INT) {
		status &= ~NM_RECORD_INT;
		NM_ACK_INT(chip, NM_RECORD_INT);
		snd_nm256_capture_update(chip);
	}

	if (status & NM_MISC_INT_1) {
		status &= ~NM_MISC_INT_1;
		NM_ACK_INT(chip, NM_MISC_INT_1);
		snd_printd("NM256: Got misc interrupt #1\n");
		snd_nm256_writew(chip, NM_INT_REG, 0x8000);
		cbyte = snd_nm256_readb(chip, 0x400);
		snd_nm256_writeb(chip, 0x400, cbyte | 2);
	}

	if (status & NM_MISC_INT_2) {
		status &= ~NM_MISC_INT_2;
		NM_ACK_INT(chip, NM_MISC_INT_2);
		snd_printd("NM256: Got misc interrupt #2\n");
		cbyte = snd_nm256_readb(chip, 0x400);
		snd_nm256_writeb(chip, 0x400, cbyte & ~2);
	}

	/* Unknown interrupt. */
	if (status) {
		snd_printd("NM256: Fire in the hole! Unknown status 0x%x\n",
			   status);
		/* Pray. */
		NM_ACK_INT(chip, status);
	}

	spin_unlock(&chip->reg_lock);
}

/*
 * Handle a potential interrupt for the device referred to by DEV_ID.
 * This handler is for the 256ZX, and is very similar to the non-ZX
 * routine.
 */

static void
snd_nm256_interrupt_zx(int irq, void *dev_id, struct pt_regs *dummy)
{
	nm256_t *chip = snd_magic_cast(nm256_t, dev_id, );
	u32 status;
	u8 cbyte;

	status = snd_nm256_readl(chip, NM_INT_REG);

	/* Not ours. */
	if (status == 0) {
		snd_nm256_intr_check(chip);
		return;
	}

	chip->badintrcount = 0;

	/* Rather boring; check for individual interrupts and process them. */

	spin_lock(&chip->reg_lock);
	if (status & NM2_PLAYBACK_INT) {
		status &= ~NM2_PLAYBACK_INT;
		NM2_ACK_INT(chip, NM2_PLAYBACK_INT);
		snd_nm256_playback_update(chip);
	}

	if (status & NM2_RECORD_INT) {
		status &= ~NM2_RECORD_INT;
		NM2_ACK_INT(chip, NM2_RECORD_INT);
		snd_nm256_capture_update(chip);
	}

	if (status & NM2_MISC_INT_1) {
		status &= ~NM2_MISC_INT_1;
		NM2_ACK_INT(chip, NM2_MISC_INT_1);
		snd_printd("NM256: Got misc interrupt #1\n");
		cbyte = snd_nm256_readb(chip, 0x400);
		snd_nm256_writeb(chip, 0x400, cbyte | 2);
	}

	if (status & NM2_MISC_INT_2) {
		status &= ~NM2_MISC_INT_2;
		NM2_ACK_INT(chip, NM2_MISC_INT_2);
		snd_printd("NM256: Got misc interrupt #2\n");
		cbyte = snd_nm256_readb(chip, 0x400);
		snd_nm256_writeb(chip, 0x400, cbyte & ~2);
	}

	/* Unknown interrupt. */
	if (status) {
		snd_printd("NM256: Fire in the hole! Unknown status 0x%x\n",
			   status);
		/* Pray. */
		NM2_ACK_INT(chip, status);
	}

	spin_unlock(&chip->reg_lock);
}

/*
 * AC97 interface
 */

/*
 * Waits for the mixer to become ready to be written; returns a zero value
 * if it timed out.
 */
/* spinlock held! */
static int
snd_nm256_ac97_ready(nm256_t *chip)
{
	int timeout = 10;
	u32 testaddr;
	u16 testb;

	testaddr = chip->mixer_status_offset;
	testb = chip->mixer_status_mask;

	/* 
	 * Loop around waiting for the mixer to become ready. 
	 */
	while (timeout-- > 0) {
		if ((snd_nm256_readw(chip, testaddr) & testb) == 0)
			return 1;
		udelay(100);
	}
	return 0;
}

/*
 */
static unsigned short
snd_nm256_ac97_read(void *private_data, unsigned short reg)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	unsigned long flags;
	int res;

	if (reg >= 128)
		return 0;

	spin_lock_irqsave(&chip->reg_lock, flags);
	if (! snd_nm256_ac97_ready(chip)) {
		spin_unlock_irqrestore(&chip->reg_lock, flags);
		return 0;
	}
	res = snd_nm256_readw(chip, chip->mixer_base + reg);
	spin_unlock_irqrestore(&chip->reg_lock, flags);
	/* Magic delay.  Bleah yucky.  */
	udelay(1000);
	return res;
}

/* 
 */
static void
snd_nm256_ac97_write(void *private_data,
		     unsigned short reg, unsigned short val)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, );
	unsigned long flags;
	int tries = 2;
	u32 base;

	base = chip->mixer_base;

	spin_lock_irqsave(&chip->reg_lock, flags);
	snd_nm256_ac97_ready(chip);

	/* Wait for the write to take, too. */
	while (tries-- > 0) {
		snd_nm256_writew(chip, base + reg, val);
		udelay(1000);  /* a little delay here seems better.. */
		if (snd_nm256_ac97_ready(chip))
			goto __ok;
	}
	snd_printd("nm256: ac97 codec not ready..\n");
__ok:
	spin_unlock_irqrestore(&chip->reg_lock, flags);
}

/* initialize the ac97 into a known state */
static void
snd_nm256_ac97_reset(void *private_data, ac97_t *ac97)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, );
	unsigned long flags;

	spin_lock_irqsave(&chip->reg_lock, flags);
	/* Reset the mixer.  'Tis magic!  */
	snd_nm256_writeb(chip, 0x6c0, 1);
	snd_nm256_writeb(chip, 0x6cc, 0x87);
	snd_nm256_writeb(chip, 0x6cc, 0x80);
	snd_nm256_writeb(chip, 0x6cc, 0x0);
	spin_unlock_irqrestore(&chip->reg_lock, flags);
}

/* create an ac97 mixer interface */
static int __init
snd_nm256_mixer(nm256_t *chip, int device, int pcm_count, int *pcm_devs)
{
	ac97_t ac97;
	int err;
	snd_kmixer_t *mixer;

	memset(&ac97, 0, sizeof(ac97));
	ac97.init = snd_nm256_ac97_reset;
	ac97.write = snd_nm256_ac97_write;
	ac97.read = snd_nm256_ac97_read;
	ac97.private_data = chip;
	if ((err = snd_ac97_mixer(chip->card, device, &ac97,
				  pcm_count, pcm_devs, &mixer)) < 0)
		return err;
	chip->ac97 = snd_magic_cast(ac97_t, mixer->private_data, -ENXIO);
	chip->mixer = mixer;
	sprintf(mixer->name, "%s", chip->card->abbreviation);

	return 0;
}

/* 
 * See if the signature left by the NM256 BIOS is intact; if so, we use
 * the associated address as the end of our audio buffer in the video
 * RAM.
 */

static int __init
snd_nm256_peek_for_sig(nm256_t *chip, int ignore_sig)
{
	/* The signature is located 1K below the end of video RAM.  */
	char *temp;
	/* Default buffer end is 5120 bytes below the top of RAM.  */
	unsigned long pointer_found = chip->buffer_end - 0x1400;
	u32 sig;

	temp = ioremap_nocache(chip->buffer_addr + chip->buffer_end - 0x400, 16);
	if (temp == NULL) {
		snd_printk("NM256: Unable to scan for card signature in video RAM\n");
		return -EBUSY;
	}

	sig = readl(temp);
	if ((sig & NM_SIG_MASK) == NM_SIGNATURE) {
		u32 pointer = readl(temp + 4);

		/*
		 * If it's obviously invalid, don't use it
		 */
		if (pointer == 0xffffffff ||
		    pointer < chip->buffer_size ||
		    pointer > chip->buffer_end) {
			if (! ignore_sig) {
				snd_printk("NM256: invalid signature found: 0x%x\n", pointer);
				return -ENODEV;
			}
		} else {
			pointer_found = pointer;
			snd_printd("NM256: Found card signature in video RAM: 0x%x\n", pointer);
		}
	} else if (! ignore_sig) {
		snd_printk("NM256: cannot find signature.  Aborted\n");
		return -ENODEV;
	}

	iounmap(temp);
	chip->buffer_end = pointer_found;

	return 0;
}

#ifdef CONFIG_PM

/*
 * APM event handler, so the card is properly reinitialized after a power
 * event.
 */
static int
snd_nm256_handle_pm_event(struct pm_dev *dev, pm_request_t rqst, void *data)
{
	nm256_t *chip = snd_magic_cast(nm256_t, dev->data, -ENXIO);
	nm256_dma_t *s;
	unsigned long flags;

	switch (rqst) {
	case PM_SUSPEND:
		spin_lock_irqsave(&chip->reg_lock, flags);
		if (chip->dma[NM_PLAYBACK].running)
			snd_nm256_playback_stop(chip);
		if (chip->dma[NM_CAPTURE].running)
			snd_nm256_capture_stop(chip);
		chip->coeffs_current = 0;
		spin_unlock_irqrestore(&chip->reg_lock, flags);
		break;
	case PM_RESUME:
		/* Perform a full reset on the hardware */
		snd_nm256_init_chip(chip);
		snd_ac97_resume(chip->ac97);
		/* start dma */
		spin_lock_irqsave(&chip->reg_lock, flags);
		s = &chip->dma[NM_PLAYBACK];
		if (s->running && s->substream) {
			//snd_nm256_set_format(chip, s, s->substream);
			snd_nm256_playback_start(chip, s, s->substream);
		}
		s = &chip->dma[NM_CAPTURE];
		if (s->running && s->substream) {
			//snd_nm256_set_format(chip, s, s->substream);
			snd_nm256_capture_start(chip, s, s->substream);
		}
		spin_unlock_irqrestore(&chip->reg_lock, flags);
		break;
	}
	return 0;
}
#endif

static int snd_nm256_free(nm256_t *chip)
{
	if (chip->dma[NM_PLAYBACK].running)
		snd_nm256_playback_stop(chip);
	if (chip->dma[NM_CAPTURE].running)
		snd_nm256_capture_stop(chip);

#ifdef CONFIG_PM
	if (chip->pm_dev)
		pm_unregister(chip->pm_dev);
#endif

	synchronize_irq();
	snd_unregister_interrupts(chip->card);

	if (chip->cport)
		iounmap(chip->cport);
	if (chip->buffer)
		iounmap(chip->buffer);
#if 0 // resource
	if (chip->res_cport)
		release_resource(chip->res_cport);
	if (chip->res_buffer)
		release_resource(chip->res_buffer);
#endif

	snd_magic_kfree(chip);
	return 0;
}

static int snd_nm256_dev_free(void *private_data)
{
	nm256_t *chip = snd_magic_cast(nm256_t, private_data, -ENXIO);
	return snd_nm256_free(chip);
}

static int __init
snd_nm256_create(snd_card_t *card, struct pci_dev *pci,
		 unsigned long play_bufsize,
		 unsigned long capt_bufsize,
		 int force_load, int ignore_sig,
		 u32 buffertop,
		 int usecache,
		 nm256_t **chip_ret)
{
	nm256_t *chip;
	int err, pval;
	int pcm_dev = 0;
	static snd_device_ops_t ops = {
		dev_free:	snd_nm256_dev_free,
	};

	*chip_ret = NULL;

	chip = snd_magic_kcalloc(nm256_t, 0, GFP_KERNEL);
	if (chip == NULL)
		return -ENOMEM;

	chip->card = card;
	chip->pci = pci;
	chip->use_cache = usecache;
	spin_lock_init(&chip->reg_lock);

	chip->dma[0].bufsize = play_bufsize;
	chip->dma[1].bufsize = capt_bufsize;

	/* 
	 * The NM256 has two memory ports.  The first port is nothing
	 * more than a chunk of video RAM, which is used as the I/O ring
	 * buffer.  The second port has the actual juicy stuff (like the
	 * mixer and the playback engine control registers).
	 */

	chip->buffer_addr = pci_resource_start(pci, 0);
	chip->cport_addr = pci_resource_start(pci, 1);

	/* Init the memory port info.  */
	/* remap control port (#2) */
#if 0 // resource
	chip->res_cport = request_mem_region(chip->cport_addr, NM_PORT2_SIZE,
					     card->abbreviation);
	if (chip->res_cport == NULL) {
		snd_printk("nm256: memory region 0x%lx (size 0x%lx) busy\n",
			   chip->cport_addr, NM_PORT2_SIZE);
		err = -EBUSY;
		goto __error;
	}
#endif
	chip->cport = ioremap_nocache(chip->cport_addr, NM_PORT2_SIZE);
	if (chip->cport == NULL) {
		snd_printk("nm256: unable to map control port %lx\n", chip->cport_addr);
		err = -ENOMEM;
		goto __error;
	}

	switch (card->type) {
	case SND_CARD_TYPE_NM256AV:
		/* Ok, try to see if this is a non-AC97 version of the hardware. */
		pval = snd_nm256_readw(chip, NM_MIXER_PRESENCE);
		if ((pval & NM_PRESENCE_MASK) != NM_PRESENCE_VALUE) {
			if (! force_load) {
				snd_printk("nm256: no ac97 is found!  (mixer presence value returns: %0x)\n", pval);
				snd_printk("  Force the driver to load by passing in the module parameter\n");
				snd_printk("    snd_force_ac97=1\n");
				snd_printk("  or try  sb16 or cs423x drivers instead.\n");
				err = -ENXIO;
				goto __error;
			}
		}
		chip->buffer_end = 2560 * 1024;
		chip->interrupt = snd_nm256_interrupt;
		chip->mixer_status_offset = NM_MIXER_STATUS_OFFSET;
		chip->mixer_status_mask = NM_MIXER_READY_MASK;
		break;
	default:
		/* Not sure if there is any relevant detect for the ZX or not.  */
		if (snd_nm256_readb(chip, 0xa0b) != 0)
			chip->buffer_end = 6144 * 1024;
		else
			chip->buffer_end = 4096 * 1024;

		chip->interrupt = snd_nm256_interrupt_zx;
		chip->mixer_status_offset = NM2_MIXER_STATUS_OFFSET;
		chip->mixer_status_mask = NM2_MIXER_READY_MASK;
		break;
	}
	
	chip->buffer_size = chip->dma[0].bufsize + chip->dma[1].bufsize;
	if (chip->use_cache)
		chip->buffer_size += NM_TOTAL_COEFF_COUNT * 4;
	else
		chip->buffer_size += NM_MAX_COEFFICIENT * 2;

	if (buffertop >= chip->buffer_size && buffertop < chip->buffer_end)
		chip->buffer_end = buffertop;
	else {
		/* get buffer end pointer from signature */
		if ((err = snd_nm256_peek_for_sig(chip, ignore_sig)) < 0)
			goto __error;
	}

	chip->buffer_start = chip->buffer_end - chip->buffer_size;
	chip->buffer_addr += chip->buffer_start;

	snd_printd("NM256: Mapping port 1 from 0x%x - 0x%x\n",
		   chip->buffer_start, chip->buffer_end);

#if 0 // resource
	chip->res_buffer = request_mem_region(chip->buffer_addr,
					      chip->buffer_size,
					      card->abbreviation);
	if (chip->res_buffer == NULL) {
		snd_printk("nm256: buffer 0x%lx (size 0x%lx) busy\n",
			   chip->buffer_addr, chip->buffer_size);
		err = -EBUSY;
		goto __error;
	}
#endif
	chip->buffer = ioremap_nocache(chip->buffer_addr, chip->buffer_size);
	if (chip->buffer == NULL) {
		err = -ENOMEM;
		snd_printk("nm256: unable to map ring buffer at %lx\n", chip->buffer_addr);
		goto __error;
	}

	/* set offsets */
	chip->dma[0].buf = chip->buffer_start;
	chip->dma[1].buf = chip->dma[0].buf + chip->dma[0].bufsize;
	if (chip->use_cache) {
		chip->all_coeff_buf = chip->dma[1].buf + chip->dma[1].bufsize;
	} else {
		chip->coeff_buf[0] = chip->dma[1].buf + chip->dma[1].bufsize;
		chip->coeff_buf[1] = chip->coeff_buf[0] + NM_MAX_COEFFICIENT;
	}

	/* acquire interrupt */
	if ((err = snd_register_interrupt(card, card->abbreviation, pci->irq,
					  SND_IRQ_TYPE_PCI,
					  chip->interrupt, (void*)chip,
					  NULL, &chip->irqptr)) < 0) {
		snd_printk("nm256: unable to grab IRQ %d\n", pci->irq);
		goto __error;
	}

	/* Fixed setting. */
	chip->mixer_base = NM_MIXER_OFFSET;

	chip->coeffs_current = 0;

	snd_nm256_init_chip(chip);

	if ((err = snd_nm256_pcm(chip, 0)) < 0)
		goto __error;
	
	if ((err = snd_nm256_mixer(chip, 0, 1, &pcm_dev) < 0))
		goto __error;

	// pci_set_master(pci); /* needed? */
	
#ifdef CONFIG_PM
	chip->pm_dev = pm_register(PM_PCI_DEV, PM_PCI_ID(pci),
				   snd_nm256_handle_pm_event);
	if (chip->pm_dev)
		chip->pm_dev->data = chip;
#endif

	if ((err = snd_device_new(card, SND_DEV_LOWLEVEL, chip, 0, &ops, NULL)) < 0)
		goto __error;

	*chip_ret = chip;
	return 0;

__error:
	snd_nm256_free(chip);
	return err;
}


static int __init snd_nm256_probe(struct pci_dev *pci,
				  const struct pci_device_id *id)
{
	static int dev = 0;
	snd_card_t *card;
	nm256_t *chip;
	int err;
	int buffer_top;

	if ((err = pci_enable_device(pci)) < 0)
		return err;

	for ( ; dev < SND_CARDS; dev++) {
		if (!snd_enable[dev]) {
			dev++;
			return -ENOENT;
		}
		break;
	}
	if (dev >= SND_CARDS)
		return -ENODEV;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_nm256_use_inc, snd_nm256_use_dec);
	if (card == NULL)
		return -ENOMEM;

	switch (pci->device) {
	case PCI_DEVICE_ID_NEOMAGIC_NM256AV_AUDIO:
		card->type = SND_CARD_TYPE_NM256AV;
		strcpy(card->abbreviation, "NM256AV");
		break;
	case PCI_DEVICE_ID_NEOMAGIC_NM256ZX_AUDIO:
		card->type = SND_CARD_TYPE_NM256ZX;
		strcpy(card->abbreviation, "NM256ZX");
		break;
	default:
		snd_printk("nm256: invalid device id 0x%x\n", pci->device);
		snd_card_free(card);
		return -EINVAL;
	}

	if (snd_vaio_hack[dev])
		buffer_top = 0x25a800;	/* this avoids conflicts with XFree86 server */
	else
		buffer_top = snd_buffer_top[dev];

	if ((err = snd_nm256_create(card, pci,
				    snd_dma_size(snd_playback_bufsize[dev], 4, 128),
				    snd_dma_size(snd_capture_bufsize[dev], 4, 128),
				    snd_force_ac97[dev], snd_ignore_signature[dev],
				    buffer_top,
				    snd_use_cache[dev],
				    &chip)) < 0) {
		snd_card_free(card);
		return err;
	}

	sprintf(card->shortname, "NeoMagic 256%s",
		(card->type == SND_CARD_TYPE_NM256AV ? "AV" : "ZX"));
	sprintf(card->longname, "%s at 0x%lx & 0x%lx, irq %ld",
		card->shortname,
		chip->buffer_addr, chip->cport_addr, chip->irqptr->irq);

	if ((err = snd_card_register(card)) < 0) {
		snd_card_free(card);
		return err;
	}

	PCI_SET_DRIVER_DATA(pci, card);
	dev++;
	return 0;
}

static void __exit snd_nm256_remove(struct pci_dev *pci)
{
	snd_card_free(PCI_GET_DRIVER_DATA(pci));
	PCI_SET_DRIVER_DATA(pci, NULL);
}


static struct pci_driver driver = {
	name: "NeoMagic 256",
	id_table: snd_nm256_ids,
	probe: snd_nm256_probe,
	remove: snd_nm256_remove,
};


static int __init alsa_card_nm256_init(void)
{
	int err;
	if ((err = pci_module_init(&driver)) < 0) {
#ifdef MODULE
		snd_printk("NeoMagic 256 audio soundchip not found or device busy\n");
#endif
		return err;
	}
	return 0;
}

static void __exit alsa_card_nm256_exit(void)
{
	pci_unregister_driver(&driver);
}

module_init(alsa_card_nm256_init)
module_exit(alsa_card_nm256_exit)
