/*
 *  card-als4000.c - driver for Avance Logic ALS4000 based soundcards.
 *  Copyright (C) 2000 by Bart Hartgers <bart@etpmod.phys.tue.nl>
 * 
 * NOTES
 *
 *  Since Avance does not provide any meaningful documentation, and I
 *  bought an ALS4000 based soundcard, I was forced to base this driver
 *  on reverse engineering.
 *
 *  The ALS4000 seems to be the PCI-cousin of the ALS100. It contains an
 *  ALS100-like SB DSP/mixer, an OPL3 synth, a MPU401 and a gameport 
 *  interface. These subsystems are mapped into ISA io-port space, 
 *  using the PCI-interface. In addition, the PCI-bit provides DMA and IRQ 
 *  services to the subsystems.
 * 
 * While ALS4000 is very similar to a SoundBlaster, the differences in
 * DMA and capturing require more changes to the SoundBlaster than
 * desirable, so I made this separate driver.
 * 
 * The ALS4000 can do real full duplex playback/capture.
 *
 * BUGS
 *   The box suggests there is some support for 3D sound, but I did not
 *   investigate this yet.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/info.h"
#include "../../include/als4000.h"
#include "../../include/sb.h"
#include "../../include/mpu401.h"

static int snd_als4000_dsp_command( als4000dsp_t *codec, unsigned cmd )
{
	int i;
	for (i = 10000; i; i--)
		if ((inb(SBP1(codec->sb_port, STATUS)) & 0x80) == 0) {
			outb(cmd, SBP1(codec->sb_port, COMMAND));
			return 1;
		}
	snd_printd(__FUNCTION__ ": timeout (0x%x)\n", cmd);
	return 0;
}

static int snd_als4000_dsp_getbyte( als4000dsp_t *codec )
{
	int i;
	for (i = 100000; i; i--) {
		if (inb(SBP1(codec->sb_port, DATA_AVAIL)) & 0x80) {
			return (unsigned) inb(SBP1(codec->sb_port, READ));
		}
	}
	return -1;
}

static int snd_als4000_dsp_reset(als4000dsp_t * codec)
{
	unsigned long flags;
	int i;
	spin_lock_irqsave(&codec->reg_lock, flags);
	outb(1, SBP1(codec->sb_port, RESET));
	udelay(10);
	outb(0, SBP1(codec->sb_port, RESET));
	udelay(30);
	for(i=0; i<1000 && !(inb(SBP1(codec->sb_port, DATA_AVAIL)) & 0x80); ++i);
	if (snd_als4000_dsp_getbyte(codec)!= 0xaa) {
		snd_printd(__FUNCTION__ ": failed at 0x%lx!!!\n", 
			   codec->sb_port);
		spin_unlock_irqrestore(&codec->reg_lock, flags);
		return -ENODEV;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return 0;
}

static int snd_als4000_dsp_version(als4000dsp_t * codec)
{
	unsigned long flags;
	int lo;
	int hi;
	spin_lock_irqsave(&codec->reg_lock, flags);
	snd_als4000_dsp_command(codec, SB_DSP_GET_VERSION);
	hi=snd_als4000_dsp_getbyte(codec);
	lo=snd_als4000_dsp_getbyte(codec);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return (hi<<8)|lo;
}

void snd_als4000_mixer_write( als4000dsp_t * codec,
			      unsigned char reg, unsigned char data)
{
	outb(reg, SBP1(codec->sb_port, MIXER_ADDR));
	udelay(10);
	outb(data, SBP1(codec->sb_port, MIXER_DATA));
	udelay(10);
}

unsigned char snd_als4000_mixer_read( als4000dsp_t * codec, unsigned char reg)
{
	unsigned char result;

	outb(reg, SBP1(codec->sb_port, MIXER_ADDR));
	udelay(10);
	result = inb(SBP1(codec->sb_port, MIXER_DATA));
	udelay(10);
	return result;
}

static void snd_als4000_set_rate( als4000dsp_t *codec, unsigned rate )
{
	if (!(codec->mode & ALS4000_MODE_RATELOCK)) {
		snd_als4000_dsp_command( codec, SB_DSP_SAMPLE_RATE_OUT );
		snd_als4000_dsp_command( codec, rate>>8 );
		snd_als4000_dsp_command( codec, rate );
	}
}

static void snd_als4000_set_capture_dma( als4000dsp_t *codec, void *buf, 
				  unsigned size )
{
	snd_als4000_gcr_write( codec, 0xa2, virt_to_bus( buf ) );
	snd_als4000_gcr_write( codec, 0xa3, (size-1) );
}

static void snd_als4000_set_playback_dma( als4000dsp_t *codec, void *buf,
					  unsigned size )
{
	snd_als4000_gcr_write( codec, 0x91, virt_to_bus( buf ) );
	snd_als4000_gcr_write( codec, 0x92, (size-1)| 0x180000 );
}

enum {
	ALS4000_FORMAT_U8_MONO=0,	ALS4000_FORMAT_S8_MONO=1,
	ALS4000_FORMAT_U16L_MONO=2,	ALS4000_FORMAT_S16L_MONO=3,
	ALS4000_FORMAT_U8_STEREO=4,	ALS4000_FORMAT_S8_STEREO=5,
	ALS4000_FORMAT_U16L_STEREO=6,	ALS4000_FORMAT_S16L_STEREO=7,
	
	ALS4000_FORMAT_WIDTH16=2,	ALS4000_FORMAT_STEREO=4,
};

static int snd_als4000_get_format( snd_pcm_runtime_t *runtime )
{
	int result;
	switch( runtime->format.format ) {
	 case SND_PCM_SFMT_U8: result = ALS4000_FORMAT_U8_MONO; break;
	 case SND_PCM_SFMT_S8: result = ALS4000_FORMAT_S8_MONO; break;
	 case SND_PCM_SFMT_U16_LE: result = ALS4000_FORMAT_U16L_MONO; break;
	 case SND_PCM_SFMT_S16_LE: result = ALS4000_FORMAT_S16L_MONO; break;
	 default:
		result = ALS4000_FORMAT_U8_MONO; printk("huh?\n");
	}
	if ( runtime->format.voices > 1 ) result+=ALS4000_FORMAT_STEREO;
	return result;
}

/* structure for setting up playback */
static struct {
	unsigned char dsp_cmd, dma_on, dma_off, format;
} playback_cmd_vals[]={
/* ALS4000_FORMAT_U8_MONO */
{ SB_DSP4_OUT8_AI, SB_DSP_DMA8_ON, SB_DSP_DMA8_OFF, SB_DSP4_MODE_UNS_MONO },
/* ALS4000_FORMAT_S8_MONO */	
{ SB_DSP4_OUT8_AI, SB_DSP_DMA8_ON, SB_DSP_DMA8_OFF, SB_DSP4_MODE_SIGN_MONO },
/* ALS4000_FORMAT_U16L_MONO */
{ SB_DSP4_OUT16_AI, SB_DSP_DMA16_ON, SB_DSP_DMA16_OFF, SB_DSP4_MODE_UNS_MONO },
/* ALS4000_FORMAT_S16L_MONO */
{ SB_DSP4_OUT16_AI, SB_DSP_DMA16_ON, SB_DSP_DMA16_OFF, SB_DSP4_MODE_SIGN_MONO },
/* ALS4000_FORMAT_U8_STEREO */
{ SB_DSP4_OUT8_AI, SB_DSP_DMA8_ON, SB_DSP_DMA8_OFF, SB_DSP4_MODE_UNS_STEREO },
/* ALS4000_FORMAT_S8_STEREO */	
{ SB_DSP4_OUT8_AI, SB_DSP_DMA8_ON, SB_DSP_DMA8_OFF, SB_DSP4_MODE_SIGN_STEREO },
/* ALS4000_FORMAT_U16L_STEREO */
{ SB_DSP4_OUT16_AI, SB_DSP_DMA16_ON, SB_DSP_DMA16_OFF, SB_DSP4_MODE_UNS_STEREO },
/* ALS4000_FORMAT_S16L_STEREO */
{ SB_DSP4_OUT16_AI, SB_DSP_DMA16_ON, SB_DSP_DMA16_OFF, SB_DSP4_MODE_SIGN_STEREO },
};
#define playback_cmd( codec ) (playback_cmd_vals[(codec)->playback_format])

/* structure for setting up capture */
enum { CMD_WIDTH8=0x04, CMD_SIGNED=0x10, CMD_MONO=0x80, CMD_STEREO=0xA0 };
static unsigned char capture_cmd_vals[]=
{
CMD_WIDTH8|CMD_MONO,			/* ALS4000_FORMAT_U8_MONO */
CMD_WIDTH8|CMD_SIGNED|CMD_MONO,		/* ALS4000_FORMAT_S8_MONO */	
CMD_MONO,				/* ALS4000_FORMAT_U16L_MONO */
CMD_SIGNED|CMD_MONO,			/* ALS4000_FORMAT_S16L_MONO */
CMD_WIDTH8|CMD_STEREO,			/* ALS4000_FORMAT_U8_STEREO */
CMD_WIDTH8|CMD_SIGNED|CMD_STEREO,	/* ALS4000_FORMAT_S8_STEREO */	
CMD_STEREO,				/* ALS4000_FORMAT_U16L_STEREO */
CMD_SIGNED|CMD_STEREO,			/* ALS4000_FORMAT_S16L_STEREO */
};	
#define capture_cmd( codec ) (capture_cmd_vals[(codec)->capture_format])

static int snd_als4000_capture_prepare( void *private_data, 
				        snd_pcm_subchn_t * subchn)
{
	unsigned long flags;
	unsigned long size;
	unsigned count;
	snd_pcm_runtime_t *runtime = subchn->runtime;
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);

	codec->capture_format = snd_als4000_get_format( runtime );
		
	size = snd_pcm_lib_transfer_size(subchn);
	count = snd_pcm_lib_transfer_fragment(subchn);
	
	if (codec->capture_format & ALS4000_FORMAT_WIDTH16) count >>=1;
	count--;

	spin_lock_irqsave(&codec->reg_lock, flags);
	
	snd_als4000_set_rate( codec, runtime->format.rate );
	snd_als4000_set_capture_dma( codec, runtime->dma_area->buf, size );
	
	snd_als4000_mixer_write( codec, 0xdc, count );
	snd_als4000_mixer_write( codec, 0xdd, count>>8 );
	
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	
	return 0;
}

static int snd_als4000_playback_prepare( void *private_data, 
					 snd_pcm_subchn_t *subchn)
{
	unsigned long flags;
	unsigned count;
	unsigned size;
	snd_pcm_runtime_t *runtime = subchn->runtime;
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	
	codec->playback_format = snd_als4000_get_format( runtime );
	
	size = snd_pcm_lib_transfer_size(subchn);
	count = snd_pcm_lib_transfer_fragment(subchn);
	
	if (codec->playback_format & ALS4000_FORMAT_WIDTH16) count >>=1;
	count--;
	
	spin_lock_irqsave(&codec->reg_lock, flags);
	
	snd_als4000_set_rate( codec, runtime->format.rate );
	snd_als4000_set_playback_dma( codec, runtime->dma_area->buf, size );
	
	snd_als4000_dsp_command(codec, SB_DSP_SPEAKER_ON );
	snd_als4000_dsp_command(codec, playback_cmd(codec).dsp_cmd );
	snd_als4000_dsp_command(codec, playback_cmd(codec).format );
	snd_als4000_dsp_command(codec, count );
	snd_als4000_dsp_command(codec, count>>8 );
	snd_als4000_dsp_command(codec, playback_cmd(codec).dma_off );
	
	spin_unlock_irqrestore( &codec->reg_lock, flags );
	
	return 0;
}

static int snd_als4000_capture_trigger(void *private_data,
				       snd_pcm_subchn_t * subchn, int cmd)
{
	unsigned long flags;
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	int result = 0;
	
	spin_lock_irqsave(&codec->reg_lock, flags);
	if (cmd == SND_PCM_TRIGGER_GO) {
		codec->mode |= ALS4000_MODE_RATELOCK_C;
		snd_als4000_mixer_write( codec, 0xde, capture_cmd(codec) );
	} else if (cmd == SND_PCM_TRIGGER_STOP) {
		codec->mode &= ~ALS4000_MODE_RATELOCK_C;
		snd_als4000_mixer_write( codec, 0xde, 0 );
	} else {
		result = -EINVAL;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

static int snd_als4000_playback_trigger(void *private_data,
					snd_pcm_subchn_t * subchn, int cmd)
{
	unsigned long flags;
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	int result = 0;

	spin_lock_irqsave(&codec->reg_lock, flags);
	if (cmd == SND_PCM_TRIGGER_GO) {
		codec->mode |= ALS4000_MODE_RATELOCK_P;
		snd_als4000_dsp_command( codec, SB_DSP_SPEAKER_ON );
		snd_als4000_dsp_command( codec, playback_cmd(codec).dma_on );
	} else if (cmd == SND_PCM_TRIGGER_STOP) {
		snd_als4000_dsp_command( codec, playback_cmd(codec).dma_off );
		snd_als4000_dsp_command( codec, SB_DSP_SPEAKER_OFF );
		codec->mode &= ~ALS4000_MODE_RATELOCK_P;
	} else {
		result = -EINVAL;
	}
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	return result;
}

static unsigned int snd_als4000_capture_pointer(void *private_data,
						snd_pcm_subchn_t * subchn)
{
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	unsigned result;
	unsigned long flags;
	spin_lock_irqsave(&codec->reg_lock, flags);	
	result = snd_als4000_gcr_read( codec, 0xa0 );
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	/*printk("capture ptr=%x\n",result);*/
	return result;
}

static unsigned int snd_als4000_playback_pointer(void *private_data,
						 snd_pcm_subchn_t * subchn)
{
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	unsigned result;
	unsigned long flags;
	spin_lock_irqsave(&codec->reg_lock, flags);	
	result = snd_als4000_gcr_read( codec, 0xa4 );
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	/*printk("playback ptr=%x\n",result);*/
	return result;
}

void snd_als4000_interrupt( snd_pcm_t *pcm )
{
	unsigned long flags;
	unsigned gcr_status;
	unsigned sb_status;
	als4000dsp_t *codec=snd_magic_cast(als4000dsp_t, pcm->private_data, );

	/* find out which bit of the ALS4000 produced the interrupt */
	gcr_status = inb( codec->gcr_port + 0xe );
	if (gcr_status & 0x80) /* playback */
		snd_pcm_transfer_done(codec->playback_subchn);
	if (gcr_status & 0x40) /* capturing */
		snd_pcm_transfer_done(codec->capture_subchn);
	if (gcr_status & 0x10) /* MPU401 interupt */
		if (codec->rmidi) snd_mpu401_uart_interrupt( codec->rmidi );
	/* release the gcr */
	outb( gcr_status, codec->gcr_port + 0xe );
	
	spin_lock_irqsave(&codec->reg_lock, flags);
	sb_status = snd_als4000_mixer_read(codec, SB_DSP4_IRQSTATUS);
	spin_unlock_irqrestore(&codec->reg_lock, flags);
	
	if (sb_status & SB_IRQTYPE_8BIT) 
		inb( SBP1( codec->sb_port, DATA_AVAIL ) );
	if (sb_status & SB_IRQTYPE_16BIT) 
		inb( SBP1( codec->sb_port, DATA_AVAIL_16 ) );
	if (sb_status & SB_IRQTYPE_MPUIN)
		inb( codec->gcr_port+0x30 ); /* this seems very strange...
					      * but it is really what OSS
					      * does.
					      */
	if (sb_status & 0x20 )
		inb( SBP1( codec->sb_port, RESET ) );
}

/*****************************************************************/

static snd_pcm_hardware_t snd_als4000_playback =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_S8 | SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE|
	SND_PCM_FMT_U16_LE,	/* formats */
	SND_PCM_RATE_CONTINUOUS | SND_PCM_RATE_8000_48000,
	4000,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	65536,			/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	2,			/* transfer block size */
	snd_pcm_lib_ioctl,
	snd_als4000_playback_prepare,
	snd_als4000_playback_trigger,
	snd_als4000_playback_pointer
};

static snd_pcm_hardware_t snd_als4000_capture =
{
	SND_PCM_CHNINFO_MMAP | SND_PCM_CHNINFO_STREAM |
	SND_PCM_CHNINFO_BLOCK | SND_PCM_CHNINFO_INTERLEAVE |
	SND_PCM_CHNINFO_MMAP_VALID,	/* flags */
	SND_PCM_FMT_S8 | SND_PCM_FMT_U8 | SND_PCM_FMT_S16_LE|
	SND_PCM_FMT_U16_LE,	/* formats */
	SND_PCM_RATE_CONTINUOUS | SND_PCM_RATE_8000_48000,
	4000,			/* min. rate */
	48000,			/* max. rate */
	1,			/* min. voices */
	2,			/* max. voices */
	64,			/* min. fragment size */
	65536,			/* max. fragment size */
	3,			/* fragment align */
	0,			/* FIFO size (unknown) */
	2,			/* transfer block size */
	snd_pcm_lib_ioctl,
	snd_als4000_capture_prepare,
	snd_als4000_capture_trigger,
	snd_als4000_capture_pointer
};

int snd_als4000_playback_open(void *private_data, snd_pcm_subchn_t * subchn)
{
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	int err = -EAGAIN;

	runtime->hw = (snd_pcm_hardware_t *) snd_kmalloc(sizeof(*runtime->hw), GFP_KERNEL);
	if (runtime->hw == NULL) return -ENOMEM;
	memcpy(runtime->hw, &snd_als4000_playback, sizeof(*runtime->hw));
	runtime->hw_free = _snd_kfree;
	snd_pcm_set_mixer(subchn, codec->kmixer->device, 
			  codec->mixer.me_playback);
	runtime->private_data = codec;
	
	if ((err=snd_pcm_dma_alloc(subchn, codec->dmaptr, "ALS4000 DSP"))<0)
		return err;
	codec->playback_subchn = subchn;
	return 0;
}

int snd_als4000_playback_close(void *private_data, snd_pcm_subchn_t * subchn)
{
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	codec->playback_subchn = NULL;
	snd_pcm_dma_free(subchn);
	return 0;
}

int snd_als4000_capture_open(void *private_data, snd_pcm_subchn_t * subchn)
{
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	snd_pcm_runtime_t *runtime = subchn->runtime;
	int err = -EAGAIN;

	runtime->hw = (snd_pcm_hardware_t *) snd_kmalloc(sizeof(*runtime->hw), GFP_KERNEL);
	if (runtime->hw == NULL) return -ENOMEM;
	memcpy(runtime->hw, &snd_als4000_capture, sizeof(*runtime->hw));
	runtime->hw_free = _snd_kfree;
	snd_pcm_set_mixer(subchn, codec->kmixer->device, 
			  codec->mixer.me_capture);
	runtime->private_data = codec;
	
	if ((err=snd_pcm_dma_alloc(subchn, codec->dmaptr, "ALS4000 DSP"))<0)
		return err;
	codec->capture_subchn = subchn;
	return 0;
}

int snd_als4000_capture_close(void *private_data, snd_pcm_subchn_t * subchn)
{
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, -ENXIO);
	codec->capture_subchn = NULL;
	snd_pcm_dma_free(subchn);
	return 0;
}

/******************************************************************/

void snd_als4000_set_addr( unsigned long gcr, int sb, int mpu, int opl,
			  int game )
{
	u32 confA=0;
	u32 confB=0;
	if (mpu>0) confB|=(mpu|1)<<16;
	if (sb>0) confB|=(sb|1);
	if (game>0) confA|=(game|1)<<16;
	if (opl>0) confA|=(opl|1);
	snd_als4000_gcr_write_addr( gcr, 0xa8, confA );
	snd_als4000_gcr_write_addr( gcr, 0xa9, confB );
}

void snd_als4000_disable( snd_pcm_t *pcm )
{
	als4000dsp_t *codec=snd_magic_cast(als4000dsp_t,pcm->private_data,);
	unsigned flags;
	spin_lock_irqsave( &codec->reg_lock, flags );
	snd_als4000_gcr_write( codec, 0x8c, 0 );
	spin_unlock_irqrestore( &codec->reg_lock, flags );
}

void snd_als4000_configure( snd_pcm_t *pcm )
{
	als4000dsp_t *codec=snd_magic_cast(als4000dsp_t,pcm->private_data,);
	unsigned long flags;
	unsigned tmp;
	int i;
	/* do some more configuration */
	spin_lock_irqsave( &codec->reg_lock, flags );
	tmp=snd_als4000_mixer_read( codec, 0xc0 );
	snd_als4000_mixer_write( codec, 0xc0, tmp|0x80 );
	/* always select DMA channel 0, since we do not actually use DMA */
	snd_als4000_mixer_write( codec, 0x81, 1 );
	snd_als4000_mixer_write( codec, 0xc0, tmp&0x7f );
	
	/* magic number. Enables interrupts(?) */
	snd_als4000_gcr_write( codec, 0x8c, 0x28000 );
	for( i=0x91; i<=0x96; ++i) snd_als4000_gcr_write( codec, i, 0 );
	
	snd_als4000_gcr_write( codec, 0x99, 
			       snd_als4000_gcr_read( codec, 0x99 ) );
	
	spin_unlock_irqrestore( &codec->reg_lock, flags );
}

static int snd_als4000_dsp_probe( snd_pcm_t *pcm )
{
	als4000dsp_t *codec = snd_magic_cast( als4000dsp_t, pcm->private_data, -ENXIO);
	int version;
	/*
	 *  initialization sequence
	 */
	if (snd_als4000_dsp_reset(codec) < 0) {
		snd_printdd("ALS4000: [0x%lx] dsp reset failed... 0x%x\n", 
			    codec->sb_port, inb(SBP1(codec->sb_port, READ)));
		return -ENODEV;
	}

	version = snd_als4000_dsp_version( codec );
	if (version < 0)
		return -ENODEV;

	snd_printdd("ALS4000: [0x%lx]: DSP chip found, version = %i.%i\n", 
		    codec->sb_port, version >> 8, version & 0xff);
	strcpy(codec->name, "ALS4000");
	sprintf(codec->name, "DSP v%i.%i", version >> 8, version & 0xff);
	return 0;
}

static void snd_als4000dsp_free(void *private_data)
{
	als4000dsp_t *codec = snd_magic_cast(als4000dsp_t, private_data, );
	snd_magic_kfree(codec);
}

int snd_als4000dsp_new_pcm(snd_card_t * card,
			   int device,
			   unsigned long gcr_port,
			   unsigned long sb_port,
			   snd_irq_t * irqptr,
			   snd_dma_t * dmaptr,
			   snd_rawmidi_t *rmidi,
			   snd_pcm_t ** rpcm)
{
	snd_pcm_t *pcm;
	als4000dsp_t *codec;
	int err;

	if ((err = snd_pcm_new(card, "ALS4000 DSP", device, 1, 1, &pcm)) < 0)
		return err;
	codec = snd_magic_kcalloc(als4000dsp_t, 0, GFP_KERNEL);
	if (codec == NULL)
		return -ENOMEM;
	spin_lock_init(&codec->reg_lock);
	codec->pcm = pcm;
	codec->card = pcm->card;
	codec->mode = 0;
	codec->playback_format = 0;
	codec->capture_format = 0;
	codec->gcr_port = gcr_port;
	codec->sb_port = sb_port;
	codec->irqptr = irqptr;
	codec->dmaptr = dmaptr;
	codec->rmidi = rmidi;
	pcm->private_data = codec;
	pcm->private_free = snd_als4000dsp_free;
	strcpy(pcm->name, "ALS4000");
	
	if (snd_als4000_dsp_probe( pcm ) < 0) {
		snd_device_free(card, pcm);
		return -ENODEV;
	}
	
	pcm->info_flags = SND_PCM_INFO_PLAYBACK | SND_PCM_INFO_CAPTURE |
	  		  SND_PCM_INFO_DUPLEX | SND_PCM_INFO_DUPLEX_RATE;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].open = snd_als4000_playback_open;
	pcm->chn[SND_PCM_CHANNEL_PLAYBACK].close = snd_als4000_playback_close;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].private_data = codec;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].open = snd_als4000_capture_open;
	pcm->chn[SND_PCM_CHANNEL_CAPTURE].close = snd_als4000_capture_close;

	*rpcm = pcm;
	return 0;
}

/*****************************************************************/

EXPORT_SYMBOL(snd_als4000_interrupt);
EXPORT_SYMBOL(snd_als4000_set_addr);
EXPORT_SYMBOL(snd_als4000_disable);
EXPORT_SYMBOL(snd_als4000_configure);
EXPORT_SYMBOL(snd_als4000dsp_new_mixer);
EXPORT_SYMBOL(snd_als4000dsp_new_pcm);
					
static int __init alsa_als4000_init(void)
{
	return 0;
}

static void __exit alsa_als4000_exit(void)
{
}

module_init(alsa_als4000_init)
module_exit(alsa_als4000_exit)
	      
