/*
    card-als4000.c - driver for Avance Logic ALS4000 based soundcards.
    Copyright (C) 2000 by Bart Hartgers <bart@etpmod.phys.tue.nl>

    Framework borrowed from Massimo Piccioni's card-als100.c.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/initval.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"
#include "../include/sb.h"
#include "../include/als4000.h"

#define CARD_NAME "ALS4000"
#define DRIVER_NAME "ALS4000"

#ifndef PCI_VENDOR_ID_AVANCE_ALS4000
#define PCI_VENDOR_ID_AVANCE_ALS4000 0x4000
#endif

MODULE_DESCRIPTION("\
Driver: Avance Logic ALS4000\n\
Card: Avance Logic ALS4000\n\
PCI: 0x4005=0x4000\n\
");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;		/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;		/* ID for this card */
int snd_sb_port[SND_CARDS] = {0x220,0x240};
int snd_mpu_port[SND_CARDS] = {0x330};
int snd_fm_port[SND_CARDS] = {0x388};
int snd_game_port[SND_CARDS] = {0x200};
int snd_dac_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 32 };
int snd_adc_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 32 };

#ifdef MODULE_PARM
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for " CARD_NAME " soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for " CARD_NAME " soundcard.");
MODULE_PARM(snd_sb_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_sb_port, "SB port # for " DRIVER_NAME " driver. [PORT12]");
MODULE_PARM(snd_game_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_game_port, "Game port # for " DRIVER_NAME " driver. [PORT12]");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for " DRIVER_NAME " driver. [PORT12]");
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_fm_port, "FM port # for " DRIVER_NAME " driver. [PORT12]");
MODULE_PARM(snd_dac_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dac_frame_size, "DAC frame size for " CARD_NAME " soundcard. [range=1-32]");
MODULE_PARM(snd_adc_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_adc_frame_size, "ADC frame size for " CARD_NAME " soundcard. [range=1-32]");
#endif	/* MODULE_PARM */

static struct snd_card_als4k {
   	struct pci_dev *pci;
	unsigned long gcr_port;
	int dev;
	snd_irq_t *irqptr;
	snd_dma_t *dmaptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;
} *snd_card_als4k_cards[SND_CARDS] = SND_DEFAULT_PTR;

typedef struct snd_card_als4k snd_card_als4k_t;

#define snd_als4k_printk(args...)	snd_printk(__FILE__": " ##args)

static void snd_card_als4k_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_als4k_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static int __init snd_card_als4k_detect( snd_card_t * card,
					 snd_card_als4k_t * acard,
					 unsigned short vendor,
					 unsigned short device )
{
	u16 word;

	if ((acard->pci = pci_find_device(vendor, device, 
					  acard->pci)) == NULL)
     		return -ENODEV;
	
	acard->gcr_port = pci_resource_start(acard->pci,0);
	/* the OSS driver only registers a range of 16, but it accesses
	 * gcr_base+0x30 on something MPU401 related....
	 */
   	if (snd_register_ioport(card, acard->gcr_port, 0x31, 
				DRIVER_NAME, NULL) < 0)
     		goto __nodev;
	
	pci_read_config_word( acard->pci, PCI_COMMAND, &word );
	pci_write_config_word( acard->pci, PCI_COMMAND, 
			       word | PCI_COMMAND_IO );
	pci_set_master( acard->pci );
		
   	return 0;
   __nodev:
   	snd_unregister_ioports( card );
   	return -ENODEV;
}

static void snd_card_als4k_interrupt( int irq, void *dev_id,
				      struct pt_regs *regs)
{
	snd_card_als4k_t *acard = (snd_card_als4k_t *) dev_id;
	if (acard) snd_als4000_interrupt( acard->pcm );
}

static int __init snd_card_als4k_resources( int dev, 
					    snd_card_als4k_t *acard,
					    snd_card_t *card )
{
	int err;
	if ((err=snd_register_interrupt(card, DRIVER_NAME, acard->pci->irq,
					SND_IRQ_TYPE_PCI, 
					snd_card_als4k_interrupt,
					acard, NULL, &acard->irqptr)) < 0)
		return err;
	
	if ((err=snd_register_dma_channel(card, DRIVER_NAME " - dac frame", 0,
					  SND_DMA_TYPE_PCI_16MB,
					  snd_dac_frame_size[dev], NULL,
					  &acard->dmaptr)) < 0)
		return err;
	acard->dmaptr->multi = 1;	/* enable multi-alloc */
       	acard->dmaptr->addressbits = 20;/* only 16MB, cheap bastards!!! */
	
	if (snd_register_ioport(card, snd_mpu_port[dev], 2,
				DRIVER_NAME " - mpu-401", NULL) < 0)
		snd_mpu_port[dev] = -1;

	if ((err=snd_register_ioport(card, snd_sb_port[dev], 16,
				     DRIVER_NAME " - dsp", NULL)) < 0)
		return err;

	if (snd_register_ioport(card, snd_fm_port[dev], 4,
				DRIVER_NAME " - opl3", NULL) < 0)
		snd_fm_port[dev] = -1;
	
	snd_als4000_set_addr( acard->gcr_port, snd_sb_port[dev],
			      snd_mpu_port[dev], snd_fm_port[dev],
			      snd_game_port[dev] );
	return 0;	
}

static void snd_card_als4k_deactivate( snd_card_als4k_t * acard )
{
	/* remove the sound stuff, but be careful about the gameport... */
	snd_als4000_disable( acard->pcm );	
	snd_als4000_set_addr( acard->gcr_port, 0, 0, 0, 
			      snd_game_port[acard->dev] );
}

static int __init snd_card_als4k_probe(int dev, snd_card_als4k_t *acard)
{
	int error;
	als4000dsp_t *codec;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;

	if (!(card = snd_card_new(snd_index[dev], snd_id[dev],
			snd_card_als4k_use_inc, snd_card_als4k_use_dec)))
		return -ENOMEM;
	card->type = SND_CARD_TYPE_ALS4000;
	card->private_data = acard;

   	if ((error = snd_card_als4k_detect( card, acard, 
					    PCI_VENDOR_ID_AVANCE,
					    PCI_VENDOR_ID_AVANCE_ALS4000 ))!=0)
     		goto __on_error;
	
	if ((error = snd_card_als4k_resources(dev, acard, card)))
		goto __on_error;

	rmidi = NULL;
	if (snd_mpu_port[dev] > 0) {
		if (snd_mpu401_uart_new(card, 0, MPU401_HW_ALS100,
				snd_mpu_port[dev], 0,
				&rmidi) < 0)
			snd_als4k_printk("no MPU-401device at 0x%x ?\n",
					   snd_mpu_port[dev]);
	}
	    
	if (snd_als4000dsp_new_pcm(card, 0,
				   acard->gcr_port,
				   snd_sb_port[dev],
				   acard->irqptr, 
				   acard->dmaptr,
				   rmidi,
				   &pcm) < 0)
		goto __on_error;
	
	snd_als4000_configure( pcm );

	codec = snd_magic_cast(als4000dsp_t, pcm->private_data, -ENXIO);

	if (snd_als4000dsp_new_mixer(codec, 0, pcm, &mixer) < 0)
		goto __on_error;

	synth = NULL;
	if (snd_fm_port[dev] > 0) {
		if (snd_opl3_new(card, 0,
				snd_fm_port[dev], snd_fm_port[dev] + 2,
				OPL3_HW_AUTO, 0, &synth) < 0)
			snd_als4k_printk("no OPL device at 0x%x-0x%x ?\n",
				snd_fm_port[dev], snd_fm_port[dev] + 2);
	}

	strcpy(card->abbreviation, "ALS4000");
	strcpy(card->shortname, "Avance Logic ALS4000");
	sprintf(card->longname, "%s soundcard, %s at 0x%lx and 0x%lx, irq %li",
		card->shortname, codec->name, codec->gcr_port,
		codec->sb_port, acard->irqptr->irq );
	if (!snd_card_register(card)) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		acard->rmidi = rmidi;
		acard->synth = synth;
		return 0;
	}

      __on_error:
	snd_card_free(card);
	return error ? error : -ENXIO;
}

#ifdef MODULE

static void __exit snd_card_als4k_free(int dev)
{
	snd_card_als4k_t *acard;

	if ((acard = snd_card_als4k_cards[dev])) {
		snd_card_als4k_cards[dev] = NULL;
		snd_card_unregister(acard->card);
		snd_card_als4k_deactivate(acard);
		snd_kfree(acard);
	}
}

#endif

static int __init alsa_card_als4k_init(void)
{
	int cards, dev;
	snd_card_als4k_t *acard,*prev=NULL;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		acard = (snd_card_als4k_t *) snd_kcalloc(
			sizeof(snd_card_als4k_t), GFP_KERNEL);
		if (acard == NULL)
			break;
		acard->pci=prev ? prev->pci : NULL;
		if (snd_card_als4k_probe(dev, acard) < 0) {
			snd_kfree(acard);
			continue;
		}
		snd_card_als4k_cards[cards++] = prev = acard;
	}

#ifdef MODULE
	if (!cards)
		snd_als4k_printk("no ALS4000 based soundcards found\n");
#endif
	return cards ? 0 : -ENODEV;
}

static void __exit alsa_card_als4k_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_card_als4k_free(dev);
}

module_init(alsa_card_als4k_init)
module_exit(alsa_card_als4k_exit)
