
package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Resources}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableResources.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableResources implements Resources {
  private final @Nullable Long nanoCpus;
  private final @Nullable Long memoryBytes;

  private ImmutableResources(
      @Nullable Long nanoCpus,
      @Nullable Long memoryBytes) {
    this.nanoCpus = nanoCpus;
    this.memoryBytes = memoryBytes;
  }

  /**
   * @return The value of the {@code nanoCpus} attribute
   */
  @JsonProperty("NanoCPUs")
  @Override
  public @Nullable Long nanoCpus() {
    return nanoCpus;
  }

  /**
   * @return The value of the {@code memoryBytes} attribute
   */
  @JsonProperty("MemoryBytes")
  @Override
  public @Nullable Long memoryBytes() {
    return memoryBytes;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Resources#nanoCpus() nanoCpus} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for nanoCpus (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableResources withNanoCpus(@Nullable Long value) {
    if (Objects.equals(this.nanoCpus, value)) return this;
    return new ImmutableResources(value, this.memoryBytes);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Resources#memoryBytes() memoryBytes} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for memoryBytes (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableResources withMemoryBytes(@Nullable Long value) {
    if (Objects.equals(this.memoryBytes, value)) return this;
    return new ImmutableResources(this.nanoCpus, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableResources} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableResources
        && equalsByValue((ImmutableResources) another);
  }

  private boolean equalsByValue(ImmutableResources another) {
    return Objects.equals(nanoCpus, another.nanoCpus)
        && Objects.equals(memoryBytes, another.memoryBytes);
  }

  /**
   * Computes a hash code from attributes: {@code nanoCpus}, {@code memoryBytes}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(nanoCpus);
    h += (h << 5) + Objects.hashCode(memoryBytes);
    return h;
  }

  /**
   * Prints the immutable value {@code Resources} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Resources{"
        + "nanoCpus=" + nanoCpus
        + ", memoryBytes=" + memoryBytes
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Resources} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Resources instance
   */
  public static ImmutableResources copyOf(Resources instance) {
    if (instance instanceof ImmutableResources) {
      return (ImmutableResources) instance;
    }
    return ImmutableResources.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableResources ImmutableResources}.
   * <pre>
   * ImmutableResources.builder()
   *    .nanoCpus(Long | null) // nullable {@link Resources#nanoCpus() nanoCpus}
   *    .memoryBytes(Long | null) // nullable {@link Resources#memoryBytes() memoryBytes}
   *    .build();
   * </pre>
   * @return A new ImmutableResources builder
   */
  public static ImmutableResources.Builder builder() {
    return new ImmutableResources.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableResources ImmutableResources}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements Resources.Builder {
    private Long nanoCpus;
    private Long memoryBytes;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Resources} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Resources instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable Long nanoCpusValue = instance.nanoCpus();
      if (nanoCpusValue != null) {
        nanoCpus(nanoCpusValue);
      }
      @Nullable Long memoryBytesValue = instance.memoryBytes();
      if (memoryBytesValue != null) {
        memoryBytes(memoryBytesValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Resources#nanoCpus() nanoCpus} attribute.
     * @param nanoCpus The value for nanoCpus (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("NanoCPUs")
    public final Builder nanoCpus(@Nullable Long nanoCpus) {
      this.nanoCpus = nanoCpus;
      return this;
    }

    /**
     * Initializes the value for the {@link Resources#memoryBytes() memoryBytes} attribute.
     * @param memoryBytes The value for memoryBytes (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("MemoryBytes")
    public final Builder memoryBytes(@Nullable Long memoryBytes) {
      this.memoryBytes = memoryBytes;
      return this;
    }

    /**
     * Builds a new {@link ImmutableResources ImmutableResources}.
     * @return An immutable instance of Resources
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableResources build() {
      return new ImmutableResources(nanoCpus, memoryBytes);
    }
  }
}
